/*****************************************************************************/
/* File: crunch.h                                                            */
/* Author: David Chatenay                                                    */
/* Last Modified: Tue Feb 04 1997                                            */
/*                                                                           */
/* Definition of macros and exported functions for libcrunch                 */
/*                                                                           */
/* Version: 1.0       Revision: 9                                            */
/*****************************************************************************/
#ifndef LIB_CRUNCH_H
#define LIB_CRUNCH_H



/*****************************************************************************/
/**************************** VERSION AND NAMING *****************************/
#define VERSION_MAJOR      1
#define VERSION_MINOR      0
#define VERSION_REVISION   9
#define FULL_VERSION "1.0.9"

#define LIB_NAME    "libcrunch"
#define FULL_NAME   "Crunch library"
#define AUTHOR_NAME "David Chatenay"
#define DATE        "Mar 14 1997"



/*****************************************************************************/
/************************** BASIC TYPE DEFINITIONS ***************************/
typedef unsigned char  byte;   /* This one for better comprehension */
typedef unsigned short word;   /* This one for history     */
typedef unsigned long  dword;  /* This one for history too */
typedef struct {
    word code_len;
    /* Codes can be long up to 256 bits... */
    /* ie up to 16 bytes... So this length */
    byte code[16];
} Code; /* A code of variable length (in bits) */



/*****************************************************************************/
/******************************* I/O FUNCTIONS *******************************/
  /*** Open for reading only */
extern int OpenR(char *name, int perm);
  /*** Open for Writing/Append/Truncate */
extern int OpenW(char *name, int perm);
  /*** Read data */
extern int Read(int fd, void *p, int s);
  /*** Read data absolute (no less) */
extern int ReadAbs(int fd, void *p, int s);
  /*** Write data (no less) */
extern int Write(int fd, void *p, int s);
  /*** Seek a position */
extern int Seek(int fd, off_t offset, int whence);
  /*** Tell the current position */
extern off_t Tell(int fd);
  /*** Secure malloc (the one that exits on errors!!!) */
extern void *Malloc(size_t t);



/*****************************************************************************/
/*************************** HASH & CRC FUNCTIONS ****************************/
  /* HASH */
extern dword RawHash(byte*, long);
#define HASH_SIZE  101  /* The maximum hash value + 1 */
#define HASH_VALUE 31   /* The hash multiplier */
extern word  SimpleHash(byte*, long);
  /* CRC */
extern word  CRC16(byte*, long);
extern dword CRC32(byte*, long);



/*****************************************************************************/
/**************************** METHOD IDENTIFYING *****************************/
  /*** Method IDs */
#define METHOD_UNKNOWN  0  /* Typical error return for IdentifyMethod */
#define METHOD_HUFFMAN  1  /* Huffman method            */
#define METHOD_AD_HUFF  2  /* Adaptative Huffman method */
#define METHOD_LZW      3  /* Lzw method                */
#define METHOD_LZSS     4  /* Lzss method               */
  /*** Error values */
#define CHECK_ERROR   -1  /* File I/O error    */
#define CHECK_INVALID -2  /* Invalid signature */
#define CHECK_CORRUPT -3  /* Corrupted file    */
#define CHECK_CRC     -4  /* CRC check error   */
#define CHECK_VERSION -5  /* Version mismatch  */
  /* Return method index, or error values defined */
extern int CheckFile(int fd);
  /* Return method index */
extern byte IdentifyMethod(int fd);



/*****************************************************************************/
/********************** FILE COMPRESION & DECOMPRESSION **********************/
  /*** HUFFMAN */
extern int HuffmanEncode(int in, int out, float *cr);
extern int HuffmanDecode(int in, int out);
  /*** ADAPT. HUFFMAN */
extern int AdaptativeHuffmanEncode(int in, int out, float *cr);
extern int AdaptativeHuffmanDecode(int in, int out);
  /*** LZW */
extern int LzwEncode(int in, int out, float *cr);
extern int LzwDecode(int in, int out);
  /*** LZSS */
extern int LzssEncode(int in, int out, float *cr);
extern int LzssDecode(int in, int out);




/*****************************************************************************/
/************************* ADVANCED FUNCTIONS ********************************/
/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/******************* LZW BLOCK COMPRESSION & DECOMPRESSION *******************/
  /*** DEFINES */
#define DICT_SIZE  4096   /* Dictionnary size */
    /* NB: DICT_SIZE can be risen up to 32768 only. */

  /*** STRUCTURES */
typedef struct lzwcell {
    byte last;           /* Last char. */
    word index;          /* Index in table */
    struct lzwcell *prev;   /* Prev char. */
} LzwCell;
typedef struct _LzwIndex {
    dword hash;         /* Hash value */
    LzwCell  *pnt;         /* Cell pointed */
    struct _LzwIndex *next; /* Next for this value. */
} LzwIndex;

  /*** LZW CONTEXT */
typedef struct {
    LzwCell Dictionnary[DICT_SIZE]; /* The dictionnary */
    LzwIndex HashTable[256];        /* The index table to hash values */
    byte  CodeSize;   /* Current code size */
    word  MaxEntries; /* 2^CodeSize */
    word  CellNumber; /* Number of used cells */
    Code  EofCode;    /* Special code for end of file */
    Code  ResetCode;  /* Special code for reset table */
    Code  SizeUpCode; /* Special code for increment code size */
} LzwContext;

  /*** LZW CONTEXT INIT */
extern void LzwInitContext(LzwContext *h);

  /*** LZW CONTEXT RESET */
extern void LzwReset(LzwContext *h);

  /*** LZW HASH TABLE DISPOSE */
extern void LzwFreeHashTable(LzwContext *h);

  /*** LZW BLOCK COMPRESS */
extern int LzwBlockEncode(LzwContext *h,
			  byte *in, dword size_in,
			  byte *out, dword *size_out);

  /*** LZW BLOCK DECOMPRESS */
extern int LzwBlockDecode(LzwContext *h,
			  byte *in, dword size_in,
			  byte *out, dword *size_out);



/*****************************************************************************/
/****************** LZSS BLOCK COMPRESSION & DECOMPRESSION *******************/
  /*** DEFINES */
    /* WARNING! You shouldn't touch these */
    /* without knowing what you do...     */
#define CBUFFER_SIZE    4096  /* Circular buffer size  */
#define CBUFFER_LOG     12    /* and its log_2 */
#define MAX_LENGTH      32    /* Maximum string length */
#define MAX_LOG         5     /* and its log_2 */
#define MIN_LENGTH      2     /* Minimum string length */

  /*** STRUCTURES */
typedef struct _LzssIndex {
    word position;
    struct _LzssIndex *next;
} LzssIndex;
typedef struct Access {
    LzssIndex *first;
    LzssIndex *last;
    word number;
} LzssAccess;

  /*** LZSS CONTEXT */
typedef struct {
    byte   CircularBuffer[CBUFFER_SIZE];  /* The circular buffer */
    LzssIndex  Dictionnary[CBUFFER_SIZE]; /* The linked array of positions */
    LzssAccess Access[256];               /* The array of access speeders */
    word   BufferLength;                  /* The length of the buffer */
    word   Begin;                         /* The beginning of the buffer */
} LzssContext;

  /*** LZSS CONTEXT INIT */
extern void LzssInitContext(LzssContext *h);

  /*** LZSS BLOCK COMPRESS */
extern int LzssBlockEncode(LzssContext *h,
			   byte *in, dword size_in,
			   byte *out, dword *size_out);

  /*** LZSS BLOCK DECOMPRESS */
extern int LzssBlockDecode(LzssContext *h,
			   byte *in, dword size_in,
			   byte *out, dword *size_out);



#endif
