/*****************************************************************************/
/* File: common.c                                                            */
/* Author: David Chatenay                                                    */
/* Last Modified: Wed Jan 15 1997                                            */
/*                                                                           */
/* Implementation of simple but useful functions (mainly front-ends with     */
/* local error-checking and action).                                         */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include "crunch.h"
#include "common.h"


/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/*                               MISCELLANOUS                                */
/*****************************************************************************/
/* GLOBAL EXPORTED VARIABLES */
byte Mask[9] = {0x00, 0x01, 0x03, 0x07, 0x0F, 0x1F, 0x3F, 0x7F, 0xFF};
byte UMask[9]= {0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01, 0x00};
byte XMask[9]= {0xFE, 0xFD, 0xFB, 0xF7, 0xEF, 0xDF, 0xBF, 0x7F, 0x00};
byte RMask[9]= {0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x00};



/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/*                            SECURE FILE I/Os                               */
/*****************************************************************************/
int OpenR(char *name, int perm)
{
    int fd;

    if ((fd = open(name, O_RDONLY, perm)) == -1) {
	ERROR("Cannot open file for reading!");
	perror(name);
	return -1;
    }
    return fd;
}


int OpenW(char *name, int perm)
{
    int fd;

    if ((fd = open(name, O_WRONLY | O_CREAT | O_TRUNC, perm)) == -1) {
	ERROR("Cannot open file for writing!");
	perror(name);
	return -1;
    }
    return fd;
}


int Read(int fd, void *p, int s)
{
    int l;

    if ((l = read(fd, p, (size_t)s)) == -1) {
	ERROR("Cannot read data!");
        perror("Reason");
    }
    DEBUG1("[Read %d]\n", l);
    return l;
}


int ReadAbs(int fd, void *p, int s)
{
    int l;

    l = read(fd, p, (size_t)s);
    if (l != s) {
        ERROR("Cannot read all data!");
	perror("Reason");
        DEBUG2("  [read %d/%d]\n", l, s);
	return -1;
    }
    return l;
}


int Write(int fd, void *p, int s)
{
    int l;

    l = write(fd, p, (size_t)s);
    DEBUG1("  [Wrote %d]\n", l);
    if (l != s) {
        ERROR("Cannot write data!");
	perror("Reason");
        DEBUG2("  [wrote %d/%d]\n", l, s);
	return -1;
    }
    return 0;
}


int Seek(int fd, off_t offset, int whence)
{
    if (lseek(fd, offset, whence) == -1) {
	ERROR("Cannot access this position!");
        perror("Reason");
	return -1;
    }
    return 0;
}


off_t Tell(int fd)
{
    off_t where;

    if ((where = lseek(fd, 0, SEEK_CUR)) == -1) {
	ERROR("Cannot tell the position!");
	perror("Reason");
	return -1;
    }
    return where;
}


/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/*                         MEMORY ALLOCATION                                 */
/*****************************************************************************/
void *Malloc(size_t s)
{
    void *p;

    p = malloc(s);
    if (!p) {
	ERROR("Cannot allocate memory!");
	perror("Reason");
	exit(-1);
    }
    return p;
}
