// Copyright (C) 1996 Keith Whitwell.
// This file may only be copied under the terms of the GNU Library General
// Public License - see the file COPYING in the lib3d distribution.

#include <Lib3d/internals/ZBuffer.H>
#include <string.h>

const uint ZBuffer::start = 0xff000000;
const float ZBuffer::scale = float(0x00ffffff);

ZBuffer::ZBuffer( uint width, uint height ) 
    : buffer(0), 
      generationMask( start ),
      width( width ), 
      height( height ), 
      active( width * height ),
      size( width * height ),
      xmin( 0 ),
      xmax( width ),
      ymin( 0 ),
      ymax( height )
{
    buffer = new uint[ size ];
    cleanBuffer();
}

ZBuffer::ZBuffer( const Device &dev ) 
    : buffer(0), 
      generationMask( start ),
      width( dev.getWidth() ), 
      height( dev.getHeight() ), 
      active( width * height ),
      size( width * height ),
      xmin( 0 ),
      xmax( width ),
      ymin( 0 ),
      ymax( height )
{
    buffer = new uint[ size ];
    cleanBuffer();
}

ZBuffer::~ZBuffer()
{
    delete [] buffer;
}


void 
ZBuffer::resize( uint newWidth, uint newHeight )
{
    uint oldSize = size;
    uint oldActive = active;
    active = newWidth * newHeight;
    width = newWidth;
    height = newHeight;

    if ( active > size ) {
	while (size < active) size = uint(size * 1.5);
    } else {
	while (size > active * 2.25) size = uint(size * .6666);
    }

    if (size != oldSize) {
	delete [] buffer;
	buffer = new uint[ size ];
    }

    if (size != oldSize || active > oldActive) {
	ymin = xmin = 0;
	xmax = width;
	ymax = height;
	cleanBuffer();
    }
}

void
ZBuffer::cleanBuffer()
{
    if_debug {
	debug() << "Cleaning x:"<<xmin<<"-"<<xmax
	        << " y:"<<ymin<<"-"<<ymax
		<< endlog;
    }

    if (xmin < xmax && ymin < ymax) {
	uint *fb = buffer + xmin + (ymin * width);
	uint wid = (xmax - xmin) * sizeof(*buffer);
	uint i = (ymax - ymin);

#ifdef __i386	
	wid = wid >> 2;
	asm("cld");
	do {
	    asm ("rep\n\t"
		 "stosl"
		 : /* no output registers */
		 : "c" (wid), "a" (-1), "D" (fb)
		 : "%ecx", "%edi" );
	    fb += width;
	} while (--i);
#else
	do {
	    memset( fb, 0xffffffff, wid );
	    fb += width;
	} while (--i);
#endif

    }

    generationMask = start;
    xmin = width;
    xmax = 0;
    ymin = height;
    ymax = 0;
}

void
ZBuffer::clear()
{
    if (generationMask == 0) {
	cleanBuffer();
	generationMask = ZBuffer::start;
    } else {
	generationMask -= generationIncrement;
    }
}

ostream &
ZBuffer::print( ostream &out ) const
{
    return Debuggable::print(out)
	<< "\n\twidth:"<<getWidth()
	<< "\n\theight:"<<getHeight()
	<< "\n\tgenerationIncrement: " << (void *)generationIncrement
	<< "\n\t    generationStart:" << (void *)start;
}












