// Copyright (C) 1996 Keith Whitwell.
// This file may only be copied under the terms of the GNU Library General
// Public License - see the file COPYING in the lib3d distribution.

#include <Lib3d/Matrix.H>
#include <Lib3d/Vector.H>
#include <math.h>


void
DeviceVector::mul( const Matrix34 &m, const Vector3 &p )
{
    v[0] = int((m.v[0][0] * p.v[0]) +
	       (m.v[0][1] * p.v[1]) +
	       (m.v[0][2] * p.v[2]) + 
	       (m.v[0][3]));
    v[1] = int((m.v[1][0] * p.v[0]) +
	       (m.v[1][1] * p.v[1]) +
	       (m.v[1][2] * p.v[2]) + 
	       (m.v[1][3]));
    v[2] = int((m.v[2][0] * p.v[0]) +
	       (m.v[2][1] * p.v[1]) +
	       (m.v[2][2] * p.v[2]) + 
	       (m.v[2][3]));
}

// assumes that p.v[W] == 1
void
DeviceVector::mul( const Matrix34 &m, const Vector4 &p )
{
    v[0] = int((m.v[0][0] * p.v[0]) +
	       (m.v[0][1] * p.v[1]) +
	       (m.v[0][2] * p.v[2]) + 
	       (m.v[0][3]));
    v[1] = int((m.v[1][0] * p.v[0]) +
	       (m.v[1][1] * p.v[1]) +
	       (m.v[1][2] * p.v[2]) + 
	       (m.v[1][3]));
    v[2] = int((m.v[2][0] * p.v[0]) +
	       (m.v[2][1] * p.v[1]) +
	       (m.v[2][2] * p.v[2]) + 
	       (m.v[2][3]));
}

void
DeviceVector::mul( const Matrix34 &m )
{
    float x = v[0];
    float y = v[1];
    float z = v[2];

    v[0] = int((m.v[0][0] * x) +
	       (m.v[0][1] * y) +
	       (m.v[0][2] * z) + 
	       (m.v[0][3]));
    v[1] = int((m.v[1][0] * x) +
	       (m.v[1][1] * y) +
	       (m.v[1][2] * z) + 
	       (m.v[1][3]));
    v[2] = int((m.v[2][0] * x) +
	       (m.v[2][1] * y) +
	       (m.v[2][2] * z) + 
	       (m.v[2][3]));
}



float 
dot( const Vector4 &v1, const Vector4 &v2 )
{
    return (v1.v[0] * v2.v[0] +
	    v1.v[1] * v2.v[1] + 
	    v1.v[2] * v2.v[2] + 
	    v1.v[3] * v2.v[3]);
}

void 
Vector3::cross( const Vector3 &v1, const Vector3 &v2 )
{
    v[0] = v1.v[1]*v2.v[2] - v1.v[2]*v2.v[1];
    v[1] = v1.v[2]*v2.v[0] - v1.v[0]*v2.v[2];
    v[2] = v1.v[0]*v2.v[1] - v1.v[1]*v2.v[0];
}


void
Vector4::mul( const Matrix34 &m, const Vector4 &p )
{
    v[0] = ((m.v[0][0] * p.v[0]) +
	    (m.v[0][1] * p.v[1]) +
	    (m.v[0][2] * p.v[2]) + 
	    (m.v[0][3] * p.v[3]));
    v[1] = ((m.v[1][0] * p.v[0]) +
	    (m.v[1][1] * p.v[1]) +
	    (m.v[1][2] * p.v[2]) + 
	    (m.v[1][3] * p.v[3]));
    v[2] = ((m.v[2][0] * p.v[0]) +
	    (m.v[2][1] * p.v[1]) +
	    (m.v[2][2] * p.v[2]) + 
	    (m.v[2][3] * p.v[3]));
    v[3] =  p.v[3];
}

void
Vector4::mul( const Matrix4 &m, const Vector4 &p )
{
    v[0] = ((m.v[0][0] * p.v[0]) +
	    (m.v[0][1] * p.v[1]) +
	    (m.v[0][2] * p.v[2]) + 
	    (m.v[0][3] * p.v[3]));
    v[1] = ((m.v[1][0] * p.v[0]) +
	    (m.v[1][1] * p.v[1]) +
	    (m.v[1][2] * p.v[2]) + 
	    (m.v[1][3] * p.v[3]));
    v[2] = ((m.v[2][0] * p.v[0]) +
	    (m.v[2][1] * p.v[1]) +
	    (m.v[2][2] * p.v[2]) + 
	    (m.v[2][3] * p.v[3]));
    v[3] = ((m.v[3][0] * p.v[0]) +
	    (m.v[3][1] * p.v[1]) +
	    (m.v[3][2] * p.v[2]) + 
	    (m.v[3][3] * p.v[3]));
}

void
Vector4::mul( const Matrix4 &m, const Vector3 &p )
{
    v[0] = ((m.v[0][0] * p.v[0]) +
	    (m.v[0][1] * p.v[1]) +
	    (m.v[0][2] * p.v[2]) + 
	    (m.v[0][3]));
    v[1] = ((m.v[1][0] * p.v[0]) +
	    (m.v[1][1] * p.v[1]) +
	    (m.v[1][2] * p.v[2]) + 
	    (m.v[1][3]));
    v[2] = ((m.v[2][0] * p.v[0]) +
	    (m.v[2][1] * p.v[1]) +
	    (m.v[2][2] * p.v[2]) + 
	    (m.v[2][3]));
    v[3] = ((m.v[3][0] * p.v[0]) +
	    (m.v[3][1] * p.v[1]) +
	    (m.v[3][2] * p.v[2]) + 
	    (m.v[3][3]));
}

void 
Vector4::homogenize()
{
    float w = 1/v[3]; 
    v[0] *= w;
    v[1] *= w;
    v[2] *= w;
    v[3] = 1;
}

Vector4::Vector4( float a, float b, float c, float d )
{
    v[0] = a;
    v[1] = b;
    v[2] = c;
    v[3] = d;
}

Vector3::Vector3( float a, float b, float c )
{
    v[0] = a;
    v[1] = b;
    v[2] = c;
}

Vector3::Vector3( float a[3] )
{
    v[0] = a[0];
    v[1] = a[1];
    v[2] = a[2];
}


void
Vector3::sub( const Vector4 &a, const Vector4 &b )
{
    v[0] = a.v[0] - b.v[0];
    v[1] = a.v[1] - b.v[1];
    v[2] = a.v[2] - b.v[2];
}

void
Vector4::sub( const Vector4 &a, const Vector4 &b )
{
    v[0] = a.v[0] - b.v[0];
    v[1] = a.v[1] - b.v[1];
    v[2] = a.v[2] - b.v[2];
    v[3] = a.v[3] - b.v[3];    
}


uint
Vector4::computeOutcodes() const
{
    uint outcodes = 0;

    if (v[Z] > v[W])  outcodes |= 0x01;
    if (v[Z] < 0)     outcodes |= 0x02;  
    if (v[Y] > v[W])  outcodes |= 0x04;
    if (v[Y] < -v[W]) outcodes |= 0x08;
    if (v[X] > v[W])  outcodes |= 0x10;
    if (v[X] < -v[W]) outcodes |= 0x20;

    return outcodes;
}


    
