// Copyright (C) 1996 Keith Whitwell.
// This file may only be copied under the terms of the GNU Library General
// Public License - see the file COPYING in the lib3d distribution.

// PostScript level 2 device driver.
// This file is contributed by Michael Ohme <mohme@informatik.uni-leipzig.de>

#include <Lib3d/Devices/PsDevice.H>
#include <string.h>
#include <stdio.h>

#ifdef HEX_OUTPUT                 // use only ASCII characters?
#define READSTRING "readhexstring"
#else  HEX_OUTPUT                 // use non ASCII characters:
#define READSTRING "readstring"	  // needs only half the file size
                                  // had some funny output with this, my
				  // knowledge of PostScript is apparently 
                                  // to small			 
#endif HEX_OUTPUT

/***********************************************************************
//
************************************************************************/



static PsDevice PsAdvertisement( Device::exemplar );
// static PsDevice PsAdvertisement2( Device::exemplar );

PsDevice::PsDevice(Exemplar e)
    : FileDevice(e) 
{ 
    FileDevice::registerChildClass(this); 
}

PsDevice::PsDevice( uint width, uint height, uint min_depth )
    : FileDevice( width, height, min_depth ), 
      out_dat(0)
{
    if (isBad()) return;	// Abort if parent constructor failed.

    if (min_depth>24) {   
	cout << getName() << ": Invalid resolution.\n" << endl;
        setBad();
        return;
    }
    
    depth = 24; 
    pixelSize = 4;

    setSize(width, height);
    rowSize = width * pixelSize;

    frameBuf = (uchar *) malloc(rowSize*height);
    if (frameBuf == 0) {
        cout << getName() << ": Unable to allocate bitmap.\n" << endl;
        setBad();
        return;
    }

    memset(frameBuf,0,rowSize*height);

    debug() << "Using:"
	    << " depth=" << depth
            << " pixelSize=" << pixelSize
	    << endlog;
}

// Never called.
//
uint
PsDevice::allocateColour(uint r, uint g, uint b) 
{
    return (r<<16)+(g<<8)+(b);
}


PsDevice::~PsDevice()
{ 
    if (frameBuf ) { free(frameBuf); frameBuf = 0; }
    if (out_dat)   { fclose(out_dat); out_dat=0; }
}


bool 
PsDevice::openFile(const char *f)
{ 
    delete [] filename;
    filename = 0;

    if (out_dat) fclose(out_dat);
    if ((out_dat=fopen(f,"a"))==0) { 
	fprintf(stderr,"Failed to open %s\n",filename); 
	return(0); 
    }

    filename = new char[strlen(f)];
    strcpy(filename, f);

    debug() << "Append PostScript output to " << filename << endlog;

    nr_pict=0;
    return(1);
}

void
PsDevice::closeFile()
{
    if (out_dat) {
	fclose(out_dat);
	out_dat = 0;
    }
    delete [] filename;
    filename = 0;
}

FileDevice *
PsDevice::cloneFileDevice( uint width, uint height, uint depth )
{  
    return new PsDevice( width, height, depth ); 
}

inline void 
PsDevice::write_char_hex(uchar c)
{
#ifdef HEX_OUTPUT
    fprintf(out_dat, c>0xf ? "%X" : "0%X" ,c);
#else  HEX_OUTPUT
    fprintf(out_dat,"%c",c);
#endif HEX_OUTPUT
}

void 
PsDevice::swapBuffers()
{ 
    if (!filename && !openFile("lib3d.eps")) {
	return;
    }

    fprintf(out_dat,
	    "%%!PS-Adobe-2.0 EPSF-2.0\n%%%%Creator Lib3D\n"
	    "%%%%BoundingBox: 0 0 %d %d\n"              // EPSF-Header
	    "%%picture %d:\n%d %d 8 [1 0 0 -1 0 %d]\n"  // columns, lines, color deepness, transformation matrix:
	    "{currentfile %d string "READSTRING" pop} false 3 colorimage\n",
	    getWidth(),
	    getHeight(),
	    ++nr_pict,
	    getWidth(),
	    getHeight(),
	    getHeight(),
	    getWidth());
    
    uchar *Buffer = frameBuf;
    uchar *endBuffer=Buffer+getRowWidth()*getHeight();

    while (Buffer<endBuffer)               // output all lines
    {
#ifdef HEX_OUTPUT
	uchar *endLine=Buffer+getRowWidth();
	for (; Buffer<endLine ;) fputc('\n',out_dat);
#endif  HEX_OUTPUT
	{ 
	    write_char_hex(*Buffer++);         //        Red
	    write_char_hex(*Buffer++);         // output Green color information
	    write_char_hex(*Buffer++);         //        Blue
	    Buffer++;                          // unused
	}
    }
    fputs("showpage\n",out_dat);
}









