// Copyright (C) 1996 Keith Whitwell.
// This file may only be copied under the terms of the GNU Library General
// Public License - see the file COPYING in the lib3d distribution.

#include <Lib3d/Camera.H>
#include <Lib3d/Viewport.H>

Camera::Camera(Node &pt)
    : Node( pt ),
      recalc( true ),
      yon(100),
      hither(1),
      field(45),
      aspect(320/200),
      lastViewport(0)
{
    world = getWorld();
    worldToCamera.setIdentity();
    cameraToCvv.setIdentity();
    worldToCvv.setIdentity();

}

Camera::~Camera()
{
}

// Inefficient when the camera is moving.
// Fix once lights are working.

void
Camera::prepareToRender( Viewport &viewport )
{
    bool dirty = isTarnished();

    if (dirty) {
	Matrix34 cameraToWorld;
	calculateObjectToWorld(cameraToWorld);
	worldToCamera.invert(cameraToWorld);

	if_debug {
	    debug() << "Recalculated camera transforms:" << endlog;
	    debug() << "cameraToWorld: " << cameraToWorld << endlog;
	    debug() << "World to camera: " << worldToCamera << endlog;
	}
    }

    recalc = recalc || (&viewport != lastViewport);
    lastViewport = &viewport;

    if (recalc) {
	D = cameraToCvv.setToCvv(hither, yon, field, aspect);

	Matrix4 t1;
	t1.setCvvPerspective(D);
	cvvToDevice.mul( t1, viewport.getDeviceTransform() ); 

	// cvvToDevice and D are static members of Node. (kludge)
    }

    if (dirty || recalc) {
	worldToCvv.mul(cameraToCvv, worldToCamera);

	recalc = false;

	if_debug {
	    debug() << "Recalculated CVV transforms:" << endlog;
	    debug() << "Camera to cvv: " << cameraToCvv << endlog;
	    debug() << "World to cvv: " << worldToCvv << endlog;
	    debug() << "cvv to Device: " << cvvToDevice << endlog;
	}
	world->recalculateHierarchy( true, worldToCvv );
    } else {
	world->recalculateHierarchy( false, worldToCvv );
    }	
}

void
Camera::notifyResize()
{
    recalc = true;
}
    
void 
Camera::setParameters( float h, float y, float f, float a )
{
    yon = y;
    hither = h;
    aspect = a;
    field = f;
    recalc = true;
}

















