// Copyright (C) 1996 Keith Whitwell.
// This file may only be copied under the terms of the GNU Library General
// Public License - see the file COPYING in the lib3d distribution.

#ifndef VectorClasses
#define VectorClasses

#include <Lib3d/internals/types.H>
#include <iostream.h>

#if 0
#define X 0
#define Y 1
#define Z 2
#define W 3

#define R 0
#define G 1
#define B 2
#else
enum { X, Y, Z, W };
enum { R, G, B };
#endif

// Vectors are row vectors, eg  | 0 1 2 |
struct Vector4;
struct Matrix34;
struct Matrix4;

struct Vector3
{
    float v[3];

    Vector3();
    Vector3( float, float, float );
    Vector3( float[3] );

    void assign( float, float, float );
    void assign( const Vector3 & );

    void mul( const Matrix34 &m, const Vector3 &p );
    void mul( const Matrix34 & );

    // Just the upper-left 3x3 matrix.
    void mul_T( const Matrix34 &m, const Vector3 &p ); 
    void mul_T( const Matrix34 & );

    void add( const Vector3 & );
    void add( const Vector3 &, const Vector3 & );
    void sub( const Vector3 &, const Vector3 & );
    void sub( const Vector4 &, const Vector4 & ); // assumed homogenous

    void clamp( float );

    void scale( float );
    void scale( const Vector3 &, float );
    void scale( const Vector3 & );

    friend inline float dot( const Vector3 &, const Vector3 & );
    void cross( const Vector3 &v1, const Vector3 &v2 );

    void normalize();
    float magnitude();

    uint computeOutcodes( float D ) const; // Against truncated pyramid cvv

    bool operator==(const Vector3 &) const;
    bool operator!=(const Vector3 &) const;

    friend ostream& operator<<( ostream &out, const Vector3 &v ) {
	return out << "(" << v.v[X] <<", "<< v.v[Y] <<", "<< v.v[Z] <<")";
    }
};

struct Vector4
{
    float v[4];

    Vector4();
    Vector4( float, float, float, float = 1);

    void mul( const Matrix34 &, const Vector4 & );
    void mul( const Matrix4 &, const Vector4 & );
    void mul( const Matrix4 &, const Vector3 & );

    void sub( const Vector4 &, const Vector4 & );

    void homogenize();

    void assign( float i, float j, float k, float l );

    Vector4 &operator=(const float tmp[4]);
    Vector4 &operator=(const Vector4 &t);

    uint computeOutcodes() const; // Against hemicube cvv.

    friend float dot( const Vector4 &, const Vector4 & );

    friend ostream& operator<<( ostream &out, const Vector4 &v ) {
	return out << "(" << v.v[X] <<", "
	                  << v.v[Y] <<", "
			  << v.v[Z] <<", "
			  << v.v[W] <<")";
    }
};

struct DeviceVector
{
    int v[3];

    DeviceVector() {}
    DeviceVector( int x, int y, int z );	

    void mul( const Matrix34 &, const Vector3 & );
    void mul( const Matrix34 &, const Vector4 & ); // v4 assumed homogenous
    void mul( const Matrix34 & );

    float project( const Matrix4 &, const Vector3 & );

    friend ostream& operator<<( ostream &out, const DeviceVector &v ) {
	return out << "(" << v.v[X] <<", "<< v.v[Y] <<", "<< v.v[Z] <<")";
    }
};

#include <Lib3d/internals/VectorInline.H>

#endif




