/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998, 1999  Sam Lantinga
    
    Improvements to FastGL - Marian Krivos - 27.01.2000

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    slouken@devolution.com
*/

#include <stdio.h>
#include <sys/time.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#ifdef __QNX__
#include <sys/select>
#endif

#include "fastgl.h"

/* The first ticks value of the application */
static struct timeval start;
/* Data to handle a single periodic alarm */
static TimerCall alarm_callback = NULL;
static unsigned int alarm_interval;

void FGTimer::StartTicks(void)
{
	/* Set first ticks value */
	gettimeofday(&start, NULL);
}

/* Get the number of milliseconds since the SDL library initialization.
 * Note that this value wraps if the program runs for more than ~49 days.
 */ 
unsigned int FGTimer::GetTicks(void)
{
	struct timeval now;
	unsigned int ticks;

	gettimeofday(&now, NULL);
	ticks=(now.tv_sec-start.tv_sec)*1000+(now.tv_usec-start.tv_usec)/1000;
	return ticks;
}

/* Wait a specified number of milliseconds before returning */
void FGTimer::Delay(unsigned int ms)
{
	int was_error;
#ifndef linux	/* Non-Linux implementations need to calculate time left */
	unsigned int then, now, elapsed;
#endif
	struct timeval tv;

	/* Set the timeout interval - Linux only needs to do this once */
#ifdef linux
	tv.tv_sec = ms/1000;
	tv.tv_usec = (ms%1000)*1000;
#else
	then = GetTicks();
#endif
	do {
		errno = 0;
#ifndef linux
		/* Calculate the time interval left (in case of interrupt) */
		now = GetTicks();
		elapsed = (now-then);
		then = now;
		if ( elapsed >= ms ) {
			break;
		}
		ms -= elapsed;
		tv.tv_sec = ms/1000;
		tv.tv_usec = (ms%1000)*1000;
#endif
		was_error = select(0, NULL, NULL, NULL, &tv);
	} while ( was_error && (errno == EINTR) );
}

void HandleAlarm(int sig)
{
	unsigned int ms;

	if ( alarm_callback ) {
		ms = (*alarm_callback)(alarm_interval);
		if ( ms != alarm_interval ) {
			FGTimer::SetTimer(ms, alarm_callback);
		}
	}
}

/* Set a callback to run after the specified number of milliseconds has
 * elapsed. The callback function is passed the current timer interval
 * and returns the next timer interval.  If the returned value is the 
 * same as the one passed in, the periodic alarm continues, otherwise a
 * new alarm is scheduled.
 *
 * To cancel a currently running timer, call SDL_SetTimer(0, NULL);
 *
 * The timer callback function may run in a different thread than your
 * main code, and so shouldn't call any functions from within itself.
 *
 * The maximum resolution of this timer is 10 ms, which means that if
 * you request a 16 ms timer, your callback will run approximately 20 ms
 * later on an unloaded system.  If you wanted to set a flag signaling
 * a frame update at 30 frames per second (every 33 ms), you might set a 
 * timer for 30 ms:
 *   SDL_SetTimer((33/10)*10, flag_update);
 *
 * If you use this function, you need to pass SDL_INIT_TIMER to SDL_Init().
 *
 * Under UNIX, you should not use raise or use SIGALRM and this function
 * in the same program, as it is implemented using setitimer().  You also
 * should not use this function in multi-threaded applications as signals
 * to multi-threaded apps have undefined behavior in some implementations.
 */
int FGTimer::SetTimer(unsigned int ms, TimerCall callback)
{
	struct itimerval timer;

	alarm_callback = NULL;
	alarm_interval = ms;
	if ( ms ) {
		/* Set a new alarm */
		alarm_callback = callback;
	}
	timer.it_value.tv_sec = (ms/1000);
	timer.it_value.tv_usec = (ms%1000)*1000;
	timer.it_interval.tv_sec = (ms/1000);
	timer.it_interval.tv_usec = (ms%1000)*1000;
	setitimer(ITIMER_REAL, &timer, NULL);
	return(0);
}

int FGTimer::TimerInit(void)
{
	struct sigaction action;

	StartTicks();
	/* Set the alarm handler (Linux specific) */
	memset(&action, 0, sizeof(action));
	action.sa_handler = &HandleAlarm;
#ifdef __linux__
	action.sa_flags = SA_RESTART;
#else
	action.sa_flags = 0;
#endif
	sigemptyset(&action.sa_mask);
	sigaction(SIGALRM, &action, NULL);
	return(0);
}

void FGTimer::TimerQuit(void)
{
	SetTimer(0, NULL);
}

#ifdef DEBUG
unsigned int prvy(unsigned int)
{
	printf("timer = %d\n", FGTimer::GetTicks());
	return 100;
}

main()
{
	FGTimer::TimerInit();
	FGTimer a(100, &prvy);
	FGTimer::Delay(2000);
}
#endif
