/*
    OpenGUI - Drawing & Windowing library

    Copyright (C) 1996,2000  Marian Krivos

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    nezmar@internet.alcatel.sk

    file2asm.c - file compiler

    Modified by: Alain Rouge
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <sys/stat.h>
#ifdef __DJGPP__
#include <io.h>
#endif
#include <unistd.h>
#include <ctype.h>

/* ********************************************************************** */

	struct RGB3
	{
		char bBlue;				 /* Blue component of the color definition */
		char bGreen;			 /* Green component of the color definition*/
		char bRed;				 /* Red component of the color definition  */
	};

	struct RGB4
	{
		char bBlue;				 /* Blue component of the color definition */
		char bGreen;			 /* Green component of the color definition*/
		char bRed;				 /* Red component of the color definition  */
		char fcOptions;			 /* Reserved, must be zero                 */
	};

	struct bitmap_file_header
	{
		unsigned short	usType;
		unsigned short	cbSizeLow, cbSizeHigh;
		short			reserve1, reserve2;
	};

	struct bitmap_info
	{
		unsigned short	imageOffset;
		short			reserve;
	};

	struct bitmap_info_header1
	{
	  unsigned long	 size;
	  unsigned short cx;			   /* Bit-map width in pels                  */
	  unsigned short cy;			   /* Bit-map height in pels                 */
	  unsigned short cPlanes;		   /* Number of bit planes                   */
	  unsigned short cBitCount;		   /* Number of bits per pel within a plane  */
	  unsigned long	 ulCompression;	   /* Compression scheme used to store the bitmap */
	  unsigned long	 cbImage;		   /* Length of bit-map storage data in chars*/
	  unsigned long	 cxResolution;	   /* x resolution of target device          */
	  unsigned long	 cyResolution;	   /* y resolution of target device          */
	  unsigned long	 cclrUsed;		   /* Number of color indices used           */
	  unsigned long	 cclrImportant;	   /* Number of important color indices      */
	};

	struct bitmap_info_header2
	{
	  unsigned long	 size;
	  unsigned long	 cx;			   /* Bit-map width in pels                  */
	  unsigned long	 cy;			   /* Bit-map height in pels                 */
	  unsigned short cPlanes;		   /* Number of bit planes                   */
	  unsigned short cBitCount;		   /* Number of bits per pel within a plane  */
	  unsigned long	 ulCompression;	   /* Compression scheme used to store the bitmap */
	  unsigned long	 cbImage;		   /* Length of bit-map storage data in chars*/
	  unsigned long	 cxResolution;	   /* x resolution of target device          */
	  unsigned long	 cyResolution;	   /* y resolution of target device          */
	  unsigned long	 cclrUsed;		   /* Number of color indices used           */
	  unsigned long	 cclrImportant;	   /* Number of important color indices      */
	};

struct in_image_bmp
{
	short	usType;
	short	cx, cy;
	short	bits;
} header;

unsigned long palette[256];

/* read_RLE8_compressed_image:
 *  For reading the 8 bit RLE compressed BMP image format.
 */
static void read_RLE8_compressed_image(FILE *f, unsigned char *bmp, int w, int h)
{
   unsigned char count, val, val0;
   int j, pos, line;
   int eolflag, eopicflag;

   eopicflag = 0;
   line = h - 1;

	while (eopicflag == 0) {
    pos = 0;                               /* x position in bitmap */
    eolflag = 0;                           /* end of line flag */

    while ((eolflag == 0) && (eopicflag == 0)) {
	count = (char)getc(f);
	val = (char)getc(f);

	if (count > 0) {                    /* repeat pixel count times */
	    for (j=0;j<count;j++) {
	       bmp[line*w+pos] = val;
	       pos++;
	    }
	}
	else {
	    switch (val) {
	       case 0:                       /* end of line flag */
			  eolflag=1;
			  break;
	       case 1:                       /* end of picture flag */
			  eopicflag=1;
			  break;
	       case 2:                       /* displace picture */
			  count = (char)getc(f);
			  val = (char)getc(f);
			  pos += count;
			  line += val;
			  break;
	       default:                      /* read in absolute mode */
			  for (j=0; j<val; j++) {
			     val0 = (char)getc(f);
		    	 bmp[line*w+pos] = val0;
			     pos++;
			  }

			  if (j%2 == 1) val0 = (char)getc(f);    /* align on word boundary */
			  break;
	    }
	 }
	 if (pos > w) eolflag=1;
   }

   line--;
   if (line < 0) eopicflag = 1;
   }
}

static FILE *process_bmp(FILE *f)
{
	int ys, one, compressed=0;
	unsigned int c, bits=0;
	int rest;
	int sz, d, i, w, h;
 	char *bmp2;
	unsigned char *src, *bp, *pom, *bitmap;
	bitmap_file_header	bmfh;
	bitmap_info			bmi;
	union {
		bitmap_info_header1	bmih1;
		bitmap_info_header2	bmih2;
	} bmih;
	union {
		RGB3 rgb3[256];
		RGB4 rgb4[256];
	} rgb;

	memset(palette, 255, sizeof(palette));
	fread(&bmfh, sizeof(bmfh), 1, f);	// read first 10 bytes

	// return if no BMP
	if (bmfh.usType != (unsigned short) ('M'*256 + 'B'))
	{
		fclose(f);
		printf("No BITMAP file!\a\n");
		return 0;
	}

	fread(&bmi, sizeof(bmi), 1, f);		// read next 4 bytes

	memset(&bmih, 0, sizeof(bmih));
	fread(&bmih, 4, 1, f);		// read size of bitmap info header

	fread(&bmih.bmih1.cx, bmih.bmih1.size - 4, 1, f);	// if 0x0c then OS/2 bitmap

	if (bmih.bmih1.size == 0x0C)
		one = 3;
	else
		one = 4;

	c = (bmi.imageOffset - (bmih.bmih1.size + 14)) / one;	// compute numbers of used colors
	if (c == 0)
	{
		w = bmih.bmih2.cx;
		h = bmih.bmih2.cy;
		bits = bmih.bmih2.cBitCount;
		if (bmih.bmih2.cPlanes != 1 || bits != 24)
		{
			fclose(f);
			printf("No supported BPP file!\a\n");
			return 0;
		}
	}
	else if (one == 3)
	{
		if ((compressed = bmih.bmih1.ulCompression)>1)
		{
			printf("No this compressed BMP supported\a\n");
			return 0;
		}
		fread(&rgb, sizeof(RGB3), c, f);
		w = bmih.bmih1.cx;
		h = bmih.bmih1.cy;
		bits = bmih.bmih1.cBitCount;
		if (bmih.bmih1.cPlanes != 1 || bits != 8)
		{
			fclose(f);
			printf("No supported BPP file!\a\n");
			return 0;
		}
		for (i = 0; i < 256; i++)
		{
			palette[i] = *((long *) ((char *) &rgb + i * one)) & 0xffffff;
		}
	}
	else if (one == 4)
	{
		if ((compressed = bmih.bmih2.ulCompression)>1)
		{
			printf("Unknown compress format!\a\n");
			return 0;
		}
		fread(&rgb, sizeof(RGB4), c, f);
		w = bmih.bmih2.cx;
		h = bmih.bmih2.cy;
		bits = bmih.bmih2.cBitCount;
		if (bmih.bmih2.cPlanes != 1 || bits != 8)
		{
			fclose(f);
			printf("No supported BPP file!\a\n");
			return 0;
		}
		memmove(palette, (const void *)&rgb, sizeof(rgb));
	}

	sz = w*h*bits/8;
	assert(bitmap = (unsigned char *) calloc(sz, 1));
	switch(bits)
	{
		case 8:
			if (!compressed) for (ys = h-1; ys>=0; ys--)
			{
				fread((unsigned char *)bitmap + ys * w, w, 1, f);
				rest = (w*bits/8)%4;
				while (rest&3)
				{
					rest++;
					fgetc(f);
				}
			}
			else read_RLE8_compressed_image(f,bitmap,w,h);
#ifdef TRUE_COLORS
			bp = (FGPixel *)calloc(areasize(w, h), 1);
		 	pom=bp;
			src=(unsigned char *)bitmap;
			sz = w*h;

			for (i = 0; i < sz; i++)
			{
				d = *((long *) ((char *) &rgb + *src * one));
				*pom = (d &= 0xffffff);
//printf("%05d - %02x %06x\n", i, *src, *pom);
				pom++;
				src++;
			}
			free(bitmap);
			bitmap = bp;
#endif
			break;
#ifdef TRUE_COLORS
		case 24:
			for (ys = h-1; ys>=0; ys--)
			{
				bmp2 = (char *)(bitmap + ys * w);
				for(int xs=0; xs<w; xs++)
				{
					fread(bmp2, 3, 1, f);
					bmp2+=4;
				}
			}
			rest = (w*bits/8)%4;
			while (rest&3)
			{
				rest++;
				fgetc(f);
			}
			break;
#endif
		default:
			printf("Bad Bitmap format!");
			fclose(f);
			free(bitmap);
			return 0;
	}
	
	FILE *out = fopen("__asd__", "wb");
	header.usType = bmfh.usType;
	header.cx     = w;
	header.cy     = h;
	header.bits   = bits;
	fwrite(&header, sizeof(header), 1, out);
	if (bits==8) fwrite(palette, sizeof(palette), 1, out);
	fwrite(bitmap,  sz, 1, out);
	fclose(out);
	fclose(f);

	return fopen("__asd__", "rb");
}

int stricmp(const char  *s1, const char *s2)
{
  while	(tolower(*s1) == tolower(*s2))
  {
	if (*s1	== 0)
		return 0;
	s1++;
	s2++;
  }
  return (int)tolower(*s1) - (int)tolower(*s2);
}

#ifdef __linux__
long	filelength(char *name)
{
    struct stat status;
    if (stat(name, &status)==-1) return 0;
    return status.st_size;
}
#endif

int main(int argc, char **argv)
{
	FILE *in,*out;
    long  sz, data[8];
    char s[50], *p=argv[1], name[50];
    int i=0, j, wasm=0 ;


   if (argc < 2)
   {
   printf("\nusage: 'file2asm filename[.ext]' [-wasm] to produce filename.asm\n");
   return 1;
   }

   if (argc == 3) if (!strcmp(argv[1],"-wasm") || !strcmp(argv[2],"-wasm")) wasm = 1;

   for(;;)
	{
      if ( (*p == '.') || (*p == '\0') ) break;
      s[i] = *p++;
      if (!(s[i]=='.' || s[i]=='\\')) i++;
	}

   s[i] = 0;
   strcpy(name,s);
	
   if (wasm)
       strcpy(s+i,".asm");
      else
       strcpy(s+i,".nsm");

   in  = fopen(argv[1], "rb");
   out = fopen(s      , "w");

   if (in==NULL || out==NULL) return 3;

#ifndef __linux__
   sz = filelength(fileno(in));
#else   
   sz = filelength(argv[1]);
#endif   
   if (sz<1) return -1; // error

   printf("process file %s %ld bytes ... ", argv[1], sz);
   fflush(stdout);

	if (strstr(argv[1], ".bmp") || strstr(argv[1], ".bmp"))
	{
		assert(in = process_bmp(in));
#ifndef __linux__
	   sz = filelength(fileno(in));
#else   
	   sz = filelength("__asd__");
#endif   
	}

   if (wasm)
    {
   fprintf(out,
   "; generated by FILE2ASM ver 2.0\n;\n.MODEL small,C\n\tassume nothing\n\tglobal _%s : near\nDATA\tsegment word public 'DATA'\n_%s:\n\tDD\t%ld\n"
   ,name,name,sz);
    }
    else
    {
   fprintf(out,
   "; generated by FILE2ASM ver 2.0\n;\nsection .data\n\tglobal _%s\n_%s:\n\tDD\t%ld\n"
   ,name,name,sz);
    }

   j = sz % (4*8);
   i = sz / (4*8);

   for(;i;i--)
   {
   fread(data,8,4,in);
   fprintf(out,"\tDD\t0x%lx, 0x%lx, 0x%lx, 0x%lx, 0x%lx, 0x%lx, 0x%lx, 0x%lx\n",
            data[0],data[1],data[2],data[3],data[4],data[5],data[6],data[7]);
   }

   for(;j;j--)
   {
   fread(data,1,1,in);
   fprintf(out,"\tDB\t0x%x\n",(unsigned char)data[0]);
   }

   if (wasm)
    fprintf(out,"\nDATA\tENDS\n\n\tend\n");
   else
    fprintf(out,"\n\tend\n");

   printf("%s done.\n", s);
   fflush(stdout);

   fclose(in);
   fclose(out);
   remove("__asd__");
   return 0;
}


