/*
    FGLMESA:  An example of integrating FGL with Mesa for OpenGL programs
    Copyright (C) 1997  Sam Lantinga, 1999 Marian Krivos

    OpenGUI - Drawing & Windowing library

    Copyright (C) 1996,2000  Marian Krivos

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
    
    Marian Krivos
    nezmar@internet.alcatel.sk
*/

#include "stdio.h"
#include "stdlib.h"
#include "config.h"
#include "base.h"
#include "_fastgl.h"
#include "fglmesa.h"


/* This function initializes the display with the given size and format,
   and then returns a GL context corresponding to the screen.  This is
   provided as a convenience only. :-)
 */
FGLMesaContext FGLMesaCreateDisplay(int width, int height, int bpp)
{
	FGPixel *screen;
	FGLMesaContext ctx;
	int mode;
	
	/* Set a display mode */
	if ( bpp != get_colordepth() )
	{
		printf("Not supported bpp!\n");
		return 0;
	}
	switch(width)
	{
		default:
	    case 320:
			mode = G320x200;
			break;
	    case 640:
			mode = G640x480;
			break;
	    case 800:
			mode = G800x600;
			break;
	    case 1024:
			mode = G1024x768;
			break;
	    case 1280:
			mode = G1280x1024;
			break;
	    case 1600:
			mode = G1600x1200;
			break;
	}

	if (!graph_set_mode(mode)) return 0;

	if (bpp != 8 && !lfb)
	{
		printf("Linear framebuffer is required for this color depth - bad hardware!\n");
		exit(-1);
	}

	screen = (FGPixel *) calloc(areasize(width,height), 1);

	if (!screen) return 0;
	
	/* Create a GL context */
	ctx = FGLMesaCreateContext(screen);
	if ( ctx == 0 ) {
		return(0);
	}

	ctx->w = width;
	ctx->h = height;
	/* Make it current */
	if ( ! FGLMesaMakeCurrent(ctx) ) {
		FGLMesaDestroyContext(ctx);
		return(0);
	}

	/* That's it!  We're ready to render */
	return(ctx);
}

/* This function takes an FGL surface and returns a GL context.
   The surface must either be in 8-bit palette or 32-bit RGB format.
 */
FGLMesaContext FGLMesaCreateContext(FGPixel *buffer)
{
	struct FGLmesa_context *ctx;
	GLenum format = (enum GLenum)0;

	/* Figure out the Mesa visual format */
	switch (get_colordepth())
	{
	case 8:
		format = OSMESA_COLOR_INDEX;
		break;
	case 32:
//		format = OSMESA_ARGB;
		format = OSMESA_RGBA;
//		format = OSMESA_BGRA;
		break;
	default:
		return 0;
	}

	/* Allocate the context and return it */
	ctx = (struct FGLmesa_context *)malloc(sizeof(*ctx));
	if ( ctx == NULL ) {
		printf("FGLmesa: Out of memory in context allocation!\n");
		return(0);
	}
	ctx->surface = buffer;
	ctx->context = OSMesaCreateContext(format, NULL);
	if ( ctx->context == 0 ) {
		printf("OSMesaCreateContext() failed!\n");
		free(ctx);
		return(0);
	}
	return(ctx);
}

/* This function makes the given context the current GL context.
 */
GLboolean FGLMesaMakeCurrent(FGLMesaContext ctx)
{
	/* Make the context current */
	if ( ! OSMesaMakeCurrent(ctx->context, ctx->surface,
			GL_UNSIGNED_BYTE, ctx->w, ctx->h) ) {
		printf("OSMesaMakeCurrent() failed!\n");
		return(GL_FALSE);
	}

	/* Tell OSMesa about the surface pitch */
	/* Note: Mesa 3.0 has a bug where OSMESA_ROW_LENGTH signifies both
	         length in bytes and length in words in different places.
	         For now, leave this commented and hope it doesn't break,
	         or Mesa will crash.
	OSMesaPixelStore(OSMESA_ROW_LENGTH, ctx->surface->pitch);
	*/

	/* Tell OSMesa that we use top-down bitmaps */
	OSMesaPixelStore(OSMESA_Y_UP, 0);

	/* Yay!  We're done! */
	return(GL_TRUE);
}

/* This function destroys a previously created GL context.
 */
void FGLMesaDestroyContext(FGLMesaContext ctx)
{
	OSMesaDestroyContext(ctx->context);
	free(ctx);
	free(ctx->surface);
}
