/*
   Copyright (C) 1996,99  Marian Krivos
   nezmar@internet.alcatel.sk

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   bitmap.cc - bitmap engine

 */

#include <setjmp.h>
#include "config.h"
#include "fastgl.h"
#include "_fastgl.h"

#include "gif.h"

#define 	TRANSP_KEY			0xFE00FE
#define     MAX_LWZ_BITS        12
#define		INTERLACE			0x40
#define		LOCALCOLORMAP		0x80

#define BitSet(byte, bit)	(((byte) & (bit)) == (bit))
#define	ReadOK(file,buffer,len) (fread(buffer, len, 1, file) != 0)
#define MKINT(a,b)		(((b)<<8)|(a))
#define NEW(x)			((x *)malloc(sizeof(x)))

/***************************************************************************
*
*  ERROR()    --  should not return
*  INFO_MSG() --  info message, can be ignored
*
***************************************************************************/

#define INFO_MSG(fmt)
#define ERROR(str) 	IError(str, 0), longjmp(setjmp_buffer, 1)

/***************************************************************************/

static int readColorMap(FILE *, int, unsigned char[GIF_MAXCOLORS][3]);
static int GetDataBlock(FILE *, unsigned char *);
static void readImage(FILE *, int, int, int, unsigned char *);

static jmp_buf setjmp_buffer;

/***************************************************************************/

#ifdef INDEX_COLORS
void Bitmap::DoPalette(int one)
{
	cUsedTable = new long[256];
	int sz = w * h, d, i;
	FGPixel *bm = bitmap;

	memset(cUsedTable, 0, sizeof(long) * 256);	// reset color table
	// calculate which colors is used in image
	for (i = 0; i < sz; i++)
	{
		cUsedTable[*bm++]++;
	}

	bm = bitmap;
	// allocating colors
	for (i = 0; i < 256; i++)
	{
		if (cUsedTable[i])
		{
			d = *((long *) ((char *) &rgb + i * one));
			d >>= 2;
			d &= 0x3f3f3f;
			if (i!=transparent_color)
				cUsedTable[i] = (char) CreateColor((char) (d >> 16), (char) (d >> 8), (char) d, -1);	// white
			else cUsedTable[i] = i;
		}
		else cUsedTable[i] = -1;
	}
	// remapping colors
	for (i = 0; i < sz; i++)
	{
		bm[i] = (char) cUsedTable[bm[i]];
	}
}
#endif

/* read_RLE8_compressed_image:
 *  For reading the 8 bit RLE compressed BMP image format.
 */
static void read_RLE8_compressed_image(FILE * f, unsigned char *bmp, int w, int h)
{
	unsigned char count, val, val0;
	int j, pos, line;
	int eolflag, eopicflag;

	eopicflag = 0;
	line = h - 1;

	while (eopicflag == 0)
	{
		pos = 0;				/* x position in bitmap */
		eolflag = 0;			/* end of line flag */

		while ((eolflag == 0) && (eopicflag == 0))
		{
			count = (char) getc(f);
			val = (char) getc(f);

			if (count > 0)
			{					/* repeat pixel count times */
				for (j = 0; j < count; j++)
				{
					bmp[line * w + pos] = val;
					pos++;
				}
			}
			else
			{
				switch (val)
				{
					case 0:		/* end of line flag */
						eolflag = 1;
						break;
					case 1:		/* end of picture flag */
						eopicflag = 1;
						break;
					case 2:		/* displace picture */
						count = (char) getc(f);
						val = (char) getc(f);
						pos += count;
						line += val;
						break;
					default:	/* read in absolute mode */
						for (j = 0; j < val; j++)
						{
							val0 = (char) getc(f);
							bmp[line * w + pos] = val0;
							pos++;
						}

						if (j % 2 == 1)
							val0 = (char) getc(f);	/* align on word boundary */
						break;
				}
			}
			if (pos > w)
				eolflag = 1;
		}
		line--;
		if (line < 0)
			eopicflag = 1;
	}
}

struct in_image_bmp
{
	short usType;
	short cx, cy;
	short bits;
};

static struct in_image_bmp header;

Bitmap::Bitmap(void *n)
: DrawBuffer()
{
	long one, bits;
	unsigned long c;
	struct in_image_bmp *hp;
	unsigned char *nn = ((unsigned char *) n) + sizeof(long);

	type = BMP_NONE;
	name = 0;
	cUsedTable = 0;
	allBitmapCounter++;
	hp = (in_image_bmp *) nn;	// skip size of file
	one = 4;
	c = 256;					// compute numbers of used colors
	memcpy(&rgb, nn + sizeof(header), sizeof(rgb));
	w = hp->cx;
	h = hp->cy;
	bits = hp->bits;

	if (bits == 8)
	{
	}

	bitmap = (FGPixel *) nn + sizeof(header) + sizeof(rgb);
	type = BMP_IMAGE;
#ifdef INDEX_COLORS
	DoPalette(one);
#endif
}

void Bitmap::turn_bitmap(int cnt, FGPixel *from, FGPixel *to)
{
	FGPixel *p = new FGPixel[w];

	for (int i = 0; i < cnt; i++)
	{
		memmove(p, to - (i * w), w);
		memmove(to - (i * w), from + (i * w), w);
		memmove(from + (i * w), p, w);
	}
	delete p;
}

static void copy_gif_palette(unsigned char *to, unsigned char *data, int cnt)
{
	int r,g,b;
	for(;cnt>0;cnt--)
	{
		r = *data++;
		g = *data++;
		b = *data++;
		*to++ = b;		
		*to++ = g;
		*to++ = r;
	}
}

void DrawBuffer::BitmapDraw(int a, int b, int c, int d)
{
	int ww, hh;

	if (type)
	{
		if (c == -1)
		{
			c = w;
			d = h;
		}
		int a1 = cx_max, b1 = cy_max;
		int oldp = ppop, oldt = transp_color;
		if (transparent_color != -1)
		{
			set_ppop(_GTRANSP);
			transp_color = transparent_color;
		}
		FGPixel *ptr = Image;

		cx_max = w;
		cy_max = h;
		Image = bitmap;
		if (c <= w && d <= h)
		{
			RamToVideo2(0, 0, a, b, c, d);
		}
		else
		{
			for (int i = 0; i <= (d / h); i++)
				for (int j = 0; j <= (c / w); j++)
				{
					ww = j == (c / w) ? c % w : w;
					hh = i == (d / h) ? d % h : h;
					RamToVideo2(0, 0, a + j * w, b + i * h, ww, hh);
				}
		}
		set_ppop(oldp);
		transp_color = oldt;
		cx_max = a1;
		cy_max = b1;
		Image = ptr;
	}
}

Bitmap::Bitmap(char *n)
: DrawBuffer()
{
	int ys, one, compressed = 0;
	unsigned int c, bits = 0;
	FILE *f=0;
	char nm[128], *n2 = n;
	int rest;
	FGPixel *bp;
	long pixel;
	unsigned char *src;
	GIFStream *gif=0;
#ifndef INDEX_COLORS
	int sz, d, i;
	FGPixel *pom;
#endif
	init();
	if (!strstr(n, ".bmp") && !strstr(n, ".BMP")
		&& !strstr(n, ".gif") && !strstr(n, ".GIF")
		&& !strstr(n, ".pcx") && !strstr(n, ".PCX"))
	{
		strcpy(nm, n);
		strcat(nm, ".bmp");
		n = nm;
	}
	char *gifp=0;
	strcpy(nm, n);
	original_name = strdup(nm);
	if ((gifp=strstr(nm, ".gif")) || (gifp=strstr(nm, ".GIF")))
	{							// decode *.gif file
		gif = GIFRead(nm);
		if (gif)
		{
			w = gif->width;
			h = gif->height;
			one = 3;
			c = gif->cmapSize;
			bits = 8;
			if (c) copy_gif_palette((unsigned char *)&rgb, gif->cmapData[0], c);
			assert(bitmap = (FGPixel *) malloc(areasize(w, h)));
			memset(bitmap, gif->background, areasize(w, h));
			GIFData *d = gif->data;
			while(d)
			{
				if (d->type == gif_image)
				{
					if (d->data.image.cmapSize)
					{
						c = d->data.image.cmapSize;
						if (c) copy_gif_palette((unsigned char *)&rgb, d->data.image.cmapData[0], c);
					}
					transparent_color = d->info.transparent;
					memcpy(bitmap, d->data.image.data, d->width*d->height);
					free(d->data.image.data);
				}
				else if(d->type == gif_text || d->type == gif_comment)
				{
					if (d->data.comment.text) free(d->data.comment.text);
				}
				GIFData *pom = d;
				d = d->next;
				free(pom);
			}
			strcpy(gifp, ".bmp");
			name = new char[strlen(nm) + 1];
			strcpy(name, nm);
			free(gif);
		}
		else
		{
			sprintf(nm, "GIF file '%s' loading error!", n);
			IError(nm, 0);
			return;
		}
	}
	else if ((gifp=strstr(nm, ".pcx")) || (gifp=strstr(nm, ".PCX")))
	{							// decode *.gif file
		char hdr[128];
		int counter, byte, data;
		
		f = fopen(n, "rb");
		if (!f)
		{
			sprintf(nm, "Bitmap '%s' not found!", n2);
			IError(nm, 0);
			return;
		}
		fread(hdr, 128, 1, f);
		if (hdr[0] != 0xA || hdr[1] != 5)
		{
error:
			sprintf(nm, "Bitmap '%s' isn't PCX 5.0 file!", n2);
			IError(nm, 0);
			fclose(f);
			return;
		}
		w = *(short *)(hdr+66);
		h = *(short *)(hdr+10)+1;
		bits = hdr[3];
		assert(bitmap = (FGPixel *) calloc(areasize(w, h), 1));
		int sz = w * h;
		src = (unsigned char *) bitmap;
		unsigned char *end = src+sz;
		while (src < end)
		{
			byte = fgetc(f);
			if (0xC0 == (0xC0 & byte))
			{
				counter = 0x3F & byte;
				data = fgetc(f);
				while(counter--)
				{
					*src++ = data;
				}
			}
			else
			{
				*src++ = byte;
			}
		}
		byte = fgetc(f);
		if (byte != 0xC) goto error;
		fread(&rgb, 768, 1, f);
		copy_gif_palette((unsigned char *)&rgb, (unsigned char *)&rgb, 256);
		fclose(f);
		f = 0;
		c = 256;
		one = 3;
		strcpy(gifp, ".bmp");
		name = new char[strlen(nm) + 1];
		strcpy(name, nm);
	}
	else
	{							// decode *.bmp file
		f = fopen(n, "rb");
		if (!f)
		{
			sprintf(nm, "Bitmap '%s' not found!", n2);
			IError(nm, 0);
			return;
		}
		name = new char[strlen(n) + 1];
		strcpy(name, n);
		if (sizeof(bmfh) != 10)
		{
			printf("Ooops, your data structure is not good aligned!!!\a\n");
			assert(0);
			abort();			// 100% sure to stop
		}
		fread(&bmfh, sizeof(bmfh), 1, f);	// read first 10 bytes
		// return if no BMP
		if (bmfh.usType != (unsigned short) ('M' * 256 + 'B'))
		{
			delete name;

			fclose(f);
			IError("No BITMAP file!", 0);
			return;
		}

		fread(&bmi, sizeof(bmi), 1, f);		// read next 4 bytes
		memset(&bmih, 0, sizeof(bmih));
		fread(&bmih, 4, 1, f);	// read size of bitmap info header
		fread(&bmih.bmih1.cx, bmih.bmih1.size - 4, 1, f);	// if 0x0c then OS/2 bitmap
		if (bmih.bmih1.size == 0x0C)
			one = 3;
		else
			one = 4;

		c = (bmi.imageOffset - (bmih.bmih1.size + 14)) / one;	// compute numbers of used colors

		if (c == 0)
		{
#ifdef INDEX_COLORS
			delete name;

			fclose(f);
			IError("True colors BMP not supported!", 0);
			return;
#else
			w = bmih.bmih2.cx;
			h = bmih.bmih2.cy;
			bits = bmih.bmih2.cBitCount;
			if (bmih.bmih2.cPlanes != 1)
			{
				delete name;

				fclose(f);
				IError("This color depth BMP is not supported!", 0);
				return;
			}
#endif
		}
		else if (one == 3)
		{
			if ((compressed = bmih.bmih1.ulCompression) > 1)
			{
				IError("No this compressed BMP supported!", 0);
				return;
			}
			fread(&rgb, sizeof(RGB3), c, f);
			w = bmih.bmih1.cx;
			h = bmih.bmih1.cy;
			bits = bmih.bmih1.cBitCount;
			if (bmih.bmih1.cPlanes != 1)
			{
				delete name;

				fclose(f);
				IError("Multi-planes BMP not supported!", 0);
				return;
			}
		}
		else if (one == 4)
		{
			if ((compressed = bmih.bmih2.ulCompression) > 1)
			{
				IError("Unknown compress format!", 0);
				return;
			}
			fread(&rgb, sizeof(RGB4), c, f);
			w = bmih.bmih2.cx;
			h = bmih.bmih2.cy;
			bits = bmih.bmih2.cBitCount;
			if (bmih.bmih2.cPlanes != 1)
			{
				delete name;

				fclose(f);
				IError("Multi-planes BMP not supported!", 0);
				return;
			}
		}
		assert(bitmap = (FGPixel *) calloc(areasize(w, h), 1));
	}
	
	if (f) switch (bits)
	{
		case 1:
			for (ys = h - 1; ys >= 0; ys--)
			{
				src = (unsigned char *) bitmap + ys * w;
				for (int xs = w / 8; xs > 0; xs--)
				{
					fread((unsigned char *) &pixel, 1, 1, f);
					*src++ = (pixel >> 7) & 1;
					*src++ = (pixel >> 6) & 1;
					*src++ = (pixel >> 5) & 1;
					*src++ = (pixel >> 4) & 1;
					*src++ = (pixel >> 3) & 1;
					*src++ = (pixel >> 2) & 1;
					*src++ = (pixel >> 1) & 1;
					*src++ = (pixel) & 1;
				}
				rest = w % 8;
				if (rest)
				{
					fread((unsigned char *) &pixel, 1, 1, f);
					*src++ = (pixel >> 7) & 1;
					if (--rest == 0)
						goto end_expand;
					*src++ = (pixel >> 6) & 1;
					if (--rest == 0)
						goto end_expand;
					*src++ = (pixel >> 5) & 1;
					if (--rest == 0)
						goto end_expand;
					*src++ = (pixel >> 4) & 1;
					if (--rest == 0)
						goto end_expand;
					*src++ = (pixel >> 3) & 1;
					if (--rest == 0)
						goto end_expand;
					*src++ = (pixel >> 2) & 1;
					if (--rest == 0)
						goto end_expand;
					*src++ = (pixel >> 1) & 1;
				}
			  end_expand:
				rest = (w * bits / 8) % 4;
				if (w % 8)
					rest++;
				while (rest & 3)
				{
					rest++;
					fgetc(f);
				}
			}
			break;
		case 4:
			for (ys = h - 1; ys >= 0; ys--)
			{
				src = (unsigned char *) bitmap + ys * w;
				for (int xs = w / 2; xs > 0; xs--)
				{
					fread((unsigned char *) &pixel, 1, 1, f);
					*src++ = (pixel >> 4) & 15;
					*src++ = pixel & 15;
				}
				rest = w % 2;
				if (rest)
				{
					fread((unsigned char *) &pixel, 1, 1, f);
					*src++ = (pixel >> 4) & 15;
				}
				rest = (w * bits / 8) % 4;
				if (w % 2)
					rest++;
				while (rest & 3)
				{
					rest++;
					fgetc(f);
				}
			}
			break;
		case 8:
			if (!compressed)
				for (ys = h - 1; ys >= 0; ys--)
				{
					fread((unsigned char *) bitmap + ys * w, w, 1, f);
					rest = (w * bits / 8) % 4;
					while (rest & 3)
					{
						rest++;
						fgetc(f);
					}
				}
			else read_RLE8_compressed_image(f, (unsigned char *) bitmap, w, h);
			break;
#ifndef INDEX_COLORS
		case 24:
			for (ys = h - 1; ys >= 0; ys--)
			{
				bp = (bitmap + ys * w);
				for (int xs = 0; xs < w; xs++)
				{
					fread(&pixel, 3, 1, f);
					*bp = DirectColor(pixel);
					bp++;
				}
			}
			rest = (w * bits / 8) % 4;
			while (rest & 3)
			{
				rest++;
				fgetc(f);
			}
			break;
#endif
#ifdef DIRECT_COLORS
		case 16:
		case 15:
			for (ys = h - 1; ys >= 0; ys--)
			{
				bp = (bitmap + ys * w);
				for (int xs = 0; xs < w; xs++)
				{
					fread(bp, 2, 1, f);
					bp++;
				}
			}
			rest = (w * bits / 8) % 4;
			while (rest & 3)
			{
				rest++;
				fgetc(f);
			}
			break;
#endif
		default:
			IError("Bad Bitmap format!", 0);
			delete name;
			fclose(f);
			free(bitmap);
			return;
	}
	if (f) fclose(f);
#ifndef INDEX_COLORS
	if (bits<=8)
	{
		bp = (FGPixel *) calloc(areasize(w, h), 1);
		pom = bp;
		src = (unsigned char *) bitmap;
		sz = w * h;

		if (transparent_color==-1) for (i = 0; i < sz; i++)
		{
			d = *((long *) ((char *) &rgb + *src * one));
			*pom = DirectColor(d);
			pom++;
			src++;
		}
		else
		{
			for (i = 0; i < sz; i++)
			{
				if (*src == transparent_color) d = TRANSP_KEY;
				else d = *((long *) ((char *) &rgb + *src * one));
				*pom++ = DirectColor(d);
				src++;
			}
			transparent_color = DirectColor(TRANSP_KEY);
		}
		free(bitmap);
		bitmap = bp;
	}
#endif
	type = BMP_FILE;
#ifdef INDEX_COLORS
	DoPalette(one);
#endif
}

void DrawBuffer::copy(FGPixel * dst, FGPixel * src)
{
	register int i = h;

	for (; i--;)
	{
		memcpy(dst, src, w * bpp);
		dst += w;
		src += w;
	}
}

DrawBuffer::DrawBuffer(int ww, int hh, FGPixel * buf)
{
	w = ww;
	h = hh;
	type = BMP_MEM;
	transparent_color = -1;
	assert(bitmap = (FGPixel *) malloc(areasize(w, h)));
	copy(bitmap, buf);
}
/*
DrawBuffer::DrawBuffer(Window * wPtr)
{
	w = wPtr->GetW();
	h = wPtr->GetH();
	type = BMP_MEM;
	transparent_color = -1;
	bitmap = wPtr->GetArray();
}
*/
DrawBuffer::DrawBuffer(int ww, int hh, BmpType t, int color, FGPixel * buf)
{
	if (ww != 0 && hh != 0)
	{
		w = ww;
		h = hh;
		type = t;
		if (!buf)
		{
			assert(bitmap = (FGPixel *) malloc(areasize(w, h)));
			FGmemset(bitmap, color, w * h);
		}
		else
			bitmap = buf;
	}
	else
		memset(this, 0, sizeof(DrawBuffer));
	transparent_color = -1;
}

DrawBuffer::~DrawBuffer()
{
	if (type)
	{
		if (type != BMP_IMAGE)
			free(bitmap);
	}
	type = 0;
}

Bitmap::~Bitmap()
{
#ifdef INDEX_COLORS
	if (cUsedTable)
	{
		for (int i = 0; i < 256; i++)
			if (cUsedTable[i] != -1)
				DeleteColor(cUsedTable[i]);
		delete cUsedTable;
	}
#endif
	if (name)
		delete name;
	if (original_name) free(original_name);

	allBitmapCounter--;
}

Bitmap::Bitmap(int ww, int hh, FGPixel color)
: DrawBuffer(ww, hh, BMP_MEM, color)
{
	init();
}

Bitmap::Bitmap(Window * wPtr)
:DrawBuffer(wPtr->GetW(), wPtr->GetH())
{
	init();
	copy(bitmap, wPtr->GetArray());
}

void Bitmap::init(void)
{
	cUsedTable = 0;
	name = original_name = 0;
	allBitmapCounter++;
	memset(&bmih, 0, sizeof(bmih));
}

int Bitmap::BitmapSave(char *nazov)
{
	int i;

	if (GetType() == BMP_NONE)
		return 0;
	char *tmpname = new char[strlen(nazov) + 16];

#ifndef INDEX_COLORS
	int j, rest;
	char *bmp2;

#endif
	nazov = strdup(nazov);
#ifdef __MSDOS__
	strlwr(nazov);
#endif
	if (strstr(nazov, ".bmp"))
		strcpy(tmpname, nazov);
	else
		sprintf(tmpname, "%s.bmp", nazov);
	bmfh.usType = 'M' * 256 + 'B';
	bmfh.reserve1 = 0;
	bmfh.reserve2 = 0;
	bmi.reserve = 0;
	bmih.bmih2.size = 0x28;
	bmih.bmih2.cx = w;
	bmih.bmih2.cy = h;
	bmih.bmih2.cPlanes = 1;
	bmih.bmih2.ulCompression = 0;
	bmih.bmih2.cbImage = w * h;
	bmih.bmih2.cxResolution = 0;
	bmih.bmih2.cyResolution = 0;
#ifdef INDEX_COLORS
	bmi.imageOffset = 0x36 + 0x400;
	bmfh.cbSizeLow = w * h + 1024 + 0x36;
	bmfh.cbSizeHigh = (w * h + 1024 + 0x36) >> 16;
	bmih.bmih2.cclrUsed = 256;
	bmih.bmih2.cclrImportant = 256;
	bmih.bmih2.cBitCount = 8;

	for (i = 0; i < 256; i++)
	{
		rgb.rgb4[i].fcOptions = 0;
		GetPaletteEntry(&rgb.rgb4[i].bRed, &rgb.rgb4[i].bGreen, &rgb.rgb4[i].bBlue, i);
		rgb.rgb4[i].bRed <<= 2;
		rgb.rgb4[i].bGreen <<= 2;
		rgb.rgb4[i].bBlue <<= 2;
	}
#else
	bmi.imageOffset = 0x36;
	bmfh.cbSizeLow = w * h + 0x36;
	bmfh.cbSizeHigh = (w * h + 0x36) >> 16;
	bmih.bmih2.cclrUsed = 0;
	bmih.bmih2.cclrImportant = 0;
	bmih.bmih2.cBitCount = 24;
#endif // index
	FILE *fp = fopen(tmpname, "wb");

	if (fp == 0)
	{
		IError("Bitmap not saved!", 0);
		delete tmpname;

		return 0;
	}
	assert(sizeof(bmfh) == 10);
	fwrite(&bmfh, sizeof(bmfh), 1, fp);
	fwrite(&bmi, sizeof(bmi), 1, fp);
	fwrite(&bmih, sizeof(bmih.bmih2), 1, fp);
#ifdef INDEX_COLORS
	fwrite(&rgb, sizeof(rgb.rgb4), 1, fp);
	turn_bitmap(h / 2, bitmap, bitmap + w * (h - 1));
	fwrite(bitmap, w * h, 1, fp);
	turn_bitmap(h / 2, bitmap, bitmap + w * (h - 1));
#else // index
	unsigned long cc, aa;

	for (i = h - 1; i >= 0; i--)
	{
		bmp2 = (char *) (bitmap + w * i);
		for (j = 0; j < w; j++)
		{
#ifdef TRUE_COLORS
			fwrite(bmp2, 3, 1, fp);
			bmp2 += 4;
#else
			cc = *(FGPixel *) bmp2;
#if (FASTGL_BPP == 15)
			aa = ((cc << 9) & 0xF80000) | ((cc << 6) & 0xF800) | ((cc << 3) & 0xF8);
#else
			aa = ((cc << 8) & 0xF80000) | ((cc << 5) & 0xFc00) | ((cc << 3) & 0xF8);
#endif // direct
			fwrite(&aa, 3, 1, fp);
			bmp2 += 2;
#endif // true
		}
#ifdef TRUE_COLORS
		rest = (w * 3) % 4;
#else
		rest = (w * 2) % 4;
#endif // true
		while (rest & 3)
		{
			rest++;
			fputc(0, fp);
		}
	}
#endif
	fclose(fp);
	free(nazov);
	delete tmpname;
	return 1;
}

/* +-------------------------------------------------------------------+ */
/* | Copyright 1990 - 1994, David Koblas. (koblas@netcom.com)          | */
/* |   Permission to use, copy, modify, and distribute this software   | */
/* |   and its documentation for any purpose and without fee is hereby | */
/* |   granted, provided that the above copyright notice appear in all | */
/* |   copies and that both that copyright notice and this permission  | */
/* |   notice appear in supporting documentation.  This software is    | */
/* |   provided "as is" without express or implied warranty.           | */
/* +-------------------------------------------------------------------+ */

GIFStream *GIFReadFP(FILE * fd)
{
	unsigned char buf[256];
	unsigned char c;
	GIFStream *stream;
	GIFData *cur, **end;
	GIF89info info;
	int resetInfo = TRUE;
	int n;

	if (fd == NULL)
		return NULL;

	if (setjmp(setjmp_buffer))
		goto out;

	if (!ReadOK(fd, buf, 6))
		ERROR("error reading magic number");

	if (strncmp((char *) buf, "GIF", 3) != 0)
		ERROR("not a GIF file");

	if ((strncmp((const char *) buf + 3, "87a", 3) != 0) &&
		(strncmp((const char *) buf + 3, "89a", 3) != 0))
		ERROR("bad version number, not '87a' or '89a'");

	if (!ReadOK(fd, buf, 7))
		ERROR("failed to read screen descriptor");

	stream = NEW(GIFStream);

	stream->width = MKINT(buf[0], buf[1]);
	stream->height = MKINT(buf[2], buf[3]);

	stream->cmapSize = 2 << (buf[4] & 0x07);
	stream->colorMapSize = stream->cmapSize;
	stream->colorResolution = ((int) (buf[4] & 0x70) >> 3) + 1;
	stream->background = buf[5];
	stream->aspectRatio = buf[6];

	stream->data = NULL;

	end = &stream->data;

	/*
	   **  Global colormap is present.
	 */
	if (BitSet(buf[4], LOCALCOLORMAP))
	{
		if (readColorMap(fd, stream->cmapSize, stream->cmapData))
			ERROR("unable to get global colormap");
	}
	else
	{
		stream->cmapSize = 0;
		stream->background = -1;
	}

	if (stream->aspectRatio != 0 && stream->aspectRatio != 49)
	{
		float r;

		r = ((float) stream->aspectRatio + 15.0) / 64.0;
		INFO_MSG(("warning - non-square pixels; to fix do a 'pnmscale -%cscale %g'",
				  r < 1.0 ? 'x' : 'y',
				  r < 1.0 ? 1.0 / r : r));
	}

	while (ReadOK(fd, &c, 1) && c != ';')
	{
		if (resetInfo)
		{
			info.disposal = (GIFDisposalType) 0;
			info.inputFlag = 0;
			info.delayTime = 0;
			info.transparent = -1;
			resetInfo = FALSE;
		}
		cur = NULL;

		if (c == '!')
		{						/* Extension */
			if (!ReadOK(fd, &c, 1))
				ERROR("EOF / read error on extention function code");
			if (c == 0xf9)
			{					/* graphic control */
				(void) GetDataBlock(fd, buf);
				info.disposal = (GIFDisposalType) ((buf[0] >> 2) & 0x7);
				info.inputFlag = (buf[0] >> 1) & 0x1;
				info.delayTime = MKINT(buf[1], buf[2]);
				if (BitSet(buf[0], 0x1))
					info.transparent = buf[3];

				while (GetDataBlock(fd, buf) != 0)
					;
			}
			else if (c == 0xfe || c == 0x01)
			{
				int len = 0;
				int size = 256;
				char *text = NULL;

				/* 
				   **  Comment or Plain Text
				 */

				cur = NEW(GIFData);

				if (c == 0x01)
				{
					(void) GetDataBlock(fd, buf);

					cur->type = gif_text;
					cur->info = info;
					cur->x = MKINT(buf[0], buf[1]);
					cur->y = MKINT(buf[2], buf[3]);
					cur->width = MKINT(buf[4], buf[5]);
					cur->height = MKINT(buf[6], buf[7]);

					cur->data.text.cellWidth = buf[8];
					cur->data.text.cellHeight = buf[9];
					cur->data.text.fg = buf[10];
					cur->data.text.bg = buf[11];

					resetInfo = TRUE;
				}
				else
				{
					cur->type = gif_comment;
				}

				text = (char *) malloc(size);

				while ((n = GetDataBlock(fd, buf)) != 0)
				{
					if (n + len >= size)
						text = (char *) realloc(text, size += 256);
					memcpy(text + len, buf, n);
					len += n;
				}

				if (c == 0x01)
				{
					cur->data.text.len = len;
					cur->data.text.text = text;
				}
				else
				{
					cur->data.comment.len = len;
					cur->data.comment.text = text;
				}
			}
			else
			{
				/*
				   **  Unrecogonized extension, consume it.
				 */
				while (GetDataBlock(fd, buf) > 0)
					;
			}
		}
		else if (c == ',')
		{
			if (!ReadOK(fd, buf, 9))
				ERROR("couldn't read left/top/width/height");

			cur = NEW(GIFData);

			cur->type = gif_image;
			cur->info = info;
			cur->x = MKINT(buf[0], buf[1]);
			cur->y = MKINT(buf[2], buf[3]);
			cur->width = MKINT(buf[4], buf[5]);
			cur->height = MKINT(buf[6], buf[7]);
			cur->data.image.cmapSize = 1 << ((buf[8] & 0x07) + 1);
			if (BitSet(buf[8], LOCALCOLORMAP))
			{
				if (readColorMap(fd, cur->data.image.cmapSize,
								 cur->data.image.cmapData))
					ERROR("unable to get local colormap");
			}
			else
			{
				cur->data.image.cmapSize = 0;

			}
			cur->data.image.data = (unsigned char *) malloc(cur->width * cur->height);
			cur->data.image.interlaced = BitSet(buf[8], INTERLACE);
			readImage(fd, BitSet(buf[8], INTERLACE),
					  cur->width, cur->height, cur->data.image.data);

			resetInfo = TRUE;
		}
		else
		{
			INFO_MSG(("bogus character 0x%02x, ignoring", (int) c));
		}

		if (cur != NULL)
		{
			*end = cur;
			end = &cur->next;
			cur->next = NULL;
		}
	}

	if (c != ';')
		ERROR("EOF / data stream");

  out:
	return stream;
}

GIFStream *GIFRead(char *file)
{
	FILE *fp = fopen(file, "rb");
	GIFStream *stream = NULL;

	if (fp != NULL)
	{
		stream = GIFReadFP(fp);
		fclose(fp);
	}
	return stream;
}

static int readColorMap(FILE * fd, int size,
						unsigned char data[GIF_MAXCOLORS][3])
{
	int i;
	unsigned char rgb[3 * GIF_MAXCOLORS];
	unsigned char *cp = rgb;

	if (!ReadOK(fd, rgb, size * 3))
		return TRUE;

	for (i = 0; i < size; i++)
	{
		data[i][0] = *cp++;
		data[i][1] = *cp++;
		data[i][2] = *cp++;
	}

	return FALSE;
}

/*
   **
 */

static int ZeroDataBlock = FALSE;

static int GetDataBlock(FILE * fd, unsigned char *buf)
{
	unsigned char count;

	if (!ReadOK(fd, &count, 1))
	{
		INFO_MSG(("error in getting DataBlock size"));
		return -1;
	}

	ZeroDataBlock = count == 0;
	if ((count != 0) && (!ReadOK(fd, buf, count)))
	{
		INFO_MSG(("error in reading DataBlock"));
		return -1;
	}
	return count;
}

/*
   **
   **
 */

/*
   **  Pulled out of nextCode
 */
static int curbit, lastbit, get_done, last_byte;
static int return_clear;

/*
   **  Out of nextLWZ
 */
static int stack[(1 << (MAX_LWZ_BITS)) * 2], *sp;
static int code_size, set_code_size;
static int max_code, max_code_size;
static int clear_code, end_code;

static void initLWZ(int input_code_size)
{
//	static int inited = FALSE;

	set_code_size = input_code_size;
	code_size = set_code_size + 1;
	clear_code = 1 << set_code_size;
	end_code = clear_code + 1;
	max_code_size = 2 * clear_code;
	max_code = clear_code + 2;

	curbit = lastbit = 0;
	last_byte = 2;
	get_done = FALSE;

	return_clear = TRUE;

	sp = stack;
}

static int nextCode(FILE * fd, int code_size)
{
	static unsigned char buf[280];
	static int maskTbl[16] =
	{
		0x0000, 0x0001, 0x0003, 0x0007,
		0x000f, 0x001f, 0x003f, 0x007f,
		0x00ff, 0x01ff, 0x03ff, 0x07ff,
		0x0fff, 0x1fff, 0x3fff, 0x7fff,
	};
	int i, j, ret, end;

	if (return_clear)
	{
		return_clear = FALSE;
		return clear_code;
	}

	end = curbit + code_size;

	if (end >= lastbit)
	{
		int count;

		if (get_done)
		{
			if (curbit >= lastbit)
				ERROR("ran off the end of my bits");
			return -1;
		}
		buf[0] = buf[last_byte - 2];
		buf[1] = buf[last_byte - 1];

		if ((count = GetDataBlock(fd, &buf[2])) == 0)
			get_done = TRUE;

		last_byte = 2 + count;
		curbit = (curbit - lastbit) + 16;
		lastbit = (2 + count) * 8;

		end = curbit + code_size;
	}

	j = end / 8;
	i = curbit / 8;

	if (i == j)
		ret = buf[i];
	else if (i + 1 == j)
		ret = buf[i] | (buf[i + 1] << 8);
	else
		ret = buf[i] | (buf[i + 1] << 8) | (buf[i + 2] << 16);

	ret = (ret >> (curbit % 8)) & maskTbl[code_size];

	curbit += code_size;

	return ret;
}

#define readLWZ(fd) ((sp > stack) ? *--sp : nextLWZ(fd))

static int nextLWZ(FILE * fd)
{
	static int table[2][(1 << MAX_LWZ_BITS)];
	static int firstcode, oldcode;
	int code, incode;
	register int i;

	while ((code = nextCode(fd, code_size)) >= 0)
	{
		if (code == clear_code)
		{
			for (i = 0; i < clear_code; ++i)
			{
				table[0][i] = 0;
				table[1][i] = i;
			}
			for (; i < (1 << MAX_LWZ_BITS); ++i)
				table[0][i] = table[1][i] = 0;
			code_size = set_code_size + 1;
			max_code_size = 2 * clear_code;
			max_code = clear_code + 2;
			sp = stack;
			do
			{
				firstcode = oldcode = nextCode(fd, code_size);
			}
			while (firstcode == clear_code);

			return firstcode;
		}
		if (code == end_code)
		{
			int count;
			unsigned char buf[260];

			if (ZeroDataBlock)
				return -2;

			while ((count = GetDataBlock(fd, buf)) > 0)
				;

			if (count != 0)
				INFO_MSG(("missing EOD in data stream"));
			return -2;
		}

		incode = code;

		if (code >= max_code)
		{
			*sp++ = firstcode;
			code = oldcode;
		}

		while (code >= clear_code)
		{
			*sp++ = table[1][code];
			if (code == table[0][code])
				ERROR("circular table entry BIG ERROR");
			code = table[0][code];
		}

		*sp++ = firstcode = table[1][code];

		if ((code = max_code) < (1 << MAX_LWZ_BITS))
		{
			table[0][code] = oldcode;
			table[1][code] = firstcode;
			++max_code;
			if ((max_code >= max_code_size) &&
				(max_code_size < (1 << MAX_LWZ_BITS)))
			{
				max_code_size *= 2;
				++code_size;
			}
		}

		oldcode = incode;

		if (sp > stack)
			return *--sp;
	}
	return code;
}

static void readImage(FILE * fd, int interlace, int width, int height,
					  unsigned char *data)
{
	unsigned char *dp, c;
	int v, xpos = 0, ypos = 0, pass = 0;

	/*
	   **  Initialize the Compression routines
	 */
	if (!ReadOK(fd, &c, 1))
		ERROR("EOF / read error on image data");

	initLWZ(c);

	if (verbose)
		INFO_MSG(("reading %d by %d%s GIF image",
				  width, height, interlace ? " interlaced" : ""));

	if (interlace)
	{
		int i;
		int pass = 0, step = 8;

		for (i = 0; i < height; i++)
		{
			dp = &data[width * ypos];
			for (xpos = 0; xpos < width; xpos++)
			{
				if ((v = readLWZ(fd)) < 0)
					goto fini;

				*dp++ = v;
			}
			if ((ypos += step) >= height)
			{
				do
				{
					if (pass++ > 0)
						step /= 2;
					ypos = step / 2;
				}
				while (ypos > height);
			}
		}
	}
	else
	{
		dp = data;
		for (ypos = 0; ypos < height; ypos++)
		{
			for (xpos = 0; xpos < width; xpos++)
			{
				if ((v = readLWZ(fd)) < 0)
					goto fini;

				*dp++ = v;
			}
		}
	}

  fini:
	if (readLWZ(fd) >= 0)
		INFO_MSG(("too much input data, ignoring extra..."));

	return;
}
