// forms_timer.H

// Emulate the Forms Timer object
// You don't want to use this if you just want a timeout, call
// Fl::add_timeout directly!

#include <FL/Fl.H>
#include <FL/Fl_Timer.H>
#include <FL/fl_draw.H>
#include <sys/time.h>
#include <stdio.h>

#define FL_TIMER_BLINKRATE	0.2

void fl_gettime(long *sec, long *usec)
/* returns current time */
{
  struct timeval tp;
  struct timezone tzp;
  gettimeofday(&tp,&tzp);
  *sec = tp.tv_sec;
  *usec = tp.tv_usec;
}

void Fl_Timer::draw() {
  int tt;
  uchar col;
  char str[32];
  if (!on || delay>0.0)
    col = color();
  else if ((int) (delay / FL_TIMER_BLINKRATE) % 2)
    col = color();
  else
    col = color2();
  draw_box(box(),col);
  if (type() == FL_VALUE_TIMER && delay>0.0) {
    float d = direction_ ? total-delay : delay;
    if (d < 60.0)
      sprintf(str,"%.1f",d);
    else {
      tt = (int) ((d+0.05) / 60.0);
      sprintf(str,"%d:%04.1f", tt, d - 60.0 * tt);
    }
    fl_font(labelfont(),labelsize());
    fl_color(labelcolor());
    fl_draw(str,x(),y(),w(),h(),FL_ALIGN_CENTER);
  } else
    draw_label();
}

void Fl_Timer::stepcb(void *v) {
  ((Fl_Timer *)v)->step();
}

void Fl_Timer::step() {
  if (!on) return;
  float lastdelay = delay;
  long sec,usec; fl_gettime(&sec,&usec);
  delay -= (float) (sec - lastsec) + (float) (usec - lastusec) / 1000000.0;
  lastsec = sec; lastusec = usec;
  if (lastdelay > 0.0 && delay <= 0.0) {
    if (type() == FL_HIDDEN_TIMER) {
      on = 0;
      delay = 0;
    } else {
      redraw();
      Fl::add_timeout(FL_TIMER_BLINKRATE,stepcb,this);
    }
    do_callback();
  } else {
    if (type() == FL_VALUE_TIMER) redraw();
    Fl::add_timeout(FL_TIMER_BLINKRATE,stepcb,this);
  }
}

int Fl_Timer::handle(int event) {
  if (event == FL_RELEASE && delay <= 0) value(0.0);
  return 0;
}

Fl_Timer::~Fl_Timer() {
  Fl::remove_timeout(stepcb,this);
}

Fl_Timer::Fl_Timer(uchar t,int x,int y,int w,int h,const char *l)
: Fl_Object(x,y,w,h,l) {
  box(FL_DOWN_BOX);
  color2(FL_RED);
  delay = 0;
  on = 0;
  direction_ = 0;
  type(t);
  if (t == FL_HIDDEN_TIMER) clear_visible();
  if (t == FL_VALUE_TIMER) align(FL_ALIGN_LEFT);
}

void Fl_Timer::value(float d) {
  delay = total = d;
  on = (d > 0.0);
  fl_gettime(&(lastsec),&(lastusec));
  if (type() != FL_HIDDEN_TIMER) redraw();
  Fl::remove_timeout(stepcb,this);
  if (on) Fl::add_timeout(FL_TIMER_BLINKRATE,stepcb,this);
}

void Fl_Timer::suspended(char d) {
  if (!d) {
    if (on) return;
    on = (delay > 0.0);
    fl_gettime(&(lastsec),&(lastusec));
    if (on) Fl::add_timeout(FL_TIMER_BLINKRATE,stepcb,this);
  } else {
    if (!on) return;
    on = 0;
    Fl::remove_timeout(stepcb,this);
  }
}
