// fl_labeltype.C

// Drawing code for the (one) common label types.
// Other label types (symbols) are in their own source files
// to avoid linking if not used.

#include <FL/Fl.H>
#include <FL/Fl_Object.H>
#include <FL/fl_draw.H>

int
fl_no_label(const Fl_Label*,int,int,int,int,uchar)
{return 1;}

int
fl_normal_label(const Fl_Label* o, int X, int Y, int W, int H, uchar align)
{
  fl_font(o->font, o->size);
  fl_color(o->color);
  fl_draw(o->value, X, Y, W, H, align);
  return 1;
}

#define MAX_LABELTYPE 32

static Fl_LabelFunction *table[MAX_LABELTYPE] = {
  fl_normal_label,
  fl_no_label,
  fl_no_label,	// fl_symbol_label
  fl_no_label,
  fl_no_label, fl_no_label, fl_no_label, fl_no_label,
  fl_no_label, fl_no_label, fl_no_label, fl_no_label,
  fl_no_label, fl_no_label, fl_no_label, fl_no_label,
};

Fl_LabelFunction* Fl::get_labeltype(uchar t) {return table[t];}

uchar Fl::set_labeltype(uchar t, Fl_LabelFunction* f) {table[t] = f; return t;}

static uchar free_labeltype = FL_FREE_LABELTYPE;
uchar Fl::set_labeltype(Fl_LabelFunction* f) {
  return set_labeltype(free_labeltype++, f);
}

////////////////////////////////////////////////////////////////

// draw label with arbitrary alignment in arbitrary box:
void Fl_Label::draw(int X, int Y, int W, int H, uchar align) const {
  if (!value) return;
  if (!table[type](this, X, Y, W, H, align))
    fl_normal_label(this, X, Y, W, H,align);
}

// The normal call for a draw() method:
void Fl_Object::draw_label() const {
  draw_label(x_+Fl::box_dx(box_),y_+Fl::box_dy(box_),
	     w_-Fl::box_dw(box_),h_-Fl::box_dh(box_));
}

// draw() can use this instead to change the bounding box:
extern char fl_draw_shortcut;
void Fl_Object::draw_label(int X, int Y, int W, int H) const {
  // quit if we are not drawing a label inside the object:
  if (align() && !(align() & FL_ALIGN_INSIDE)) return;
  if (flags()&SHORTCUT_LABEL) fl_draw_shortcut = 1;
  label_.draw(X,Y,W,H,align());
  fl_draw_shortcut = 0;
}

// Parent object can call this to draw an outside lable at a random position:
void Fl_Object::draw_outside_label(int X, int Y, int W, int H, uchar a) const {
// caller must do these tests if wanted, Fl_Tabs did NOT want them:
//if (!visible()) return;
//if (!align() || (align() & FL_ALIGN_INSIDE)) return;
  if (flags()&SHORTCUT_LABEL) fl_draw_shortcut = 1;
  label_.draw(X,Y,W,H,a);
  fl_draw_shortcut = 0;
}

// Parents normally call this to draw outside labels:
void Fl_Object::draw_outside_label() const {
  if (!visible()) return;
  // quit if we are not drawing a label outside the object:
  if (!align() || (align() & FL_ALIGN_INSIDE)) return;
  // invent a box that is outside the object:
  uchar align = this->align();
  int X = x();
  int Y = y();
  int W = w();
  int H = h();
  if (align & FL_ALIGN_TOP) {
    if (fl_clipped && fl_current_clip.y > Y) return;
    Y = Y-H; align ^= (FL_ALIGN_BOTTOM|FL_ALIGN_TOP);
  } else if (align & FL_ALIGN_BOTTOM) {
    Y = Y+H; align ^= (FL_ALIGN_BOTTOM|FL_ALIGN_TOP);
    if (fl_clipped && fl_current_clip.b < Y) return;
  } else if (align & FL_ALIGN_LEFT) {
    if (fl_clipped && fl_current_clip.x > X) return;
    X = X-H; W=H; align=FL_ALIGN_RIGHT;
  } else if (align & FL_ALIGN_RIGHT) {
    X = X+W; W=H; align=FL_ALIGN_LEFT;
    if (fl_clipped && fl_current_clip.r < X) return;
  }
  if (flags()&SHORTCUT_LABEL) fl_draw_shortcut = 1;
  label_.draw(X,Y,W,H,align);
  fl_draw_shortcut = 0;
}
