/*	fl_font.C

	Routines to take a "font" and "size" and set up the
	GL context so that fl_draw(char*) will draw in that font.

	This file is GLX specific and would probably be replaced
	for any other window system with totally different code.

	This code is also the main reason only one Fl can be
	created.  There should be a seperate list of Fl_Font
	structures for each Fl...

*/

#include <config.h>
#include <FL/Fl.H>
#include <FL/fl_draw.H>
#include <FL/x.H>
#include "Fl_Font.H"

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

Fl_Font::Fl_Font(const char *name, uchar num) {
  font = XLoadQueryFont(fl_display, name);
  if (!font) font = XLoadQueryFont(fl_display, "fixed");
#if HAVE_GL
  madelistbase = 0;
#endif
  number = num;
}

Fl_Font *fl_current_font;

Fl_Font::~Fl_Font() {
#if HAVE_GL
// Delete list created by gl_draw().  This is not done by this code
// as it will link in GL unnecessarily.  There should be some kind
// of "free" routine pointer, or a subclass?
// if (madelistbase) {
//  int base = font->min_char_or_byte2;
//  int size = font->max_char_or_byte2-base+1;
//  int base = 0; int size = 256;
//  glDeleteLists(listbase+base,size);
// }
#endif
  if (this == fl_current_font) fl_current_font = 0;
  XFreeFont(fl_display, font);
}

////////////////////////////////////////////////////////////////

#define FL_MAXFONT 32

struct fontentry {
  const char *name;
  char **xlist;
  int n;
  Fl_Font *first;	// list of ones we have found so far
};

static fontentry fonttab[FL_MAXFONT] = {
{"*helvetica-medium-r-no*"},
{"*helvetica-bold-r-no*"},
{"*helvetica-medium-o-no*"},
{"*helvetica-bold-o-no*"},
{"*courier-medium-r-no*"},
{"*courier-bold-r-no*"},
{"*courier-medium-o-no*"},
{"*courier-bold-o-no*"},
{"*times-medium-r-no*"},
{"*times-bold-r-no*"},
{"*times-medium-i-no*"},
{"*times-bold-i-no*"},
{"*symbol-*"},
{"*screen-medium-*"},	// was "Type"
{"*international-*"},	// was "Iris"
{"*zapf dingbats-*"},
};

static Fl_Font *fixed;

#define MAXSIZE 32767

// return a pointer to a number we think is "point size":
char *fl_find_fontsize(char *name) {
  int n = 0;
  char *c;
  // first try after the 7th dash
  for (c=name; *c;) if (*c++=='-') {n++; if (n==7 && isdigit(*c)) return c;}
  // now try trailing digits:
  while (c>name && isdigit(*(c-1))) c--;
  if (c>name && *c) return c;
  // we lose:
  return 0;
}

static Fl_Font *find(uchar fnum, int size) {
  fontentry *s = fonttab+(fnum%FL_MAXFONT);
  if (!s->name) s = fonttab; // use 0 if fnum undefined
  Fl_Font *f;
  for (f = s->first; f; f = f->next)
    if (f->minsize <= size && f->maxsize >= size) return f;
  fl_open_display();
  if (!s->xlist) {
    s->xlist = XListFonts(fl_display, s->name, 100, &(s->n));
    if (!s->xlist) {	// use fixed if no matching font...
      if (!fixed) {
	fixed = new Fl_Font("fixed",fnum);
	fixed->minsize = 0;
	fixed->maxsize = 32767;
      }
      s->first = fixed;
      return fixed;
    }
  }
  // search for largest <= font size:
  char *name = s->xlist[0]; int ptsize = 0;	// best one found so far
  int matchedlength = 32767;
  char namebuffer[128];	// holds scalable font name
  for (int n=0; n < s->n; n++) {

    char *thisname = s->xlist[n];
    char *c = fl_find_fontsize(thisname);
    int thissize = c ? atoi(c) : MAXSIZE;
    int thislength = strlen(thisname);
    if (thissize == size && thislength < matchedlength) {
      name = thisname;
      ptsize = size;
      matchedlength = thislength;
    } else if (!thissize && ptsize!=size) {
      // whoa!  A scalable font!  Use unless exact match found:
      int l = c-thisname;
      memcpy(namebuffer,thisname,l);
#if 0 // this works if you don't want stdio
      namebuffer[l++] = size/100+'0';
      namebuffer[l++] = (size/10)%10+'0';
      namebuffer[l++] = (size%10)+'0';
#else
      l += sprintf(namebuffer+l,"%d",size);
#endif
      while (isdigit(*c)) c++;
      strcpy(namebuffer+l,c);
      name = namebuffer;
      ptsize = size;
    } else if (!ptsize ||	// no fonts yet
	       thissize < ptsize && ptsize > size || // current font too big
	       thissize > ptsize && thissize <= size || // current too small
	       thissize == ptsize && thislength < matchedlength // better
      ) {
      name = thisname; ptsize = thissize;
      matchedlength = thislength;
    }
  }

  if (ptsize != size) { // see if we already found this unscalable font:
    for (f = s->first; f; f = f->next) {
      if (f->minsize <= ptsize && f->maxsize >= ptsize) {
	if (f->minsize > size) f->minsize = size;
	if (f->maxsize < size) f->maxsize = size;
	return f;
      }
    }
  }

  // okay, we definately have some name, make the font:
//  if (fnum>5) printf("using %s (%d%s)\n",name,size,name==namebuffer?" scaled":"");
  f = new Fl_Font(name,fnum);
  if (ptsize < size) {f->minsize = ptsize; f->maxsize = size;}
  else {f->minsize = size; f->maxsize = ptsize;}
  f->next = s->first;
  s->first = f;
  return f;

}

////////////////////////////////////////////////////////////////
// Public interface:

void fl_font(uchar fnum, int size) {
  Fl_Font *f = fl_current_font;
  if (f && f->number == fnum && f->minsize <= size && f->maxsize >= size);
  else fl_current_font = find(fnum,size);
}

const char *Fl::get_font(uchar fnum) {
  return fonttab[fnum%FL_MAXFONT].name;
}

uchar Fl::set_font(uchar fnum, const char *name) {
  fontentry *s = fonttab+(fnum%FL_MAXFONT);
  if (s->name) {
    if (!strcmp(s->name, name)) {s->name = name; return fnum;}
    if (s->xlist) {XFreeFontNames(s->xlist); s->xlist = 0;}
    if (fl_current_font->number == fnum) fl_current_font = 0;
    for (Fl_Font *f = s->first; f && f!=fixed;) {
      Fl_Font *n = f->next; delete f; f = n;
    }
    s->first = 0;
  }
  s->name = name;
  return fnum;
}

static uchar free_font = FL_FREE_FONT;
uchar Fl::set_font(const char *name) {return set_font(free_font++, name);}

int fl_height() {
  return (fl_current_font->font->ascent + fl_current_font->font->descent);
}

int fl_descent() {
  return fl_current_font->font->descent;
}

float fl_width(const char *c) {
  XCharStruct *p = fl_current_font->font->per_char;
  if (!p) return strlen(c)*fl_current_font->font->min_bounds.width;
  int base = fl_current_font->font->min_char_or_byte2;
  int w = 0;
  while (*c) w += p[(uchar)(*c++)-base].width;
  return w;
}

float fl_width(const char *c, int n) {
  XCharStruct *p = fl_current_font->font->per_char;
  if (!p) return n*fl_current_font->font->min_bounds.width;
  int base = fl_current_font->font->min_char_or_byte2;
  int w = 0;
  while (n--) w += p[(uchar)(*c++)-base].width;
  return w;
}

float fl_width(uchar c) {
  XCharStruct *p = fl_current_font->font->per_char;
  if (!p) return fl_current_font->font->min_bounds.width;
  int base = fl_current_font->font->min_char_or_byte2;
  return p[c-base].width;
}

void fl_draw(const char *str, int n, int x, int y) {
  XSetFont(fl_display, fl_gc, fl_current_font->font->fid);
  XDrawString(fl_display, fl_window, fl_gc, x, y, str, n);
}

void fl_draw(const char *str, int x, int y) {
  fl_draw(str, strlen(str), x, y);
}

// end of fl_font.C
