// fl_draw.C

// Implementation of fl_draw(const char*,int,int,int,int,uchar)

// Used to draw all the labels and text, this routine:

// handles '\t' and '\n' to make multiple lines expands all other
// control characters to ^X aligns inside or against the edges of a
// box

// There is a lot of similarity to code in Fl_Input, and these used to
// be the same code.  I split it because the Fl_Input was adding too
// many state variables and arguments to this code.

#include <FL/fl_draw.H>
#include <string.h>

#define MAXBUF 1024

char fl_draw_shortcut;	// set by fl_labeltypes.C
static char* underline_at;

// Copy string p..e to the buffer, replacing characters with ^X and \nnn
// as necessary.  Truncate if necessary so the resulting string and
// null terminator fits in a buffer of size n.  Return pointer to
// start of next line and set n to length of output string.
static const char * expand(const char *p, char *buf, int &n) {
  char *o = buf;
  char *e = buf+(MAXBUF-4);
  underline_at = 0;
  while (o<e) {
    int c = *p++ & 255;
    if (c < ' ' || c == 127) {
      if (!c || c=='\n') {p--; break;}
      if (c == '\t') {
	for (c = (o-buf)%8; c<8 && o<e; c++) *o++ = ' ';
      } else {
	*o++ = '^';
	*o++ = c ^ 0x40;
      }
    } else if (c >= 128 && c < 0xA0) {
      *o++ = '\\';
      *o++ = (c>>6)+'0';
      *o++ = ((c>>3)&7)+'0';
      *o++ = (c&7)+'0';
    } else if (c == '&' && fl_draw_shortcut && *p) {
      if (*p == '&') {*o++ = c; p++;}
      else underline_at = o;
    } else {
      *o++ = c;
    }
  }
  *o = 0;
  n = o-buf;
  return p;
}

void fl_draw(
    const char *str,	// the (multi-line) string
    int x, int y, int w, int h,	// bounding box
    uchar align,
    void (*callthis)(const char *,int,int,int)
) {
  if (!str || !*str) return;

  // add some left/right border to make them look nicer:
  if (w > 11) {w -= 6; x += 3;}

  const char *p,*e;
  char buf[MAXBUF];
  int buflen;

  /* count how many lines and put the last one into the buffer: */
  int lines;
  for (p=str,lines=0; ;) {
    e = expand(p,buf,buflen);
    lines++;
    if (!*e) break;
    if (*e == '\n') e++;
    p = e;
  }

  /* figure out vertical position of the first line: */
  int ypos;
  int height = fl_height();
  if (align & FL_ALIGN_BOTTOM) ypos = y+h-(lines-1)*height;
  else if (align & FL_ALIGN_TOP) ypos = y+height;
  else ypos = y+(h-lines*height)/2+height;

  /* now draw all the lines */
  int desc = fl_descent();
  for (p=str; ; ypos += height) {
    if (lines>1) e = expand(p,buf,buflen);

    int xpos;
    if (align & FL_ALIGN_LEFT) xpos = x;
    else if (align & FL_ALIGN_RIGHT) xpos = x+w-int(fl_width(buf)+.5);
    else xpos = x+int((w-fl_width(buf))/2);

    callthis(buf,buflen,xpos,ypos-desc);

    if (underline_at)
      callthis("_",1,xpos+int(fl_width(buf,underline_at-buf)),ypos-desc);

    if (!*e) break;
    if (*e == '\n') e++;
    p = e;
  }
}

void fl_draw(
  const char *str,	// the (multi-line) string
  int x, int y, int w, int h,	// bounding box
  uchar align) {
  fl_draw(str, x, y, w, h, align, fl_draw);
}
