// fl_color.C

// The fl "colormap".  This allows ui colors to be stored in 8-bit
// locations, and provides a level of indirection so that global color
// changes can be made.  Not to be confused with the X colormap, which
// I try to hide completely.

// SGI compiler seems to have problems with unsigned char arguments
// being used to index arrays.  So I always copy them to an integer
// before use.

#include <config.h>
#include <FL/Fl.H>
#include <FL/x.H>
#include <FL/fl_draw.H>

// the FL "colormap" so UI colors can be stored in 8 bits:
static unsigned fl_cmap[256] = {
#include "fl_cmap.h" // this is a file produced by "cmap.C":
};

// Translations to x pixel numbers:
Fl_XMap fl_xmap[256];

static void map(int i /* uchar does not work, bug in Irix compiler? */) {
#if HAVE_XCOLORMAP
  Fl_XMap &xmap = fl_xmap[i];
  unsigned c = fl_cmap[i];
  XColor xcol;
  xcol.red=(c>>16)&0xFF00; xcol.green=(c>>8)&0xFF00; xcol.blue=c&0xFF00;

  static XColor *allcolors;
  static int numcolors;
  if (!numcolors && XAllocColor(fl_display, fl_colormap, &xcol)) {
    xmap.mapped = 1;
    xmap.r = xcol.red>>8;
    xmap.g = xcol.green>>8;
    xmap.b = xcol.blue>>8;
    xmap.pixel = xcol.pixel;
  } else {
    for (;;) {
      // query whole colormap if not yet done:
      if (!numcolors) {
	numcolors = fl_visual->colormap_size;
	if (!allcolors) allcolors = new XColor[numcolors];
	for (int p = numcolors; p--;) allcolors[p].pixel = p;
	XQueryColors(fl_display, fl_colormap, allcolors, numcolors);
      }
      // find least-squares match:
      int mindist = 0x7FFFFFFF;
      unsigned int bestmatch = 0;
      for (int p=numcolors; p--;) {
	XColor &a = allcolors[p];
	int d, t;
	t = int(xcol.red>>8)-int(a.red>>8); d = t*t;
	t = int(xcol.green>>8)-int(a.green>>8); d += t*t;
	t = int(xcol.blue>>8)-int(a.blue>>8); d += t*t;
	if (d < mindist) {bestmatch = p; mindist = d;}
      }
      if (XAllocColor(fl_display, fl_colormap, allcolors+bestmatch)) {
	XColor &p = allcolors[bestmatch];
	xmap.mapped = 2;
	xmap.r = p.red>>8;
	xmap.g = p.green>>8;
	xmap.b = p.blue>>8;
	xmap.pixel = p.pixel;
	if (p.pixel != bestmatch) {
	  // this indicates that something has happened to
	  // colormap and we better reread it...
	  numcolors = 0;
	}
	break;
      } else {
	// it may fail (unlikely) if another client has managed to
	// free & reallocate that cell, so try again:
	numcolors = 0;
      }
    }
  }
#else
  Fl_XMap &xmap = fl_xmap[i];
  unsigned c = fl_cmap[i];
  xmap.r = c>>24;
  xmap.g = c>>16;
  xmap.b = c>>8;
  xmap.pixel = fl_xpixel(xmap.r,xmap.g,xmap.b);
  xmap.mapped = 1;
#endif
}

#if HAVE_OVERLAY
ulong fl_alloc_color(Colormap cm, uchar ii) {
  int i = ii;
  unsigned c = fl_cmap[i];
  XColor xcol;
  xcol.red=(c>>16)&0xFF00; xcol.green=(c>>8)&0xFF00; xcol.blue=c&0xFF00;
  XAllocColor(fl_display, cm, &xcol);
  return xcol.pixel;
}
uchar fl_overlay;
#endif

ulong fl_xpixel(uchar ii) {
  int i = ii;
#if HAVE_OVERLAY
  if (fl_overlay) {
    switch (i) {
    case FL_RED: return 1;
    case FL_BLACK: return 2;
    case FL_WHITE: return 3;
    default:
      if (i < FL_GRAY_RAMP) return 1;
      else if (i < FL_DARK1) return 2;
      else if (i > FL_LIGHT1) return 3;
      else return 1;
    }
  }
#endif
  if (!fl_xmap[i].mapped) map(i);
  return fl_xmap[i].pixel;
}

void fl_color(uchar i) {
  XSetForeground(fl_display, fl_gc, fl_xpixel(i));
}

uchar Fl::set_color(uchar ii, unsigned c) {
  int i = ii;
  if (fl_xmap[i].mapped && fl_cmap[i] != c) {
#if HAVE_XCOLORMAP
    XFreeColors(fl_display, fl_colormap, &(fl_xmap[i].pixel), 1, 0);
#endif
    fl_xmap[i].mapped = 0;
  }
  fl_cmap[i] = c;
  return i;
}

static uchar free_color = FL_FREE_COLOR;
uchar Fl::set_color(unsigned c) {return Fl::set_color(free_color++, c);}

unsigned Fl::get_color(uchar ii) {
  int i = ii;
  return fl_cmap[i];
}

uchar Fl::set_color(uchar i, uchar red, uchar green, uchar blue) {
  return Fl::set_color(i,
	((unsigned)red<<24)+((unsigned)green<<16)+((unsigned)blue<<8));
}

uchar Fl::set_color(uchar red, uchar green, uchar blue) {
  return Fl::set_color(
	((unsigned)red<<24)+((unsigned)green<<16)+((unsigned)blue<<8));
}

void Fl::get_color(uchar ii, uchar &red, uchar &green, uchar &blue) {
  int i = ii;
  unsigned c = fl_cmap[i];
  red   = uchar(c>>24);
  green = uchar(c>>16);
  blue  = uchar(c>>8);
}
