// fl_boxtype.c

// Box drawing code for the common box types and the table of
// boxtypes.  Other box types are in seperate files so they are not
// linked in if not used.

#include <FL/Fl.H>
#include <FL/Fl_Object.H>
#include <FL/fl_draw.H>

////////////////////////////////////////////////////////////////

void fl_no_box(int, int, int, int, uchar) {}

void fl_thin_down_frame(int x, int y, int w, int h, uchar) {
  if (w >= 1 && h >= 1) {
    fl_color(FL_DARK3); fl_yxline(x, y+h-1, y, x+w-1);
    if (w >= 2 && h >= 2)
      fl_color(FL_LIGHT3); fl_xyline(x+1, y+h-1, x+w-1, y+1);
  }
}

void fl_thin_down_box(int x, int y, int w, int h, uchar c) {
  fl_thin_down_frame(x,y,w,h,c);
  fl_color(c); fl_rectf(x+1, y+1, w-2, h-2);
}

void fl_thin_up_frame(int x, int y, int w, int h, uchar) {
  if (w >= 1 && h >= 1) {
    fl_color(FL_LIGHT3); fl_yxline(x, y+h-1, y, x+w-1);
    if (w >= 2 && h >= 2)
      fl_color(FL_DARK3); fl_xyline(x+1, y+h-1, x+w-1, y+1);
  }
}

void fl_thin_up_box(int x, int y, int w, int h, uchar c) {
  fl_thin_up_frame(x,y,w,h,c);
  fl_color(c); fl_rectf(x+1, y+1, w-2, h-2);
}

void fl_up_frame(int x, int y, int w, int h, uchar) {
  if (w >= 3 && h >= 4) {
    fl_color(FL_LIGHT3); fl_xyline(x+1, y+1, x+w-2);
    fl_color(FL_LIGHT2); fl_yxline(x+1, y+h-2, y+2, x+w-2);
    fl_color(FL_LIGHT1); fl_yxline(x+2, y+h-3, y+3, x+w-3);
    fl_color(FL_DARK1); fl_xyline(x+3, y+h-4, x+w-3, y+3);
    fl_color(FL_DARK2); fl_xyline(x+2, y+h-3, x+w-2, y+2);
    fl_color(FL_DARK3); fl_xyline(x+1, y+h-2, x+w-2);
  }
  fl_color(FL_GRAY0); fl_rect(x,y,w,h);
}

void fl_up_box(int x, int y, int w, int h, uchar c) {
  fl_up_frame(x,y,w,h,c);
  fl_color(c); fl_rectf(x+3,y+4,w-6,h-8);
}

void fl_down_frame(int x, int y, int w, int h, uchar) {
  if (w >= 3 && h >= 4) {
    fl_color(FL_DARK1); fl_xyline(x+0, y, x+w-2);
    fl_color(FL_DARK2); fl_yxline(x+0, y+h-1, y+1, x+w-2);
    fl_color(FL_DARK3); fl_yxline(x+1, y+h-2, y+2, x+w-3);
    fl_color(FL_LIGHT3); fl_xyline(x+2, y+h-3, x+w-2, y+2);
    fl_color(FL_LIGHT2); fl_xyline(x+1, y+h-2, x+w-1, y);
    fl_color(FL_LIGHT1); fl_xyline(x+0, y+h-1, x+w-1);
    fl_color(FL_GRAY0); fl_rect(x+2, y+3, w-4, h-6);
  }
}

void fl_down_box(int x, int y, int w, int h, uchar c) {
  fl_down_frame(x,y,w,h,c);
  fl_color(c); fl_rectf(x+3, y+4, w-6, h-8);
}

void fl_engraved_frame(int x, int y, int w, int h, uchar) {
  fl_color(FL_LIGHT3); fl_rect(x+1, y+1, w-1, h-1);
  fl_color(FL_DARK3);
  fl_yxline(x, y+h-2, y, x+w-2);
  fl_xyline(x+2, y+h-2, x+w-2, y+2);
}

void fl_engraved_box(int x, int y, int w, int h, uchar c) {
  fl_engraved_frame(x,y,w,h,c);
  fl_color(c); fl_rectf(x+2, y+2, w-4, h-4);
}

void fl_embossed_frame(int x, int y, int w, int h, uchar) {
  fl_color(FL_DARK3); fl_rect(x+1, y+1, w-1, h-1);
  fl_color(FL_LIGHT3);
  fl_yxline(x, y+h-1, y, x+w-1);
  fl_xyline(x+2, y+h-2, x+w-2, y+2);
}

void fl_embossed_box(int x, int y, int w, int h, uchar c) {
  fl_embossed_frame(x,y,w,h,c);
  fl_color(c); fl_rectf(x+2, y+2, w-4, h-4);
}

void fl_rectbound(int x, int y, int w, int h, uchar bgcolor) {
//   if (w<0) {x += w; w = -w;}
//   if (h<0) {y += h; h = -h;}
//   if (!w) w = 1;
//   if (!h) h = 1;
  fl_color(bgcolor); fl_rectf(x+1, y+1, w-2, h-2);
  fl_color(FL_BLACK); fl_rect(x, y, w, h);
}

void fl_rectf(int x, int y, int w, int h, uchar c) {
  fl_color(c);
  fl_rectf(x, y, w, h);
}

void fl_border_frame(int x, int y, int w, int h, uchar c) {
  fl_color(c);
  fl_rect(x, y, w, h);
}

////////////////////////////////////////////////////////////////

struct {
  Fl_BoxFunction *f;
  uchar dx, dy, dw, dh;
} table[] = {
// must match list in Enumerations.H!!!
  {fl_no_box,		0,0,0,0},		
  {fl_rectf,		0,0,0,0}, // FL_FLAT_BOX
  {fl_up_box,		3,4,6,8},
  {fl_down_box,		3,4,6,8},
  {fl_up_frame,		3,4,6,8},
  {fl_down_frame,	3,4,6,8},
  {fl_thin_up_box,	1,1,2,2},
  {fl_thin_down_box,	1,1,2,2},
  {fl_thin_up_frame,	1,1,2,2},
  {fl_thin_down_frame,	1,1,2,2},
  {fl_engraved_box,	2,2,4,4},
  {fl_embossed_box,	2,2,4,4},
  {fl_engraved_frame,	2,2,4,4},
  {fl_embossed_frame,	2,2,4,4},
  {fl_rectbound,	1,1,2,2}, // FL_BORDER_BOX, 
  {fl_no_box,		1,1,2,2}, // _FL_SHADOW_BOX,
  {fl_border_frame,	1,1,2,2},
  {fl_no_box,		1,1,2,2}, // _FL_SHADOW_FRAME,
  {fl_no_box,		4,4,8,8}, // _FL_ROUNDED_BOX,
  {fl_no_box,		4,4,8,8}, // _FL_RSHADOW_BOX,
  {fl_no_box,		4,4,8,8}, // _FL_ROUNDED_FRAME
  {fl_no_box,		4,4,8,8}, // _FL_RFLAT_BOX,
  {fl_no_box,		8,4,16,8}, // _FL_ROUND_UP_BOX
  {fl_no_box,		8,4,16,8}, // _FL_ROUND_DOWN_BOX,
  {fl_no_box,		0,0,0,0}, // _FL_DIAMOND_UP_BOX
  {fl_no_box,		0,0,0,0}, // _FL_DIAMOND_DOWN_BOX
  {fl_no_box,		4,4,8,8}, // _FL_OVAL_BOX,
  {fl_no_box,		4,4,8,8}, // _FL_OVAL_SHADOW_BOX,
  {fl_no_box,		4,4,8,8}, // _FL_OVAL_FRAME
  {fl_no_box,		4,4,8,8}, // _FL_OVAL_FLAT_BOX,
  // 8 slots for user-defined extras:
  {fl_no_box},
  {fl_no_box},
  {fl_no_box},
  {fl_no_box},
  {fl_no_box},
  {fl_no_box},
  {fl_no_box},
  {fl_no_box}
};

Fl_BoxFunction* Fl::get_boxtype(uchar t) {return table[t].f;}

int Fl::box_dx(uchar t) {return table[t].dx;}
int Fl::box_dy(uchar t) {return table[t].dy;}
int Fl::box_dw(uchar t) {return table[t].dw;}
int Fl::box_dh(uchar t) {return table[t].dh;}

uchar Fl::set_boxtype(uchar t, Fl_BoxFunction* f) {table[t].f=f; return t;}

static uchar free_boxtype = FL_FREE_BOXTYPE;
uchar Fl::set_boxtype(Fl_BoxFunction* f){return set_boxtype(free_boxtype++,f);}

uchar Fl::set_boxtype(uchar t, uchar f) {table[t] = table[f]; return t;}

void Fl::set_box_deltas(uchar t, uchar a, uchar b, uchar c, uchar d) {
  table[t].dx = a;
  table[t].dy = b;
  table[t].dw = c;
  table[t].dh = d;
}

void fl_draw_box(uchar t,int x,int y,int w,int h,uchar c) {
  if (t) table[t].f(x,y,w,h,c);
}

extern Fl_Object *fl_boxcheat; // hack set by Fl_Window.C

void Fl_Object::draw_box() const {
  int t = box_;
  if (!t) return;
  if (this == fl_boxcheat) {
    fl_boxcheat = 0;
    if (t == FL_FLAT_BOX) return;
    t += 2; // convert box to frame
  }
  table[t].f(x_,y_,w_,h_,color_);
}

void Fl_Object::draw_box(uchar b,uchar c) const {
  if (b) table[b].f(x_,y_,w_,h_,c);
}
