// Fl_arg.C

// OPTIONAL initialization code for a program using FL.
// You do not need to call this!  Feel free to make up your own switches.

#include <FL/Fl.H>
#include <FL/x.H>
#include <FL/Fl_Window.H>
#include <FL/filename.H>
#include <FL/fl_draw.H>
#include <ctype.h>
#include <string.h>

static int match(const char *a, const char *match, int atleast = 1) {
  const char *b = match;
  while (*a && (*a == *b || tolower(*a) == *b)) {a++; b++;}
  return !*a && b >= match+atleast;
}

// flags set by previously parsed arguments:
static char iconic;
static char arg_called;
static char return_i;
static const char *name;
static const char *fg;
static const char *bg;
static const char *bg2;
static const char *geometry;
static const char *title;

// consume a switch from argv.  Returns number of words eaten, 0 on error:
int Fl::arg(int argc, char **argv, int &i) {
  arg_called = 1;
  const char *s = argv[i];

  if (!s) return 1;	// something removed by calling program?

  // a word that does not start with '-', or a word after a '--', or
  // the word '-' by itself all start the "non-switch arguments" to
  // a program.  Return 0 to indicate that we don't understand the
  // word, but set a flag (return_i) so that args() will return at
  // that point:
  if (s[0] != '-' || !s[1]) {return_i = 1; return 0;}
  if (s[1] == '-') {
    if (!s[2]) {i++; return_i = 1; return 0;}
    s++; // make "--word" work the same as "-word" for Gnu compatability
  }
  s++; // point after the dash

  if (match(s, "iconic")) {
      iconic = 1;
      i++;
      return 1;
  }

  const char *v = argv[i+1];
  if (i >= argc-1 || !v)
      return 0;	// all the rest need an argument, so if missing it is an error

  if (match(s, "display")) {

    Fl::display(v);

  } else if (match(s, "geometry")) {

    int flags, gx, gy; unsigned int gw, gh;
    flags = XParseGeometry(v, &gx, &gy, &gw, &gh);
    if (!flags) return 0;
    geometry = v;

  } else if (match(s, "title")) {
    title = v;

  } else if (match(s, "name")) {
    name = v;

  } else if (match(s, "bg2", 3) || match(s, "background2", 11)) {
    bg2 = v;

  } else if (match(s, "bg") || match(s, "background")) {
    bg = v;

  } else if (match(s, "fg") || match(s, "foreground")) {
    fg = v;

  } else return 0; // unrecognized

  i += 2;
  return 2;
}

// consume all switches from argv.  Returns number of words eaten.
// Returns zero on error.  'i' will either point at first word that
// does not start with '-', at the error word, or after a '--', or at
// argc.  If your program does not take any word arguments you can
// report an error if i < argc.

int Fl::args(int argc, char** argv, int& i, int (*cb)(int,char**,int&)) {
  arg_called = 1;
  if (!i) i = 1; // skip argv[0]
  while (i < argc) {
    if (cb && cb(argc,argv,i)) continue;
    if (!arg(argc,argv,i)) return return_i ? i : 0;
  }
  return i;
}

static void parsecolor(const char *name, void (*func)(uchar,uchar,uchar)) {
  XColor x;
  if (!name) return;
  if (!XParseColor(fl_display, fl_colormap, name, &x))
    Fl::abort("Unknown color: %s", name);
  func(x.red>>8, x.green>>8, x.blue>>8);
}

// show a main window, use any parsed arguments
void Fl_Window::show(int argc, char **argv) {
  if (!arg_called) Fl::args(argc,argv);

  fl_open_display(); // this may call Fl::abort() if display name wrong

  // set colors first, so background_pixel is correct:
  parsecolor(fg,Fl::foreground);
  parsecolor(bg,Fl::background);
  parsecolor(bg2,Fl::background2);
  fg = bg = bg2 = 0; // don't do it again

  if (geometry) {
    int flags = 0, gx = x(), gy = y(); unsigned int gw = w(), gh = h();
    flags = XParseGeometry(geometry, &gx, &gy, &gw, &gh);
    if (flags & XNegative) gx = Fl::w()-w()+gx;
    if (flags & YNegative) gy = Fl::h()-h()+gy;
//  int mw,mh; minsize(mw,mh);
//  if (mw > gw) gw = mw;
//  if (mh > gh) gh = mh;
    Fl_Object *r = resizable();
    if (!r) resizable(this);
    if (flags & (XValue | YValue))
      x(-1), resize(gx,gy,gw,gh);
    else
      size(gw,gh);
    resizable(r);
    geometry = 0;
  }

  if (name) {xclass(name); name = 0;}
  else if (!xclass()) xclass(filename_name(argv[0]));
  if (title) {label(title); title = 0;}
  else if (!label()) label(xclass());
  show();

  // set the command string, used by state-saving window managers:
  char *p; const char *q; char buffer[2048];
  p = buffer;
  for (int i=0; i<argc; i++) for (q = argv[i]; (*p++ = *q++););
  XChangeProperty(fl_display, xid(), XA_WM_COMMAND, XA_STRING, 8, 0,
		  (unsigned char *)buffer, p-buffer-1);

  if (iconic) {iconize(); iconic = 0;}
}

// Calls useful for simple demo programs, with automatic help message:

const char * const Fl::help =
" -display host:n.n\n"
" -geometry WxH+X+Y\n"
" -title windowtitle\n"
" -name classname\n"
" -iconic\n"
" -fg color\n"
" -bg color\n"
" -bg2 color";

void Fl::args(int argc, char **argv) {
  int i = 0;
  if (Fl::args(argc,argv,i) < argc) Fl::abort("options are:\n%s", Fl::help);
}

// end of Fl_Window_arg.C
