// Fl_Window.C

// The FL user interface library version 0.98
// Copyright (C) 1998 Digital Domain

// The Fl_Window is a window in the FL library.
// This file contains the huge amount of crap you have to do
// to correctly make a window appear under an X window manager.

// "subwindows" are identified by having parent() non-zero.  They
// act significantly different than regular windows.  Search for the
// differences by searching for parent().

#include <FL/Fl.H>
#include <FL/x.H>
#include <FL/Fl_Window.H>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>

void Fl_Window::_Fl_Window() {
  type(FL_WINDOW);
  box(FL_FLAT_BOX);
  xid_ = 0;
  xclass_ = 0;
  labeltype(FL_NO_LABEL);
  resizable(0);
  nextwindow_ = 0;
  set_flag(FL_BORDER|INVISIBLE);
  label(label());
  xjunk = 0;
}

Fl_Window::Fl_Window(int X,int Y,int W, int H, const char *l)
: Fl_Group(X, Y, W, H, l) {
  _Fl_Window();
  set_flag(FL_FORCE_POSITION);
}

Fl_Window::Fl_Window(int W, int H, const char *l)
  : Fl_Group(0, 0, W, H, l) {
  _Fl_Window();
}

// return non-zero if there are other windows mapped:
int Fl_Window::other_windows() {
  for (Fl_Window *t = Fl::first(); t; t = t->nextwindow_)
    if (t!=this && !t->parent()) return 1;
  return 0;
}

extern int fl_eventdx, fl_eventdy;

int Fl_Window::handle(int event) {
  if (parent()) {
    if (!children()) return 0; // speed up subwindows with no children
    int dx = x(); Fl::e_x -= dx; fl_eventdx += dx;
    int dy = y(); Fl::e_y -= dy; fl_eventdy += dy;
    int r = Fl_Group::handle(event);
    Fl::e_x += dx; fl_eventdx -= dx;
    Fl::e_y += dy; fl_eventdy -= dy;
    return r;
  }
  return Fl_Group::handle(event);
}

Fl_Window *Fl_Object::window() const {
  for (Fl_Object *o = parent(); o; o = o->parent())
    if (o->type()>=FL_WINDOW) return (Fl_Window*)o;
  return 0;
}

int Fl_Window::x_root() const {
  Fl_Window *p = window();
  if (p) return p->x_root() + x();
  return x();
}

int Fl_Window::y_root() const {
  Fl_Window *p = window();
  if (p) return p->y_root() + y();
  return y();
}

////////////////////////////////////////////////////////////////
#include <stdio.h>

Fl_Window *fl_resize_bug_fix;
void Fl_Window::resize(int X,int Y,int W,int H) {
  if (shown() && this != fl_resize_bug_fix) {
    // tell X window manager to change window size:
    if (!(flags()&FL_FORCE_POSITION) && X == x() && Y == y())
      XResizeWindow(fl_display, xid(), W, H);
    else if (W != w() || H != h())
      XMoveResizeWindow(fl_display, xid(), X, Y, W, H);
    else
      XMoveWindow(fl_display, xid(), X, Y);
    if (!parent()) return; // wait for response from server
//  XSync(fl_display,0); // this seems to reduce flashing a lot
  } else {
    // This is to avoid echoing resize events back to window manager:
    fl_resize_bug_fix = 0;
  }
  if (X != x() || Y != y()) set_flag(FL_FORCE_POSITION);
  if (W != w() || H != h()) {
    x(0); y(0);
    Fl_Group::resize(0,0,W,H);
//  If we leave the damage bits set, FL will probably redraw the
//  window before it gets the Expose events from X, and then draw
//  the window unnecessarily a second time.  So I turn them off.
//  Unfortunately this will only work if the subsequent expose
//  event is sent for the entire window.  "ForgetGravity" should
//  cause this, but some X implemtations may not do this correctly.
    clear_damage(); // wait for Expose events from X
  }
  x(X); y(Y);
}

////////////////////////////////////////////////////////////////
// The really ugly X part...

// How "show" works under X...

// A subclass must be allowed to use any visual to create the windows
// (otherwise there is no reason to have any subclasses).  Therefore,
// show() is a virtual function.  See the bottom of this file for the
// implementation of Fl_Window's show().

// show() must select a "visual" and a "colormap".  Usually
// an implementation will do this just once and stash the results in
// static variables.  For the colormap, notice that fl_color (and
// thus all the fl drawing routines) assumme they use fl_colormap
// at all times.  For now fl_colormap is always the default colormap
// of the display.  If your window requires it's own colormap, you
// must use your own code to draw into it, or rewrite fl_color to
// recognize your colormap (it does this already for overlays)

// The visual and colormap (through the virtual function
// colormap()) are then passed to make_xid(visual).  This will
// create the X window and add it to the list of displayed windows.
// It also sends a lot of crap to the X server and the window manager,
// and you don't have much control over it (otherwise the function
// would have a hundred arguments!)  As needed I may add controls
// through hack static variables like "fl_override_redirect".

// However, this does not map the window.  Because it is often useful
// to do even more ugly X stuff after the window is created (such
// as forcing it iconized) the map is deferred until the next time
// Fl::flush() is called.  This is done with an ugly flag bit set on
// the window object that Fl::flush() checks.

static const int inactiveEventMask =
ExposureMask|StructureNotifyMask|EnterWindowMask;

static const int XEventMask =
ExposureMask|StructureNotifyMask
|KeyPressMask|FocusChangeMask
|ButtonPressMask|ButtonReleaseMask
|EnterWindowMask|LeaveWindowMask
|PointerMotionMask;

char fl_override_redirect; // hack for menus
int fl_background_pixel = -1; // hack to get X to erase window

// visual is a void* pointer to avoid having Fl_Window.H needing xlib.h:
void Fl_Window::make_xid(void *pvisual) {
  XVisualInfo *visual = (XVisualInfo *)pvisual;

  int root = parent() ? window()->xid() : RootWindow(fl_display,fl_screen);

  XSetWindowAttributes attr;
  int mask = CWBorderPixel|CWColormap|CWEventMask|CWBitGravity;
  attr.event_mask = !parent()&&active() ? XEventMask : inactiveEventMask;
  attr.colormap = colormap();
  attr.border_pixel = 0;
  attr.bit_gravity = 0; // StaticGravity;
  if (fl_override_redirect) {
    attr.override_redirect = 1;
    attr.save_under = 1;
    mask |= CWOverrideRedirect|CWSaveUnder;
  }
  if (fl_background_pixel >= 0) {
    attr.background_pixel = fl_background_pixel;
    mask |= CWBackPixel;
  }
  xid_ = XCreateWindow(fl_display, root,
		       x(), y(), w(), h(),
		       0,		// border width
		       visual->depth, InputOutput,
		       visual->visual,
		       mask, &attr);
  nextwindow_ = Fl::first();
  Fl::first_ = this;

  if (!parent() && !fl_override_redirect) {
    // Communicate all kinds 'o junk to the X Window Manager:

    label(label(),iconlabel());

    // set the WM_PROTOCOLS to WM_DELETE_WINDOW and _WM_QUIT_APP:
    Atom atoms[2];
    atoms[0] = fl_wm_delete_window;
    atoms[1] = fl__wm_quit_app;
    XChangeProperty(fl_display, xid(), fl_wm_protocols,
		    XA_ATOM, 32, 0, (uchar *)atoms, 2);
    sendxjunk();

    char buffer[2048];

    // set the class property, which controls the icon used:
    if (xclass_) {
      char *p; const char *q;
      // truncate on any punctuation, because they break XResource lookup:
      for (p = buffer, q = xclass_; isalnum(*q)||(*q&128);) *p++ = *q++;
      *p++ = 0;
      // create the capitalized version:
      q = buffer;
      *p = toupper(*q++); if (*p++ == 'X') *p++ = toupper(*q++);
      while ((*p++ = *q++));
      XChangeProperty(fl_display, xid(), XA_WM_CLASS, XA_STRING, 8, 0,
		      (unsigned char *)buffer, p-buffer-1);
    }

    // You may need to remove this.  Some window managers crap out
    // when given this:
#if 0
    if (modal() && nextwindow_ && nextwindow_->visible())
      XSetTransientForHint(fl_display, xid(), nextwindow_->xid());
#endif

  }

  XInstallColormap(fl_display, colormap());
  clear_damage(); // wait for X expose events
  if (parent()) {
    XMapRaised(fl_display, xid());
  } else {
    if (modal()) fl_fix_focus(); // grab the focus if modal
    Fl::damage(1); // make sure Fl::flush searches list
    set_flag(FL_MAPRAISED); // make Fl::flush() call mapraised()
  }
}

// Fl::flush() will call this later:

void Fl_Window::mapraised() {
  clear_flag(FL_MAPRAISED);	// turn off the flag.
  if (visible()) return; // broken X servers map some windows w/o asking
  // we must map them in the order show() was called, but the linked list
  // is in backwards order, so recursively search forward for them:
  for (Fl_Window *w = nextwindow_; w; w = w->nextwindow_)
    if (w->flags() & FL_MAPRAISED) {w->mapraised(); break;}
  XMapRaised(fl_display, xid());
  clear_damage(); // wait for X expose events
}

////////////////////////////////////////////////////////////////
// Resize information:

struct XJunk {
  long prop[5]; // see the file /usr/include/X11/Xm/MwmUtil.h
  XSizeHints hints;
};

void Fl_Window::sendxjunk() {
  if (!xid()) return; // it's not mapped yet!
  if (parent()) return; // it's not a window manager window!

  if (!xjunk) { // default, back compatable with earlier versions of FL
    Fl_Object *o = resizable();
    if (o) {	// it is resizable:
      int minw = o->w(); if (minw > 100) minw = 100;
      int minh = o->h(); if (minh > 100) minh = 100;
      o->size_range(minw,minh,0,0);
    } else {	// fixed-size:
      size_range(w(), h(), w(), h());
    }
    return; // because this recursively called here
  }

  XJunk &j = *(XJunk*)(xjunk);
  j.prop[0] = border() ? 1 : 3; // current state of border
  if (flags() & FL_FORCE_POSITION) {
    j.hints.flags |= USPosition;
    j.hints.x = x();
    j.hints.y = y();
  } else
    j.hints.flags &= ~USPosition;

  XSetWMNormalHints(fl_display, xid(), &(j.hints));
  XChangeProperty(fl_display, xid(),
		  fl__motif_wm_hints, fl__motif_wm_hints,
		  32,0,(unsigned char *)(j.prop),5);
}

// propagate size changes up the hierarchy to main window, and set the
// xjunk to the results:
void Fl_Object::size_range(
  int minw, int minh, int maxw, int maxh, int dw, int dh, int aw)
{
  Fl_Object *o = this;
  for (;;) {
    Fl_Window *w = o->window();
    if (!w) break;
    minw += w->w()-o->w();
    maxw += w->w()-o->w();
    minh += w->h()-o->h();
    maxh += w->h()-o->h();
    o = w;
  }
  if (o->type() < FL_WINDOW) return; // oops
  Fl_Window *w = (Fl_Window*)o;
  if (!w->xjunk) w->xjunk = new XJunk;
  XJunk &j = *(XJunk*)(w->xjunk);
  // j.prop[0] = MWM_HINTS_FUNCTIONS {| MWM_HINTS_DECORATIONS if !border()}
  j.prop[1] = 1; // MWM_FUNC_ALL
  j.prop[2] = 0; // no decorations
  if (minw != maxw || minh != maxh) { // resizable
    j.hints.min_width = minw;
    j.hints.min_height = minh;
    j.hints.flags = PMinSize;
    if (maxw >= minw || maxh >= minh) {
      j.hints.max_width = maxw;
      j.hints.max_height = maxh;
      j.hints.flags = PMinSize|PMaxSize;
      // unfortunately we can't set only one maximum size, so we have
      // to guess the other one:
      if (maxw < minw) j.hints.max_width = Fl::w()-10;
      if (maxh < minh) j.hints.max_height = Fl::h()-30;
    }
    if (dw < 1) dw = 1;
    if (dh < 1) dh = 1;
    if (dw>1 || dh>1) {
      j.hints.width_inc = dw;
      j.hints.height_inc = dh;
      j.hints.flags |= PResizeInc;
    }
    if (aw) {
      // stupid X!  It could insist that the corner go on the
      // straight line between min and max...
      j.hints.min_aspect.x = j.hints.max_aspect.x = minw;
      j.hints.min_aspect.y = j.hints.max_aspect.y = minh;
      j.hints.flags |= PAspect;
    }
//  This is removed as it is incompatable with some Forms programs:
//  j.prop[2] = 4; // turn on MWM_DECOR_RESIZEH if !border()
  } else { // not resizable
    j.hints.min_width = j.hints.max_width = minw;
    j.hints.min_height= j.hints.max_height= minh;
    j.hints.flags = PMinSize|PMaxSize;
    j.prop[1] = 3+16;	// MWM_FUNC_ALL | MWM_FUNC_RESIZE | MWM_FUNC_MAXIMIZE
  }
  w->sendxjunk();
}

////////////////////////////////////////////////////////////////

// set a label without using those foul XTextProperty structures:
static inline void
SetStringProperty(Display *d, Window w, Atom prop, const char *value) {
  XChangeProperty(d,w,prop,XA_STRING,8,0,(unsigned char *)value,strlen(value));
}

void Fl_Window::iconlabel(const char *iname) {
  iconlabel_ = iname;
  if (shown() && !parent()) {
    SetStringProperty(fl_display, xid(), XA_WM_ICON_NAME, iname ? iname : "");
  }
}

void Fl_Window::label(const char *name,const char *iname) {
  Fl_Object::label(name);
  if (shown() && !parent()) {
    SetStringProperty(fl_display, xid(), XA_WM_NAME, name ? name : "");
  }
  iconlabel(iname);
}

// returns pointer to the filename, or null if name ends with '/'
const char *filename_name(const char *name) {
  const char *p,*q;
  for (p=q=name; *p;) if (*p++ == '/') q = p;
  return q;
}

// set both the label and iconlabel:
// the assumption here is that the label is a filename for the displayed data
void Fl_Window::label(const char *name) {
  label(name, name ? filename_name(name) : 0);
}

////////////////////////////////////////////////////////////////
// hide() destroys the X window:

void Fl_Window::hide() {
  if (!shown()) return;

  // remove from the list of windows:
  Fl_Window **pp = &Fl::first_;
  for (; *pp != this; pp = &(*pp)->nextwindow_) if (!*pp) return;
  *pp = nextwindow_;

  // recursively remove any subwindows:
  for (Fl_Window *w = Fl::first(); w;) {
    if (w->window() == this) {w->hide(); w = Fl::first();}
    else w = w->nextwindow_;
  }

  XDestroyWindow(fl_display, xid());
  xid_ = 0;
  set_flag(INVISIBLE);
  handle(FL_HIDE);

  // Make sure no events are sent to this window:
  if (contains(Fl::pushed())) Fl::pushed(Fl::first());
  if (this == fl_xmousewin) fl_xmousewin = 0;
  if (this == fl_xfocus) fl_xfocus = 0;
  fl_fix_focus();

  // Try to move the selection to another window:
  if (this == fl_selection_window) {
    fl_selection_window = 0;
    Fl::selection_owner(Fl::selection_owner());
  }
}

Fl_Window::~Fl_Window() {
  hide();
}

////////////////////////////////////////////////////////////////

#include <FL/fl_draw.H>

// meaning of bits in damage():
// 1 = one or more children need redrawing (from Fl_Group)
// 2 = damage box needs redrawing
// 4 = redraw the whole thing

// X expose events call this:
void Fl_Window::expose(int X,int Y,int W,int H) {
  int R = X+W;
  int B = Y+H;
//if (X<=x() && Y<=y() && R>=x()+w() && T>=y()+h()) {redraw(); return;}
  if (damage_box.r>0) {
    if (X < damage_box.x) damage_box.x = X;
    if (Y < damage_box.y) damage_box.y = Y;
    if (R > damage_box.r) damage_box.r = R;
    if (B > damage_box.b) damage_box.b = B;
  } else {
    damage_box.x = X;
    damage_box.y = Y;
    damage_box.r = R;
    damage_box.b = B;
  }
  damage(2);
}

void Fl_Window::draw() {
  int savex = x(); x(0);
  int savey = y(); y(0);
  Fl_Group::draw();
  x(savex);
  y(savey);
}

////////////////////////////////////////////////////////////////
// Implement the virtual functions for the base Fl_Window class:

ulong Fl_Window::colormap() const {return fl_colormap;}

// If the box is a filled rectangle, we can make the redisplay *look*
// faster by using X's background pixel erasing.  We can make it
// actually *be* faster by drawing the frame only, this is done by
// setting fl_boxcheat, which is seen by code in fl_drawbox.C:
Fl_Window *fl_boxcheat;
static inline int can_boxcheat(uchar b) {return (b==1 || (b&2) && b<=15);}

void Fl_Window::show() {
  if (!shown()) {
    fl_open_display();
    if (can_boxcheat(box())) fl_background_pixel = fl_xpixel(color());
    make_xid(fl_visual);
    fl_background_pixel = -1;
  }
  if (!(flags()&FL_MAPRAISED)) XMapRaised(fl_display, xid());
}

// state variables passed to drawing routines:
Fl_Window *Fl_Window::current_;
Window fl_window;
GC fl_gc;

// make X drawing go into this window (called by subclass flush() impl.)
void Fl_Window::make_current(ulong xid) {
  static GC gc;	// the GC used by all X windows
  if (!gc) gc = XCreateGC(fl_display, xid, 0, 0);
  fl_window = xid;
  fl_gc = gc;
  current_ = this;
}

void Fl_Window::flush() {
  if (damage() != 1 && can_boxcheat(box())) {
    fl_boxcheat = this;
    if (damage()!=2) XClearWindow(fl_display,xid());
  }
  make_current(xid());
  if (damage()==2) {
    // notice that this is *not* done if bit 1 is set, indicating
    // a child needs redrawing!
    fl_clip(damage_box.x, damage_box.y,
	    damage_box.r-damage_box.x, damage_box.b-damage_box.y);
    draw();
    fl_pop_clip();
  } else {
    draw();
  }
  clear_damage();
  damage_box.r = 0;
}

//// End of Fl_Window.C
