// Fl_Tabs.C

// This is the "file card tabs" interface to allow you to put lots and lots
// of buttons and switches in a panel, as popularized by many toolkits.

// Each child object is a card, and it's label() is printed on the card tab.
// Clicking the tab makes that card visible.

#include <FL/Fl.H>
#include <FL/Fl_Tabs.H>
#include <FL/fl_draw.H>

#define TABSIZE 30
#define BORDER 10

int Fl_Tabs::tab_right(int xpos, const Fl_Object *o) {
  int r;
  if (o->label()) {
    fl_font(o->labelfont(), o->labelsize());
    r = xpos+int(fl_width(o->label())+10.5);
  } else 
    r = xpos+20;
  int t = 20*(children()-1-find(*o));
  if (r+t > x()+w()) {r = x()+w()-t; if (r < xpos+20) r = xpos+20;}
  return r;
}

// this is used by fluid to pick tabs:
Fl_Object *Fl_Tabs::which(int event_x, int event_y) {
  if (event_y > y()+TABSIZE ||
      event_y < y() ||
      event_x < x()) return 0;
  int X = x();
  Fl_Object*const* a = array();
  for (int i=children(); i--;) {
    Fl_Object* o = *a++;
    X = tab_right(X,o);
    if (event_x < X) return o;
  }
  return 0;
}

int Fl_Tabs::handle(int event) {

  Fl_Object *o;

  switch (event) {

  case FL_PUSH:
    if (Fl::event_y() > y()+TABSIZE) goto DEFAULT;
  case FL_DRAG:
  case FL_RELEASE:
    o = which(Fl::event_x(), Fl::event_y());
    if (event == FL_RELEASE) {push(0); if (o) value(o);}
    else push(o);
    return 1;

  default:
  DEFAULT:
    return Fl_Group::handle(event);

  }
}

int Fl_Tabs::push(Fl_Object *o) {
  if (push_ == o) return 0;
  push_ = o;
  damage(2);
  return 1;
}

int Fl_Tabs::value(Fl_Object *o) {
  if (value_ == o) return 0;
  if (o) o->show();
  if (value_) value_->hide();
  value_ = o;
  redraw();
  return 1;
}

void Fl_Tabs::draw() {
  Fl_Object *v = value_;
  if (!v) {
    // If value() has not been called, make exactly one object visible:
    Fl_Object*const* a = array();
    for (int i=children(); i--;) {
      Fl_Object* o = *a++;
      if (v) o->hide();
      else if (o->visible()) v = o;
    }
    value_ = v;
  }
  if (damage()&~3) { // redraw the entire thing:
    fl_draw_box(box(),x(),y()+TABSIZE,w(),h()-TABSIZE,v?v->color():color());
    if (v) v->redraw_child();
  }
  if (damage()&1) { // redraw the child
    if (v) v->draw_child();
  }
  if (damage()&2) {
    int vx1 = 0;
    int vx2 = 0;
    int X = x();
    fl_clip(x(),y(),w(),TABSIZE);
    Fl_Object*const* a = array();
    for (int i=children(); i--;) {
      Fl_Object* o = *a++;
      int x1 = X;
      X = tab_right(X,o);
      if (o == v) {vx1 = x1; vx2 = X; continue;}
      fl_draw_box(o == push_ ? box()+1 : box(),
		  x1, y(), X-x1, TABSIZE+Fl::box_dh(box()), o->color());
      o->draw_outside_label(x1, y(), X-x1, TABSIZE+3, FL_ALIGN_CENTER);
    }
    if (X < x()+w()) {
      fl_color(color());
      fl_rectf(X,y(),x()+w()-X,TABSIZE);
    }
    fl_pop_clip();
    if (v) {
      fl_clip(x(),y(),w(),TABSIZE+Fl::box_dy(box()));
      fl_draw_box(box(),vx1,y(),vx2-vx1,TABSIZE+Fl::box_dh(box()),v->color());
      v->draw_outside_label(vx1,y(),vx2-vx1,TABSIZE+3,FL_ALIGN_CENTER);
      fl_pop_clip();
    }
  }
}

Fl_Tabs::Fl_Tabs(int X,int Y,int W, int H, const char *l) :
  Fl_Group(X,Y,W,H,l) {
    box(FL_UP_BOX);
    value_ = push_ = 0;
}
