// Fl_Slider.C

#include <FL/Fl.H>
#include <FL/Fl_Slider.H>
#include <FL/fl_draw.H>
#include <math.h>

void Fl_Slider::_Fl_Slider() {
  align(FL_ALIGN_BOTTOM);
  when(FL_WHEN_CHANGED);
  min = 0;
  max = 1;
  value_ = 0.5; // this initial value is compatable with Forms
  step_ = 0;
  size_ = 0.08;
  slider_ = 0; // FL_UP_BOX;
  page_ = increment_ = 0;
}

Fl_Slider::Fl_Slider(int x,int y,int w,int h, const char *l)
: Fl_Object(x,y,w,h,l) {
  box(FL_DOWN_BOX);
  _Fl_Slider();
}

Fl_Slider::Fl_Slider(uchar t,int x,int y,int w,int h, const char *l)
: Fl_Object(x,y,w,h,l) {
  type(t);
  box(t==FL_HOR_NICE_SLIDER || t==FL_VERT_NICE_SLIDER ?
      FL_FLAT_BOX : FL_DOWN_BOX);
  _Fl_Slider();
}

int Fl_Slider::value(float v) {
  clear_changed();
  if (value_ != v) {value_ = v; damage(2); return 1;}
  return 0;
}

void Fl_Slider::bounds(float a, float b) {
  if (min != a || max != b) {min = a; max = b; damage(2);}
}

void Fl_Slider::size(float v) {
  if (v <=  0.001) v = 0.001;
  if (v > 1) v = 1;
  if (size_ != v) {size_ = v; damage(2);}
}

int Fl_Slider::value(int p, int w, int t, int l) {
//	p = position, first line displayed
//	w = window, number of lines displayed
//	t = top, number of first line
//	l = length, total number of lines
  step(1);
  if (p+w > t+l) l = p+w-t;
  size(w >= l ? 1.0 : float(w)/float(l));
  bounds(t, l-w+t);
  page(w > increment() ? w-increment() : 0);
  return value(p);
}

// All slider interaction is done as though the slider ranges from
// zero to one, and the left (bottom) edge of the slider is at the
// given position.  Since when the slider is all the way to the
// right (top) the left (bottom) edge is not all the way over, a
// position on the object itself covers a wider range than 0-1,
// actually it ranges from 0 to 1/(1-size).

static void draw_bg(uchar type, uchar box,
		    int x, int y, int w, int h, uchar c, int BW) {
  fl_draw_box(box,x,y,w,h,c);
  if (type == FL_VERT_NICE_SLIDER) {
    fl_color(FL_BLACK);
    fl_rectf(x+w/2-2, y+BW, 4, h-2*BW);
  } else if (type == FL_HOR_NICE_SLIDER) {
    fl_color(FL_BLACK);
    fl_rectf(x+BW, y+h/2-2, w-2*BW, 4);
  }
}

void Fl_Slider::draw(int x, int y, int w, int h) {
  float val;

  if (min == max)
    val = 0.5;
  else {
    val = (value_ - min)/(max - min);
    if (val > 1.0) val = 1.0;
    else if (val < 0.0) val = 0.0;
  }

  int X,S,W;
  int BW = Fl::box_dx(box());
  W = (horizontal() ? w : h) - 2*BW;
  if (type()==FL_HOR_FILL_SLIDER || type() == FL_VERT_FILL_SLIDER) {
    S = int(val*W+.5);
    if (min>max) {S = W-S; X = h-BW-S;}
    else X = BW;
  } else {
    S = int(size_*W+.5); if (S<8) S = 8;
    X = BW+int(val*(W-S)+.5);
  }
  int xsl,ysl,wsl,hsl;
  if (horizontal()) {
    xsl = x+X;
    wsl = S;
    ysl = y+BW;
    hsl = h-2*BW;
  } else {
    ysl = y+X;
    hsl = S;
    xsl = x+BW;
    wsl = w-2*BW;
  }

  if (damage()&128) { // complete redraw
    draw_bg(type(),box(),x,y,w,h,color(),BW);
  } else { // partial redraw, clip off new position of slider
    if (X > BW) {
      if (horizontal()) fl_clip(x,ysl,X,hsl);
      else fl_clip(xsl,y,wsl,X);
      draw_bg(type(),box(),x,y,w,h,color(),BW);
      fl_pop_clip();
    }
    if (X+S < W+BW) {
      if (horizontal()) fl_clip(xsl+wsl,ysl,x+w-BW-xsl-wsl,hsl);
      else fl_clip(xsl,ysl+hsl,wsl,y+h-BW-ysl-hsl);
      draw_bg(type(),box(),x,y,w,h,color(),BW);
      fl_pop_clip();
    }
  }

  uchar box1 = slider_;
  if (!box1) {box1 = box()&-2; if (!box1) box1 = FL_UP_BOX;}
  if (type() == FL_VERT_NICE_SLIDER) {
    fl_draw_box(box1, xsl, ysl, wsl, hsl, FL_GRAY);
    int d = (hsl-4)/2;
    fl_draw_box(FL_THIN_DOWN_BOX,xsl+2, ysl+d, wsl-4, hsl-2*d, color2());
  } else if (type() == FL_HOR_NICE_SLIDER) {
    fl_draw_box(box1, xsl, ysl, wsl, hsl, FL_GRAY);
    int d = (wsl-4)/2;
    fl_draw_box(FL_THIN_DOWN_BOX,xsl+d, ysl+2, wsl-2*d, hsl-4, color2());
  } else {
    if (wsl>0 && hsl>0) fl_draw_box(box1, xsl, ysl, wsl, hsl, color2());
  }

  draw_label(xsl,ysl,wsl,hsl);
}

void Fl_Slider::draw() {
  draw(x(),y(),w(),h());
}

int Fl_Slider::handle(int event, int x, int y, int w, int h) {
  switch (event) {
  case FL_PUSH:
    if (!Fl::event_inside(x,y,w,h)) return 0;
  case FL_DRAG: {
    if (size() >= 1 || min==max) return 1;
    float W, X;
    int BW = Fl::box_dx(box());
    if (horizontal()) {
      W = w-BW;
      X = Fl::event_x()-x-BW;
    } else {
      W = h-BW;
      X = Fl::event_y()-y-BW;
    }
    float val = (value_ - min)/(max - min);
    static float offcenter;
    float v;
    float S = size_*W+.5; if (S<8) S = 8;
    if (type() == FL_HOR_FILL_SLIDER || type() == FL_VERT_FILL_SLIDER) {
      v = X/W;
      if (event == FL_PUSH) {
	offcenter = v-val;
	if (offcenter < -size_/2 || offcenter > size_/2) offcenter = 0;
	else return 1;
      }
    } else {
      v = X/(W-S);
      if (event == FL_PUSH) {
	offcenter = v-val;
	if (offcenter < 0) offcenter = 0;
	else if (offcenter > S/(W-S)) offcenter = S/(W-S);
	else return 1;
      }
    }
    v -= offcenter;
    if (v < 0) {
      offcenter = v+offcenter;
      if (offcenter<0) offcenter=0;
      v = 0;
    } else if (v > 1) {
      offcenter =  v+offcenter-1;
      if (offcenter > S/(W-S)) offcenter = S/(W-S);
      v = 1;
    }
    // if (Fl::event_shift()) v = val+(v-val)*.05;
    v = v*(max-min)+min;
    if (step()) v = int(v/step()+.5)*step();
    if (value_ != v) {
      value_ = v; damage(2);
      if (when() & FL_WHEN_CHANGED) do_callback();
      else set_changed();
    }}
    return 1;
  case FL_RELEASE:
    if (when() & FL_WHEN_RELEASE
	&& (changed() || when()&FL_WHEN_NOT_CHANGED)) {
      clear_changed(); do_callback();}
    return 1;
#if 0
// Allow shortcut keys for arrows and page up/down to move the slider
// This does not work very well as extra arrow keys ignored by other
// objects will unexpectedly move the slider.  Still, it is an interesting
// UI test...
  case FL_SHORTCUT: {
    if (size() >= 1) return 0;
// put this line in to require mouse to point at slider:
//  if (!Fl::event_inside(this)) return 0;
    float increment;
    if (this->increment()) increment = this->increment();
    else if (step()) increment = step();
    else for (increment=1; increment && increment>=fabs(max-min); increment /= 10);
    float page;
    if (this->page_) page = this->page_;
    else {page = fabs(max-min)*(1+size_)*size_;
    if (step()) page = int(page/step()+.5)*step();}
    float v = value_;
    float delta = 0;
    if (horizontal()) {
      switch (Fl::event_key()) {
      case FL_Left: delta = -increment; break;
      case FL_Right: delta = increment; break;
      default: return 0;
      }
    } else {
      switch (Fl::event_key()) {
      case FL_Down: delta = -increment; break;
      case FL_Up: delta = increment; break;
      case FL_Page_Down: delta = -page; break;
      case FL_Page_Up: delta = page; break;
      case FL_Home: v = max; break;
      case FL_End: v = min; break;
      default: return 0;
      }
    }
    if (min<=max) {
      v += delta;
      if (v<min) v = min;
      else if (v>max) v = max;
    } else {
      v -= delta;
      if (v<max) v = max;
      else if (v>min) v = min;
    }
    if (value_ != v) {value_ = v; damage(2); set_changed();}
    else return 0; // delete this to make arrows stop at ends
    goto JUMP;} // jump up to the callback code
#endif
  default:
    return 0;
  }
}

int Fl_Slider::handle(int event) {
  return handle(event,x(),y(),w(),h());
}

// End of Fl_Slider.C
