// Fl_Poll.C

// Written by Bill Spitzak

#include <config.h>
#include <FL/Fl_Poll.H>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#if !HAVE_POLL
#include <unistd.h>
#ifdef __sgi // on IRIX select is broken in C++, fix it:
inline static void bzero(void *b, int l) {memset(b,0,l);}
extern "C" int	select( int, fd_set *, fd_set *, fd_set *, struct timeval * );
#endif
#endif

// you must substitue whatever function returns current time in ticks:
#include <sys/times.h>
static clock_t Clock() {
  struct tms foo;
  return times(&foo);
}
#undef CLK_TCK
#define CLK_TCK 100	// CLK_TCK did not work on IRIX

Fl_Poll fl_poll;

void Fl_Poll::clear() {
  nfds = numtimeouts = numtests = 0;
  idle = 0;
  initclock = 0;
#if !HAVE_POLL
  FD_ZERO(&fdsets[0]);
  FD_ZERO(&fdsets[1]);
  FD_ZERO(&fdsets[2]);
  maxfd = 0;
#endif
}

void Fl_Poll::add(int n, short events, void (*cb)(int, void*), void *v) {
  int i;
  if (nfds < MAXFD) {i = nfds; nfds++;} else {i = MAXFD-1;}
  fds[i].fd = n;
  fds[i].events = events;
#if !HAVE_POLL
  if (events & POLLIN) FD_SET(n, &fdsets[0]);
  if (events & POLLOUT) FD_SET(n, &fdsets[1]);
  if (events & POLLERR) FD_SET(n, &fdsets[2]);
  if (n > maxfd) maxfd = n;
#endif
  fd[i].cb = cb;
  fd[i].arg = v;
}

void Fl_Poll::remove(int n) {
  int i,j;
  for (i=j=0; i<nfds; i++) {
    if (fds[i].fd == n);
    else {if (j<i) {fd[j]=fd[i]; fds[j]=fds[i];} j++;}
  }
  nfds = j;
#if !HAVE_POLL
  FD_CLR(n, &fdsets[0]);
  FD_CLR(n, &fdsets[1]);
  FD_CLR(n, &fdsets[2]);
  if (n == maxfd) maxfd--;
#endif
}

// Timeouts are insert-sorted into order.  This works good if there
// are only a small number:

void Fl_Poll::add_timeout(float t, void (*cb)(void *), void *v) {
  int i;
  if (numtimeouts<MAXTIMEOUT) numtimeouts++;
  for (i=0; i<numtimeouts-1; i++) {
    if (timeout[i].time > t) {
      for (int j=numtimeouts-1; j>i; j--) timeout[j] = timeout[j-1];
      break;
    }
  }
  timeout[i].time = t;
  timeout[i].cb = cb;
  timeout[i].arg = v;
}

void Fl_Poll::remove_timeout(void (*cb)(void *), void *v) {
  int i,j;
  for (i=j=0; i<numtimeouts; i++) {
    if (timeout[i].cb == cb && timeout[i].arg==v) ;
    else {if (j<i) timeout[j]=timeout[i]; j++;}
  }
  numtimeouts = j;
}

void Fl_Poll::add_test(int (*t)(void *), void (*cb)(void *), void *v) {
  int a;
  for (a=0; ; a++) {
    if (a >= numtests) {
      if (numtests < MAXTEST) numtests++; else a--;
      break;
    }
    if (!test[a].test || test[a].test==t && test[a].cb==cb && test[a].arg==v)
      break;
  }
  test[a].test = t;
  test[a].cb = cb;
  test[a].arg = v;
}

void Fl_Poll::remove_test(int (*t)(void *),void (*cb)(void *), void *v) {
  int a;
  for (a=0; ; a++) {
    if (a >= numtests) return;
    if (test[a].test==t && test[a].cb == cb && test[a].arg == v) break;
  }
  numtests--;
  for (; a<numtests; a++) test[a] = test[a+1];
}

////////////////////////////////////////////////////////////////

inline int Fl_Poll::timeout_ready() {
  return (numtimeouts && timeout[0].time<=0);
}

float Fl_Poll::reset() {
  unsigned long newclock = Clock();
  if (!initclock) {prevclock = newclock; initclock = 1; return 0;}
  float t = float(newclock-prevclock)/CLK_TCK;
  prevclock = newclock;
  for (int i=0; i<numtimeouts; i++) timeout[i].time -= t;
  return t;
}

// warning: this only works if timeout_ready() is true:
void Fl_Poll::call_timeouts() {
  struct {
    void (*cb)(void *);
    void *arg;
  } temp[MAXTIMEOUT];
  int i,j,k;
  // copy all expired timeouts to temp array:
  for (i=j=0; j<numtimeouts && timeout[j].time <= 0; i++,j++) {
    temp[i].cb = timeout[j].cb;
    temp[i].arg= timeout[j].arg;
  }
  // remove them from source array:
  for (k=0; j<numtimeouts;) timeout[k++] = timeout[j++];
  numtimeouts = k;
  // and then call them:
  for (k=0; k<i; k++) temp[k].cb(temp[k].arg);
}

// I have attempted to write these to not call reset() unnecessarily,
// since on some systems this results in a time-wasting context
// switch to the system when Clock() is called.  This means that
// only reset() and wait(n) are guaranteed to reset the clock that
// timeouts are measured from.  wait() and check() and ready() may
// or may not change the clock.

// Kludgey macros to get info out of local variables and do callbacks:
#if HAVE_POLL
#define do_callbacks() { \
  for (int i=0;i<nfds; i++) if (fds[i].revents) \
    fd[i].cb(fds[i].fd, fd[i].arg); \
}
#else
#define do_callbacks() { \
  for (int i=0; i<nfds; i++) { \
    int f = fds[i].fd; \
    short revents = 0; \
    if (FD_ISSET(f,&fdt[0])) revents |= POLLIN; \
    if (FD_ISSET(f,&fdt[1])) revents |= POLLOUT; \
    if (FD_ISSET(f,&fdt[2])) revents |= POLLERR; \
    if (fds[i].events & revents) fd[i].cb(f, fd[i].arg); \
  } \
}
#endif

int Fl_Poll::wait() {
  for (int i=numtests; i--;) if (test[i].test(test[i].arg)) {
    test[i].cb(test[i].arg);
    return 0;
  }
#if !HAVE_POLL
  fd_set fdt[3];
  fdt[0] = fdsets[0];
  fdt[1] = fdsets[1];
  fdt[2] = fdsets[2];
#endif
  if (!numtimeouts && !idle) {
    // do the no-timeout version fast, don't call Clock():
    initclock = 0;
#if HAVE_POLL
    int n = ::poll(fds, nfds, -1);
#else
    int n = ::select(maxfd+1,&fdt[0],&fdt[1],&fdt[2],0);
#endif
    if (n>0) do_callbacks();
    return n;
  } else {
    reset();	// subtract time before the call
#if HAVE_POLL
    int t;	// wait for the next timeout, or 0 if already ready
    if (idle || timeout[0].time <= 0) t = 0;
    else t = int(timeout[0].time*1000);
    int n = ::poll(fds, nfds, t);
#else
    timeval t;	// wait for the next timeout, or 0 if already ready
    if (idle || timeout[0].time <= 0) {
      t.tv_sec = 0;
      t.tv_usec = 0;
    } else {
      t.tv_sec = int(timeout[0].time);
      t.tv_usec = int(1000000 * (timeout[0].time-t.tv_sec));
    }
    int n = ::select(maxfd+1,&fdt[0],&fdt[1],&fdt[2],&t);
#endif
    reset();	// expire timeouts during poll call
    if (n>0) do_callbacks();
    if (timeout_ready()) call_timeouts();
    if (idle) idle();
    return n;
  }
}

float Fl_Poll::wait(float time) {
  time -= reset();
  for (int i=numtests; i--;) if (test[i].test(test[i].arg)) {
    test[i].cb(test[i].arg);
    return time;
  }
#if HAVE_POLL
  int t;
  float t1 = idle ? 0 : timeout[0].time;
  if (!numtimeouts || time < t1) t1 = time;
  if (t1 > 0) t = int(t1*1000);
  else t = 0;
  int n = ::poll(fds, nfds, t);
#else
  timeval t;
  float t1 = idle ? 0 : timeout[0].time;
  if (!numtimeouts || time < t1) t1 = time;
  if (t1 > 0) {
    t.tv_sec = int(t1);
    t.tv_usec = int(1000000 * (t1-t.tv_sec));
  } else {
    t.tv_sec = 0;
    t.tv_usec = 0;
  }
  fd_set fdt[3];
  fdt[0] = fdsets[0];
  fdt[1] = fdsets[1];
  fdt[2] = fdsets[2];
  int n = ::select(maxfd+1,&fdt[0],&fdt[1],&fdt[2],&t);
#endif
  time -= reset();
  if (n>0) do_callbacks();
  if (timeout_ready()) call_timeouts();
  if (idle) idle();
  return time;
}

int Fl_Poll::check() {
  for (int i=numtests; i--;) if (test[i].test(test[i].arg)) {
    test[i].cb(test[i].arg);
    return 0;
  }
#if HAVE_POLL
  int n = ::poll(fds, nfds, 0);
#else
  timeval t;
  t.tv_sec = 0;
  t.tv_usec = 0;
  fd_set fdt[3];
  fdt[0] = fdsets[0];
  fdt[1] = fdsets[1];
  fdt[2] = fdsets[2];
  int n = ::select(maxfd+1,&fdt[0],&fdt[1],&fdt[2],&t);
#endif
  if (numtimeouts) reset();
  if (n>0) do_callbacks();
  if (timeout_ready()) call_timeouts();
//if (idle) idle(); // ??? should it do this ???
  return n;
}

int Fl_Poll::ready() {
  for (int i=numtests; i--;) if (test[i].test(test[i].arg)) return 1;
  if (numtimeouts) {reset(); if (timeout_ready()) return 1;}
//if (idle) return 1; // ??? should it do this ???
#if HAVE_POLL
  return ::poll(fds, nfds, 0);
#else
  timeval t;
  t.tv_sec = 0;
  t.tv_usec = 0;
  fd_set fdt[3];
  fdt[0] = fdsets[0];
  fdt[1] = fdsets[1];
  fdt[2] = fdsets[2];
  return ::select(maxfd+1,&fdt[0],&fdt[1],&fdt[2],&t);
#endif
}
