// Fl_Overlay_Window.C

// A window using double-buffering and able to draw an overlay
// on top of that.  Uses the hardware to draw the overlay if
// possible, otherwise it just draws in the front buffer.

#include <config.h>
#include <FL/Fl.H>
#include <FL/Fl_Overlay_Window.H>
#include <FL/fl_draw.H>
#include <FL/x.H>

void Fl_Overlay_Window::show() {
  Fl_Double_Window::show();
  if (overlay_ && overlay_ != this) overlay_->show();
}

void Fl_Overlay_Window::hide() {
  Fl_Double_Window::hide();
}

void Fl_Overlay_Window::flush() {
  if (overlay_ != this) {
    Fl_Double_Window::flush();
    return;
  }

  if (damage()&4) { // overlay damaged:
    // force double window to update everywhere, thus erasing overlay:
    damage_box.x = damage_box.y = 0;
    damage_box.r = w(); damage_box.b = h();
    clear_damage(damage()|2);
  }

  // draw background, which may erase parts of overlay:
  Fl_Double_Window::flush();

  // and then draw the overlay:
  fl_window = xid();
  draw_overlay();
}

void Fl_Overlay_Window::resize(int X, int Y, int W, int H) {
  Fl_Double_Window::resize(X,Y,W,H);
  if (overlay_ && overlay_!=this) overlay_->resize(0,0,w(),h());
}

Fl_Overlay_Window::~Fl_Overlay_Window() {
  hide();
//  delete overlay; this is done by ~Fl_Group
}

#if !HAVE_OVERLAY

int Fl_Overlay_Window::can_do_overlay() {return 0;}

void Fl_Overlay_Window::redraw_overlay() {overlay_ = this; damage(4);}

#else

XVisualInfo *fl_overlay_visual(int);

class _Fl_Overlay : public Fl_Window {
  friend class Fl_Overlay_Window;
  void flush();
  void show();
  ulong colormap() const;
public:
  _Fl_Overlay(int x, int y, int w, int h) :
    Fl_Window(x,y,w,h) {}
};

static XVisualInfo *vis;
static Colormap colormap_;
static int tried;

int Fl_Overlay_Window::can_do_overlay() {
  if (!tried) {
    vis = fl_overlay_visual(1);
    if (!vis) vis = fl_overlay_visual(0);
    tried = 1;
  }
  return vis != 0;
}

ulong _Fl_Overlay::colormap() const {return colormap_;}

void _Fl_Overlay::show() {
  if (shown()) {Fl_Window::show(); return;}
  if (!colormap_)
    colormap_ =
      XCreateColormap(fl_display, RootWindow(fl_display,fl_screen),
		      vis->visual, AllocNone);
  fl_background_pixel = 0; // clear
  make_xid(vis);
  fl_background_pixel = -1;
}

static GC gc;	// gc shared by all overlays

void _Fl_Overlay::flush() {
  fl_window = xid();
  if (!gc) gc = XCreateGC(fl_display, xid(), 0, 0);
  fl_gc = gc;
  // set the colors *after* the -bg switches have been parsed:
  if (tried == 1) {
    fl_alloc_color(colormap_,FL_RED);
    fl_alloc_color(colormap_,FL_BLACK);
    fl_alloc_color(colormap_,FL_WHITE);
    tried = 2;
  }
  fl_overlay = 1;
  Fl_Overlay_Window *w = (Fl_Overlay_Window *)parent();
  if (damage()==2) {
    fl_clip(damage_box.x, damage_box.y,
	    damage_box.r-damage_box.x, damage_box.b-damage_box.y);
    w->draw_overlay();
    fl_pop_clip();
  } else {
    XClearWindow(fl_display,xid());
    w->draw_overlay();
  }
  clear_damage();
  damage_box.r = 0;
  fl_overlay = 0;
}

void Fl_Overlay_Window::redraw_overlay() {
  if (!overlay_) {
    if (can_do_overlay()) {
      overlay_ = new _Fl_Overlay(0,0,w(),h());
      add(*overlay_);
    } else {
      overlay_ = this;	// fake the overlay
    }
  }
  if (shown()) {
    if (overlay_ == this)
      damage(4);
    else if (!overlay_->shown())
      overlay_->show();
    else
      overlay_->redraw();
  }
}

#endif

// End of Fl_Overlay_Window.C
