// Fl_Object.C

// The FL user interface library version 0.98
// Copyright (C) 1998 Digital Domain

#include <FL/Fl.H>
#include <FL/x.H>
#include <FL/Fl_Object.H>
#include <FL/Fl_Group.H>

////////////////////////////////////////////////////////////////
// for compatability with Forms, all objects without callbacks are
// inserted into a "queue" when they are activated, and the forms
// compatability interaction functions (fl_do_events, etc) will
// read one object at a time from this queue and return it:

const int QUEUE_SIZE = 20;

static Fl_Object *obj_queue[QUEUE_SIZE];
static int obj_head, obj_tail;

void Fl_Object::default_callback(Fl_Object *o, void *v) {
  for (Fl_Object *p = o->parent(); p; p = p->parent())
    if (p->callback() != default_callback) {
      p->do_callback(o,v);
      return;
    }
  obj_queue[obj_head++] = o;
  if (obj_head >= QUEUE_SIZE) obj_head = 0;
  if (obj_head == obj_tail) {
    obj_tail++;
    if (obj_tail >= QUEUE_SIZE) obj_tail = 0;
  }
}

Fl_Object *Fl::readqueue() {
  if (obj_tail==obj_head) return 0;
  Fl_Object *o = obj_queue[obj_tail++];
  if (obj_tail >= QUEUE_SIZE) obj_tail = 0;
  return o;
}
    
////////////////////////////////////////////////////////////////

int Fl_Object::handle(int) {return 0;}

Fl_Object::Fl_Object(int X, int Y, int W, int H, const char* L) {

  x_ = X; y_ = Y; w_ = W; h_ = H;

  label_.value	= L;
  label_.type	= FL_NORMAL_LABEL;
  label_.font	= FL_HELVETICA;
  label_.size	= FL_NORMAL_SIZE;
  label_.color	= FL_BLACK;
  callback_	= default_callback;
  user_data_ 	= 0;
  type_		= 0;
  flags_	= 0;
  damage_	= 0;
  box_		= FL_NO_BOX;
  color_	= FL_GRAY;
  color2_	= FL_GRAY;
  align_	= FL_ALIGN_CENTER;
  when_		= FL_WHEN_RELEASE;

  parent_ = 0;
  if (Fl_Group::current()) Fl_Group::current()->add(this);
}

void Fl_Object::resize(int X, int Y, int W, int H) {
  x_ = X;
  y_ = Y;
  w_ = W;
  h_ = H;
  for (Fl_Object *p = parent(); p; p = p->parent())
    if (p->box() || !p->parent()) {p->redraw(); break;}
}

int Fl_Object::take_focus() {
  if (!activevisible()) return 0;
  if (!handle(FL_FOCUS)) return 0; // see if it wants it
  if (contains(Fl::focus())) return 1; // it called Fl::focus for us
  Fl::focus(this);
  return 1;
}

extern Fl_Object *fl_selection_requestor; // object that did paste()

// get rid of pointers to this object:
static void throw_focus(Fl_Object *o) {
  if (o->contains(Fl::pushed())) Fl::pushed(0);
  if (o->contains(Fl::selection_owner())) Fl::selection_owner(0);
  if (o->contains(fl_selection_requestor)) fl_selection_requestor = 0;
  int fix = 0;
  if (o->contains(Fl::belowmouse())) {Fl::belowmouse(0); fix = 1;}
  if (o->contains(Fl::focus())) {Fl::focus(0); fix = 1;}
  if (fix) fl_fix_focus();
}

// Destruction does not remove from any parent group!  And groups when
// destroyed destroy all their children.  This is convienent and fast.
// However, it is only legal to destroy a "root" such as an Fl_Window,
// and automatic destructors may be called.
// This also assummes that ::fl is the one holding this object.
Fl_Object::~Fl_Object() {
  parent_ = 0; // kludge to prevent ~Fl_Group from destroying again
  throw_focus(this);
}

void Fl_Object::activate() {
  if (active()) return;
  flags_ &= ~INACTIVE;
  if (inside(Fl::focus())) Fl::focus()->take_focus();
}

void Fl_Object::deactivate() {
  if (!active()) return;
  flags_ |= INACTIVE;
  throw_focus(this);
}

void Fl_Object::show() {
  if (visible()) return;
  clear_flag(INVISIBLE);
  handle(FL_SHOW);
  if (inside(Fl::focus())) Fl::focus()->take_focus();
  //  if (!align() || (align()&FL_ALIGN_INSIDE))
  redraw();
  //  else
  //    redraw_background(this);
}

void Fl_Object::hide() {
  if (!visible()) return;
  set_flag(INVISIBLE);
  handle(FL_HIDE);
  throw_focus(this);
  for (Fl_Object *p = parent(); p; p = p->parent())
    if (p->box() || !p->parent()) {p->redraw(); break;}
}

// return true if object is inside (or equal to) this:
// Returns false for null objects.
int Fl_Object::contains(const Fl_Object *o) const {
  for (; o; o = o->parent_) if (o == this) return 1;
  return 0;
}

////////////////////////////////////////////////////////////////

#include <FL/Fl_Window.H>
#include <FL/fl_draw.H>

void Fl_Object::redraw() {damage(~0);}

void Fl_Object::damage(uchar flags) {
  if (!visible()) return;
  damage_ |= flags;
  // mark all the parents as having children bad but not bad themselves:
  if (type() >= FL_WINDOW) {
    Fl::damage(1);
  } else if (parent())
    parent()->damage(1);
}

// a parent draw() implementation calls this on each child to redraw
// it, this decides if this is needed and then does it:
void Fl_Object::draw_child() {
  if (!damage() || !visible() || type() >= FL_WINDOW) return;
  if (fl_clipped && (fl_current_clip.r<=x() || fl_current_clip.b<=y()
	|| fl_current_clip.x>=x()+w() || fl_current_clip.y>=y()+h())) return;
  draw();	
  damage_ = 0;
}

// if a parent draw() draws something that it knows obscures a child,
// it should call this on that child:
void Fl_Object::redraw_child() {
  if (!visible() || type() >= FL_WINDOW) return;
  if (fl_clipped && (fl_current_clip.r<=x() || fl_current_clip.b<=y()
	|| fl_current_clip.x>=x()+w() || fl_current_clip.y>=y()+h())) return;
  if (damage()) {
    // try to fool it into drawing this later with draw_child():
    damage_ = ~0;
    parent()->damage_ |= 1;
    return;
  }
  damage_ = ~0;	// force redraw on
  draw();
  damage_ = 0;
}
