// Fl_Menu.C

// The FL user interface library version 0.98
// Copyright (C) 1998 Bill Spitzak, Digital Domain

#include <FL/Fl.H>
#include <FL/Fl_Menu_Window.H>
#include <FL/Fl_Menu.H>
#include <FL/fl_draw.H>

int Fl_Menu::size() const {
  const Fl_Menu *m;
  for (m=this; m->text; m = m->next());
  return m-this+1;
}

const Fl_Menu *Fl_Menu::next(int n) const {
  if (n < 0) return 0; // this is so selected==-1 returns NULL
  const Fl_Menu *m = this;
  int nest = 0;
  while (n>0) {
    if (!m->text) {
      if (--nest < 0) return 0;
    } else if (*(m->text)=='/') nest++;
    m++;
    if (!nest && m->visible()) n--;
  }
  return m;
}

uchar Fl_Menu::textsize_ = FL_NORMAL_SIZE;
uchar Fl_Menu::textfont_ = FL_HELVETICA+FL_BOLD+FL_ITALIC;

////////////////////////////////////////////////////////////////

// each vertical menu has one of these:
class menuwindow : public Fl_Menu_Window {
  void draw();
  void drawentry(const Fl_Menu *,int i,int erase);
public:
  int handle(int);
  int itemheight;	// zero == menubar
  int numitems;
  int selected;
  int drawn_selected;	// last redraw has this selected
  int H;
  const Fl_Menu *menu;
  menuwindow(const Fl_Menu* m, int X, int Y, int W, int H,
	     const Fl_Menu* picked, const char *title,
	     int menubar = 0);
  void set_selected(int);
  int find_selected(int mx, int my);
  int titlex(int);
  void autoscroll(int);
  void position(int x, int y);
};

#define BW	3 // border thickness
#define LEFT	7 // between left edge of item and edge of box
#define RIGHT	9 // between right edge of item and edge of box
#define BOTTOM	4 // between bottom item and bottom of box
#define LEADING 4 // extra vertical leading
#define TOP	5 // between top item and top of box
#define VSHADOW	5 // shadow on bottom
#define HSHADOW 5 // shadow on right

// width of label, including effect of & characters:
static int width(const char *t) {
  float r = fl_width(t);
  while (*t) if (*t++ == '&' && *t) {r -= fl_width('&'); t++;}
  return int(r);
}

menuwindow::menuwindow(const Fl_Menu* m, int X, int Y, int Wp, int Hp,
		       const Fl_Menu* picked, const char *t, 
		       int menubar)
  : Fl_Menu_Window(0,0,t)
{
  menu = m;
  drawn_selected = -1;
  clear_border();
  box(0);
  set_modal();
  selected = -1;
  {int i = 0;
  if (m) for (const Fl_Menu* m1=m; ; m1 = m1->next(), i++) {
    if (picked) {
      if (m1 == picked) {selected = i; picked = 0;}
      else if (m1 > picked) {selected = i-1; picked = 0; Wp = Hp = 0;}
    }
    if (!m1->text) break;
  }
  numitems = i;}

  if (menubar) {
    itemheight = 0;
    x(X); y(Y); w(Wp); h(H=Hp);
    return;
  }

  fl_font(Fl_Menu::textfont(), Fl_Menu::textsize());
  itemheight = int(fl_height())+LEADING;

  int hotKeysw = 0;
  int W = t ? width(t) : 0;
  if (m) for (; m->text; m = m->next()) {
    const char *t = m->text;
    int w1;
    if (m->submenu()) {w1 = 13; if (*t=='/') t++;}
    else if (m->checkbox()) w1 = 14;
    else w1 = 0;
    if (*t=='_') t++;
    w1 += width(t);
    if (w1 > W) W = w1;
    if (m->shortcut) {
      w1 = int(fl_width(fl_shortcut_label(m->shortcut))) + 8;
      if (w1 > hotKeysw) hotKeysw = w1;
    }
  }
  if (selected >= 0 && !Wp) X -= W/2;
  W += hotKeysw+LEFT+RIGHT; if (Wp > W) W = Wp;
  H = (numitems ? itemheight*numitems-LEADING : 0)+TOP+BOTTOM;

  x(X); w(W+HSHADOW);
  h(t ? H+itemheight-LEADING+TOP+BOTTOM+VSHADOW+2 : H+1+VSHADOW);
  if (selected >= 0)
    Y = Y+(Hp-itemheight)/2-selected*itemheight-2;
  else
    Y = Y+Hp;
  y(Y+H+VSHADOW-h());

  show();
}

void menuwindow::position(int X, int Y) {
  Fl_Menu_Window::position(X,Y);
  x(X); y(Y); // don't wait for response from X
}

// scroll so item i is visible on screen
void menuwindow::autoscroll(int i) {
  int Y = y()+h()-(VSHADOW + BOTTOM + (numitems-i)*itemheight - LEADING + 1);
  if (Y < 0) Y = -Y+10;
  else {
    Y = Y+itemheight-Fl::h();
    if (Y <= 0) return;
    Y = -Y-10;
  }
  position(x(),y()+Y);
}

////////////////////////////////////////////////////////////////

static void drawPopupBox(int x,int y,int w,int h) {
//clearcolor();
//fl_rectf(x+HSHADOW, y-VSHADOW, w, VSHADOW);
//fl_rectf(x+w, y, HSHADOW, h+1);
  fl_color(FL_GRAY0); fl_50percent2();
  fl_rectf(x+HSHADOW, y+h, w, VSHADOW);
  fl_rectf(x+w, y+VSHADOW-1, HSHADOW, h+1-VSHADOW);
  fl_100percent();
  fl_rect(x,y-1,w,h+1);
  fl_rectf(x+1,y+h-BW,w-2,BW-1);
  fl_rectf(x+w-BW,y,BW-1,h-BW);
  fl_color(FL_LIGHT3);
  fl_rect(x+1,y,BW-1,h-BW);
  fl_rect(x+BW,y,w-BW-2,BW-1);
  fl_color(FL_LIGHT1);
  fl_50percent();
  fl_rectf(x+1,y+h-BW,w-2,BW-1);
  fl_rectf(x+w-BW,y,BW-1,h-BW);
  fl_100percent();
  fl_rectf(x+BW, y+BW-1, w-2*BW, h-2*BW+1);
}

extern char fl_draw_shortcut;

void menuwindow::drawentry(const Fl_Menu *m, int i, int erase) {
  if (!m) return; // this happens if -1 is selected item and redrawn

  int x = LEFT-3;
  int W = this->w()-HSHADOW;
  int w = W-(LEFT+RIGHT-6);
  int y = h()-(VSHADOW + BOTTOM + (numitems-i)*itemheight - LEADING + 1);
  int h = itemheight - LEADING;

  if (i == selected) {
    fl_color(FL_GRAY0); fl_50percent();
    fl_rectf(BW+2, y, W-2*BW-3, h+1);
    fl_color(FL_WHITE); fl_100percent();
    fl_rectf(BW+1, y-1, W-2*BW-3, h+1);
  } else if (erase) {
    fl_color(FL_LIGHT1);
    fl_rectf(BW+1, y-1, W-2*BW-2, h+2);
  }

  const char *t = m->text; if (*t=='/') t++;
  if (*t == '_') {
    t++;
    fl_color(FL_DARK3);
    fl_xyline(BW-1,y+h+1,W-BW);
    fl_color(FL_LIGHT3);
    fl_xyline(BW,y+h+2,W-BW);
  }

  if (m->checkbox()) {
    fl_color(FL_LIGHT3);
    fl_rectf(x+4, y+3, 10, 10);
    fl_color(FL_DARK3);
    fl_rectf(x+3, y+2, 10, 10);
    fl_color(FL_LIGHT1); fl_50percent();
    fl_rectf(x+4, y+3, 9, 9);
    fl_color(FL_BLACK); fl_100percent();
    if (m->checked()) fl_rectf(x+5, y+4, 7, 7);
    x += 14; w -= 14;
  } else {
    fl_color(FL_BLACK);
  }
  fl_draw_shortcut = 1;
  fl_draw(t, x, y, w, h, FL_ALIGN_LEFT);
  fl_draw_shortcut = 0;

  if (m->submenu()) {
    fl_color(FL_BLACK);
    fl_polygon(x+w-10,y+2, x+w-10,y+2+10, x+w,y+2+5);
  } else if (m->shortcut) {
    fl_draw(fl_shortcut_label(m->shortcut), x, y, w, h, FL_ALIGN_RIGHT);
  }

  if (!m->active()) {
    fl_color(FL_LIGHT1); fl_50percent();
    fl_rectf(BW, y-1, W-2*BW, h+2);
    fl_100percent();
  }
}


void menuwindow::draw() {

  fl_font(Fl_Menu::textfont(), Fl_Menu::textsize());

  if (damage() & 6) {	// complete redraw
    if (label()) {
      drawPopupBox(0, 1,
		   width(label())+LEFT+RIGHT,
		   h()-(H+VSHADOW+3)-1);
      fl_color(FL_BLACK);
      fl_draw_shortcut = 1;
      fl_draw(label(), LEFT, h()-(H+VSHADOW+BW+BOTTOM), 0,0,
	      FL_ALIGN_LEFT+FL_ALIGN_BOTTOM);
      fl_draw_shortcut = 0;
    }
    if (menu) {
      drawPopupBox(0, h()-H-VSHADOW, w()-HSHADOW, H);
      const Fl_Menu *m; int i;
      for (m=menu, i=0; m->text; i++, m = m->next()) drawentry(m,i,0);
    }
  } else {
/*
  if (damage() & 2) { // shadow on edge
  clearcolor();
  fl_rectf(w()-HSHADOW, 0, HSHADOW, H+VSHADOW+1);
  fl_color(FL_GRAY0); fl_50percent2();
  fl_rectf(w()-HSHADOW, 0, HSHADOW, H);
  fl_100percent();
  } */
    if (damage() & 1 && selected!=drawn_selected) { // change selection
      drawentry(menu->next(drawn_selected),drawn_selected,1);
      drawentry(menu->next(selected),selected,1);
    }
  }	    
  drawn_selected = selected;
}

void menuwindow::set_selected(int i) {
  if (i != selected) {selected = i; damage(1);}
}

////////////////////////////////////////////////////////////////

#define XBORDER 16 // horizontal spacing for Fl_MenuBar

// Return item in menubar pointed to by (relative) coordinates x,y
// or return -1 if none:
static const Fl_Menu*
find_in_menubar(const Fl_Menu* menu, int mx, int* returni = 0) {
  if (!menu) return 0;
  int x = BW; int i = 0;
  const Fl_Menu* m = menu;
  fl_font(Fl_Menu::textfont(), Fl_Menu::textsize());
  for (; ; m = m->next(), i++) {
    if (!m->text) return 0;
    const char *title = m->text; if (*title=='/') title++;
    x += width(title) + XBORDER;
    if (x > mx) break;
  }
  if (returni) *returni = i; return m;
}

int menuwindow::find_selected(int mx, int my) {
  if (!menu || !menu->text) return -1;
  mx -= x();
  my -= y();
  if (my <= 0 || my >= h()) return -1;
  if (!itemheight) {
    int i;
    if (find_in_menubar(menu, mx, &i)) return i;
    return -1;
  }
  if (mx <= 0 || mx >= w()) return -1;
  my -= h()-(VSHADOW + BOTTOM + numitems*itemheight - LEADING + 2);
  if (my <= 0) return -1;
  int i = my/itemheight;
  if (i>=numitems) i = numitems-1;
  return i;
}

// return horizontal position for item i in a menubar:
int menuwindow::titlex(int i) {
  const Fl_Menu* m;
  int x = BW;
  fl_font(Fl_Menu::textfont(), Fl_Menu::textsize());
  for (m=menu; i--; m = m->next()) {
    const char *title = m->text; if (*title=='/') title++;
    x += width(title) + XBORDER;
  }
  return x;
}

// match shortcuts & label shortcuts, don't search submenus:
// returns menu item and index
static const Fl_Menu *find_shortcut(const Fl_Menu *menu, int & i) {
  const Fl_Menu *m1 = menu;
  for (int ii = 0; m1 && m1->text; m1 = m1->next(1),ii++) {
    if (m1->activevisible() &&
	(Fl::test_shortcut(m1->shortcut)
	 || Fl_Object::test_shortcut(m1->text))) {i = ii; return m1;}
  }
  return 0;
}

////////////////////////////////////////////////////////////////

// Because the menus are modal, all the events will go to the top one.
// But the handle method needs to look at all of them to find out
// what item the user is pointing at.  And it needs a whole lot
// of other state variables to determine what is going on with
// the currently displayed menus.
// So the main loop (handlemenu()) puts all the state in a structure
// and puts a pointer to it in a static location, so the handle()
// on menus can refer to it and alter it.  The handle() method
// changes variables in this state to indicate what item is
// picked, but does not actually alter the display, instead the
// main loop does that.  This is because the X mapping and unmapping
// of windows is slow, and we don't want to fall behind the events.

struct menustate {
  int j; // which menu selected item is in, j<0 means none
  int i; // which item in that menu
  const Fl_Menu *m; // the selected item, zero for none
  menuwindow *p[20];
  int nummenus;
  int menubar; // if true p[0] is a menubar
  int state; // 0 at first, 1 after push, 2 when done
};
static menustate *p;

static int forward(int j) { // go to next item in menu j if possible
  menustate &p = *(::p);
  menuwindow &m = *(p.p[j]);
  int ii = j == p.j ? p.i : m.selected;
  while (++ii < m.numitems) {
    const Fl_Menu* m1 = m.menu->next(ii);
    if (m1->activevisible()) {p.m = m1; p.i = ii; p.j = j; return 1;}
  }
  return 0;
}

static int backward(int j) { // previous item in menu j if possible
  menustate &p = *(::p);
  menuwindow &m = *(p.p[j]);
  int ii = j == p.j ? p.i : m.selected;
  while (--ii >= 0) {
    const Fl_Menu* m1 = m.menu->next(ii);
    if (m1->activevisible()) {p.m = m1; p.i = ii; p.j = j; return 1;}
  }
  return 0;
}

static void findit(int j,int i) { // turn i,j into m
  menustate &p = *(::p);
  p.i = i; p.j = j;
  p.m = (j >= 0) ? p.p[j]->menu->next(i) : 0;
  if (p.m && !p.m->activevisible()) p.m = 0;
}

int menuwindow::handle(int e) {
  menustate &p = *(::p);
  switch (e) {
  case FL_KEYBOARD:
    switch (Fl::event_key()) {
    case FL_Up:
      if (p.j < 0) p.j = 0;
      if (p.menubar && p.j==0) ;
      else if (backward(p.j));
      else if (p.menubar && p.j==1) findit(0, p.p[0]->selected);
      break;
    case FL_Down:
      if (p.j < 0) p.j = 0;
      if (p.j || !p.menubar) forward(p.j);
      else if (p.j < p.nummenus-1) forward(p.j+1);
      break;
    case FL_Right:
      if (p.menubar && (p.j<=0 || p.j==1 && p.nummenus==2)) forward(0);
      else if (p.j < p.nummenus-1) forward(p.j+1);
      break;
    case FL_Left:
      if (p.menubar && p.j<=1) backward(0);
      else if (p.j>0) findit(p.j-1, p.p[p.j-1]->selected);
      break;
    case FL_Enter:
      p.state = 2;
      break;
    case FL_Escape:
      p.m = 0;
      p.state = 2;
      break;
    default: { // see if it is a shortcut for currently displayed menus
      for (int j = p.nummenus; j--;) {
	int i; const Fl_Menu *m = find_shortcut(p.p[j]->menu,i);
	if (m) {
	  p.m = m; p.i = i; p.j = j;
	  if (!m->submenu()) p.state = 2;
	  return 1;
	}
      }
      } break;
    }
    break;
  case FL_MOVE:
    if (Fl::event_is_click()) break; goto J1;
  case FL_PUSH:
    p.state = 1; goto J1;
  case FL_RELEASE:
  case FL_DRAG: J1: {
    int mx = Fl::event_x_root();
    int my = Fl::event_y_root();
    int i=0,j;
    for (j=p.nummenus-1,i=0; j>=0; j--) {
      i = p.p[j]->find_selected(mx,my);
      if (i >= 0) break;
    }
    findit(j,i);}
    if (e == FL_RELEASE && (p.state || !Fl::event_is_click())) p.state = 2;
    break;
  default:
    return Fl_Window::handle(e);
  }
  return 1;
}

// This internal version of popup() takes coordinates translated to
// screen space, and extra flags to indicate the menubar:
static const Fl_Menu *popup(
    const Fl_Menu *m, int X, int Y, int W, int H, const Fl_Menu *picked,
    const char *t, int menubar=0) {

  if (!m || !m->text) return 0;

  Fl::grab();
  menuwindow mw(m, X, Y, W, H, picked, t, menubar);
  menustate p; ::p = &p;
  p.p[0] = &mw;
  p.nummenus = 1;
  p.menubar = menubar;
  p.state = 0;
  if (mw.selected>=0) findit(0, mw.selected); else {
    p.j = -1; p.i = 0; p.m = 0; picked = 0;}
  menuwindow *fakemenu = 0; // kludge for buttons in menubar

  if (picked && p.m) goto NOWAIT;

  // the main loop, runs until p.state goes to 2:
  for (;;) {
    {const Fl_Menu *old = p.m;
    Fl::wait();
    if (p.state == 2) break; // done.
    if (p.m == old) continue;}

    if (p.m) {	// we are pointing at something
      if (p.i == p.p[p.j]->selected) { // same item as before
	for (int k=p.j+1; k<p.nummenus; k++) p.p[k]->set_selected(-1);
	while (p.nummenus > p.j+2) delete p.p[--p.nummenus];
      } else {	// different item, submenus must change
      NOWAIT:
	if (p.m == picked) picked = 0; // stop the startup code if item found
	delete fakemenu; fakemenu = 0;
	while (p.nummenus > p.j+1) delete p.p[--p.nummenus];
	p.p[p.j]->autoscroll(p.i);
	if (p.m->submenu()) {
	  const char *title = p.m->text;
	  const Fl_Menu *menutable;
	  if (*title == '/') {menutable = p.m+1; title++;}
	  else menutable = (Fl_Menu*)(p.m)->argument;
	  int nX,nY;
	  if (!p.j && p.menubar) {	// menu off a menubar:
	    nX = p.p[p.j]->x() + p.p[p.j]->titlex(p.i);
	    nY = p.p[p.j]->y() + p.p[p.j]->h();
	    picked = 0;
	  } else {
	    nX = p.p[p.j]->x() + p.p[p.j]->w()-HSHADOW;
	    nY = p.p[p.j]->y() + p.p[p.j]->h()-VSHADOW-p.p[p.j]->H
		+ p.i*p.p[p.j]->itemheight;
	    title = 0;
	  }
	  if (picked) {
	    menuwindow *n = new menuwindow(menutable,X,Y,W,H,picked,title);
	    p.p[p.nummenus++] = n;
	    // move all earlier menus to line up with this new one:
	    if (n->selected>=0) {
	      int dy = n->y()-nY;
	      int dx = n->x()-nX;
	      for (int i=0; i<=p.j; i++) {
		menuwindow *t = p.p[i];
		t->position(t->x()+dx, t->y()+dy+1);
	      }
	      findit(p.nummenus-1, n->selected);
	      goto NOWAIT;
	    }
	  } else {
	    p.p[p.nummenus++] =
	      new menuwindow(menutable,nX,nY,0,0,0,title);
	  }
	} else if (!p.j && p.menubar) {
	  // kludge so buttons in menubar "highlight"
	  fakemenu = new menuwindow(0,
				    p.p[p.j]->x()+p.p[p.j]->titlex(p.i),
				    p.p[p.j]->y()+p.p[p.j]->h(),0,0,
				    0, p.m->text);
	}
	p.p[p.j]->set_selected(p.i);
	picked = 0; // stop the startup code if no more submenus
      }
    } else { // pointing at nothing
      if (!fakemenu) p.p[p.nummenus-1]->set_selected(-1);
    }
  }
  delete fakemenu;
  while (p.nummenus>1) delete p.p[--p.nummenus];
  mw.hide();
  Fl::release();
  return p.m;
}

////////////////////////////////////////////////////////////////

const Fl_Menu*
Fl_Menu::popup(int x, int y, const char *title) const {
  return ::popup(this,
		 x+Fl::event_x_root()-Fl::event_x(),
		 y+Fl::event_y_root()-Fl::event_y(), 0, 0,
		 0, title);
}

const Fl_Menu*
Fl_Menu::popup(int x, int y, int w, int h,
	       const Fl_Menu* p, const char* title) const {
  return ::popup(this,
		 x+Fl::event_x_root()-Fl::event_x(),
		 y+Fl::event_y_root()-Fl::event_y(), w, h,
		 p, title);
}

const Fl_Menu *Fl_Menu::test_shortcut() const {
  const Fl_Menu *m = this;
  const Fl_Menu *ret = 0;
  if (m) for (; m->text; m = m->next()) {
    if (m->activevisible()) {
      // return immediately any match of an item in top level menu:
      if (Fl::test_shortcut(m->shortcut)) return m;
      // if (Fl_Object::test_shortcut(m->text)) return m;
      // only return matches from lower menu if nothing found in top menu:
      if (!ret && m->submenu()) {
	const Fl_Menu* s = (m->text[0]=='/') ? m+1:(const Fl_Menu*)m->argument;
	ret = s->test_shortcut();
      }
    }
  }
  return ret;
}

/////////////////////////////////////////////////////////////////

#include <FL/Fl_Menu_Bar.H>

void Fl_Menu_Bar::draw() {
  draw_box();
  if (!menu() || !menu()->text) return;
  fl_font(textfont(), textsize());
  const Fl_Menu *m;
  int X = x()+LEFT+3;
  for (m=menu(); m->text; m = m->next()) {
    const char *title = m->text; if (*title=='/') title++;
    fl_color(m->active() ? labelcolor() : 8);
    fl_draw_shortcut = 1;
    fl_draw(title,X,y(),0,h(),FL_ALIGN_LEFT);
    X += width(title) + XBORDER;
  }
  fl_draw_shortcut = 0;
}

int Fl_Menu_Bar::handle(int event) {
  const Fl_Menu *v;
  int i;
  switch (event) {
  case FL_PUSH:
    v = find_in_menubar(menu(), Fl::event_x()-x());
    if (!v) return 0;
  J1:
    Fl_Menu::textfont(textfont());
    Fl_Menu::textsize(textsize());
    v = popup(menu(),
	      x()+Fl::event_x_root()-Fl::event_x(),
	      y()+Fl::event_y_root()-Fl::event_y(), w(), h(),
	      v, 0, 1);
    picked(v);
    return 1;
  case FL_SHORTCUT:
    v = menu()->test_shortcut();
    if (v) {picked(v); return 1;}
    v = find_shortcut(menu(),i);
    if (v) goto J1;
    return 0;
  default:
    return 0;
  }
}

// end of Fl_Menu.C
