// Fl_Input.C

// This is the "user interface", it decodes user actions into what to
// do to the text.  See also Fl_Input_.C, where the text is actually
// manipulated (and some ui, in particular the mouse, is done...).
// In theory you can replace this code with another subclass to change
// the keybindings.

#include <FL/Fl.H>
#include <FL/Fl_Input.H>
#include <FL/fl_draw.H>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>

void Fl_Input::draw() {
  if (type() == FL_HIDDEN_INPUT) return;
  if (damage() & 128)
    draw_box(box(), Fl::focus()==this ? color2() : color());
//draw_label(); // it should never do this, since it obscures text
  Fl_Input_::drawtext(x()+Fl::box_dx(box())+3, y()+Fl::box_dy(box()),
		      w()-Fl::box_dw(box())-6, h()-Fl::box_dh(box()));
}

// kludge so shift causes selection to extend:
int Fl_Input::shift_position(int p) {
  return position(p, Fl::event_shift() ? mark() : p);
}
int Fl_Input::shift_up_down_position(int p) {
  return up_down_position(p, Fl::event_shift());
}

static char quotenext;
#define ctrl(x) (x^0x40)

// If you define this symbol as zero you will get the peculiar
// original FL behavior where moving off the end of an input
// field will move the cursor into the next field:
// define it as 1 to prevent cursor movement from going to next field:
#define NORMAL_INPUT_MOVE 0

int Fl_Input::handle_key() {
  int i;

  const char *text = Fl::event_text();
  char pquotenext = quotenext; quotenext = 0;
  char key = text[0];

  if (pquotenext && key) {
    return replace(position(),mark(),text);
  }

  if (Fl::event_alt()) { // reserved for shortcuts
    quotenext = pquotenext;
    return 0;
  }

  switch (Fl::event_key()) {
  case FL_Left:
//case FL_KP_Left:
    key = ctrl('B'); break;
  case FL_Right:
//case FL_KP_Right:
    key = ctrl('F'); break;
  case FL_Up:
//case FL_KP_Up:
    key = ctrl('P'); break;
  case FL_Down:
//case FL_KP_Down:
    key = ctrl('N'); break;
  case FL_Delete:
//case FL_KP_Delete:
    key = ctrl('D'); break;
  case FL_Home:
//case FL_KP_Home:
    key = ctrl('A'); break;
  case FL_End:
//case FL_KP_End:
    key = ctrl('E'); break;
  case FL_BackSpace:
    if (mark() != position()) cut();
    else cut(-1);
    return 1;
  case FL_Enter:
  case FL_KP_Enter:
    if (when() & FL_WHEN_ENTER_KEY) {
      position(size(), 0);
      maybe_do_callback();
      return 1;
    } else if (type() == FL_MULTILINE_INPUT)
      return replace(position(), mark(), "\n", 1);
    else 
      return 0;	// reserved for shortcuts
  case FL_Tab:
    if (Fl::event_ctrl() || type()!=FL_MULTILINE_INPUT) return 0;
    break;
  case FL_Escape:
    return 0;	// reserved for shortcuts (Forms cleared field)
  }

  switch(key) {
  case 0:	// key did not translate to any text
    quotenext = pquotenext; // allow user to hit shift keys after ^Q
    return 0;
  case ctrl('A'):
    if (type() == FL_MULTILINE_INPUT)
      for (i=position(); i && index(i-1)!='\n'; i--) ;
    else
      i = 0;
    return shift_position(i) + NORMAL_INPUT_MOVE;
  case ctrl('B'):
    return shift_position(position()-1) + NORMAL_INPUT_MOVE;
  case ctrl('C'): // copy
    return copy();
  case ctrl('D'):
    if (mark() != position()) return cut();
    else return cut(1);
  case ctrl('E'):
    if (type() == FL_MULTILINE_INPUT)
      for (i=position(); index(i) && index(i)!='\n'; i++) ;
    else
      i = size();
    return shift_position(i) + NORMAL_INPUT_MOVE;
  case ctrl('F'):
    return shift_position(position()+1) + NORMAL_INPUT_MOVE;
  case ctrl('K'):
    if (position()>=size()) return 0;
    if (type() == FL_MULTILINE_INPUT) {
      if (index(position()) == '\n')
	i = position() + 1;
      else 
	for (i=position()+1; index(i) && index(i) != '\n'; i++);
    } else
      i = size();
    cut(position(), i);
    return copy_cuts();
  case ctrl('N'):
    if (type()!=FL_MULTILINE_INPUT) return 0;
    for (i=position(); index(i)!='\n'; i++)
      if (!index(i)) return NORMAL_INPUT_MOVE;
    shift_up_down_position(i+1);
    return 1;
  case ctrl('P'):
    if (type()!=FL_MULTILINE_INPUT) return 0;
    for (i = position(); i > 0 && index(i-1) != '\n'; i--) ;
    if (!i) return NORMAL_INPUT_MOVE;
    shift_up_down_position(i-1);
    return 1;
  case ctrl('Q'):
    quotenext = 1;
    return 1;
  case ctrl('U'):
    return cut(0, size());
  case ctrl('V'):
  case ctrl('Y'):
    Fl::paste(*this);
    return 1;
  case ctrl('X'):
  case ctrl('W'):
    copy();
    return cut();
  case ctrl('Z'):
  case ctrl('_'):
    return undo();
  }

  return replace(position(), mark(), text);
}

int Fl_Input::handle(int event) {
  switch (event) {

  case FL_FOCUS:
    switch (Fl::event_key()) {
    case FL_Right:
      position(0);
      break;
    case FL_Left:
      position(size());
      break;
    case FL_Down:
      up_down_position(0);
      break;
    case FL_Up:
      up_down_position(size());
      break;
    case FL_Tab:
      position(size(),0);
      break;
    }
    break;

  case FL_UNFOCUS:
    quotenext = 0;
    break;

  case FL_KEYBOARD:
    return handle_key();

  case FL_PUSH:
    quotenext = 0;
    if (Fl::event_button() == 2) {
      Fl::paste(*this);
      if (Fl::focus()==this) return 1; // remove line for Motif behavior
    }
    if (Fl::focus() != this) {
      Fl::focus(this);
      handle(FL_FOCUS); // cause minimal update
    }
    break;

  case FL_DRAG:
  case FL_RELEASE:
    if (Fl::event_button() == 2) return 0;
    break;
  }
  return Fl_Input_::handletext(event,
	x()+Fl::box_dx(box())+3, y()+Fl::box_dy(box()),
	w()-Fl::box_dw(box())-6, h()-Fl::box_dh(box()));
}

Fl_Input::Fl_Input(int x, int y, int w, int h, const char *l)
: Fl_Input_(x, y, w, h, l) {}
