// Fl_Group.C

// The Fl_Group is the only defined container type in FL.

// Fl_Window itself is a subclass of this, and most of the event
// handling is designed so windows themselves work correctly.

#include <FL/Fl.H>
#include <FL/Fl_Group.H>
#include <stdlib.h>

Fl_Group* Fl_Group::current_;

// Hack: A single child is stored in the pointer to the array, while
// multiple children are stored in an allocated array:
Fl_Object*const* Fl_Group::array() const {
  return children_ <= 1 ? (Fl_Object**)(&array_) : array_;
}

int Fl_Group::find(const Fl_Object* o) const {
  Fl_Object*const* a = array();
  int i; for (i=0; i < children_; i++) if (*a++ == o) break;
  return i;
}

int Fl_Group::handle(int event) {

  Fl_Object*const* a = array();
  int i;
  Fl_Object* o;

  switch (event) {

  case FL_FOCUS:
    if (savedfocus_ && savedfocus_->take_focus()) return 1;
    for (i = children(); i--;) if ((*a++)->take_focus()) return 1;
    return 0;

  case FL_UNFOCUS:
    o = Fl::focus();
    while (o && o->parent() != this) o = o->parent();
    savedfocus_ = o;
    return 0;

  case FL_KEYBOARD:
    return navigation();

  case FL_SHORTCUT:
    for (i = children(); i--;) {
      o = a[i];
      if (o->activevisible() && Fl::event_inside(o) && o->handle(FL_SHORTCUT))
	return 1;
    }
    for (i = children(); i--;) {
      o = a[i];
      if (o->activevisible() && !Fl::event_inside(o) && o->handle(FL_SHORTCUT))
	return 1;
    }
    if (Fl::event_key() == FL_Enter) return navigation(FL_Down);
    return 0;

  case FL_ENTER:
  case FL_MOVE:
    for (i = children(); i--;) {
      o = a[i];
      if (o->activevisible() && Fl::event_inside(o)) {
	if (o->contains(Fl::belowmouse())) {
	  return o->handle(FL_MOVE);
	} else if (o->handle(FL_ENTER)) {
	  if (!o->contains(Fl::belowmouse())) Fl::belowmouse(o);
	  return 1;
	}
      }
    }
    Fl::belowmouse(this);
    return 1;

  case FL_PUSH:
    for (i = children(); i--;) {
      o = a[i];
      if (o->activevisible() && Fl::event_inside(o)) {
	if (o->handle(FL_PUSH)) {
	  if (Fl::pushed() && !o->contains(Fl::pushed())) Fl::pushed(o);
	  return 1;
	}
      }
    }
    return 0;

  case FL_DEACTIVATE:
  case FL_ACTIVATE:
    for (i = children(); i--;) {
      o = *a++;
      if (o->active()) o->handle(event);
    }
    return 1;

  case FL_HIDE:
  case FL_SHOW:
    for (i = children(); i--;) {
      o = *a++;
      if (o->visible()) o->handle(event);
    }
    return 1;

  default:
    return 0;

  }
}

// translate the current keystroke into up/down/left/right for navigation:
#define ctrl(x) (x^0x40)
int navkey() {
  switch (Fl::event_key()) {
  case FL_Tab:
    return (Fl::event_shift() ? FL_Left : FL_Right);
  case FL_Right:
  case FL_KP_Right:
    return FL_Right;
  case FL_Left:
  case FL_KP_Left:
    return FL_Left;
  case FL_Up:
  case FL_KP_Up:
    return FL_Up;
  case FL_Down:
  case FL_KP_Down:
    return FL_Down;
  default:
    switch (Fl::event_text()[0]) {
    case ctrl('N') : return FL_Down;
    case ctrl('P') : return FL_Up;
    case ctrl('F') : return FL_Right;
    case ctrl('B') : return FL_Left;
    }
  }
  return 0;
}

//void Fl_Group::focus(Fl_Object *o) {Fl::focus(o); o->handle(FL_FOCUS);}

#if 0
const char *nameof(Fl_Object *o) {
  if (!o) return "NULL";
  if (!o->label()) return "<no label>";
  return o->label();
}
#include <stdio.h>
#endif

// try to move the focus in response to a keystroke:
int Fl_Group::navigation(int key) {
  if (children() <= 1) return 0;
  if (!key) {key = navkey(); if (!key) return 0;}
  Fl_Object *focus_ = Fl::focus();
  int old_i;
  for (old_i=0;;old_i++) {
    if (old_i >= children_) return 0;
    if (array_[old_i]->contains(focus_)) break;
  }
  int i = old_i;

  for (;;) {
    switch (key) {
    case FL_Right:
    case FL_Down:
      i++; if (i >= children_) i = 0;
      break;
    case FL_Left:
    case FL_Up:
      if (i) i--; else i = children_-1;
      break;
    default:
      return 0;
    }
    if (i == old_i) return 0;
    Fl_Object* o = array_[i];
    switch (key) {
    case FL_Down:
    case FL_Up:
      if (o->x() >= focus_->x()+focus_->w() ||
	  o->x()+o->w() <= focus_->x()) continue;
    }
    if (o->take_focus()) return 1;
  }
}

////////////////////////////////////////////////////////////////

Fl_Group::Fl_Group(int X,int Y,int W,int H,const char *l)
: Fl_Object(X,Y,W,H,l) {
  type(FL_GROUP);
  align(FL_ALIGN_TOP);
  children_ = 0;
  array_ = 0;
  savedfocus_ = 0;
  resizable_ = this;
}

Fl_Group::~Fl_Group() {
  Fl_Object*const* a = array();
  for (int i=children(); i--;) {
    Fl_Object* o = *a++;
    // test the parent to see if child already destructed:
    if (o->parent() == this) delete o;
  }
  if (children() > 1) free((void*)array_);
}

void Fl_Group::insert(Fl_Object &o, int i) {
  if (o.parent_) return;	// ignore double adds
  o.parent_ = this;
  if (children_ == 0) { // use array pointer to point at single child
    array_ = (Fl_Object**)&o;
  } else if (children_ == 1) { // go from 1 to 2 children
    Fl_Object* t = (Fl_Object*)array_;
    array_ = (Fl_Object**)malloc(2*sizeof(Fl_Object*));
    array_[!i] = t; array_[i] = &o;
  } else {
    if (!(children_ & (children_-1))) // double number of children
      array_ = (Fl_Object**)realloc((void*)array_,
				    2*children_*sizeof(Fl_Object*));
    for (int j = children_; j > i; j--) array_[j] = array_[j-1];
    array_[i] = &o;
  }
  children_++;
}

void Fl_Group::add(Fl_Object &o) {insert(o, children_);}

void Fl_Group::remove(Fl_Object &o) {
  int i = find(o);
  if (i >= children_) return;
  if (&o == savedfocus_) savedfocus_ = 0;
  o.parent_ = 0;
  children_--;
  if (children_ == 1) { // go from 2 to 1 child
    Fl_Object *t = array_[!i];
    free((void*)array_);
    array_ = (Fl_Object**)t;
  } else if (children_ > 1) { // delete from array
    for (; i < children_; i++) array_[i] = array_[i+1];
  }
}

////////////////////////////////////////////////////////////////

// set the dimensions of a group to surround contents
void Fl_Group::resize() {
  if (!children()) return;
  Fl_Object*const* a = array();
  Fl_Object* o = *a++;
  int rx = o->x();
  int ry = o->y();
  int rw = rx+o->w();
  int rh = ry+o->h();
  for (int i=children_-1; i--;) {
    o = *a++;
    if (o->x() < rx) rx = o->x();
    if (o->y() < ry) ry = o->y();
    if (o->x()+o->w() > rw) rw = o->x()+o->w();
    if (o->y()+o->h() > rh) rh = o->y()+o->h();
  }
  x(rx); y(ry); w(rw-rx); h(rh-ry);
}

void Fl_Group::resize(int X, int Y, int W, int H) {
  int dx = X-x();
  int dy = Y-y();
  int dw = W - w();
  int dh = H - h();

  Fl_Object*const* a = array();
  if (!resizable() || !dw && !dh) {
    Fl_Object::resize(X,Y,W,H);
    if (dx || dy || dh) {
      for (int i=children_; i--;) {
	Fl_Object* o = *a++;
	o->resize(o->x()+dx, o->y()+dy, o->w(), o->h());
      }
    }
    return;
  }

  Fl_Object *rbox = resizable(); if (!rbox) rbox = this;
  int ox = rbox->x();
  int ow = rbox->w();
  int nw = ow + dw;
  // reject resizes that make the resizable negative:
  if (nw < 0) {dw += -nw; W += -nw; nw = 0;}
  int oy = rbox->y();
  int oh = rbox->h();
  int nh = oh + dh;
  // reject resizes that make the resizable negative:
  if (nh < 0) {dh += -nh; H += -nh; nh = 0;}

  Fl_Object::resize(X,Y,W,H);
  for (int i=children_; i--;) {
    Fl_Object* o = *a++;
    int l = o->x();
    int r = l+o->w();
    int b = o->y();
    int t = b+o->h();
    // most of these extra if's are to make it act as though
    // zero-width objects can be resized to bigger than zero
    // width, rather than having both edges move together:
    if (ow <= 0) {
      if (r <= l) {
	if (l > ox) l += dw;
	if (r >= ox+ow) r += dw;
      } else {
	if (l >= ox) l += dw;
	if (r > ox+ow) r += dw;
      }
    } else { // this is the normal case
      if (l >= ox+ow) l+=dw; else if (l > ox) l = ox+((l-ox)*nw+ow/2)/ow;
      if (r >= ox+ow) r+=dw; else if (r > ox) r = ox+((r-ox)*nw+ow/2)/ow;
    }
    if (oh <= 0) {
      if (t <= b) {
	if (b > oy) b += dh;
	if (t >= oy+oh) t += dh;
      } else {
	if (b >= oy) b += dh;
	if (t > oy+oh) t += dh;
      }
    } else { // this is the normal case
      if (b >= oy+oh) b+=dh; else if (b > oy) b = oy+((b-oy)*nh+oh/2)/oh;
      if (t >= oy+oh) t+=dh; else if (t > oy) t = oy+((t-oy)*nh+oh/2)/oh;
    }
    o->resize(l+dx, b+dy, r-l, t-b);
  }
}

// also see Fl_Window::draw()
void Fl_Group::draw() {
  Fl_Object*const* a = array();
  if (damage()&~1) { // redraw the entire thing:
    draw_box();
    draw_label();
    for (int i=children_; i--;) {
      Fl_Object* o = *a++;
      o->redraw_child();
      o->draw_child();
      o->draw_outside_label();
    }
  } else {	// only redraw the children that need it
    for (int i=children_; i--;) (*a++)->draw_child();
  }
}
