/*	Fl_Gl_Window.C

	A window to draw GL output into.

*/

#include <config.h>
#if HAVE_GL

#include <FL/Fl.H>
#include <FL/x.H>
#include <FL/Fl_Gl_Window.H>
#include <GL/glx.h>

#include "Fl_Gl_Choice.H"

////////////////////////////////////////////////////////////////

// The symbol SWAP_TYPE defines what is in the back buffer after doing
// a glXSwapBuffers().

// The OpenGl documentation says that the contents of the backbuffer
// are "undefined" after glXSwapBuffers().  However, if we know what
// is in the backbuffers then we can save a good deal of time.  For
// this reason you can define some symbols to describe what is left in
// the back buffer.

// The default of SWAP_SWAP works on an SGI, and will also work (but
// is sub-optimal) on machines that should be SWAP_COPY or SWAP_NODAMAGE.

// contents of back buffer after glXSwapBuffers():
#define UNDEFINED 0 	// unknown
#define SWAP 1		// former front buffer
#define COPY 2		// unchanged
#define NODAMAGE 3	// unchanged even by X expose() events

#ifdef MESA
#define SWAP_TYPE NODAMAGE
#else
#define SWAP_TYPE SWAP
#endif

////////////////////////////////////////////////////////////////

int Fl_Gl_Window::can_do(int a, const int *b) {
  return Fl_Gl_Choice::find(a,b) != 0;
}

void Fl_Gl_Window::show() {
  if (!xid()) {
    if (!g) {
      g = Fl_Gl_Choice::find(mode_,alist);
      if (!g) Fl::abort("Insufficient GL support");
    }
    make_xid(g->vis);
    if (overlay && overlay != this) overlay->show();
  }
  Fl_Window::show();
}

void Fl_Gl_Window::invalidate() {valid(0); if (overlay) overlay->valid(0);}

ulong Fl_Gl_Window::colormap() const {return g->colormap;}

int Fl_Gl_Window::mode(int m, const int *a) {
  if (m == mode_ && a == alist) return 0;
  mode_ = m; alist = a;
  if (shown()) {
    Fl_Gl_Choice *g1 = g;
    g = Fl_Gl_Choice::find(mode_,alist);
    if (!g || g->vis->visualid != g1->vis->visualid || g->d != g1->d) {
      hide(); show();
    }
  }
  return 1;
}

uchar fl_rgbmode;

extern GLXContext fl_first_context; // in Fl_Gl_Choice.C

void Fl_Gl_Window::make_current() {
  if (!context) {
    context = glXCreateContext(fl_display, g->vis, fl_first_context, 1);
    if (!fl_first_context) fl_first_context = (GLXContext)context;
    valid(0);
  }
  glXMakeCurrent(fl_display, xid(), (GLXContext)context);
}

void Fl_Gl_Window::ortho() {
  glLoadIdentity();
  glViewport(0, 0, w(), h());
  glOrtho(0, w(), 0, h(), -1, 1);
}

void Fl_Gl_Window::swap_buffers() {
  glXSwapBuffers(fl_display,xid());
}

void Fl_Gl_Window::flush() {
  make_current();
#if HAVE_OVERLAY
  fl_overlay = g->o;
#endif
  fl_rgbmode = g->r;
  if (g->d) {
#if SWAP_TYPE == UNDEFINED

    clear_damage(~0); draw();
    glXSwapBuffers(fl_display,xid());

#elif SWAP_TYPE == NODAMAGE

    if ((damage()&128) || !valid()) draw();
    glXSwapBuffers(fl_display,xid());

#elif SWAP_TYPE == COPY

    if ((damage()&128) || !valid()) draw();
    else if (damage()&2) {
      glScissor(damage_box.x, h()-damage_box.b,
		damage_box.r-damage_box.x, 
		damage_box.b-damage_box.y);
      glEnable(GL_SCISSOR_TEST);
      draw();
      glDisable(GL_SCISSOR_TEST);
    }
    glXSwapBuffers(fl_display,xid());

#elif SWAP_TYPE == SWAP

    if (overlay == this) {

      // Use CopyPixels, even though it is much slower, on assumption
      // that it is faster than drawing, to erase overlay
      if ((damage()&128) || !valid()) draw();
      else if (damage()&2) {
	glScissor(damage_box.x, h()-damage_box.b,
		  damage_box.r-damage_box.x, 
		  damage_box.b-damage_box.y);
	glEnable(GL_SCISSOR_TEST);
	draw();
	glDisable(GL_SCISSOR_TEST);
      }
      static GLXContext ortho_context;
      if (!ortho_context) {
	// we use a seperate context for this because rasterpos must be 0
	// and depth test needs to be off:
	ortho_context = glXCreateContext(fl_display,g->vis,fl_first_context,1);
	glXMakeCurrent(fl_display, xid(), ortho_context);
	glDisable(GL_DEPTH_TEST);
	glReadBuffer(GL_BACK);
	glDrawBuffer(GL_FRONT);
      } else
	glXMakeCurrent(fl_display, xid(), ortho_context);
      glCopyPixels(0,0,w(),h(),GL_COLOR);
      glXMakeCurrent(fl_display, xid(), (GLXContext)context);

    } else {

      if ((damage()|damage1_)&128 || !valid()) draw();
      else if ((damage()|damage1_)&2) {
	if (!damage_box1.r) {
	  damage_box1.x = damage_box.x;
	  damage_box1.y = damage_box.y;
	  damage_box1.r = damage_box.r;
	  damage_box1.b = damage_box.b;
	} else {
	  if (damage_box.x < damage_box1.x) damage_box1.x = damage_box.x;
	  if (damage_box.y < damage_box1.y) damage_box1.y = damage_box.y;
	  if (damage_box.r > damage_box1.r) damage_box1.r = damage_box.r;
	  if (damage_box.b > damage_box1.b) damage_box1.b = damage_box.b;
	}
	glScissor(damage_box1.x, h()-damage_box1.b,
		  damage_box1.r-damage_box1.x, damage_box1.b-damage_box1.y);
	glEnable(GL_SCISSOR_TEST);
	draw();
	glDisable(GL_SCISSOR_TEST);
      }
      damage_box1.x = damage_box.x;
      damage_box1.y = damage_box.y;
      damage_box1.r = damage_box.r;
      damage_box1.b = damage_box.b;
      damage1_ = damage();
      glXSwapBuffers(fl_display,xid());
    }

#else
    BARF;
#endif

    if (overlay==this) { // fake overlay in front buffer
      glDrawBuffer(GL_FRONT);
      draw_overlay();
      glDrawBuffer(GL_BACK);
      glFlush();
    }

  } else {	// single-buffered context is simpler:

    if (damage()!=2 || !valid()) {
      draw();
    } else {
      glScissor(damage_box.x, h()-damage_box.b,
		damage_box.r-damage_box.x, damage_box.b-damage_box.y);
      glEnable(GL_SCISSOR_TEST);
      draw();
      glDisable(GL_SCISSOR_TEST);
    }
    // fake the overlay, this will work very badly for a single buffer!!!
    if (overlay==this) draw_overlay();
    glFlush();

  }
  clear_damage();
  damage_box.r = 0;
#if HAVE_OVERLAY
  fl_overlay = 0;
#endif
  fl_rgbmode = 0;
  valid(1);
}

void Fl_Gl_Window::resize(int X,int Y,int W,int H) {
  if (W != w() || H != h()) valid(0);
  Fl_Window::resize(X,Y,W,H);
}

void Fl_Gl_Window::hide() {
  if (context) {
    glXMakeCurrent(fl_display,0,0); // Mesa crashes if this is not done
#ifdef GLX_MESA_release_buffers
    glXReleaseBuffersMESA(fl_display,xid());
#endif
    if (context != fl_first_context)
      glXDestroyContext(fl_display,(GLXContext)context);
    context = 0;
  }
  Fl_Window::hide();
}

Fl_Gl_Window::~Fl_Gl_Window() {
  hide();
//  delete overlay; this is done by ~Fl_Group
}

void Fl_Gl_Window::init() {
  box(FL_NO_BOX);
  mode_ = FL_RGB | FL_DEPTH | FL_DOUBLE;
  alist = 0;
  context = 0;
  g = 0;
  overlay = 0;
}

void Fl_Gl_Window::draw_overlay() {}

#endif
