// Fl_Double_Window.C

// A double-buffered window.  This is achieved by using the Xdbe extension,
// or a pixmap if that is not available.

// On systems that support double buffering "naturally" the base
// Fl_Window class will probably do double-buffer and this subclass
// does nothing.

#include <config.h>
#include <FL/Fl.H>
#include <FL/Fl_Double_Window.H>
#include <FL/x.H>
#include <FL/fl_draw.H>

#if HAVE_XDBE

#include <X11/extensions/Xdbe.h>

static int use_xdbe;

static int can_xdbe() {
  static int tried;
  if (!tried) {
    tried = 1;
    int event_base, error_base;
    if (!XdbeQueryExtension(fl_display, &event_base, &error_base)) return 0;
    Drawable root = RootWindow(fl_display,fl_screen);
    int numscreens = 1;
    XdbeScreenVisualInfo *a = XdbeGetVisualInfo(fl_display,&root,&numscreens);
    if (!a) return 0;
    for (int j = 0; j < a->count; j++)
      if (a->visinfo[j].visual == fl_visual->visualid) {use_xdbe = 1; break;}
    XdbeFreeVisualInfo(a);
  }
  return use_xdbe;
}
#endif

void Fl_Double_Window::show() {
  if (shown()) {Fl_Window::show(); return;}
  fl_open_display();
  make_xid(fl_visual);
}

void Fl_Double_Window::flush() {
  if (damage()==2 && pixmap()) { // expose events
    make_current(pixmap_); // sets fl_gc
#if HAVE_XDBE
    if (use_xdbe) {
      fl_clip(damage_box.x, damage_box.y,
	      damage_box.r-damage_box.x, damage_box.b-damage_box.y);
      draw();
      fl_pop_clip();
      XdbeSwapInfo s;
      s.swap_window = xid();
      s.swap_action = XdbeCopied;
      XdbeSwapBuffers(fl_display,&s,1);
    } else
#endif
      XCopyArea(fl_display, pixmap_, xid(), fl_gc,
		damage_box.x, damage_box.y,
		damage_box.r-damage_box.x, damage_box.b-damage_box.y,
		damage_box.x, damage_box.y);
  } else {
    if (!pixmap_) {
#if HAVE_XDBE
      if (can_xdbe())
	pixmap_ = XdbeAllocateBackBufferName(fl_display,xid(),XdbeCopied);
      else
#endif
      pixmap_ = XCreatePixmap(fl_display,xid(),w(),h(),fl_visual->depth);
    }
    make_current(pixmap_);
    draw();
#if HAVE_XDBE
    if (use_xdbe) {
      XdbeSwapInfo s;
      s.swap_window = xid();
      s.swap_action = XdbeCopied;
      XdbeSwapBuffers(fl_display,&s,1);
    } else
#endif
      XCopyArea(fl_display,pixmap_,xid(),fl_gc,0,0,w(),h(),0,0);
  }
  damage_box.r = 0;
  clear_damage();
}

void Fl_Double_Window::resize(int X,int Y,int W,int H) {
  int ow = w();
  int oh = h();
  Fl_Window::resize(X,Y,W,H);
#if HAVE_XDBE
  if (use_xdbe) return;
#endif
  if (pixmap_ && (ow != w() || oh != h())) {
    XFreePixmap(fl_display, pixmap_);
    pixmap_ = 0;
  }
}

void Fl_Double_Window::hide() {
  if (pixmap_) {
#if HAVE_XDBE
    if (!use_xdbe)
#endif
      XFreePixmap(fl_display, pixmap_);
    pixmap_ = 0;
  }
  Fl_Window::hide();
}

Fl_Double_Window::~Fl_Double_Window() {
  hide();
}
