// Fl_Counter.H

// A numerical value with up/down step buttons.  From Forms.

#include <FL/Fl.H>
#include <FL/Fl_Counter.H>
#include <FL/fl_draw.H>
#include <stdio.h>

void Fl_Counter::draw() {
  int i; uchar boxtype[5];

  boxtype[0] = box();
  if (boxtype[0] == FL_UP_BOX) boxtype[0] = FL_DOWN_BOX;
  for (i=1; i<5; i++)
    if (mouseobj == i)
      boxtype[i] = down_box()?down_box():box()+1;
    else
      boxtype[i] = box();

  int xx[5], ww[5];
  if (type() == FL_NORMAL_COUNTER) {
    int W = w()*15/100;
    xx[1] = x();	 ww[1] = W;
    xx[2] = x()+1*W;     ww[2] = W;
    xx[0] = x()+2*W;     ww[0] = w()-4*W;
    xx[3] = x()+w()-2*W; ww[3] = W;
    xx[4] = x()+w()-1*W; ww[4] = W;
  } else {
    int W = w()*20/100;
    xx[2] = x();	 ww[2] = W;
    xx[0] = x()+W;	 ww[0] = w()-2*W;
    xx[3] = x()+w()-1*W; ww[3] = W;
  }

  char str[64];
  sprintf(str,"%.*f",precision(),value());
  if (type() == FL_NORMAL_COUNTER) {
    fl_draw_box(boxtype[1],xx[1],y(),ww[1],h(),color());
    fl_draw_symbol("@-4<<",xx[1],y(),ww[1],h(),color2());
  }
  fl_draw_box(boxtype[2],xx[2],y(),ww[2],h(),color());
  fl_draw_symbol("@-4<",xx[2],y(),ww[2],h(),color2());
  fl_draw_box(boxtype[0],xx[0],y(),ww[0],h(),color());
  fl_font(labelfont(),labelsize());
  fl_color(labelcolor());
  fl_draw(str,xx[0],y(),ww[0],h(),FL_ALIGN_CENTER);
  fl_draw_box(boxtype[3],xx[3],y(),ww[3],h(),color());
  fl_draw_symbol("@-4>",xx[3],y(),ww[3],h(),color2());
  if (type() == FL_NORMAL_COUNTER) {
    fl_draw_box(boxtype[4],xx[4],y(),ww[4],h(),color());
    fl_draw_symbol("@-4>>",xx[4],y(),ww[4],h(),color2());
  }
  draw_label();
}

void Fl_Counter::increment() {
  if (!mouseobj) return;
  float v = value();
  switch (mouseobj) {
  case 1: v -= lstep(); break;
  case 2: v -= step(); break;
  case 3: v += step(); break;
  case 4: v += lstep(); break;
  }
  if (v<min) v = min;
  if (v>max) v = max;
  if (value(v)) {
    if (when()&FL_WHEN_CHANGED) do_callback(); else set_changed();
  }
}

#define INITIALREPEAT .5
#define REPEAT .1

void Fl_Counter::repeat_callback(void *v) {
  Fl_Counter *b = (Fl_Counter*)v;
  if (b->mouseobj) {
    Fl::add_timeout(REPEAT,repeat_callback,b);
    b->increment();
  }
}

int Fl_Counter::calc_mouseobj() {
  if (type() == FL_NORMAL_COUNTER) {
    int W = w()*15/100;
    if (Fl::event_inside(x(),y(),W,h())) return 1;
    if (Fl::event_inside(x()+W,y(),W,h())) return 2;
    if (Fl::event_inside(x()+w()-2*W,y(),W,h())) return 3;
    if (Fl::event_inside(x()+w()-W,y(),W,h())) return 4;
  } else {
    int W = w()*20/100;
    if (Fl::event_inside(x(),y(),W,h())) return 2;
    if (Fl::event_inside(x()+w()-W,y(),W,h())) return 3;
  }
  return -1;
}

int Fl_Counter::handle(int event) {
  int i;
  switch (event) {
  case FL_RELEASE:
    if (mouseobj) {
      Fl::remove_timeout(repeat_callback,this);
      mouseobj = 0;
      redraw();
    }
    if (!(when() & FL_WHEN_RELEASE)) return 1;
    if (changed() || when()&FL_WHEN_NOT_CHANGED) {
      clear_changed(); do_callback();}
    return 1;
  case FL_PUSH:
  case FL_DRAG:
    i = calc_mouseobj();
    if (i != mouseobj) {
      Fl::remove_timeout(repeat_callback,this);
      mouseobj = i;
      if (i) Fl::add_timeout(INITIALREPEAT,repeat_callback,this);
      increment();
      redraw();
    }
    return 1;
  default:
    return 0;
  }
}

Fl_Counter::~Fl_Counter() {
  Fl::remove_timeout(repeat_callback,this);
}

Fl_Counter::Fl_Counter(int x,int y,int w,int h,const char *l)
  : Fl_Object(x,y,w,h,l) {
    box(FL_UP_BOX);
    down_box(0);
    color2(FL_INACTIVE_COLOR); // was FL_BLUE
    align(FL_ALIGN_BOTTOM);
    min = -1000000.0;
    max = 1000000.0;
    step_ = 0.1;
    lstep_ = 1.0;
    value_ = 0.0;
    precision_ = 1;
    mouseobj = 0;
}

int Fl_Counter::value(float v) {
  clear_changed();
  if (v != value_) {value_ = v; redraw(); return 1;}
  return 0;
}
