// Fl_Browser_.C

// This is the base class for browsers.  To be useful it must be
// subclassed and several virtual functions defined.  The
// Forms-compatable browser and the file chooser's browser are
// subclassed off of this.

// Yes, I know this should be a template...

// This has been designed so that the subclass has complete control
// over the storage of the data, although because next() and prev()
// functions are used to index, it works best as a linked list or as a
// large block of characters in which the line breaks must be searched
// for.

// A great deal of work has been done so that the "height" of a data
// object does not need to be determined until it is drawn.  Very
// useful if actually figuring out the size of an object requires
// accessing image data or doing stat() on a file or doing some other
// slow operation.

#include <FL/Fl.H>
#include <FL/Fl_Object.H>
#include <FL/Fl_Slider.H>
#include <FL/Fl_Browser_.H>
#include <FL/fl_draw.H>

/* redraw bits:
   1 = redraw children (the scrollbar)
   2 = redraw one or two items
   4 = redraw all items
*/

static void scrollbar_callback(Fl_Object *s,void *) {
  ((Fl_Browser_ *)(s->parent()))->position(int(((Fl_Slider *)s)->value()));
}

// you can override this function to get different types of
// scrollbars, to put the scrollbar on the right, or to force
// the scrollbar on at all times:
void Fl_Browser_::show_scrollbar(int scrolling) {
  if (scrolling) {
    if (!scrollbar) {
      scrollbar = new Fl_Slider(x(),y(),21,h(),0);
      scrollbar->callback(scrollbar_callback);
      scrollbar->parent((Fl_Group*)this);
      redraw();
    } else if (!scrollbar->visible()) {
      scrollbar->show();
      redraw();
    }
  } else {
    if (scrollbar && scrollbar->visible()) {
      scrollbar->hide();
      redraw();
    }
  }
}

void Fl_Browser_::resize(int X,int Y,int W,int H) {
  if (scrollbar)
    scrollbar->resize(
	scrollbar->x()>x() ? X+W-scrollbar->w() : X, Y, scrollbar->w(), H);
  Fl_Object::resize(X,Y,W,H);
}

// Cause minimal update to redraw the given item:
void Fl_Browser_::redraw_line(void *l) {
  if (!redraw1 || redraw1 == l) {redraw1 = l; damage(2);}
  else if (!redraw2 || redraw2 == l) {redraw2 = l; damage(2);}
  else damage(4);
}

// Figure out top() based on position():
void Fl_Browser_::update_top() {
  if (!top_) top_ = item_first();
  if (position_ != real_position_) {
    void *l;
    int ly;
    int y = position_;
    // start from either head or current position, whichever is closer:
    if (!top_ || y <= real_position_/2) {
      l = item_first();
      ly = 0;
    } else {
      l = top_;
      ly = real_position_-offset_;
    }
    if (!l) {
      top_ = 0;
      offset_ = 0;
      real_position_ = 0;
    } else {
      int h = item_quick_height(l);
      // step through list until we find line containing this point:
      while (ly > y) {
	void *l1 = item_prev(l);
	if (!l1) {ly = 0; break;} // hit the top
	l = l1;
	h = item_quick_height(l);
	ly -= h;
      }
      while (ly+h <= y) {
	void *l1 = item_next(l);
	if (!l1) {y = ly+h-1; break;}
	l = l1;
	ly += h;
	h = item_quick_height(l);
      }
      // use it:
      top_ = l;
      offset_ = y-ly;
      real_position_ = y;
    }
    damage(4);
  }
}

// Change position(), top() will update when update_top() is called
// (probably by draw() or handle()):
void Fl_Browser_::position(int y) {
  if (y < 0) y = 0;
  if (y == position_) return;
  position_ = y;
  if (y != real_position_) redraw_lines();
}

// Tell whether item is currently displayed:
int Fl_Browser_::displayed(void *x) const {
  int yy = h()-Fl::box_dh(box())+offset_;
  for (void *l = top_; l && yy > 0; l = item_next(l)) {
    if (l == x) return 1;
    yy -= item_height(l);
  }
  return 0;
}

// Insure this item is displayed:
// Messy because we have no idea if it is before top or after bottom:
void Fl_Browser_::display(void *x) {
  if (!top_) top_ = item_first();
  if (x == item_first()) {position(0); return;}
  void *l = top_;
  int Y = -offset_;
  // see if it is at the top or just above it:
  if (l == x) {position(real_position_+Y); return;} // scroll up a bit
  void *lp = item_prev(l);
  if (lp == x) {position(real_position_+Y-item_quick_height(lp)); return;}
  int H = h()-Fl::box_dh(box());
  // search forward for it:
  for (; l; l = item_next(l)) {
    int h1 = item_quick_height(l);
    if (l == x) {
      if (Y <= H) { // it is visible or right at bottom
	Y = Y+h1-H; // find where bottom edge is
	if (Y > 0) position(real_position_+Y); // scroll down a bit
      } else {
	position(real_position_+Y-(H-h1)/2); // center it
      }
      return;
    }
    Y += h1;
  }
  // search backward for it, if found center it:
  l = lp;
  Y = -offset_;
  for (; l; l = item_prev(l)) {
    int h1 = item_quick_height(l);
    Y -= h1;
    if (l == x) {
      if (Y + h1 >= 0) position(real_position_+Y);
      else position(real_position_+Y-(H-h1)/2);
      return;
    }
  }
}

// redraw, has side effect of updating top and setting scrollbar:
void Fl_Browser_::draw() {
  update_top();
  int X;
  int Y = y()+Fl::box_dy(box());
  int W;
  int H = h()-Fl::box_dh(box());
  int full_height_ = full_height();
J1:
  // see if scrollbar needs to be switched on/off:
  if (position_ || full_height_>H) {
    show_scrollbar(1);
  } else {
    top_ = item_first(); real_position_ = offset_ = 0;
    show_scrollbar(0);
  }
  X = leftedge();
  if (damage() & 128) { // redraw the box if full redraw
    fl_draw_box(box(),X,y(),x()+w()-X,h(),color());
  }
  X += Fl::box_dx(box());
  W = w() - Fl::box_dw(box());
  if (scrollbar && scrollbar->visible()) W -= scrollbar->w();
  fl_clip(X,Y,W,H);
  // for each line, draw it if full redraw or scrolled.  Erase background
  // if not a full redraw or if it is selected:
  void *l = top();
  int yy = Y-offset_;
  for (; l && yy<Y+H; l = item_next(l)) {
    int hh = item_height(l); if (hh <= 0) continue;
    if ((damage()&(4|1)) || l == redraw1 || l == redraw2) {
      if (item_selected(l)) {fl_color(color2()); fl_rectf(X,yy,W,hh);}
      else if (!(damage()&128)) {fl_color(color()); fl_rectf(X,yy,W,hh);}
      item_draw(l,X,yy,W,hh);
    }
    yy += hh;
  }
  // erase the area below last line:
  if (!(damage()&128) && yy < Y+H) {
    fl_color(color());
    fl_rectf(X,yy,W,Y+H-yy);
  }
  fl_pop_clip();
  redraw1 = redraw2 = 0;

  // see if changes to full_height caused by calls to slow_height
  // caused scrollbar state to change, in which case we have to redraw:
  full_height_ = full_height();
  if (scrollbar && scrollbar->visible()) {
    if (!position_ && full_height_ <= H) goto J1;
    scrollbar->step(1);
    scrollbar->increment(incr_height());
    scrollbar->value(position_,H,0,full_height_);
    if (damage()&128) scrollbar->redraw_child();
    scrollbar->draw_child();
  } else {
    if (position_ || full_height_ > H) goto J1;
  }
  draw_label();
}

// Quick way to delete and reset everything:
void Fl_Browser_::new_list() {
  top_ = 0;
  position_ = real_position_ = 0;
  selection_ = 0;
  offset_ = 0;
  redraw();
}

// Tell it that this item is going away, and that this must remove
// all pointers to it:
void Fl_Browser_::deleting(void *l) {
  if (displayed(l)) redraw_lines();
  if (l == selection_) selection_ = 0;
  if (l == top_) {
    real_position_ -= offset_;
    offset_ = 0;
    top_ = item_next(l);
    if (!top_) top_ = item_prev(l);
  }
}

void Fl_Browser_::replacing(void *a,void *b) {
  if (a == selection_) selection_ = b;
  if (a == top_) top_ = b;
}

void *Fl_Browser_::find_item(int my) {
  update_top();
  int Y = y()+Fl::box_dy(box());
  int H = h()-Fl::box_dh(box());
  void *l;
  int yy = Y-offset_;
  for (l = top_; l; l = item_next(l)) {
    int hh = item_height(l); if (hh <= 0) continue;
    yy += hh;
    if (my <= yy || yy>=Y+H) return l;
  }
  return 0;
}

int Fl_Browser_::select(void *l,int i,int docallbacks) {
  if (type() == FL_MULTI_BROWSER) {
    selection_ = l;
    if ((!i)==(!item_selected(l))) return 0;
    item_select(l,i);
    redraw_line(l);
  } else {
    if (i && selection_ == l) return 0;
    if (!i && selection_ != l) return 0;
    if (selection_) {
      item_select(selection_,0);
      redraw_line(selection_);
      selection_ = 0;
    }
    if (i) {
      item_select(l,1);
      selection_ = l;
      redraw_line(l);
    }
  }	    
  Fl::event_clicks(0);
  if (docallbacks) do_callback();
  return 1;
}

int Fl_Browser_::deselect(int docallbacks) {
  if (type() == FL_MULTI_BROWSER) {
    int change = 0;
    for (void *p = item_first(); p; p = item_next(p))
      change |= select(p,0,docallbacks);
    return change;
  } else {
    if (!selection_) return 0;
    item_select(selection_,0);
    redraw_line(selection_);
    selection_ = 0;
    return 1;
  }
}

int Fl_Browser_::select_only(void *l,int docallbacks) {
  if (!l) return deselect(docallbacks);
  int change = 0;
  if (type() == FL_MULTI_BROWSER) {
    for (void *p = item_first(); p; p = item_next(p))
      if (p != l) change |= select(p,0,docallbacks);
  }
  change |= select(l,1,docallbacks);
  display(l);
  return change;
}

int Fl_Browser_::leftedge() {
  return (scrollbar&&scrollbar->visible()&&scrollbar->x()<=x() ?
	  scrollbar->w() : 0) + x();
}

int Fl_Browser_::handle(int event) {
  int X = leftedge() + Fl::box_dx(box());
  int Y = y() + Fl::box_dy(box());
  // int W = w() - Fl::box_dw(box());
  // if (scrollbar && scrollbar->visible()) W -= scrollbar->w();
  int H = h() - Fl::box_dh(box());
  int my;
  static char inscrollbar;
  static char change;
  static char whichway;
  static int py;
  static void *prev_selection;
  switch (event) {
  case FL_PUSH:
    if (scrollbar && scrollbar->visible() && Fl::event_x() < X)
      {inscrollbar = 1; break;}
    if (type() == FL_SELECT_BROWSER) deselect();
    inscrollbar = 0;
    my = py = Fl::event_y();
    change = 0;
    prev_selection = selection_;
    if (type() == FL_NORMAL_BROWSER || !top_)
      ;
    else if (type() == FL_MULTI_BROWSER) {
      void *l = find_item(my);
      selection_ = l;
      whichway = 1;
      if (Fl::event_shift()) { // toggle selection:
	if (l) {
	  whichway = !item_selected(l);
	  change = select(l,whichway,when() & FL_WHEN_CHANGED);
	}
      } else {
	change = select_only(l,when() & FL_WHEN_CHANGED);
      }
    } else {
      change = select_only(find_item(my),when() & FL_WHEN_CHANGED);
    }
    return 1;
  case FL_DRAG:
    if (inscrollbar) break;
    // do the scrolling first:
    my = Fl::event_y();
    if (my < Y && my < py) {
      int p = real_position_+my-Y;
      if (p<0) p = 0;
      position(p);
    } else if (my > Y+H && my > py) {
      int p = real_position_+my-(Y+H);
      int h = full_height()-H; if (p > h) p = h;
      if (p<0) p = 0;
      position(p);
    }
    if (type() == FL_NORMAL_BROWSER || !top_)
      ;
    else if (type() == FL_MULTI_BROWSER) {
      void *l = find_item(my);
      void *t,*b; // this will be the range to change
      if (my > py) { // go down
	t = selection_ ? item_next(selection_) : 0;
	b = l ? item_next(l) : 0;
      } else {	// go up
	t = l;
	b = selection_;
      }
      for (; t && t != b; t = item_next(t))
	change |= select(t,whichway,when() & FL_WHEN_CHANGED);
      if (l) selection_ = l;
    } else {
      void *l;
      if (Fl::event_x()<x() || Fl::event_x()>x()+w()) l = prev_selection;
      else l = find_item(my);
      select_only(l,when() & FL_WHEN_CHANGED);
      change = (l != prev_selection);
    }
    py = my;
    return 1;
  case FL_RELEASE:
    if (inscrollbar) break;
    if (type() == FL_SELECT_BROWSER) {
      void *t = selection_; deselect(); selection_ = t;
    }
    if (change) {
      if (when() & FL_WHEN_RELEASE) do_callback();
      else if (!(when()&FL_WHEN_CHANGED)) set_changed();
    } else {
      if (when() & FL_WHEN_NOT_CHANGED) do_callback();
    }
    return 1;
  case FL_SHORTCUT:
    if (type() == FL_HOLD_BROWSER) {
      void *l = selection_; if (!l) l = top_; if (!l) l = item_first();
      if (l) switch (Fl::event_key()) {
      case FL_Down:
	while ((l = item_next(l)))
	  if (item_height(l)>0) {select_only(l,1); break;}
	return 1;
      case FL_Up:
	while ((l = item_prev(l))) if (item_height(l)>0) {
	  select_only(l,1); break;}
	return 1;
      }
    }
    if (!top_ || !scrollbar || !scrollbar->visible()) break;
    switch (Fl::event_key()) {
    case FL_Up:
      my = item_quick_height(top_); if (my<10) my = 10;
      position(real_position_-my);
      return 1;
    case FL_Down:
      my = item_quick_height(top_); if (my<10) my = 10;
      position(real_position_+my);
      return 1;
    case FL_Page_Up: position(real_position_-(h()-8)); return 1;
    case FL_Page_Down: position(real_position_+(h()-8)); return 1;
    case FL_Home: position(0); return 1;
    case FL_End: position(full_height()-h()+Fl::box_dh(box())); return 1;
    }
    break;
  }

  // pass all other events to scrollbar:
  if (scrollbar && scrollbar->visible()) return scrollbar->handle(event);
  return 0;
}

Fl_Browser_::Fl_Browser_(int x,int y,int w,int h, const char *l)
  : Fl_Object(x,y,w,h,l) {
  box(FL_DOWN_BOX);
  align(FL_ALIGN_BOTTOM);
  position_ = real_position_ = 0;
  offset_ = 0;
  top_ = 0;
  when(FL_WHEN_RELEASE | FL_WHEN_NOT_CHANGED);
  selection_ = 0;
  scrollbar = 0;
  selection_color(FL_WHITE); // Forms used yellow (3)
}

// Default versions of some of the virtual functions:

int Fl_Browser_::item_quick_height(void *l) const {
  return item_height(l);
}

int Fl_Browser_::incr_height() const {
  return item_quick_height(item_first());
}

int Fl_Browser_::full_height() const {
  int t = 0;
  for (void *p = item_first(); p; p = item_next(p))
    t += item_quick_height(p);
  return t;
}

void Fl_Browser_::item_select(void *,int ) {}

int Fl_Browser_::item_selected(void *l) const {return l==selection_;}

// end of Fl_Browser_.C
