// Fl_Adjuster.C

// Fl object for drag-adjusting a floating point value.

#include <FL/Fl.H>
#include <FL/Fl_Adjuster.H>
#include <FL/Fl_Bitmap.H>
#include <FL/fl_draw.H>

#include "fastarrow.h"
static Fl_Bitmap fastarrow(fastarrow_bits,fastarrow_width,fastarrow_height);
#include "mediumarrow.h"
static Fl_Bitmap mediumarrow(mediumarrow_bits,mediumarrow_width,mediumarrow_height);
#include "slowarrow.h"
static Fl_Bitmap slowarrow(slowarrow_bits,slowarrow_width,slowarrow_height);

void Fl_Adjuster::draw() {
  int dx,dy,W,H;
  if (w()>=h()) {
    dx = W = w()/3;
    dy = 0; H = h();
  } else {
    dx = 0; W = w();
    dy = H = h()/3;
  }
  fl_draw_box(drag==1?(uchar)FL_DOWN_BOX:box(),x(),y()+2*dy,W,H,color());
  fl_draw_box(drag==2?(uchar)FL_DOWN_BOX:box(),x()+dx,y()+dy,W,H,color());
  fl_draw_box(drag==3?(uchar)FL_DOWN_BOX:box(),x()+2*dx,y(),W,H,color());
  fl_color(labelcolor());
  fl_draw(&fastarrow,x(),y()+2*dy,W,H);
  fl_draw(&mediumarrow,x()+dx,y()+dy,W,H);
  fl_draw(&slowarrow,x()+2*dx,y(),W,H);
  draw_label();
}

int Fl_Adjuster::handle(int event) {
  float v;
  int delta;
  float s;
  int mx = Fl::event_x();
  static int changed;
  switch (event) {
  case FL_PUSH:
    ix = mx;
    if (w()>=h())
      drag = 3*(mx-x())/w() + 1;
    else
      drag = 3-3*(Fl::event_y()-y()-1)/h();
    ival = val;
    redraw();
    changed = 0;
    return 1;
  case FL_DRAG:
    if (w() >= h()) {
      delta = x()+(drag-1)*w()/3;	// left edge of button
      if (mx < delta)
	delta = mx-delta;
      else if (mx > delta+w()/3) // right edge of button
	delta = mx-delta-w()/3;
      else
	delta = 0;
    } else {
      if (mx < x())
	delta = mx-x();
      else if (mx > x()+w())
	delta = mx-x()-w();
      else
	delta = 0;
    }
    s = step_ ? step_ : (max-min)/100;
    switch (drag) {
    case 2: v = ival + delta*s/10; break;
    case 3: v = ival + delta*s/100; break;
    default:v = ival + delta*s; break;
    }
    if (ival > min && v < min) v = min;
    if (ival < max && v > max) v = max;
    if (v != val) {val = v; changed = 1; do_callback();}
    return 1;
  case FL_RELEASE:
    if (!changed) {
      if (Fl::event_state()&15) delta = -10;
      else delta = 10;
      s = step_ ? step_ : (max-min)/100;
      switch (drag) {
      case 2: v = ival + delta*s/10; break;
      case 3: v = ival + delta*s/100; break;
      default:v = ival + delta*s; break;
      }
      if (ival > min && v < min) v = min;
      if (ival < max && v > max) v = max;
      if (v != val) {val = v; do_callback();}
    }
    drag = 0;
    redraw();
    return 1;
  }
  return 0;
}

Fl_Adjuster::Fl_Adjuster(int x,int y,int w,int h,const char *l)
: Fl_Object(x,y,w,h,l) {
    align(FL_ALIGN_BOTTOM);
    box(FL_UP_BOX);
    min = 0;
    max = 1;
    val = .5;
    step_ = 0;
    drag = 0;
}
