// Fl.C

// The FL user interface library version 0.98
// Copyright (C) 1998 Digital Domain

// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.

// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// Written by Bill Spitzak spitzak@d2.com
// Digital Domain
// 300 Rose Avenue
// Venice, CA 90291

#include <config.h>
#define _FL_IMPLEMENTATION_H
#include <FL/Fl.H>
#include <FL/x.H>
#include <FL/Fl_Poll.H>
#include <FL/Fl_Window.H>
#include <stdlib.h>

// Fl *fl;	// for back-compatabilitye fl->foo() calls
int Fl::damage_;
Fl_Object *Fl::belowmouse_;
Fl_Object *Fl::pushed_;
Fl_Object *Fl::focus_;
Fl_Object *Fl::selection_owner_;
Display *fl_display;
int fl_screen;
XVisualInfo *fl_visual;
Colormap fl_colormap;	// this is the one used by fl_color()
int Fl::e_x, Fl::e_y, Fl::e_x_root, Fl::e_y_root;
unsigned int Fl::e_state;
int Fl::e_clicks;
int Fl::e_is_click;
int Fl::e_keysym;
char *Fl::e_text;
ulong Fl::e_time;
Fl_Window *Fl::first_;

static void do_queued_events(void *);

static void fd_callback(int,void *) {
  if (XEventsQueued(fl_display,1)) do_queued_events(0);
}

static int test(void *) {return XQLength(fl_display);}

Atom fl_wm_delete_window;
Atom fl__wm_quit_app;
Atom fl_wm_protocols;
Atom fl__motif_wm_hints;

void fl_open_display() {
  if (fl_display) return;

  Display *d = XOpenDisplay(0);
  if (!d) Fl::abort("Can't open display: %s",XDisplayName(0));

  fl_display = d;

  fl_wm_delete_window = XInternAtom(d,"WM_DELETE_WINDOW",0);
  fl__wm_quit_app = XInternAtom(d,"_WM_QUIT_APP",0);
//fl_wm_save_yourself = XInternAtom(d,"WM_SAVE_YOURSELF",0);
  fl_wm_protocols = XInternAtom(d,"WM_PROTOCOLS",0);
  fl__motif_wm_hints = XInternAtom(d,"_MOTIF_WM_HINTS",0);
  fl_poll.add(ConnectionNumber(d), POLLIN, fd_callback);
  fl_poll.add_test(test, do_queued_events);

  fl_screen = DefaultScreen(fl_display);
// construct an XVisualInfo that matches the default Visual:
  XVisualInfo templt; int num;
  templt.visualid = XVisualIDFromVisual(DefaultVisual(fl_display,fl_screen));
  fl_visual = XGetVisualInfo(fl_display, VisualIDMask, &templt, &num);
  fl_colormap = DefaultColormap(fl_display,fl_screen);
}

void fl_close_display() {
  fl_poll.remove(ConnectionNumber(fl_display));
  fl_poll.remove_test(test, do_queued_events);
  XCloseDisplay(fl_display);
}

int Fl::wait()		{flush(); return fl_poll.wait();}
float Fl::wait(float t)	{flush(); return fl_poll.wait(t);}
float Fl::reset()	{return fl_poll.reset();}
int Fl::check()		{flush(); return fl_poll.check();}
int Fl::ready()		{return fl_poll.ready();}
void Fl::add_timeout(float t,void (*cb)(void*),void* v) {
  fl_poll.add_timeout(t,cb,v);}
void Fl::remove_timeout(void (*cb)(void*), void* v) {
  fl_poll.remove_timeout(cb,v);}
void Fl::add_fd(int fd, int w, void (*cb)(int, void*), void* v) {
  fl_poll.add(fd,w,cb,v);}
void Fl::add_fd(int fd, void (*cb)(int, void*), void* v) {
  fl_poll.add(fd,POLLIN,cb,v);}
void Fl::remove_fd(int fd) {fl_poll.remove(fd);}
void Fl::set_idle(void (*cb)()) {fl_poll.set_idle(cb);}

int Fl::h() /*const*/ {
  fl_open_display();
  return DisplayHeight(fl_display,fl_screen);}
int Fl::w() /*const*/ {
  fl_open_display();
  return DisplayWidth(fl_display,fl_screen);}

int Fl::event_inside(int x,int y,int w,int h) /*const*/ {
  int mx = event_x();
  int my = event_y();
  return (mx >= x && mx < x+w && my >= y && my < y+h);
}

int Fl::event_inside(const Fl_Object *o) /*const*/ {
  return event_inside(o->x(),o->y(),o->w(),o->h());
}

void Fl::mouse_position(int &x, int &y) {
  fl_open_display();
  Window root = RootWindow(fl_display, fl_screen);
  Window c; int mx,my,cx,cy; unsigned int mask;
  XQueryPointer(fl_display,root,&root,&c,&mx,&my,&cx,&cy,&mask);
  x = mx;
  y = my;
}

////////////////////////////////////////////////////////////////

void Fl::flush() {
  if (!fl_display) return;
  if (damage()) {
    damage_ = 0;
    for (Fl_Window *f = first(); f; f = f->nextwindow()) {
      if (f->flags() & Fl_Window::FL_MAPRAISED) f->mapraised();
      if (f->damage() && f->visible()) f->flush();
    }
  }
  XFlush(fl_display);
}

void Fl::redraw() {
  for (Fl_Window *w = first(); w; w = w->nextwindow()) w->redraw();
}

////////////////////////////////////////////////////////////////

void Fl::focus(Fl_Object *o) {
  Fl_Object *p = focus_;
  if (o != p) {
    for (; p && !p->contains(o); p = p->parent()) p->handle(FL_UNFOCUS);
    focus_ = o;
  }
}

void Fl::belowmouse(Fl_Object *o) {
  Fl_Object *p = belowmouse_;
  if (o != p) {
    event_is_click(0);
    for (; p && !p->contains(o); p = p->parent()) p->handle(FL_LEAVE);
    belowmouse_ = o;
  }
}

// These are used to adjust the coordinates of mouse events sent directly
// to children of child windows:
int fl_eventdx, fl_eventdy; // set by Fl_Window.C
static int pushed_deltax, pushed_deltay;

void Fl::pushed(Fl_Object *o) {
  pushed_deltax = fl_eventdx; pushed_deltay = fl_eventdy;
  pushed_ = o;
}

Fl_Window *fl_xfocus;	// which window X thinks has focus
Fl_Window *fl_xmousewin; // which window X thinks has FL_ENTER

static inline int modal() {
  return Fl::first()->modal();
}

// Update them all according to current modal state.
// This is called whenever a window is added or hidden, and whenever
// X says the focus or mouse window have changed:
void fl_fix_focus() {
  Fl_Window *w = fl_xfocus;
  if (w) {
    if (modal()) w = Fl::first();
    if (!w->contains(Fl::focus()))
      if (!w->take_focus()) Fl::focus(w);
  } else
    Fl::focus(0);

  if (Fl::pushed()) {
    if (modal() && !Fl::first()->contains(Fl::pushed()))
      Fl::pushed_ = Fl::first();
    return; // don't change belowmouse() when pushed() is true
  }

  w = fl_xmousewin;
  if (w) {
    if (modal()) w = Fl::first();
    if (!w->contains(Fl::belowmouse())) {
      Fl::belowmouse(w); w->handle(FL_ENTER);}
  } else
    Fl::belowmouse(0);
}

Fl_Object *fl_selection_requestor; // object that did paste()
Fl_Window *fl_selection_window; // window X thinks selection belongs to

// call this to free a selection (or change the owner):
void Fl::selection_owner(Fl_Object *owner) {
  if (owner && !fl_selection_window) {
    Fl_Window *w = fl_selection_window = Fl::first();
    if (w)		// we lose if there are no windows
      XSetSelectionOwner(fl_display, XA_PRIMARY, w->xid(), Fl::event_time());
  }
  if (selection_owner_ && owner != selection_owner_)
    selection_owner_->handle(FL_SELECTIONCLEAR);
  selection_owner_ = owner;
}

////////////////////////////////////////////////////////////////

Fl_Window* Fl_Window::find(ulong xid) {
  Fl_Window *window;
  for (Fl_Window **pp = &Fl::first_; (window = *pp); pp = &window->nextwindow_)
    if (window->xid() == xid) {
      if (window != Fl::first_ && !Fl::first_->modal()) {
	// make this window be first to speed up searches
	*pp = window->nextwindow_;
	window->nextwindow_ = Fl::first_;
	Fl::first_ = window;
      }
      return window;
    }
  return 0;
}

XEvent fl_xevent; // the last xevent read

// record event mouse position and state from an XEvent
// checkdouble is true for button and keyboard down events

static void set_event_xy(int checkdouble=0) {
  static int px, py;
  static ulong ptime;
  Fl::e_x_root = fl_xevent.xbutton.x_root;
  Fl::e_y_root = fl_xevent.xbutton.y_root;
  Fl::e_x = fl_xevent.xbutton.x;
  Fl::e_y = fl_xevent.xbutton.y;
  Fl::e_state = fl_xevent.xbutton.state;
  Fl::e_time = fl_xevent.xbutton.time;
  if (abs(Fl::e_x_root-px)+abs(Fl::e_y_root-py)>3 || Fl::e_time>=ptime+1000) {
    Fl::e_is_click = 0;
  }
  if (checkdouble) {
    if (Fl::e_is_click == Fl::e_keysym)
      Fl::e_clicks++;
    else {
      Fl::e_clicks = 0;
      Fl::e_is_click = Fl::e_keysym;
    }
    px = Fl::e_x_root;
    py = Fl::e_y_root;
    ptime = Fl::e_time;
  }
}

struct handler_link {
  int (*handle)(int);
  const handler_link *next;
};

static int backstop_handle(int event) {
  // raise windows that are clicked on:
  if (event == FL_PUSH && Fl::first()) Fl::first()->show();
  return 1;
}

static const handler_link backstop_link = {backstop_handle, 0};

static const handler_link *handlers = &backstop_link;

void Fl::add_handler(int (*h)(int)) {
  handler_link *l = new handler_link;
  l->handle = h;
  l->next = handlers;
  handlers = l;
}

static void send_event(int event, Fl_Object *object) {
  if (object && object->handle(event)) return;
  for (const handler_link *h = handlers; ; h = h->next)
    if (h->handle(event)) return;
}

static void send_pushed(int event) {
  // adjust event position if being sent to a child X window:
  Fl::e_x -= pushed_deltax;
  Fl::e_y -= pushed_deltay;
  send_event(event, Fl::pushed());
}

void Fl::default_atclose(Fl_Window* window, void* v) {
  if (!window) exit(0); // Quit
  if (!window->other_windows()) // no other windows?
    // recursively call this to quit, so user's atclose() is called:
    atclose(0,v);
  else
    // hide this window if there are others:
    window->hide();
}
void (*Fl::atclose)(Fl_Window*, void*) = Fl::default_atclose;

static void do_queued_events(void *) {

  while (XEventsQueued(fl_display,QueuedAfterReading)) {
    XNextEvent(fl_display, &fl_xevent);
    Fl_Window *window = Fl_Window::find(fl_xevent.xany.window);
    if (!window) goto DEFAULT;

    switch (fl_xevent.type) {

    case ClientMessage:
      if ((Atom)(fl_xevent.xclient.data.l[0])==fl_wm_delete_window) {
	// Close box clicked
	if (modal()) {
	  if (window != Fl::first()) break;
	} else {
	  Fl::atclose(window, window->user_data());
	}
	window->do_callback();
      } else if ((Atom)fl_xevent.xclient.data.l[0]==fl__wm_quit_app) {
	// 4DWM "Quit" picked off window menu
	Fl::atclose(0, window->user_data());
      } else goto DEFAULT;
      break;

    case UnmapNotify:
      window->clear_visible();
      send_event(FL_HIDE, window);
      break;

    case MapNotify:
      window->set_visible();
      send_event(FL_SHOW, window);
      break;

    case Expose:
      if (modal() && window!=Fl::first()) Fl::first()->show();
      window->expose(fl_xevent.xexpose.x, fl_xevent.xexpose.y,
		     fl_xevent.xexpose.width, fl_xevent.xexpose.height);
      break;

    case ButtonPress:
      Fl::e_keysym = Fl::BUTTON + fl_xevent.xbutton.button;
      set_event_xy(1);
      Fl::e_state |= (0x80 << fl_xevent.xbutton.button);
      if (modal()) {
	window = Fl::first();
	Fl::e_x = Fl::e_x_root - window->x_root();
	Fl::e_y = Fl::e_y_root - window->y_root();
      }
      if (Fl::pushed()) {send_pushed(FL_PUSH); break;}
      Fl::pushed_ = window; // make sure something is turned on
      send_event(FL_PUSH, window);
      break;

    case ButtonRelease:
      Fl::e_keysym = Fl::BUTTON + fl_xevent.xbutton.button;
      set_event_xy();
      Fl::e_state &= ~(0x80 << fl_xevent.xbutton.button);
      { // unfortunately pushed() must be zero when the callback() is done:
	Fl_Object* o = Fl::pushed_; Fl::pushed_ = 0;
	if (!o) o = Fl::first();
	Fl::e_x -= pushed_deltax;
	Fl::e_y -= pushed_deltay;
	send_event(FL_RELEASE, o);
      }
      fl_fix_focus();
      if (window->contains(Fl::belowmouse())) window->handle(FL_MOVE);
      break;

    case FocusIn:
      Fl::e_keysym = 0; // make sure it is not confused with navigation key
      fl_xfocus = window;
      fl_fix_focus();
      break;

    case FocusOut:
      Fl::e_keysym = 0; // make sure it is not confused with navigation key
      fl_xfocus = 0;
      fl_fix_focus();
      break;

    case KeyPress: {
      static int got_backspace;
      static XComposeStatus compose;
      static char buffer[21];
      KeySym keysym;
      int len = XLookupString(&(fl_xevent.xkey),buffer,20,&keysym,&compose);
      if (!len) {
	if (keysym < 0x400) {
	  // turn all latin-2,3,4 characters into 8-bit codes:
	  buffer[0] = keysym;
	  len = 1;
	} else if (keysym >= 0xFF95 && keysym < 0xFFA0) {
	  // turn numeric keypad into numbers even if NumLock is off
	  // This lookup table turns the XK_KP_* functions back into the ascii
	  // characters.  This won't work on non-PC layout keyboards, but are
	  // there any of those left??
	  buffer[0] = "7486293150."[keysym-0xFF95];
	  len = 1;
#if 0
	} else if (keysym >= 0xFFA0 && keysym < 0xFFBE) {
	  // turn other numeric keys into codes, even if Xlib didn't do it
	  // I'm not sure if any Xlib's don't do this:
	  buffer[0] = keysym - 0xFF80;
	  len = 1;
#endif
	}
      }
      buffer[len] = 0;
      if (!got_backspace) {
	// Backspace kludge: until user hits the backspace key, assumme
	// it is missing and use the Delete key for that purpose:
	if (keysym == FL_Delete) keysym = FL_BackSpace;
	else if (keysym == FL_BackSpace) got_backspace = 1;
      }
      Fl::e_keysym = keysym;
      Fl::e_text = buffer;
      set_event_xy(1);
      if (len == 1) {
	if (Fl::event_alt()) buffer[0] |= 128; // Meta key
	if (Fl::event_ctrl() && keysym == '-') buffer[0] = 0x1F; // ^_
      }

      fl_xfocus = window; fl_fix_focus();
    
      Fl_Object *o;
      // Try it as keystroke, sending it to focus and all parents:
      for (o = Fl::focus(); o; o = o->parent())
	if (o->handle(FL_KEYBOARD)) goto BREAK;

      // Try it as shortcut, sending to mouse object and all parents:
      for (o = Fl::belowmouse(); o; o = o->parent())
	if (o->handle(FL_SHORTCUT)) goto BREAK;

      // Try sending shortcut to all other windows:
      if (!modal())
	for (window=Fl::first(); window;
	     window=window->nextwindow())
	  if (window != o && !window->parent() &&
	      window->handle(FL_SHORTCUT)) goto BREAK;

      // try using add_handle() functions:
      send_event(FL_SHORTCUT,0);
    BREAK:;}
      break;

    case EnterNotify:
      // We only need child enter events to force the colormap on:
      XInstallColormap(fl_display, window->colormap());
      if (window->parent()) goto DEFAULT;
      // X sends extra enter events to the parent when mouse enters
      // child, we have to ignore them:
      if (fl_xevent.xcrossing.detail == NotifyInferior) goto DEFAULT;
      // okay, this event is good, use it:
      set_event_xy();
      Fl::e_state = fl_xevent.xcrossing.state;
      fl_xmousewin = window; fl_fix_focus();
      break;

    case LeaveNotify:
      if (window->parent()) goto DEFAULT;
      if (fl_xevent.xcrossing.detail == NotifyInferior) goto DEFAULT;
      set_event_xy();
      Fl::e_state = fl_xevent.xcrossing.state;
      if (window == fl_xmousewin) {fl_xmousewin = 0; fl_fix_focus();}
      break;

    case MotionNotify:
      set_event_xy();
      fl_xmousewin = window;
      if (modal()) {
	window = Fl::first();
	Fl::e_x = Fl::e_x_root - window->x_root();
	Fl::e_y = Fl::e_y_root - window->y_root();
      }
      fl_fix_focus();
      if (Fl::pushed()) send_pushed(FL_DRAG);
      else send_event(FL_MOVE,window);
      break;

    case ConfigureNotify: {
      int x = fl_xevent.xconfigure.x;
      int y = fl_xevent.xconfigure.y;
      int w = fl_xevent.xconfigure.width;
      int h = fl_xevent.xconfigure.height;
      // avoid bug (?) in 4DWM, it reports position of 0,0 on resize:
      if (!x && !y && !window->parent()) {
	Window r, c; int X, Y; unsigned int m;
	XQueryPointer(fl_display, window->xid(),	
		      &r, &c, &x, &y, &X, &Y, &m);
	x = x-X; y = y-Y;
      }
      fl_resize_bug_fix = window;
      window->resize(x, y, w, h);
      fl_resize_bug_fix = 0;}
      break;

    default:
    DEFAULT:
      send_event(0,0);
    }
  }
}

// End of Fl.C //
