/*	Fl_Window_Type.C

	The object describing an Fl_Window.  This is also all the code
	for interacting with the overlay, which allows the user to
	select, move, and resize the children objects.

*/

#include <FL/Fl.H>
#include <FL/Fl_Overlay_Window.H>
#include <FL/fl_message.H>
#include <FL/fl_draw.H>
#include "Fl_Object_Type.H"
#include <math.h>
#include <stdlib.h>
#include "alignment_panel.H"
#include <stdio.h>

int gridx = 10;
int gridy = 10;
int snap = 3;

void alignment_cb(Fl_Input *i, long v) {
  int n = atoi(i->value());
  if (n < 0) n = 0;
  switch (v) {
  case 1: gridx = n; break;
  case 2: gridy = n; break;
  case 3: snap  = n; break;
  }
}

void show_alignment_cb(Fl_Object *, void *) {
  make_alignment_window();
  char buf[128];
  sprintf(buf,"%d",gridx); horizontal_input->value(buf);
  sprintf(buf,"%d",gridy); vertical_input->value(buf);
  sprintf(buf,"%d",snap); snap_input->value(buf);
  alignment_window->show();
}

////////////////////////////////////////////////////////////////

class Fl_Window_Type : public Fl_Object_Type {

  friend class Overlay_Window;
  int mx,my;		// mouse position during dragging
  int x1,y1;		// initial position of selection box
  int bx,by,br,bt;	// bounding box of selection
  int dx,dy;
  int drag;		// which parts of bbox are being moved
  int numselected;	// number of children selected
  enum {LEFT=1,RIGHT=2,BOTTOM=4,TOP=8,DRAG=16,BOX=32};
  void draw_overlay();
  void newdx();
  void newposition(Fl_Type *,int &x,int &y,int &w,int &h);
  int handle(int);
  virtual void setlabel(const char *);
  void write_code();
  Fl_Object_Type *_make() {return 0;} // we don't call this
  Fl_Object *object(int,int,int,int) {return 0;}
  int recalc;		// set by fix_overlay()

public:

  uchar modal;
  const char *xclass;

  Fl_Type *make();
  virtual const char *type_name() {return "Fl_Window";}

  void open();

  void fix_overlay();	// update the bounding box, etc

  virtual void write_properties();
  virtual void read_property(const char *);
  virtual int read_fdesign(const char*, const char*);
};

static int overlays_invisible;

// The following Fl_Object is used to simulate the windows.  It has
// an overlay for the fluid ui, and special-cases the FL_NO_BOX.

class Overlay_Window : public Fl_Overlay_Window {
  void draw();
  void draw_overlay();
public:
  Fl_Window_Type *window;
  int handle(int);
  Overlay_Window(int w,int h) : Fl_Overlay_Window(w,h) {}
  void resize(int,int,int,int);
};
void Overlay_Window::draw() {
  const int CHECKSIZE = 8;
  // see if box is clear or a frame or rounded:
  if (!box() || (box()>=4&&!(box()&2)) || box()>=_FL_ROUNDED_BOX) {
    // if so, draw checkerboard so user can see what areas are clear:
    for (int y = 0; y < h(); y += CHECKSIZE) 
      for (int x = 0; x < w(); x += CHECKSIZE) {
	fl_color(((y/(2*CHECKSIZE))&1) != ((x/(2*CHECKSIZE))&1) ? FL_WHITE:0);
	fl_rectf(x,y,CHECKSIZE,CHECKSIZE);
      }
  }
  Fl_Overlay_Window::draw();
}

void Overlay_Window::draw_overlay() {
  window->draw_overlay();
}
int Overlay_Window::handle(int e) {
  return window->handle(e);
}

Fl_Type *Fl_Window_Type::make() {
  Fl_Type *p = Fl_Type::current;
  if (!p) {
    fl_message("Please select a function");
    return 0;
  }
  while (p->parent) p = p->parent;
  Fl_Window_Type *o = new Fl_Window_Type();
  if (!this->o) this->o = new Fl_Window(100,100); // template object
  o->factory = this;
  o->drag = 0;
  o->numselected = 0;
  Overlay_Window *w = new Overlay_Window(100,100);
  w->window = o;
  o->o = w;
  o->isparent = 1;
  o->add(p);
  o->modal = 0;
  o->xclass = 0;
  return o;
}

////////////////////////////////////////////////////////////////

// Double-click on window object shows the window, or if already shown,
// it shows the control panel.
void Fl_Window_Type::open() {
  Overlay_Window *w = (Overlay_Window *)o;
  if (w->visible()) {
    w->show();
    Fl_Object_Type::open();
  } else {
    Fl_Object *p = w->resizable();
    if (!p) w->resizable(w);
    w->show();
    w->resizable(p);
  }
}

// control panel items:
#include "object_panel.H"

void modal_cb(Fl_Light_Button* i, void* v) {
  if (v == LOAD) {
    if (current_object->level > 1) {i->hide(); return;}
    i->show();
    i->value(((Fl_Window_Type *)current_object)->modal);
  } else {
    ((Fl_Window_Type *)current_object)->modal = i->value();
  }
}

void border_cb(Fl_Light_Button* i, void* v) {
  if (v == LOAD) {
    if (current_object->level > 1) {i->hide(); return;}
    i->show();
    i->value(((Fl_Window*)(current_object->o))->border());
  } else {
    ((Fl_Window*)(current_object->o))->border(i->value());
  }
}

void xclass_cb(Fl_Input* i, void* v) {
  if (v == LOAD) {
    if (current_object->level > 1) {i->hide(); return;}
    i->show();
    i->value(((Fl_Window_Type *)current_object)->xclass);
  } else {
    Fl_Window_Type* w = (Fl_Window_Type*)current_object;
    storestring(i->value(),w->xclass);
    ((Fl_Window*)(w->o))->xclass(w->xclass);
  }
}

////////////////////////////////////////////////////////////////

void Fl_Window_Type::setlabel(const char *n) {
  if (o) ((Fl_Window *)o)->label(n);
}

// make() is called on this object when user picks window off New menu:
static Fl_Window_Type window_factory_i;
Fl_Type *window_factory() {return &window_factory_i;}

// Resize from window manager, try to resize it back to a legal size.
// This is not proper X behavior, but works on 4DWM and fvwm
void Overlay_Window::resize(int X,int Y,int W,int H) {
  if (!visible() || W==w() && H==h()) {
    Fl_Overlay_Window::resize(X,Y,W,H);
    return;
  }
  Fl_Overlay_Window::resize(X,Y,W,H);
  int nw = gridx&&W!=w() ? ((W+gridx/2)/gridx)*gridx : W;
  int nh = gridy&&H!=h() ? ((H+gridy/2)/gridy)*gridy : H;
  if (resizable()) {
    // resize child cannot be made smaller than 0x0:
    int mw = w()-resizable()->w();
    if (nw < mw) nw = mw;
    int mh = h()-resizable()->h();
    if (nh < mh) nh = mh;
  } else {
    // make sure new window size surrounds the objects:
    int b = 0;
    int r = 0;
    for (Fl_Type *o=window->next; o && o->level>window->level; o=o->next) {
      if (o->o->x()+o->o->w() > r) r = o->o->x()+o->o->w();
      if (o->o->y()+o->o->h() > b) b = o->o->y()+o->o->h();
    }
    if (nh < b) nh = b;
    if (nw < r) nw = r;
  }
  // If changed, tell the window manager.  Skip really big windows
  // that might be bigger than screen:
  if (nw != W && nw < Fl::w()-100 || nh != H && nh < Fl::h()-100) size(nw,nh);
}

static int moved; // turned on if user drags the mouse

// calculate actual move by moving mouse position (mx,my) to
// nearest multiple of gridsize, and snap to original position
void Fl_Window_Type::newdx() {
  int dx, dy;
  if (Fl::event_alt()) {
    dx = mx-x1;
    dy = my-y1;
  } else {
    int dx0 = mx-x1;
    int ix = (drag&RIGHT) ? br : bx;
    dx = gridx ? ((ix+dx0+gridx/2)/gridx)*gridx - ix : dx0;
    if (dx0 > snap) {
      if (dx < 0) dx = 0;
    } else if (dx0 < -snap) {
      if (dx > 0) dx = 0;
    } else 
      dx = 0;
    int dy0 = my-y1;
    int iy = (drag&BOTTOM) ? by : bt;
    dy = gridy ? ((iy+dy0+gridy/2)/gridy)*gridy - iy : dy0;
    if (dy0 > snap) {
      if (dy < 0) dy = 0;
    } else if (dy0 < -snap) {
      if (dy > 0) dy = 0;
    } else 
      dy = 0;
  }
  if (this->dx != dx || this->dy != dy) {
    moved = 1;
    this->dx = dx; this->dy = dy;
    ((Overlay_Window *)(this->o))->redraw_overlay();
  }
}

// Move an object according to dx and dy calculated above
void Fl_Window_Type::newposition(Fl_Type *o,int &X,int &Y,int &R,int &T) {
  X = o->o->x();
  Y = o->o->y();
  R = X+o->o->w();
  T = Y+o->o->h();
  if (!drag) return;
  if (drag&DRAG) {
    X += dx;
    Y += dy;
    R += dx;
    T += dy;
  } else {
    if (drag&LEFT) if (X==bx) X += dx; else if (X<bx+dx) X = bx+dx;
    if (drag&BOTTOM) if (Y==by) Y += dy; else if (Y<by+dy) Y = by+dy;
    if (drag&RIGHT) if (R==br) R += dx; else if (R>br+dx) R = br+dx;
    if (drag&TOP) if (T==bt) T += dy; else if (T>bt+dx) T = bt+dx;
  }
  if (R<X) {int n = X; X = R; R = n;}
  if (T<Y) {int n = Y; Y = T; T = n;}
}

void Fl_Window_Type::draw_overlay() {
  if (recalc) {
    bx = o->w(); by = o->h(); br = 0; bt = 0;
    numselected = 0;
    for (Fl_Type *o=next; o && o->level>level; o=o->next)
      if (o->selected) {
	numselected++;
	if (o->o->x() < bx) bx = o->o->x();
	if (o->o->y() < by) by = o->o->y();
	if (o->o->x()+o->o->w() > br) br = o->o->x()+o->o->w();
	if (o->o->y()+o->o->h() > bt) bt = o->o->y()+o->o->h();
      }
    recalc = 0;
  }
  fl_color(1);
  if (drag==BOX && (x1 != mx || y1 != my)) {fl_rect(x1,y1,mx-x1,my-y1);}
  if (overlays_invisible) return;
  if (selected) fl_rect(0,0,o->w(),o->h());
  if (!numselected) return;
  int bx,by,br,bt;
  bx = o->w(); by = o->h(); br = 0; bt = 0;
  for (Fl_Type *o=next; o && o->level>level; o=o->next)
    if (o->selected) {
      int x,y,r,t;
      newposition(o,x,y,r,t);
      fl_rect(x,y,r-x,t-y);
      if (x < bx) bx = x;
      if (y < by) by = y;
      if (r > br) br = r;
      if (t > bt) bt = t;
    }
  if (selected) return;
  if (numselected>1) fl_rect(bx,by,br-bx,bt-by);
  fl_rectf(bx,by,5,5);
  fl_rectf(br-5,by,5,5);
  fl_rectf(br-5,bt-5,5,5);
  fl_rectf(bx,bt-5,5,5);
}

// Calculate new bounding box of selected objects:
void Fl_Window_Type::fix_overlay() {
  overlays_invisible = 0;
  recalc = 1;
  ((Overlay_Window *)(this->o))->redraw_overlay();
}

// do that for every window (when selected set changes):
void redraw_overlays() {
  for (Fl_Type *o=Fl_Type::first; o; o=o->next)
    if (o->level==1) ((Fl_Window_Type*)o)->fix_overlay();
}

void toggle_overlays(Fl_Object *,void *) {
  overlays_invisible = !overlays_invisible;
  for (Fl_Type *o=Fl_Type::first; o; o=o->next)
    if (o->level==1) ((Overlay_Window*)(o->o))->redraw_overlay();
}

extern void select(Fl_Type *,int);
extern void select_only(Fl_Type *);
extern void deselect();
extern Fl_Type* in_this_only;
extern void fix_group_size(Fl_Type *t);
extern Fl_Type* tabs_test(Fl_Type *o, int x, int y);

#include <FL/Fl_Menu.H>
extern Fl_Menu Main_Menu[];

int Fl_Window_Type::handle(int event) {
  switch (event) {
  case FL_PUSH:
    x1 = mx = Fl::event_x();
    y1 = my = Fl::event_y();
    if (Fl::event_button() >= 3) {
      in_this_only = this; // modifies how some menu items work.
      const Fl_Menu *m = Main_Menu->popup(mx,my);
      if (m && m->callback) m->callback(this->o, m->argument);
      in_this_only = 0;
      return 1;
    }
    if (numselected && !overlays_invisible && !(Fl::event_shift()) &&
	mx<br+5 && mx>bx-5 && my<bt+5 && my>by-5) {
      drag = 0;
      int w1 = (br-bx)/3; if (w1 > 5) w1 = 5;
      if (mx>=br-w1) drag |= RIGHT;
      else if (mx<bx+w1) drag |= LEFT;
      w1 = (bt-by)/3; if (w1 > 5) w1 = 5;
      if (my<=by+w1) drag |= BOTTOM;
      else if (my>bt-w1) drag |= TOP;
      if (!drag) drag = DRAG;
    } else
      drag = BOX;
    moved = 0;
    return 1;
  case FL_DRAG:
    mx = Fl::event_x();
    my = Fl::event_y();
    newdx();
    return 1;
  case FL_RELEASE:
    mx = Fl::event_x();
    my = Fl::event_y();
    newdx();
    if (drag != BOX && moved) {
      if (dx || dy) {
	Fl_Type *o;
	for (o=next; o && o->level>level; o=o->next)
	  if (o->selected) {
	    int x,y,r,t;
	    newposition(o,x,y,r,t);
	    o->o->resize(x,y,r-x,t-y);
	    modflag = 1;
	    this->o->redraw();
	  }
	for (o=next; o && o->level>level; o=o->next) fix_group_size(o);
	fix_overlay();
      }
    } else if ((Fl::event_clicks() || Fl::event_ctrl())) {
      Fl_Object_Type::open();
    } else {
      if (mx<x1) {int t = x1; x1 = mx; mx = t;}
      if (my<y1) {int t = y1; y1 = my; my = t;}
      int n = 0;
      int toggle = Fl::event_shift();
      Fl_Type *o;
      Fl_Type *selection = this;
      // clear selection on everything:
      if (!toggle) deselect(); else Fl::event_is_click(0);
      // select everything in box:
      for (o=next; o && o->level>level; o=o->next) {
	for (Fl_Object *o1 = o->o; o1; o1 = o1->parent())
	  if (!o1->visible()) goto CONTINUE;
	if (Fl::event_inside(o->o)) selection = o;
	if (o->o->x()>=x1 && o->o->y()>y1 &&
	    o->o->x()+o->o->w()<mx && o->o->y()+o->o->h()<my) {
	  n++;
	  select(o, toggle ? !o->selected : 1);
	}
      CONTINUE:;
      }
      // if nothing in box, select what was clicked on:
      if (!n) {
	if (!toggle) selection = tabs_test(selection,mx,my);
	select(selection, toggle ? !selection->selected : 1);
      }
    }
    drag = 0;
    return 1;

  case FL_KEYBOARD: {
    if (Fl::event_key() == FL_Escape) {((Fl_Window*)o)->hide(); return 1;}
    // find current child:
    Fl_Type *o = Fl_Type::current;
    while (o && !o->o) o = o->parent;
    if (!o) return 0;
    Fl_Type *p = o->parent;
    while (p && p != this) p = p->parent;
    if (!p) {o = next; if (!o || o->level <= level) return 0;}
    p = o;
    // try to navigate to another child:
    for (;;) {
      switch (Fl::event_key()) {
      case FL_Tab:
	if (Fl::event_shift()) goto LEFT;
      case FL_Right:
      case FL_Down:
	o = o->next; break;
      case FL_Left:
      case FL_Up:
      LEFT:
	o = o->prev; break;
      default:
	return 0;
      }
      if (!o || o->level <= level) {o = p; break;}
      if (!o->o) continue;
      switch (Fl::event_key()) {
      case FL_Up:
      case FL_Down:
	if (o->o->x() >= p->o->x()+p->o->w() ||
	    o->o->x()+o->o->w() <= p->o->x()) continue;
      }
      break;
    }
    // select it:
    deselect(); select(o,1);
    } return 1;

  case FL_SHORTCUT: {
    in_this_only = this; // modifies how some menu items work.
    const Fl_Menu *m = Main_Menu->test_shortcut();
    if (m && m->callback) m->callback(this->o, m->argument);
    in_this_only = 0;
    return (m != 0);
  }
  default:
    return 0;
  }
}

////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <string.h>

extern FILE *code_file;

void Fl_Window_Type::write_code() {
  const char *t = type_name();
  if (subclass()) t = subclass();
  if (name()) {
    fprintf(code_file, " if (!(w=%s)) {\n", name());
  }
  fprintf(code_file, " {%s* o = new %s(%d, %d", t, t, o->w(), o->h());

  if (label() && *label()) {
    fprintf(code_file, ", ");
    write_cstring(label());
  }
  fprintf(code_file, ");\n");
  if (name()) fprintf(code_file, "  w = %s = o;\n", name());
  else fprintf(code_file, "  w = o;\n");
  if (modal) fprintf(code_file, "  w->set_modal();\n");
  if (!((Fl_Window*)o)->border()) fprintf(code_file, "  w->clear_border();\n");
  write_object_code();
  fprintf(code_file, " }\n w->begin();\n");
  Fl_Type *child;
  for (child = next; child && child->level > level; child = child->next)
    if (child->level == level+1) child->write_code();
  fprintf(code_file, " w->end();\n");
  if (name()) fprintf(code_file, " }\n");
}

void Fl_Window_Type::write_properties() {
  Fl_Object_Type::write_properties();
  if (modal) write_string("modal");
  if (!((Fl_Window*)o)->border()) write_string("noborder");
  if (xclass) {write_string("xclass"); write_word(xclass);}
}

void Fl_Window_Type::read_property(const char *c) {
  if (!strcmp(c,"modal")) {
    modal = 1;
  } else if (!strcmp(c,"noborder")) {
    ((Fl_Window*)o)->border(0);
  } else if (!strcmp(c,"xclass")) {
    storestring(read_word(),xclass);
    ((Fl_Window*)o)->xclass(xclass);
  } else {
    Fl_Object_Type::read_property(c);
  }
}

int Fl_Window_Type::read_fdesign(const char* name, const char* value) {
  int x;
  o->box(FL_NO_BOX); // because fdesign always puts an Fl_Box next
  if (!strcmp(name,"Width")) {
    if (sscanf(value,"%d",&x) == 1) o->size(x,o->h());
  } else if (!strcmp(name,"Height")) {
    if (sscanf(value,"%d",&x) == 1) o->size(o->w(),x);
  } else if (!strcmp(name,"NumberofObjects")) {
    return 1; // we can figure out count from file
  } else if (!strcmp(name,"border")) {
    if (sscanf(value,"%d",&x) == 1) ((Fl_Window*)o)->border(x);
  } else if (!strcmp(name,"title")) {
    label(value);
  } else {
    return Fl_Object_Type::read_fdesign(name,value);
  }
  return 1;
}
