// Fl_Group_Type.C

// Object describing an Fl_Group
// Probably also links to Fl_Window_Type.C

// Also the Fl_Tabs object, with special stuff to select tab items
// and insure that only one is visible.

#include <FL/Fl.H>
#include <FL/Fl_Group.H>
#include <FL/fl_message.H>
#include "Fl_Object_Type.H"

class igroup : public Fl_Group {
public:
  void resize(int,int,int,int);
  igroup(int x,int y,int w,int h) : Fl_Group(x,y,w,h) {}
};
// I override the group's resize to not mess with children except 
// to move them if group is dragged:
void igroup::resize(int X, int Y, int W, int H) {
  if ((X != x() || Y != y()) && W==w() && H==h()) {
    Fl_Object*const* a = array();
    for (int i=children(); i--;) {
      Fl_Object* o = *a++;
      Fl_Type* t = (Fl_Type*)(o->user_data());
      if (!t->selected) { // selected ones will be moved by Fl_Window_Type
	o->resize(o->x()+X-x(), o->y()+Y-y(), o->w(), o->h());
      }
    }
  }
  Fl_Object::resize(X,Y,W,H);
  redraw();
}

void fix_group_size(Fl_Type *t) {
  if (!t || !t->isparent || !(t->o)) return;
  int X = t->o->x();
  int Y = t->o->y();
  int R = X+t->o->w();
  int B = Y+t->o->h();
  for (Fl_Type *n = t->next; n && n->level > t->level; n = n->next) {
    if (!(n->o) || n->isparent) continue;
    int x = n->o->x();	if (x < X) X = x;
    int y = n->o->y();	if (y < Y) Y = y;
    int r = x+n->o->w();if (r > R) R = r;
    int b = y+n->o->h();if (b > B) B = b;
  }
  t->o->resize(X,Y,R-X,B-Y);
}

class Fl_Group_Type : public Fl_Object_Type {
public:
  virtual const char *type_name() {return "Fl_Group";}
  Fl_Object *object(int x,int y,int w,int h) {
    return new igroup(x,y,w,h);}
  Fl_Object_Type *_make() {return new Fl_Group_Type();}
  Fl_Type *make();
  void write_code();
};
static Fl_Group_Type Fl_Group_type;	// the "factory"
Fl_Type *group_factory() {return &Fl_Group_type;}

Fl_Type *Fl_Group_Type::make() {
  Fl_Type *t = Fl_Object_Type::make();
  t->isparent = 1;
  return t;
}

extern int force_parent;

void group_cb(Fl_Object *, void *) {
  // Find the current object:
  Fl_Type *q;
  q = Fl_Type::current;
  while (q && !q->o) q = q->parent;
  if (!q || q->level <= 1) {
    fl_message("Please select objects to group");
    return;
  }
  force_parent = 1;
  Fl_Type *n = Fl_Group_type.make();
  n->move_before(q);
  n->o->resize(q->o->x(),q->o->y(),q->o->w(),q->o->h());
  for (Fl_Type *t = Fl_Type::first; t;) {
    if (!t->o || t->level != n->level || t == n || !t->selected) {
      t = t->next; continue;}
    Fl_Type *nxt = t->remove();
    t->add(n);
    t = nxt;
  }
  fix_group_size(n);
}

void ungroup_cb(Fl_Object *, void *) {
  // Find the group:
  Fl_Type *q,*n;
  q = Fl_Type::current;
  while (q && !q->o) q = q->parent;
  if (q) q = q->parent;
  if (!q || q->level <= 1) {
    fl_message("Please select objects in a group");
    return;
  }
  for (n = q->next; n && n->level > q->level; n = n->next) {
    if (n->level == q->level+1 && !n->selected) {
      fl_message("Please select all objects in group");
      return;
    }
  }
  for (n = q->next; n && n->level > q->level;) {
    Fl_Type *nxt = n->remove();
    n->insert(q);
    n = nxt;
  }
  delete q;
}

////////////////////////////////////////////////////////////////

#include <stdio.h>

extern FILE *code_file;

void Fl_Group_Type::write_code() {
  const char *t = type_name();
  if (subclass()) t = subclass();
  fprintf(code_file, " {%s* o = new %s(%d, %d, %d, %d", t, t,
	  o->x(), o->y(), o->w(), o->h());
  if (label() && *label()) {
    fprintf(code_file, ", ");
    write_cstring(label());
  }
  fprintf(code_file, ");\n");
  if (name()) fprintf(code_file, "  %s = o;\n", name());
  write_object_code();
  fprintf(code_file, "  o->begin();\n");
  Fl_Type *child;
  for (child = next; child && child->level > level; child = child->next)
    if (child->level == level+1) child->write_code();
  fprintf(code_file, "  o->end();\n");
  fprintf(code_file, " }\n");
}

////////////////////////////////////////////////////////////////

#include <FL/Fl_Tabs.H>

class itabs : public Fl_Tabs {
public:
  void resize(int,int,int,int);
  itabs(int x,int y,int w,int h) : Fl_Tabs(x,y,w,h) {}
};
// I override the group's resize to not mess with children except 
// to move them if group is dragged:
void itabs::resize(int X, int Y, int W, int H) {
  if ((X != x() || Y != y()) && W==w() && H==h()) {
    Fl_Object*const* a = array();
    for (int i=children(); i--;) {
      Fl_Object* o = *a++;
      Fl_Type* t = (Fl_Type*)(o->user_data());
      if (!t->selected) { // selected ones will be moved by Fl_Window_Type
	o->resize(o->x()+X-x(), o->y()+Y-y(), o->w(), o->h());
      }
    }
  }
  Fl_Object::resize(X,Y,W,H);
  redraw();
}

static const char tabs_type_name[] = "Fl_Tabs";

class Fl_Tabs_Type : public Fl_Group_Type {
public:
  virtual const char *type_name() {return tabs_type_name;}
  Fl_Object *object(int x,int y,int w,int h) {
    return new itabs(x,y,w,h);}
  Fl_Object_Type *_make() {return new Fl_Tabs_Type();}
};
static Fl_Tabs_Type Fl_Tabs_type;	// the "factory"
Fl_Type *tabs_factory() {return &Fl_Tabs_type;}

// This is called when user clicks on an object in the window.  See
// if it is a tab title, and adjust visibility and return new selection:
// If none, return o unchanged:

Fl_Type *tabs_test(Fl_Type *o, int x, int y) {
  if (o->type_name() != tabs_type_name) return o;
  Fl_Tabs *t = (Fl_Tabs*)(o->o);
  Fl_Object *a = t->which(x,y);
  if (a) {t->value(a); return (Fl_Type*)(a->user_data());}
  return o;
}

// This is called when o is created.  If it is in the tab group make
// sure it is visible:

void group_add(Fl_Type *o, Fl_Object *before) {
  Fl_Type *p = o->parent;
  ((Fl_Group*)(p->o))->insert(*(o->o),before);
  if (p->type_name() == tabs_type_name) {
    ((Fl_Tabs*)(p->o))->value(0);
  }
  o->redraw();
}

// This is called when o is deleted.  If it is in the tab group make
// sure it is not visible:

void group_remove(Fl_Type *o) {
  Fl_Type *p = o->parent;
  ((Fl_Group*)(p->o))->remove(o->o);
  if (p->type_name() == tabs_type_name) {
    Fl_Tabs *t = (Fl_Tabs*)(p->o);
    if (t->value() == o->o) t->value(0);
  }
  o->redraw();
}

// move, don't change selected value:
void group_move(Fl_Type *o,Fl_Object *before) {
  Fl_Type *p = o->parent;
  ((Fl_Group*)(p->o))->remove(o->o);
  ((Fl_Group*)(p->o))->insert(*(o->o),before);
  o->redraw();
}

