/*	Fl_Function_Type_Type.C

	Type describing a C function output by Fluid.

*/

#include <FL/Fl.H>
#include "Fl_Type.H"
#include <FL/fl_show_input.H>
#include <string.h>
#include <stdio.h>
#include <ctype.h>

////////////////////////////////////////////////////////////////
// quick check of any C code for legality, returns an error message
// these currently require comments to parse correctly!

static char buffer[128]; // for error messages

// check a quoted string ending in either " or ' or >:
const char *_q_check(const char * & c, int type) {
  for (;;) switch (*c++) {
  case '0':
    sprintf(buffer,"missing %c",type);
    return buffer;
  case '\\':
    if (*c) c++;
    break;
  default:
    if (*(c-1) == type) return 0;
  }
}

// check normal code, match braces and parenthesis:
const char *_c_check(const char * & c, int type) {
  const char *d;
  for (;;) switch (*c++) {
  case 0:
    if (!type) return 0;
    sprintf(buffer, "missing %c", type);
    return buffer;
  case '{':
    if (type) goto UNEXPECTED;
    d = _c_check(c,'}'); if (d) return d; break;
  case '(':
    d = _c_check(c,')'); if (d) return d; break;
  case '\"':
    d = _q_check(c,'\"'); if (d) return d; break;
  case '\'':
    d = _q_check(c,'\''); if (d) return d; break;
  case '}':
  case ')':
  case '#':
  UNEXPECTED:
    if (type == *(c-1)) return 0;
    sprintf(buffer, "unexpected %c", *(c-1));
    return buffer;
  }
}

const char *c_check(const char *c, int type) {
  return _c_check(c,type);
}

////////////////////////////////////////////////////////////////

int function_type_count;

class Fl_Function_Type : public Fl_Type {
public:
  Fl_Type *make();
  void write_declare();
  void write_code();
  void open();
  int ismain() {return name_ == 0;}
  virtual const char *type_name() {return "Function";}
  virtual const char *title() {
    return name() ? name() : "main()";
  }
};

Fl_Type *Fl_Function_Type::make() {
  Fl_Function_Type *o = new Fl_Function_Type();
  function_type_count++;
  char buf[128];
  sprintf(buf,"function%d()",function_type_count);
  o->name_ = strdup(buf);
  o->isparent = 1;
  o->add(0);
  o->factory = this;
  return o;
}

void Fl_Function_Type::open() {
  const char *c = name();
  const char *message = "Function declaration, or blank for main():";
  for (;;) {
    c = fl_show_input(message,c);
    if (!c) return; // user hit cancel
    while (isspace(*c)) c++;	
    if (!*c) {name(0); return;} // main() declaration
    message = c_check(c); if (message) continue;
    const char *d = c;
    for (; *d != '('; d++) if (isspace(*d) || !*d) break;
    if (*d != '(') {message = "must be name(arguments), try again:"; continue;}
    name(c); break;
  }
}

static Fl_Function_Type function_factory_i;
Fl_Type *function_factory() {return &function_factory_i;}

extern FILE *code_file;
extern FILE *header_file;

void Fl_Function_Type::write_declare() {
  ::write_declare("#include <FL/Fl.H>");
}

void Fl_Function_Type::write_code() {
// we need to see if it creates any objects, otherwise we will get
// an "unused variable" error:
  Fl_Type *child;
  int haveobjects = 0;
  for (child = next; child && child->level > level; child = child->next)
    if (child->o) {haveobjects = 1; break;}
  fprintf(code_file, "\n");
  if (ismain()) {
    fprintf(code_file, "int main(int argc, char **argv) {\n");
    fprintf(code_file, " Fl_Window *w;\n");
    //if (haveobjects) fprintf(code_file, "  Fl_Object *o;\n");
  } else if (haveobjects) {
    fprintf(header_file, "Fl_Window *%s;\n", name());
    fprintf(code_file,"Fl_Window *%s {\n"
	    " Fl_Window *w;\n",name());
  } else {
    fprintf(header_file, "void %s;\n", name());
    fprintf(code_file,"void %s {\n",name());
  }
  for (child = next; child && child->level > level; child = child->next)
    if (child->level == level+1) child->write_code();
  if (ismain())
    fprintf(code_file, " w->show(argc, argv);\n for (;;) Fl::wait();\n}\n");
  else if (haveobjects)
    fprintf(code_file," return w;\n}\n");
  else
    fprintf(code_file,"}\n");
}
