// Fl_Object.H

// The FL user interface library version 0.98
// Copyright (C) 1998 Digital Domain

#ifndef Fl_Object_H
#define Fl_Object_H

#include "Enumerations.H"

class Fl_Object;
class Fl_Window;

typedef void (Fl_Callback )(Fl_Object*, void*);
typedef void (Fl_Callback0)(Fl_Object*);
typedef void (Fl_Callback1)(Fl_Object*, long);

struct Fl_Label {
  const char* value;
  uchar type;
  uchar font;
  uchar size;
  uchar color;
  void draw(int,int,int,int, uchar align) const ;
};

class Fl_Object {
  friend class Fl_Group;

  Fl_Group* parent_;
  Fl_Callback* callback_;
  void* user_data_;
  Fl_Label label_;
  short x_,y_,w_,h_;
  uchar type_;
  uchar flags_;
  uchar damage_;
  uchar box_;
  uchar color_;
  uchar color2_;
  uchar align_;
  uchar when_;

  virtual void draw() = 0;

// these are not implemented, they are private so default ones don't exist:
  Fl_Object & operator=(const Fl_Object &);
  Fl_Object(const Fl_Object &);

protected:

  Fl_Object(int,int,int,int,const char* =0);

  uchar flags() const {return flags_;}
  void set_flag(int c) {flags_ |= c;}
  void clear_flag(int c) {flags_ &= ~c;}
  enum {INACTIVE=1, INVISIBLE=2, SHORTCUT_LABEL=64, CHANGED=128};
  void clear_damage(uchar c = 0) {damage_ = c;}
  void damage(uchar c);
  void x(int v) {x_ = v;}
  void y(int v) {y_ = v;}
  void w(int v) {w_ = v;}
  void h(int v) {h_ = v;}

// should only be called by draw():

  void draw_box() const;
  void draw_box(uchar b, uchar c) const;
  void draw_label(int, int, int, int) const;
  void draw_label() const;

public:

// should only be called by draw() on a child:

  void draw_outside_label() const ;
  void draw_outside_label(int,int,int,int,uchar) const ;
  void draw_child();
  void redraw_child();

// these are public for internal reasons:

  void parent(Fl_Group* p) {parent_ = p;}
  void type(uchar t) {type_ = t;}
  Fl_Label &fl_label() {return label_;}
  void set_visible() {flags_ &= ~INVISIBLE;}
  void clear_visible() {flags_ |= INVISIBLE;}

// real public functions:

  virtual ~Fl_Object();

  virtual int handle(int);

  Fl_Group* parent() const {return parent_;}
  uchar type() const {return type_;}

  int x() const {return x_;}
  int y() const {return y_;}
  int w() const {return w_;}
  int h() const {return h_;}
  virtual void resize(int,int,int,int);
  void position(int X,int Y) {resize(X,Y,w_,h_);}
  void size(int W,int H) {resize(x_,y_,W,H);}
  void size_range(int minw, int minh, int maxw=0, int maxh=0,
		  int dw = 0, int dh = 0, int aspect=0);

  uchar align() const {return align_;}
  void align(uchar a) {align_ = a;}
  uchar box() const {return box_;}
  void box(uchar a) {box_ = a;}
  uchar color() const {return color_;}
  void color(uchar a) {color_ = a;}
  uchar color2() const {return color2_;}
  void color2(uchar a) {color2_ = a;}
  void color(uchar a,uchar b) {color_=a; color2_=b;}
  const char* label() const {return label_.value;}
  void label(const char* a) {label_.value=a;}
  void label(uchar a,const char* b) {label_.type = a; label_.value = b;}
  uchar labeltype() const {return label_.type;}
  void labeltype(uchar a) {label_.type = a;}
  uchar labelcolor() const {return label_.color;}
  void labelcolor(uchar a) {label_.color=a;}
  uchar labelfont() const {return label_.font;}
  void labelfont(uchar a) {label_.font=a;}
  uchar labelsize() const {return label_.size;}
  void labelsize(uchar a) {label_.size=a;}
  Fl_Callback* callback() const {return callback_;}
  void callback(Fl_Callback* c, void* p) {callback_=c; user_data_=p;}
  void callback(Fl_Callback* c) {callback_=c;}
  void callback(Fl_Callback0*c) {callback_=(Fl_Callback*)c;}
  void callback(Fl_Callback1*c, long p=0) {callback_=(Fl_Callback*)c; user_data_=(void*)p;}
  void* user_data() const {return user_data_;}
  void user_data(void* v) {user_data_ = v;}
  long argument() const {return (long)user_data_;}
  void argument(long v) {user_data_ = (void*)v;}
  uchar when() const {return when_;}
  void when(uchar i) {when_ = i;}

  int visible() const {return !(flags_&INVISIBLE);}
  void show();
  void hide();
  int active() const {return !(flags_&INACTIVE);}
  void activate();
  void deactivate();
  int activevisible() const {return !(flags_&(INACTIVE|INVISIBLE));}
  int changed() {return flags_&CHANGED;}
  void set_changed() {flags_ |= CHANGED;}
  void clear_changed() {flags_ &= ~CHANGED;}
  int take_focus();

  int contains(const Fl_Object*) const ;
  int inside(const Fl_Object* o) const {return o ? o->contains(this) : 0;}
  void redraw();
  uchar damage() const {return damage_;}
  void do_callback() {callback_(this,user_data_);}
  void do_callback(Fl_Object* o,void* arg=0) {callback_(o,arg);}
  void do_callback(Fl_Object* o,long arg) {callback_(o,(void*)arg);}
  int test_shortcut();
  static int test_shortcut(const char*);

  static void default_callback(Fl_Object*, void*);
  Fl_Window* window() const ;
};

// reserved type numbers (necessary for my cheapo RTTI) start here.
// grep the header files for "RESERVED_TYPE" to find the next available
// number.
#define FL_RESERVED_TYPE 100

#endif
