// Fl_Menu.H

// Definition of the structure to define a popup menu and
// used to construct an Fl_Menubar.

// A menu is an *array* of these structures.  The structure
// is designed so it is relatively easy to define an entire
// hierarchy of menus with a C initialization constant.

#ifndef Fl_Menu_H
#define Fl_Menu_H

// these typedefs are the same as in Fl_Object.H:
typedef void (Fl_Callback )(Fl_Object *, void *);
typedef void (Fl_Callback0)(Fl_Object *);
typedef void (Fl_Callback1)(Fl_Object *, long);

enum {	// values for flags in an Fl_Menu (matching old gl PUP symbols):
  FL_PUP_NONE	= 0,
  FL_PUP_GREY	= 1,
  FL_PUP_GRAY	= 1,
  FL_PUP_BOX	= 2,
  FL_PUP_CHECK	= 4,
  FL_PUP_RADIO	= 8,
  FL_PUP_INVISIBLE = 16,
  FL_PUP_SUBMENU = 32
};

// special leading characters:
//	'/' = submenu starting with next item
//	' ' = ignored
//	'_' = draw a line under this item

// An Fl_Menu is an *array* of these:
struct Fl_Menu {
  const char *text;	// initial character is "type", NULL = end of menu
  const char *shortcut;
  Fl_Callback *callback;
  void *argument;
  int flags;

  const Fl_Menu *popup(int x, int y, const char *title = 0) const;
  const Fl_Menu *popup(int x, int y, int w, int h,
		       const Fl_Menu* = 0, const char* title = 0) const;
  const Fl_Menu *test_shortcut() const;

  // insert new items into existing Fl_Menu array:
  int add(const char *,const char *,Fl_Callback *,void * =0,int = 0);
  int size() const ;

  // advance N items, skipping submenus:
  const Fl_Menu *next(int=1) const;
  Fl_Menu *next(int i=1) {return (Fl_Menu*)(((const Fl_Menu*)this)->next(i));}

  // control appearance:
  static uchar textsize_, textfont_;
  static void textfont(uchar x) {textfont_ = x;}
  static void textsize(uchar x) {textsize_ = x;}
  static uchar textfont() {return textfont_;}
  static uchar textsize() {return textsize_;}

  // item tests/changes:
  int checked() const {return flags&FL_PUP_CHECK;}
  void check() {flags |= FL_PUP_CHECK;}
  void uncheck() {flags &= ~FL_PUP_CHECK;}
  int active() const {return !(flags&FL_PUP_GREY);}
  void activate() {flags &= ~FL_PUP_GREY;}
  void deactivate() {flags |= FL_PUP_GREY;}
  int visible() const {return !(flags&FL_PUP_INVISIBLE);}
  void show() {flags &= ~FL_PUP_INVISIBLE;}
  void hide() {flags |= FL_PUP_INVISIBLE;}
  int activevisible() const {return !(flags&(FL_PUP_GREY|FL_PUP_INVISIBLE));}
  // warning: these tests will crash on the null end-of-menu item:
  int submenu() const {return *text=='/' || flags&FL_PUP_SUBMENU;}
  int checkbox() const {return flags&(FL_PUP_CHECK|FL_PUP_BOX|FL_PUP_RADIO);}
};

#endif
