// sockbuf.cc
//
//	Derivation from streambuf to allow streams over BSD sockets.
//

#include "sockbuf.h"
#include <errno.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <assert.h>

#define COMMBUFSIZE	4096

sockbuf :: sockbuf (void)
{
    m_sfd = EOF;
    m_commbuf = new char [COMMBUFSIZE];
    assert (m_commbuf != NULL);
    setbuf (m_commbuf, COMMBUFSIZE);
}

sockbuf :: sockbuf (const socket_t& sfd)
{
    m_sfd = sfd;
    m_commbuf = new char [COMMBUFSIZE];
    assert (m_commbuf != NULL);
    setbuf (m_commbuf, COMMBUFSIZE);
}

sockbuf :: sockbuf
(const char * hostname, port_t port, int socktype, int streamtype)
{
    m_sfd = EOF;
    m_commbuf = new char [COMMBUFSIZE];
    assert (m_commbuf != NULL);
    setbuf (m_commbuf, COMMBUFSIZE);
    open (hostname, port, socktype, streamtype);
}

streambuf * sockbuf :: setbuf (char *p, int len)
{
    setg (p, p, p);	// point gptr to the end to force underflow
    setp (p, p+len);
    return (this);
}

sockbuf * sockbuf :: attach (const socket_t& sfd)
{
    // If you don't close the file before calling this, tough.
    m_sfd = sfd;
    return (this);
}

int sockbuf :: sync (void)
{
    streampos n = pptr() - pbase();
    if (n == 0)
	return (0);
    int bw = sys_write (pbase(), n);
    if (bw == 0 && errno != EAGAIN)
	return (EOF);
    memmove (pbase(), pbase() + bw, COMMBUFSIZE - bw);
    pbump (-bw);
    return (0);
}

int sockbuf :: underflow (void)
{
    char c;
    setg (eback(), eback(), eback());
    if (sys_read (&c, 1) != 1)
	return (EOF);
    *eback() = c;
    setg (eback(), eback(), eback() + 1);
    return ((egptr() - eback() > 0) ? 0 : EOF);
}

int sockbuf :: overflow (int ch)
{
    streamsize n = pptr() - pbase();
    if (n && sync())
	return (EOF);
    if (ch != EOF) {
	char cbuf[1];
	cbuf[0] = ch;
	if (sys_write (cbuf, 1) != 1)
	    return (EOF);
    }
    pbump (-n);	// reset pptr to beginning
    return (0);
}

streamsize sockbuf :: xsputn (const char* text, streamsize n)
{
    return (sync() == EOF ? 0 : sys_write (text, n));
}

streamsize sockbuf :: xsgetn (char* text, streamsize n)
{
    return (sync() == EOF ? 0 : sys_read (text, n));
}

streamsize sockbuf :: sys_read (char* text, streamsize n)
{
streamsize br = 0;

    m_bcount = 0;
    while (m_bcount < n) {
	br = ::read (m_sfd, &text[m_bcount], n - m_bcount);
	if (br >= 0)
	    m_bcount += br;
	else
	    break;
    }
    return (m_bcount);
}

streamsize sockbuf :: sys_write (const char* text, streamsize n)
{
streamsize bw = 0;

    m_bcount = 0;
    while (m_bcount < n) {
	bw = ::write (m_sfd, &text[m_bcount], n - m_bcount);
	if (bw >= 0)
	    m_bcount += bw;
	else
	    break;
    }
    return (m_bcount);
}

sockbuf * sockbuf :: open 
(const char * hostname, port_t port, int socktype, int streamtype)
{
struct sockaddr_in sockAddr;
struct hostent * pHostEnt;

    memset (&sockAddr, 0, sizeof(struct sockaddr_in));
    sockAddr.sin_family = streamtype;    // This will be the default
    sockAddr.sin_port = htons (port);
    if ((sockAddr.sin_addr.s_addr = inet_addr (hostname)) == INADDR_NONE) {
	if ((pHostEnt = gethostbyname (hostname)) == NULL) {
	    errno = ECONNREFUSED;
	    return (NULL);
	}
	if (pHostEnt->h_addr == NULL) {
	    errno = ECONNREFUSED;
	    return (NULL);
	}
	memcpy (&sockAddr.sin_addr, pHostEnt->h_addr, sizeof(long int));
	sockAddr.sin_family = pHostEnt->h_addrtype;
    }

    if ((m_sfd = socket (sockAddr.sin_family, socktype, 0)) < 0) {
	return (NULL);
    }
    if (connect (m_sfd, (struct sockaddr*)&sockAddr, sizeof(sockAddr)) < 0) {
	::close (m_sfd);
	return (NULL);
    }
    return (this);
}

sockbuf * sockbuf :: close (void)
{
    if (::close (m_sfd) != 0)
	return (this);
    else
	return (NULL);
}

sockbuf :: ~sockbuf (void)
{
    assert (m_commbuf != NULL);
    delete [] m_commbuf;
}

