# PLC.pm v0.14 Copyright (c) 2004 by Dan Baker (dan.baker@bigfoot.com)
# Last edited 1/13/2004  7:00am

## Note: if you don't use any serial port connections,
## and don't want to install Device::SerialPort, set
## DisableSerialPortStuff = 1
## in the rc file (usually /plc/PLCproc.rc)

#
#	Licensed under the Artistic License (See the file 'Artistic')
######## when done, send T3 close to all open connections????????????
##### Added variable $UseConnectedMessages in code ------ TEST

# v0.14   Added DisableSerialPortStuff parameter in rc file
#         Changed 0.1.4 to numeric 0.14 for version checking
# v0.1.3x Internal TEST version for Ethernet->NET-ENI->MicroLogix
# v0.1.3  minor corrections to comments in this PLC.pm
# v0.1.2  fixed Floating-point encoding for writes

# Still need to correct terminology in many places.
# Also, the perl code in this module is unbelievably ugly, but it works.
# Someday I may pretty it up (if time permits), for better readability.

package PLC;

# Supported: 	 Communications:
# PLC5		 DataHighwayPlus(Note 1), Serial DF1, Ethernet(Note 2)
# PLC5/Ethernet	 DataHighwayPlus(Note 1), Serial DF1, Ethernet
# MicroLogix	 Serial DF1, Ethernet(Note 3)
# SLC500 Fixed	 DH485(Note 4)
# SLC501/502	 DH485(Note 4)
# SLC503	 Serial DF1, DH485(Note 4)
# SLC504	 Serial DF1, DataHighwayPlus(Note 1)
# SLC505	 Serial DF1, Ethernet

#  Note 1 - DataHighwayPlus requires one of the following:
#          a. 1770-DF2 module (Serial port to DH+)
#       or b. ControlLogix Gateway with one 1756-ENET module (ethernet)
#             and at least one 1756-DHRIO module (DH+)
#  Note 2 - For ethernet communications to a non-ethernet PLC5,
#           an ethernet sidecar (1785-ENET) is required.
#  Note 3 - For ethernet communications to a MicroLogix or SLC503/504,
#           a 1761-NET-ENI module is required.
#  Note 4 - For DH485 communications, a 1770-KF3 module and
#           at least one 1747-AIC module is required.

use vars qw($VERSION $REVDATE $COPYRIGHT);
$VERSION = "0.14";
$REVDATE = "1/13/2004";
$COPYRIGHT = "\nPLC.pm v$VERSION Copyright (c) 2004 by Dan Baker"
           . '(dan.baker@bigfoot.com)'
           . "\na module to communicate with industrial PLC"
           . " controllers\n(supports Allen-Bradley PLC's and SLC's)\n\n";
$COPYRIGHT .= "Running on $^O\n\n" if $^O; # add operating system name

## For best results, use the included program 'tns_server' to prevent
## conflicts between programs with the TNS parameter in Allen-Bradley
## communication packets. See the README for more details.

# NOTE: (ControlLogixGateway, DataHighwayPlus, PLC5, MicroLogix, and SLC are
# registered trademarks of Allen-Bradley Company Incorporated.
# Allen-Bradley has no connection at all with this software,
# and I have no connection with Allen-Bradley,
# so don't expect them to give you free technical support.

# Information on Allen-Bradley's DF1 protocol can be found at:
#		http://www.ab.com/manuals/cn/protocol.html
# and in:	http://www.ab.com/manuals/cn/17706516.pdf
# Everything I know about these communication protocols comes from
# the above AB manual (1770-6.5.16), from analysing RS232 and Ethernet packets,
# and (for ethernet PLC stuff) from Ron Gage's abplc package.

# 6/2002 - Found additional information on CIP ethernet communications
# in http://www.automation.rockwell.com/enabled/pdf/eipexp1.pdf

# I originally wrote a module for ControlLogix Gateway to DH+ only,
# then another module for RS232 DF1, and then one for straight Ethernet.
# I have been using the ControlLogix Gateway stuff since around mid-1999
# continuously, but I haven't used the RS232 stuff very much, and the
# Ethernet-PLC support is very recent. This module is a Frankensteinian
# conglomeration of all three, so I really need some feedback from
# users/testers.

# For PLC areas other than Section 0 (Data Table Files),
# I have adopted Ron Gage's designations (XA,XB,XC,XD for Sections 3,4,5,&6,
# PN for Program Name, RG for Rung Data, etc.)

#  Send questions, ideas, etc. to Dan Baker (dan.baker@bigfoot.com)

# Note: For PLC5's Input and Output Data is specified in OCTAL
# i.e. I:10/17 for the top bit in decimal word 8 (decimal I:8/15)
# All other addresses, and all addresses for SLC's are specified in decimal.

# Allen-Bradley PLC5's and SLC's supported
#### using RS232: (direct to PLC/SLC or via DF2/DF3 module) -----
# Computer <-RS232-> (PLC5 or SLC503/SLC504/SLC505)
# Computer <-RS232-> KF2 <- DH+ -> (PLC5 or SLC504)
# Computer <-RS232-> KF3 <-DH485-> 1747AIC <----> (SLC500)

# A typical entry in /plc/PLCproc.rc for an RS232 connected PLC is:

# [MYPLC] # (these 7 lines would be uncommented in the rc file)
# CommType = RS232
# PLCtype = PLC5
# RS232device = /dev/ttyS1
# RS232bps = 9600      # 9600 bits/second
# RS232parity = 'N'    # no parity
# RS232crc_bcc = 'CRC' # using CRC error detection

#### using Ethernet: to ControlLogix Gateway Ethernet module, then through ---
#                 DHRIO module(s) to PLC/SLC on DataHighwayPlus
# Computer <-Ethernet-> (ControlLogix Gateway) <- DH+ -> (PLC5-Ethernet/SLC505)
# The ControlLogix Gateway needs a 1756-ENET ethernet module,
# a ControlLogix Gateway Rack (i.e. 1756-A7A), a Power Supply,
# and at least one 1756-DHRIO module.


# You should have a working setup (using RSLinx, etc.)
# before trying this software for the first time.

# A typical entry in /plc/PLCproc.rc for a CL-Gateway connected SLC504 is:

# [MY504] # (these 7 lines would be uncommented in the rc file)
# CommType = ControlLogixGateway
# PLCtype = SLC504
# GatewayIP = 192.168.0.10 # IP address of the ENET module in the Gateway
# GatewaySlot = 1 # the DHRIO module is in slot 1 (slot 0 is first slot)
# GatewayChan = 1 # each DHRIO has 2 DH+ channels (1=A,2=B)
# DHnode = 005    # the SLC504's DH+ node address

# (At work, I have a 1756-ENET in slot 0, and DHRIO's in slots 1-4)

#### or directly to Ethernet PLC/SLC -----------------------------------------
# Computer <-Ethernet-> (PLC5/Ethernet or SLC505)

# A typical entry in /plc/PLCproc.rc for an Ethernet-connected PLC is:

# [ETHPLC] # (these 4 lines would be uncommented in the rc file)
# CommType = EthernetPLC
# PLCtype = PLC5
# PLC_IP = 192.168.0.20 # IP address of the Ethernet PLC


require 5.002;
use strict;
use IO::Socket;
use POSIX;
STDOUT->autoflush(1);
STDERR->autoflush(1);

delete @ENV{qw(IFS CDPATH ENV BASH_ENV PATH)}; # secure %ENV

use Exporter();
use vars qw(@ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);
@ISA = qw(Exporter);
@EXPORT = qw();
@EXPORT_OK = qw();
%EXPORT_TAGS = ();
use vars qw(%PLCproc $PLCname $tns $commandline @DontRetry);
use vars qw($DisableSerialPortStuff);
use constant LinuxRules => 1; # always true

sub log2 { # calculate the log (base 2) of a number
 return (log shift) / (log 2);
}

sub qprint {
 print @_ unless $PLC::quiet;
}

sub qprinterr {
 print STDERR @_ unless $PLC::quiet && $PLC::quiet >= 2;
}

sub sig_x  {
 print "\n--- sub PLC::sig_x\n" if ($PLC::debug and $PLC::debug >= 5);
 my ($signal) = @_;
 PLC::abort1 (9, "++ Received SIG$signal signal ++");
}

sub sigalrm  {
 print "\n--- sub PLC::sig_alrm\n" if ($PLC::debug and $PLC::debug >= 5);
 PLC::abort1 (8, "++ Timeout Alarm ++");
}

sub sigint  {
 print "\n--- sub PLC::sigint\n" if ($PLC::debug and $PLC::debug >= 5);
 PLC::abort1 (10, "++ User Interrupt ++");
}

sub sigsegv {
 print "\n--- sub PLC::sigsegv\n" if ($PLC::debug and $PLC::debug >= 5);
 PLC::abort1 (11, "++ Segmentation Fault ++");
}

$SIG{ALRM} = \&sigalrm;
$SIG{TERM} = \&sig_x;
$SIG{HUP}  = \&sig_x;
$SIG{QUIT} = \&sig_x;
$SIG{ABRT} = \&sig_x;
$SIG{PIPE} = \&sig_x;
$SIG{INT}  = \&sigint;
$SIG{SEGV} = \&sigsegv;

use vars qw(%AvailPLCs @array $cmd $fnc $plc $s $i $junk $rcdir $rcfile);
use vars qw($debug $hex $bin $AlwaysInitGatewayChannel $errorcode @errormsg);
use vars qw($retries $retry_delay $lockdir $writeserial_delay $scanfor_delay);
use vars qw($ForceRS232autodetect $quiet $UseConnectedMessages $PingFirst);
use vars qw($tns_server_host);

$retries = 6; # default retries - override with retries parm in rc file
$retry_delay = 3; # default delay - override with retry_delay parm in rc file
$errorcode = 0;
$debug = 0;
$quiet = 0;
$hex = 0;
$bin = 0;
$AlwaysInitGatewayChannel = 1;
$UseConnectedMessages = 1; # ControlLogix Gateway use Class 3 messages = 1
$ForceRS232autodetect = 0;
$writeserial_delay = 0.05;
$scanfor_delay = 0.2;
$tns_server_host = 'localhost'; # default to this machine

use vars qw($NeedSerialPortMessage $OS_win);

BEGIN {
 $rcdir = '/plc/';
 $rcfile = 'PLCproc.rc';
# check for the parameter DisableSerialPortStuff in rc file,
# and if set true, don't try to load Device::SerialPort:
 if (open(RCFILE,"$PLC::rcdir$PLC::rcfile")) {
  my @lines = <RCFILE>;
  for (@lines) {
   chomp;
   s/^\s+//; # skip leading whitespace
   s/[#;].*$//; # ignore comments
   s/\s+$//; # skip trailing whitespace
   if (/^DisableSerialPortStuff/i) { # DisableSerialPortStuff
    my ($junk1,$parm1) = split('=',$_);
    $parm1 =~ s/^\s+//; # skip leading whitespace
    if ( ($parm1 =~ /^Y/i) || ($parm1 == 1) ) {
     $PLC::DisableSerialPortStuff = $parm1;
     last;
    }
   }
  } # end for @lines
  close(RCFILE);
 }
 unless ($PLC::DisableSerialPortStuff) {
  $NeedSerialPortMessage = "\n\nATTENTION:\n" .
   "#####################################################################\n" .
   "## THE SerialPort MODULE IS REQUIRED IN ORDER TO USE THIS PROGRAM. ##\n" .
   "## YOU CAN GET IT FROM http://www.cpan.org (sorry I shouted)       ##\n" .
   "## (see the README file if you don't use any serial communication) ##\n" .
   "#####################################################################\n\n";
  $OS_win = ($^O =~ /MSWin/) ? 1 : 0;
  if ($OS_win) {
   eval "use Win32::SerialPort 0.13";
   die "$NeedSerialPortMessage$@\n" if ($@);
  }
  else {
   eval "use Device::SerialPort";
   die "$NeedSerialPortMessage$@\n" if ($@);
  }
 } # end unless ($PLC::DisableSerialPortStuff)
} # End BEGIN

$lockdir = '/var/lock';
$tns = ( ($$ + 100) & 0xFFFF);
$commandline = "$0 " . join(' ',@ARGV);

## most routines set $PLC::errorcode when there is a problem
## $PLC::errormsg[$PLC::errorcode] returns the appropriate message:
## (also see subroutine PLC::ABerrormessage for error codes
##  returned from PLC's)
@errormsg = (
  "No Error",						#  0
  "No PLC name",					#  1
  "Can't open serial device",				#  2
  "No Socket",						#  3
  "Can't open rc file",					#  4
  "Can't get channel handle",				#  5
  "PLC returned error code",				#  6
  "Read Timeout",					#  7
  "Unknown Processor Type",				#  8
  "No Data Address",					#  9
  "No CMD",						# 10
  "No FNC",						# 11
  "Can't specify write to bit level",			# 12
  "Unsupported CMD-FNC",				# 13
  "Unable to generate DF1 packet",			# 14
  "No hash ref in subroutine call",			# 15
  "No write data specified",				# 16
  "No Gateway (or NET-ENI) IP",				# 17
  "No Gateway Slot #",					# 18
  "No Gateway Channel",					# 19
  "No DH+ node",					# 20
  "Lost Socket",					# 21
  "No IP specified for this PLC",			# 22
  "Failed to get connection handle from ethernet PLC",	# 23
  "No serial port device",				# 24
  "No commport object",					# 25
  "RS232 read error",					# 26
  "RS232 write error",					# 27
  "Error dialing modem",				# 28
  "Error generating CRC or BCC",			# 29
  "No packet to send",					# 30
  "Failed to get first init packet from gateway",	# 31
  "Failed to get second init packet from gateway",	# 32
  "Failed to get third init packet from gateway",	# 33
  "Failed to get connection handle from gateway",	# 34
  "Failed to get channel handle packet from gateway",	# 35
  "Invalid data address",				# 36
  "Received data in unknown format",			# 37
  "Unable to initialize modem",				# 38
  "Modem says BUSY",					# 40
  "Modem says NO DIALTONE",				# 41
  "Modem says NO CARRIER",				# 42
  "Received TERM, HUP, QUIT, ABRT, or PIPE signal",	# 43
  "Received SIGINT signal",				# 44
  "Segmentation fault",					# 45
  "Timed Out",						# 46
);

# If $PLC::errorcode is in this list, don't retry in PLC::PLCcommand()
@DontRetry = qw(
 1 4 8 9 10 11 12 13 14 15 16 17 18 19 20 22 24 30 36
);

my @DataTypes = (
"End/No Type", "Bit(B)", "Bit String(BS)", "Byte String(CS)", "Integer(I)",
"Timer(T)", "Counter(C)", "Control(R)", "Floating Point(F)", "Array(A)",
"Unknown(0x0A)", "Unknown(0x0B)", "Unknown(0x0C)", "Rung Data","Unknown(0x0E)",
"Address", "BCD(D)", "Unknown(0x11)", "Unknown(0x12)", "Unknown(0x13)",
"Unknown(0x14)", "PID Block(PD)", "Message Block(MG)", "Unknown(0x17)",
"Unknown(0x18)", "Unknown(0x19)", "Unknown(0x1A)", "Unknown(0x1B)",
"Unknown(0x1C)", "SfcStatus(SC)", "String(ST)", "Unknown(0x1F)",
"Block Transfer(BT)"
);

# The TNS is a transaction number used to identify PCCC packets
# to/from a PLC or SLC. Each time you send a command (with the current TNS),
# you increment the TNS for next time. If you send a command to a PLC using
# the same TNS you used the last time, it may ignore your command.
# If the program exits and then restarts, it might use the same TNS
# again, and then your command would be ignored. Or in the case of
# ethernet or CL-gateway, you might have several programs communicating
# with a PLC or PLC's all at once. For this reason, it is good practice
# to have a 'TNS server' that will dispense a different (ascending)
# number each time. See the PLC.pm README and the included program
# 'tns_server'. If tns_server is running, we just connect to it
# on port 56283 on the local machine (or change the host name like this:
# $PLC::tns_server_host = 'tns_servername'; # name of PC running tns_server
# if you are running tns_server on another machine), and read our TNS.
# Otherwise, we fall back to our own tns counter.

sub tns {
# First try to connect to a tns server on tcp port 56283
 print "\n--- sub PLC::tns\n" if ($PLC::debug and $PLC::debug >= 5);
 my $serverhost = $PLC::tns_server_host;
 if ($serverhost =~ /^([@-_\w.]+)$/) {
  $serverhost = $&; # untaint serverhost
 }
 if ($serverhost =~  /^(\d+\.\d+\.\d+\.\d+)$/) {
  $serverhost = $&; # untaint serverhost if IPV4 dotted quad
 }

 print "Using tns_server host \'$serverhost\'\n"
  if ($PLC::debug and $PLC::debug >= 5);
 my $tns1;
 my $save_warning_state = $^W; $^W = 0;
 my $sock = new IO::Socket::INET( PeerAddr => $serverhost,
				  PeerPort => 56283, # tns_server listens here
				  Proto    => 'tcp',
				  Timeout  => 1 );
 $^W = $save_warning_state;
 if ($sock) {
  $tns1 = <$sock>;
  close ($sock);
  chomp $tns1;
 }
 if (defined $tns1) {
  $tns = $tns1;
  printf("Received TNS = %04X from tns_server\n", $tns) if $PLC::debug;
 }
 else { # just use our own tns
  $tns++;
  $tns = 1 if $tns > 65535;
  printf("Using TNS = %04X (no tns_server)\n", $tns) if $PLC::debug;
 }
 return $tns;
} # end sub tns

sub is_in {
 my $target = shift or return undef;
 for (@_) {
  $_ == $target && return 1;
 }
 return 0;
}

sub istrue {
 my $value = shift;
 return ( (defined $value) && (($value =~ /^Y/i ) || ($value == 1 )) );
}

sub snooze {
 select(undef,undef,undef,$_[0]);
}

sub antichop { # chop off first char of a string and return it
 return(substr(shift,0,1,''));
}

sub firstchar {
 return(substr($_[0],0,1));
}

sub getopts { # this allows options and parameters in any order
no strict 'refs';
 my ($char,$x,@ary1,@ary2,$parm1,%opts,@opts,@opts1,@opts2,%optlen,$arg);
 my $opt_str = shift or return undef;

 while ($opt_str) {
  $x = antichop($opt_str);
  $optlen{$x} = 1;
  if (firstchar($opt_str) eq ':') {
   antichop($opt_str);
   $optlen{$x} = 2;
  }
 }
 @opts = keys %optlen;
 for (@opts) {
  if    ($optlen{$_} == 1) { push(@opts1,$_) }
  elsif ($optlen{$_} == 2) { push(@opts2,$_) }
 }

 @ary1 = @ARGV;

 while (@ary1) {
  $parm1 = shift @ary1;
  if ($parm1 !~ /^-/) { push(@ary2,$parm1) } # no leading '-'
  elsif ($parm1 eq '-') {
   push(@ary2,(@ary1)); # no more options
   @ary1 = ();
  }
  elsif ($parm1 eq '--') {
   push(@ary2, '-'); # convert to single '-'
  }
  elsif ($parm1 =~ /^-[@opts]/) { # it is an option or series of options
   antichop $parm1; # get rid of leading '-'
   while ($parm1) {
    $char = antichop($parm1);
    if ($char =~ /^[@opts]$/) { # is this character a valid option?
     if ($optlen{$char} == 1) { # standalone option, no argument
      $opts{$char}++;
      ${"main::opt_$char"} = $opts{$char};
     }
     elsif ($optlen{$char} == 2) {
      if ( length $parm1 ) {
       $arg = $parm1; # argument is the rest of this parm
       $parm1 = '';
      }
      else { # argument must be next parm
       $arg = shift @ary1;
      }
      $opts{$char} = $arg;
      ${"main::opt_$char"} = $opts{$char};
     }
     else { qprinterr "Unknown error processing option '$char'\n" }
    }
    else { # bad option, complain
     qprinterr "Hey, $char is not a valid option!\n";
    }
   } # end while $parm1
  }
  else { # starts with '-', but not an option
   push(@ary2,$parm1);
  }
 }
 @ARGV = @ary2;
} # end sub getopts

sub debug {
 print "\n--- sub PLC::debug\n" if ($PLC::debug and $PLC::debug >= 5);
 if (defined $_[0]) {
  $debug = $_[0];
 }
 return $debug;
}

sub abort1 {
 print "\n--- sub PLC::abort1\n" if ($PLC::debug and $PLC::debug >= 5);
 my $exitcode = int(shift) || 1;
 my $abortstr = "++ Aborting ++\n" . shift;
 qprinterr "\nProgram $0\nPLC.pm v$VERSION RevDate $REVDATE\n";
 qprinterr "\n$abortstr\n";
 exit $exitcode;
}

####################################
sub readPLCdata {
 print "\n--- sub PLC::readPLCdata\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my @return_array = ();
 my $return_string;
 my $offset;
 my ($file,$element,$data_address);
 my $PLCname = shift;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 my $data_addr = shift;
 unless ($data_addr) {
  $PLC::errorcode = 9; # No Data Address
  return undef;
 }
 my $elements = shift || 1;
### read_rcfile($PLCname) or return undef;

# if data type 'ST' and elements > 1, call readPLCdata multiple times
 if ( ($data_addr =~ /^ST/i) && ($elements > 1) ) {
  print "Reading $elements string (ST) elements\n" if $PLC::debug;
  for (1 .. $elements) { # call readPLCdata2 once for each string
   $offset = $_ - 1;
# calculate data address with offset
   ($file,$element) = split ':', $data_addr;
   $element += $offset;
   $data_address = "$file:$element";
   ($return_string) = readPLCdata2($PLCname,$data_address,1);
   return undef unless defined ($return_string);
   push @return_array, $return_string;
  }
  return @return_array; # return an array of string (ST) values
 } # end if datatype == ST and elements > 1
 else {
  return readPLCdata2($PLCname,$data_addr,$elements);
 }
} # end sub readPLCdata

####################################
sub readPLCdata2 {
 print "\n--- sub PLC::readPLCdata2\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 my $data_addr = shift;
 unless ($data_addr) {
  $PLC::errorcode = 9; # No Data Address
  return undef;
 }
 my $elements = shift || 1;
 read_rcfile($PLCname) or return undef;
 if ($PLCproc{$PLCname}->{'PLCtype'} =~ /^PLC/i) {
   return PLCcommand($PLCname, 0x0F,0x68,$data_addr,$elements);
 }
 elsif ($PLCproc{$PLCname}->{'PLCtype'} =~ /^SLC/i) {
   return PLCcommand($PLCname, 0x0F,0xA2,$data_addr,$elements);
 }
 elsif ($PLCproc{$PLCname}->{'PLCtype'} =~ /^MicroLogix/i) {
   return PLCcommand($PLCname, 0x0F,0xA2,$data_addr,$elements);
 }
 else {
 $PLC::errorcode = 8; # Unknown Processor Type
  return undef;
 }
} # end sub readPLCdata2

####################################
sub read_program_name {
 print "\n--- sub PLC::read_program_name\n"
  if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 my @ret;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 read_rcfile($PLCname) or return undef;
 my $PLCtype = $PLCproc{$PLCname}->{'PLCtype'};
 unless ($PLCtype) {
  $PLC::errorcode = 8; # Unknown Processor Type
   return undef;
 }
 $PLCproc{$PLCname}->{recv_ProgName} = 'UNKNOWN';
 if ($PLCtype =~ /^PLC/i) {
  @ret = PLCcommand($PLCname, 0x0F,0x68,'PN0:0.0',8);
  $PLCproc{$PLCname}->{recv_ProgName} = pack("C*",@ret) if defined $ret[0];
 }
 elsif ($PLCtype =~ /^SLC/i) {
  @ret = PLCcommand($PLCname, 0x0F,0xA2,'PN0:0.0',8);
  $PLCproc{$PLCname}->{recv_ProgName} = pack("C*",@ret) if defined $ret[0];
 }
 elsif ($PLCtype =~ /^MicroLogix/i) {
  @ret = PLCcommand($PLCname, 0x0F,0xA2,'PN0:0.0',8);
  $PLCproc{$PLCname}->{recv_ProgName} = pack("C*",@ret) if defined $ret[0];
 }
 else {
 $PLC::errorcode = 8; # Unknown Processor Type
  return undef;
 }
 $PLCproc{$PLCname}->{recv_ProgName} =~ s/\0.*//;
 return $PLCproc{$PLCname}->{recv_ProgName};
} # end sub read_program_name

####################################
sub program_mode {
 print "\n--- sub PLC::program_mode\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 read_rcfile($PLCname) or return undef;
 my $REMPROG5 = chr(0x00);
 my $REMPROG500 = chr(0x01);
 my $REMPROG1000 = chr(0x01);
 my $PLCtype = $PLCproc{$PLCname}->{'PLCtype'};
 unless ($PLCtype) {
  $PLC::errorcode = 8; # Unknown Processor Type
   return undef;
 }
 if ($PLCtype =~ /^PLC/i) {
  return PLCcommand($PLCname, 0x0F,0x3A,$REMPROG5); # Change Mode (PLC5 only)
 }
 elsif ($PLCtype =~ /^SLC/i) {
# Change Mode (SLC 500,5/03,5/04,MicroLogix only)
  return PLCcommand($PLCname, 0x0F,0x80,$REMPROG500);
 }
 elsif ($PLCtype =~ /^MicroLogix/i) {
# Change Mode (SLC 500,5/03,5/04,MicroLogix only)
  return PLCcommand($PLCname, 0x0F,0x80,$REMPROG500);
 }
 else {
  $PLC::errorcode = 8; # Unknown Processor Type
   return undef;
 }
}

####################################
sub run_mode {
 print "\n--- sub PLC::run_mode\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 read_rcfile($PLCname) or return undef;
 my $REMRUN5  = chr(0x02);
 my $REMRUN500  = chr(0x06);
 my $REMRUN1000  = chr(0x02);
 my $PLCtype = $PLCproc{$PLCname}->{'PLCtype'};
 unless ($PLCtype) {
  $PLC::errorcode = 8; # Unknown Processor Type
   return undef;
 }
 if ($PLCtype =~ /^PLC/i) {
  return PLCcommand($PLCname, 0x0F,0x3A,$REMRUN5);  # Change Mode (PLC5 only)
 }
 elsif ($PLCtype =~ /^SLC/i) {
# Change Mode (SLC 500,5/03,5/04,MicroLogix only)
  return PLCcommand($PLCname, 0x0F,0x80,$REMRUN500);
 }
 elsif ($PLCtype =~ /^MicroLogix/i) {
# Change Mode (SLC 500,5/03,5/04,MicroLogix only)
  return PLCcommand($PLCname, 0x0F,0x80,$REMRUN500);
 }
 else {
  $PLC::errorcode = 8; # Unknown Processor Type
   return undef;
 }
}

sub init {
##################
# initialize Comms depending on CommType
##################
 print "\n--- sub PLC::init\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 $PLC::PLCname = $PLCname; ###XXX
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }

 if (Protocol($PLCname) =~ /^CIP_encap/i) { # CL Gateway or NET_ENI
  my $GatewayIP = $PLCproc{$PLCname}->{'GatewayIP'};
  unless ($GatewayIP) {
   $PLC::errorcode = 17; # No Gateway IP
   return undef;
  }
  ABgate3::ABinit($GatewayIP) or return undef;
  if (PLC::istrue($PLCproc{$PLCname}->{NET_ENI})) { ###ZZZ
  ($PLCproc{$PLCname}->{'GatewaySlot'},$PLCproc{$PLCname}->{'GatewayChan'})
   = (0,0);
  }
  my ($slotnum,$channel)
   = ($PLCproc{$PLCname}->{'GatewaySlot'},$PLCproc{$PLCname}->{'GatewayChan'});
  unless (defined $slotnum) {
   $PLC::errorcode = 18; # No Gateway Slot #
   return undef;
  }
  unless (defined $channel) {
   $PLC::errorcode = 19; # No Gateway Channel
   return undef;
  }
  return ABgate3::ABinitchan($GatewayIP,$slotnum,$channel);
 }
 elsif (Protocol($PLCname) =~ /^Eth/i) { # Ethernet PLC (or SLC)
  return ABeth1::eth_connect($plc);
 }
 else { # default to RS232 if Protocol unknown
  return DF1a::init($PLCname);
 }
} # end sub init

## PLCcommand can be called in hash context or array context:
# Example of hash context:
# @data_array = PLC::PLCcommand(PLCname   => $PLCname,
#				CMD       => $cmd,
#				FNC       => $fnc,
#				DataAddr  => $data_addr,
#				Elements  => $elements,
#				WriteData => \@writedata);
#
# Example of array context:
# @data_array
#  = PLC::PLCcommand($PLCname,$cmd,$fnc,$data_addr,$elements,@writedata);
#
# always returns an array of data values (or undef if there is an error)
sub PLCcommand {
 print "\n--- sub PLC::PLCcommand\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $mightbehash = 0;
 my ($PLCname,$cmd,$fnc,$data_addr,$elements,@writedata,%parmhash);
 if ( ((scalar @_) % 2) == 0 ) { # even # of parameters? might be hash
  (%parmhash) = @_;
  $mightbehash = 1;
  for (keys %parmhash) { # any undefined parameter names?
   unless (
      $_ eq 'PLCname'
   or $_ eq 'CMD' or $_ eq 'Cmd'
   or $_ eq 'FNC' or $_ eq 'Fnc'
   or $_ eq 'DataAddr' or $_ eq 'Data'
   or $_ eq 'Elements'
   or $_ eq 'WriteData'
   ) {
    $mightbehash = 0; last; # assume parameters passed in array format
   }
  }
 }
 if ($mightbehash) {
  print "Called with hash parameters\n" if ($PLC::debug and $PLC::debug >= 5);
  $PLCname   = $parmhash{PLCname};
  $cmd       = $parmhash{CMD} || $parmhash{Cmd} || 0;
  $fnc       = $parmhash{FNC} || $parmhash{Fnc} || 0;
  $data_addr = $parmhash{DataAddr} || $parmhash{Data};
  $elements  = $parmhash{Elements};
  if (ref $parmhash{WriteData} && ref $parmhash{WriteData} eq 'ARRAY') {
   @writedata = @{$parmhash{WriteData}};
  }
  elsif ($parmhash{WriteData}) {
   @writedata = ($parmhash{WriteData}); # not an arrayref, just 1 element
  }
 }
 else {
  print "Called with array parameters\n" if ($PLC::debug and $PLC::debug >= 5);
  ($PLCname,$cmd,$fnc,$data_addr,$elements,@writedata) = @_;
 }

 my @ret;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 unless (defined $cmd) {
  $PLC::errorcode = 10; # No CMD
  return undef;
 }
 unless (defined $fnc) {
  $PLC::errorcode = 11; # No FNC
  return undef;
 }
 unless ( (defined $data_addr) || ($cmd == 6 && $fnc == 3) ) {
  $PLC::errorcode = 9; # No Data Address
  return undef;
 }
 $data_addr = uc $data_addr;

 read_rcfile($PLCname) or return undef;

 if (@writedata && defined $writedata[0]) {
  @{$PLCproc{$PLCname}->{send_writedata}} = @writedata;
  $elements ||= scalar(@{$PLCproc{$PLCname}->{send_writedata}});
 }
 else { $PLCproc{$PLCname}->{send_writedata} = undef } ##QQQQQ
 $elements ||= 1;

 my $tries = $plc->{retries} || $PLC::retries;
 my $retry_delay = $plc->{retry_delay} || $PLC::retry_delay;
 do {{ # double-curly-do allows 'next' + 'last' statements
  PLC::snooze($retry_delay) if $PLC::errorcode;

  $PLCproc{$PLCname}->{send_cmd} = $cmd;
  $PLCproc{$PLCname}->{send_fnc} = $fnc;
  $PLCproc{$PLCname}->{send_data_addr} = $data_addr;
  $PLCproc{$PLCname}->{send_elements} = $elements;

# For some commands, $data_addr is raw data, not a PLC data address
  if ( ( CmdFnc($PLCproc{$PLCname}) eq "6-00" )     # 6-00 echo function
  ||   ( CmdFnc($PLCproc{$PLCname}) eq "F-3A" )     # PLC5 set mode command
  ||   ( CmdFnc($PLCproc{$PLCname}) eq "F-80" ) ) { # SLC  set mode command
   @{$PLCproc{$PLCname}->{send_data_array}} = unpack "C*", "$data_addr";
  }

  $PLCproc{$PLCname}->{PLCname} = $PLCname;
  $PLCproc{$PLCname}->{remoteIP} = $PLCproc{$PLCname}->{PLC_IP};
  $PLCproc{$PLCname}->{remoteIP} ||= $PLCname; # try DNS if no IP in rc file

# generate RS232 or Ethernet or ControlLogix Gateway data packet,
  unless (defined gen_df1($PLCproc{$PLCname})) { # generate required values
   return undef;
  }
  print("\nCommType = ", CommType($PLCname), "\n") if $debug;
  print("\nProtocol = ", Protocol($PLCname), "\n\n") if $debug;
  @ret = sendDF1pkt($PLCproc{$PLCname});

  if ($PLC::errorcode) { #####XXXX ALSO CHECK STS and EXTSTS
   if ($PLC::debug) {
    print "\n++ ERROR - $PLC::errormsg[$PLC::errorcode]\n";
    print "++ PLC = $PLCname, address = $data_addr \n";
    print "Re-trying ...\n" if $tries > 1;
   }
  } # end if errorcode
  $tries = 0 if is_in($errorcode,@DontRetry); # some conditions we don't retry
 }} while ( ($PLC::errorcode) && (--$tries > 0) );
 return @ret;
} # end sub PLCcommand

sub safestring {
 my $str1 = shift;
 my $str2 = '';
 for (unpack "C*", ($str1)) {
  if (chr($_) eq "\\") { $str2 .= "\\\\" } # escape backslash character
  elsif (chr($_) =~ /^[\x20-\x7E]$/) {
   $str2 .= chr($_); # printable character
  }
  elsif (chr($_) eq "\n") { $str2 .= '\\n' }
  elsif (chr($_) eq "\r") { $str2 .= '\\r' }
  else { # non-printable character
   $str2 .= sprintf "\\%02X",$_; # display in hex
  }
 }
 return $str2;
} # end sub safestring

sub superdebugprint {
 print "\n--- sub PLC::superdebugprint\n"
  if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $hashref = shift;
 unless ($hashref) {
  $PLC::errorcode = 15; # No hash ref in subroutine call
  return undef;
 }
 print "\n\n";
 for ( sort keys %{$hashref} ) {
  if ( defined $hashref->{$_} ) {
   if ( ref($hashref->{$_}) ) {
    if ( ref($hashref->{$_}) eq 'ARRAY' ) {
     my $fmtstr = '%d';
     if ($_ =~ /byte/) {$fmtstr = '0x%02X'}
     elsif ($_ =~ /word/) {$fmtstr = '0x%04X'}
     print $_," " x (15 - length $_) ,"= ";
     $i = 0;
     for $junk ( @{$hashref->{$_}} ) {
      if ( defined $junk ) {
       if ($junk !~ /^[\x20-\x7E]*$/) { # don't print non-printable ASCII
        PLC::PrintBytes($junk);
       }
       elsif (length ($fmtstr)) {
        if ($junk !~ /^\d+$/) { $fmtstr = '%s' }
        printf("$fmtstr ",$junk);
       } # end if length
       else {print "$junk "}
       if (++$i >= 8) {$i=0; print "\n", " " x 17;}

      } # end if defined
     } # end for
     print "\n";
    } # end if ARRAY
   } # end if ref
###########################################

   elsif ( $_ eq 'recv_string'   ) {
    print($_," " x (15 - length $_) ,"= ");
    print "'",safestring($hashref->{$_}),"'    (String Value)\n";
   } # end elsif

   elsif (
   ($hashref->{$_} !~ /^[\x20-\x7E]*$/) # don't print non-printable ASCII
   ) {
    print($_," " x (15 - length $_) ,"= ");
    PLC::PrintBytes($hashref->{$_});
   } # end elsif

   elsif   ( ($_ !~ /packet/)
   &&      ($_ !~ /pkt/)
   &&      ($_ !~ /sys_addr/) ) {
    print $_," " x (15 - length $_) ,"= ";
    if ($hashref->{$_} =~ /^\d+$/) { # numeric data
     printf("0x%02X\n", $hashref->{$_});
    }
    else { # not numeric
     print("$hashref->{$_}\n");
    }
   } # end elsif
  } # end if defined $_
 } # end for sort keys
}

sub read_rcfile {
 print "\n--- sub PLC::read_rcfile\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift; # call with no parameters to just populate %PLCproc hash
 $PLC::DisableSerialPortStuff ||= 0;
 print "\n\n--- DisableSerialPortStuff = $PLC::DisableSerialPortStuff\n\n"
  if $PLC::debug;

# Read defaults and PLC-specific values from rc file:
 if (open(RCFILE,"$rcdir$rcfile")) {
  my $readparms = 1; # default to reading parms from rc file into %PLCproc
  my @lines = <RCFILE>;
  for (@lines) {
   chomp;
   s/^\s+//; # skip leading whitespace
   s/[#;].*$//; # ignore comments
   s/\s+$//; # skip trailing whitespace
   if (/^\[(.+)\]/i) { # [PLCname]
    for ( split(',', uc $1) ) {
     unless ($_ eq '*') {
      $AvailPLCs{$_} = 'N/A';	# 'keys %PLC::AvailPLCs' ...
     }				# is a list of all available PLC's
    }				# in rc file
   }
   if ($PLCname) {
    if ( (/^\[$PLCname\]/i) # [PLCname]
    ||   (/^\[.*,$PLCname\]/i) # [XXX,PLCname]
    ||   (/^\[$PLCname,.*\]/i) # [PLCname,YYY]
    ||   (/^\[.*,$PLCname,.*\]/i) # [XXX,PLCname,YYY]
    || (/^\[\*\]/) ) {$readparms = 1} # [*] is for all PLCname's
    elsif (/^\[.*\]/) {$readparms = 0} # some other PLC name
    elsif ($readparms) {
     if ($_) {
      s/\s/=/ unless /=/;
      s/\s*=\s*/=/; # now line should be in the format KEY=VALUE
      my ($key,$value) = split('=',$_);
      if ($value =~ /^0/) {$value = oct($value)} # hex, octal, or binary number
      $value =~ s/[\'\"]//g;
      if ( ($key ne 'PLCname') && (defined $key) && (defined $value) ) {
       $PLCproc{$PLCname}->{$key} = $value;
       $AvailPLCs{$PLCname} = $value if $key =~ /^PLCtype$/i;
      }
     }
    }
   } # end if PLCname
  } # end for @lines
  close(RCFILE);

  if (defined $PLCname) {
   if (PLC::istrue($PLCproc{$PLCname}->{NET_ENI})) { ###ZZ
    if ( $PLCproc{$PLCname}->{'PLC_IP'} ) {
     $PLCproc{$PLCname}->{'GatewayIP'} = $PLCproc{$PLCname}->{'PLC_IP'}; ##ZZ
    }
   }

   if (CommType($PLCname) =~ /^Eth/i) { # Ethernet PLC or SLC
    $PLCproc{$PLCname}->{'Protocol'} = 'EthernetPLC';
   }
   elsif (CommType($PLCname) =~ /^Control/i) {  # ControlLogix Gateway
    $PLCproc{$PLCname}->{'Protocol'} = 'CIP_encap';
   }
   elsif (CommType($PLCname) =~ /^NET_ENI/i) {  # 1761-NET-ENI
    $PLCproc{$PLCname}->{'Protocol'} = 'CIP_encap';
    $PLCproc{$PLCname}->{'NET_ENI'} = 'Yes';
   }
   else {
    $PLCproc{$PLCname}->{'Protocol'} = 'RS232';
   }

# If NET_ENI is true, default to Unconnected Messages:
   if ( PLC::istrue($PLCproc{$PLCname}->{'NET_ENI'}) ) {
    $PLCproc{$PLCname}->{'NET_ENI'} = "Yes"; #ZZZZZ
    $PLCproc{$PLCname}->{'UseConnectedMessages'} ||= 0;
    $PLCproc{$PLCname}->{'GatewaySlot'} ||= 0;
    $PLCproc{$PLCname}->{'GatewayChan'} ||= 0;
    $PLCproc{$PLCname}->{'DHnode'} ||= 0;
   } #ZZZZZ

# TEST -- setting $UseConnectedMessages to 0 is experimental #######ZZZ
   if ( (defined $PLCproc{$PLCname}->{'UseConnectedMessages'})
   && ( ($PLCproc{$PLCname}->{'UseConnectedMessages'} =~ /^N/i )
   ||   ($PLCproc{$PLCname}->{'UseConnectedMessages'} == 0 ) ) ) {
    $PLC::UseConnectedMessages = 0;
   }
  } # end if defined $PLCname
 } # end if open
 else { # couldn't open rc file
  qprinterr "\n\n++ Unable to open rc file $rcdir$rcfile ++\n\n";
  $PLC::errorcode = 4; # Can't open rc file
  if ( ! -d $rcdir ) {
   abort1(4,
    "\n++ Can't find directory $rcdir, did you create it? See README ++\n\n");
  }
  elsif ( ! -r $rcdir ) {
   abort1(4,
    "\n++ Can't read directory $rcdir, check permissions, See README ++\n\n");
  }
  elsif ( ! -f "$rcdir$rcfile" ) {
   abort1(4,
    "\n++ Can't find $rcdir$rcfile, and I need it! See README ++\n\n");
  }
  elsif ( ! -r "$rcdir$rcfile" ) {
   abort1(4,
    "\n++ Can't read $rcdir$rcfile, check permissions, See README ++\n\n");
  }
  else {
   abort1(4, "\n++ Can't read $rcdir$rcfile, See README");
  }
 }
 return LinuxRules; # always true
} # end sub read_rcfile

sub packdf1a {
# generate "base" PCCC packet (common to most commands/functions)
 print "\n--- sub PLC::packdf1a\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $hashref = shift;
 unless ($hashref) {
  $PLC::errorcode = 15; # No hash ref in subroutine call
  return undef;
 }
 $hashref->{send_src} = $hashref->{'RS232src'};
 $hashref->{send_dst} = $hashref->{'RS232dst'};
 unless (defined $hashref->{send_dst}) {
  $hashref->{send_dst} = $hashref->{'DHnode'};
 }
 $hashref->{send_dst} ||= 0;
 $hashref->{send_src} ||= 0;
 $hashref->{send_control} ||= 0;
 $hashref->{send_lsap} ||= 0;
 $hashref->{send_elements} ||= 0;
 $hashref->{send_sts} ||= 0;
 $hashref->{send_tns} = PLC::tns();
 unless (defined $hashref->{send_cmd}) {
  $PLC::errorcode = 10; # No CMD
  return undef;
 }
 unless (defined $hashref->{send_fnc}) {
  $PLC::errorcode = 11; # No FNC
  return undef;
 }

 if ( exists($hashref->{Protocol})
 && ( ( $hashref->{Protocol} =~ /^RS232/i )  # RS232 DF1
 || ( $hashref->{Protocol} =~ /^CIP_encap/i ) ) ) { # CL_Gateway or NET_ENI
  $hashref->{send_df1packet} = (pack "CCCCSC",
    ($hashref->{send_dst}, $hashref->{send_src},
     $hashref->{send_cmd},$hashref->{send_sts},$hashref->{send_tns},
     $hashref->{send_fnc}) );
 }
 else { # default to Ethernet                          # Ethernet PLC only(?)
  $hashref->{send_df1packet} = (pack "CCCCCCSC",
    ($hashref->{send_dst},$hashref->{send_control},
     $hashref->{send_src},$hashref->{send_lsap},
     $hashref->{send_cmd},$hashref->{send_sts},$hashref->{send_tns},
     $hashref->{send_fnc}) );
 }
 unless ($hashref->{send_df1packet}) {
  $PLC::errorcode = 14; # Unable to generate PCCC packet
  return undef;
 }
} # end sub packdf1a

sub packwritedata {
 print "\n--- sub PLC::packwritedata\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $hashref = shift;
 unless ($hashref) {
  $PLC::errorcode = 15; # No hash ref in subroutine call
  return undef;
 }
 if ($hashref->{send_datatype} ne 'ST') {
  unless (defined (@{$hashref->{send_writedata}})) {
   $PLC::errorcode = 16; # No write data specified
   return undef;
  }
 }
 my ($writedata,$floatx,$floatword,$floatsign,$floatexponent,$floatmantissa);
 $hashref->{send_writedatapkt} = '';
 for (@{$hashref->{send_writedata}}) {
  $writedata = $_;
  if ($hashref->{send_datatype} eq 'ST') { $writedata ||= '' }
  if ( ($hashref->{send_datatype} eq 'F')     # Floating point value
    || ( ( $hashref->{send_datatype} eq "PD") # PID Block Data Subelement
       && ( defined $hashref->{send_subelement})
       && ( $hashref->{send_subelement} >= 2) ) ) {
   $floatx = abs($writedata);
   if ($writedata < 0) {$floatsign = 1} else {$floatsign = 0}
   if ($floatx < (2 ** -126)) {
    $floatexponent = -127;
    $floatmantissa = 0;
   }
   else {
    if ( log2($floatx) < int( log2($floatx) ) ) {
     $floatexponent =int( log2($floatx) ) - 1;
    }
    else { $floatexponent =int( log2($floatx) ) }
    $floatmantissa = $floatx / ( 2 ** $floatexponent);
   }
   $floatword = ( (($floatsign << 31) & 0x80000000)
		| ((($floatexponent + 127) << 23) & 0x7F800000)
		| ( int(($floatmantissa * 0x00800000) + 0.5 ) & 0x007FFFFF) );
   if ($PLC::debug and $PLC::debug >= 5) {
    print "\n==========\nfloatsign = $floatsign\n";
    print "floatexponent = $floatexponent\n";
    print "floatmantissa = $floatmantissa\n";
    printf("floatword = %08X\n",$floatword);
   }
   if (CmdFnc($hashref) eq "F-00") { # word range write (PLC only)
    $hashref->{send_writedatapkt}
     .= pack("SS", (($floatword / 65536), ($floatword % 65536) ) );
   }
   else { # typed write
    $hashref->{send_writedatapkt}
     .= pack("SS", (($floatword % 65536), ($floatword / 65536) ) );
   }
  }
  elsif ($hashref->{send_datatype} eq 'ST') { # String data
   if ( (CmdFnc($hashref) eq "F-AA")	   	# Need to shuffle bytes ...
   &&   (!defined $hashref->{send_subelement}) )  { # for SLC strings
    my $len = length $writedata;
    if ( ($len % 2) == 1 ) { $writedata .= chr(0) }
    my @str = unpack("n*",$writedata);
    $writedata = pack("S*",($len,@str));
   }
   $hashref->{send_writedatapkt} = $writedata;
   $hashref->{send_datasize} = length $writedata;
  }
  elsif ($hashref->{send_datatype} eq 'A') { # ASCII data
   if (CmdFnc($hashref) eq "F-AA") {		# Need to shuffle bytes ...
    my $len = length $writedata; # for SLC ASCII
    if ( ($len % 2) == 1 ) { $writedata .= ' ' }
    my @str = unpack("n*",$writedata);
    $writedata = pack("S*",@str);
   }
   $hashref->{send_writedatapkt} .= $writedata;
   $hashref->{send_datasize} = length $hashref->{send_writedatapkt};
   $hashref->{send_totaltrans} = $hashref->{send_datasize};
  }
  elsif ($hashref->{send_datasize} == 1) { # other byte or character data
   $hashref->{send_writedatapkt} .= pack("C", ( $writedata ) );
  }
  else { # Integer value
   if ($writedata =~ /^0/) {
    $writedata = oct($writedata); # hex, octal, or binary number
   }
   $hashref->{send_writedatapkt} .= pack("S", ( $writedata ) );
  }
 } # end for @writedata

 if ( ($hashref->{send_IDvalue} == 3)
   && (length($hashref->{send_writedatapkt}) == 0) ) { # empty string
  $hashref->{send_typedatapkt} = pack("CC",(0x39 ,0x00));
 }
 elsif ( ($hashref->{send_IDvalue} < 8)
   && ($hashref->{send_datasize} < 8) ) {
  $hashref->{send_typedatapkt}
    = pack("CCSC", ( 0x9A, 0x09, (1 + length $hashref->{send_writedatapkt}),
   (($hashref->{send_IDvalue} << 4) | ($hashref->{send_datasize} & 0x0F))
   ) );
 }
 elsif ( $hashref->{send_IDvalue} < 8 ) {
  $hashref->{send_typedatapkt}
    = pack("CCSCC", ( 0x9A, 0x09, (2 + length $hashref->{send_writedatapkt}),
   (($hashref->{send_IDvalue} << 4) | 0x09) ,
   ($hashref->{send_datasize})
   ) );
 }
 elsif ( $hashref->{send_datasize} < 8 ) {
  $hashref->{send_typedatapkt}
   = pack("CCSCC", ( 0x9A, 0x09,
   (2 + length($hashref->{send_writedatapkt})),
   ( 0x90 | ($hashref->{send_datasize})),
   ($hashref->{send_IDvalue}) ) );
 }
 else {
  $hashref->{send_typedatapkt}
    = pack("CCSCCC", (0x9A, 0x09,(3 + length $hashref->{send_writedatapkt}),
  0x99, $hashref->{send_IDvalue}, $hashref->{send_datasize} ) );
 }
 return LinuxRules; # always true
} # end sub packwritedata

sub DF1addrpack1 {
# for hash reference parameter:
# ( Convert section,file,element,subelement to PLC system address )
# takes an array of 1-4 numbers, calculates mask byte, and for each number:
#  if < 255, pack it as 1 byte into return string
#  else prepend 0xFF, and pack it as 1 word into return string
 print "\n--- sub PLC::DF1addrpack1\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $hashref = shift;
 unless ($hashref) {
  $PLC::errorcode = 15; # No hash ref in subroutine call
  return undef;
 }
 my ($Section,$File,$Element,$Subelement)
  = ($hashref->{send_section},$hashref->{send_datafile},
     $hashref->{send_element},$hashref->{send_subelement});

 my $DF1addrstr;
 my $Mask;
 my $PLCsystemaddr;
 my $DF1packet;
 if (defined $Subelement) {$Mask = 0x0F}
 elsif (defined $Element) {$Mask = 0x07}
 elsif (defined $File) {$Mask = 0x03}
 else {$Mask = 0x01}

 $DF1addrstr = chr($Mask); # start with mask byte
 my $addrparam;
 for $addrparam ($Section,$File,$Element,$Subelement) {
  if ( defined $addrparam ) {
   print "packing $addrparam ...\n" if $PLC::debug;
   if ($addrparam < 255) { $DF1addrstr .= chr($addrparam) }
   else { $DF1addrstr .= pack "CS", ( 0xFF, $addrparam )  }
  }
 }
 my @x = unpack "C*", ($DF1addrstr);
 if ($PLC::debug) {
  print "PLC system address bytes : ";
  if (scalar @x) {
   for (@x) { printf "%02X ", $_ }
  }
  else {print "[no data] "}
  print "\n";
 }
 $hashref->{send_sys_addr} = $DF1addrstr;
 return $DF1addrstr;
} # end sub DF1addrpack1

sub gen_df1 {
 print "\n--- sub PLC::gen_df1\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $hashref = shift;
 unless ($hashref) {
  $PLC::errorcode = 15; # No hash ref in subroutine call
  return undef;
 }
 my ($writedata,$floatx,$floatword,$floatsign,$floatexponent,$floatmantissa);
 my ($Xword,$Xtrans,$Xfile,$Xsubelement);
 my $datatypedone = 0;
 my $pktoffset = 0;

 $hashref->{PLCtype} ||= '';
 $hashref->{send_dst} ||= 0;
 $hashref->{send_control} ||= 5;
 $hashref->{send_src} ||= 0;
 $hashref->{send_lsap} ||= 0;
 $hashref->{send_sts} ||= 0;
 $hashref->{send_elements} ||= 0;
 $hashref->{send_offset} = $pktoffset || 0;

 unless ( (CmdFnc($hashref) =~ /^[67]-/)    # CMD 6 and 7 functions
   ||     (CmdFnc($hashref) eq "F-3A")     # PLC set mode
   ||     (CmdFnc($hashref) eq "F-80") ) { # SLC set mode

  if ( ($hashref->{send_data_addr} =~ ':')
    || ($hashref->{send_data_addr} =~ '/') ) {
   ($Xfile,$Xword,$junk) = split ':',$hashref->{send_data_addr};
   if ( !defined $Xword ) { # maybe no ':' in address (i.e. B3/45)
    ($Xfile,$Xword,$junk) = split '/',$hashref->{send_data_addr};
    $Xword = "0/$Xword";
   }
   if (defined $junk) {	 # more than one ':' in address, not good
    $PLC::errorcode = 36; # Invalid data address
    return undef;
   }
  }
  else { # Just a file, no element, maybe? (i.e. for read section size)
   $Xfile = $hashref->{send_data_addr};
  }

  @array = split '', $Xfile;
  $hashref->{send_datatype} = "";
  $hashref->{send_datafile} = "";
  undef $hashref->{send_element}; ### NEW 10/30/2003
  undef $hashref->{send_subelement}; ### NEW 10/30/2003

  foreach (@array) {
   if ($_ =~ /[A-Z]/i && !$datatypedone) {$hashref->{send_datatype} .= uc $_}
   else {$hashref->{send_datafile} .= $_; $datatypedone = 1}
  }

  if ($hashref->{send_datafile} eq "") {
   if    ($hashref->{send_datatype} eq 'S') {
    $hashref->{send_datafile} = 2;
   }
   elsif ($hashref->{send_datatype} eq 'I') {
    $hashref->{send_datafile} = 1;
   }
   elsif ($hashref->{send_datatype} eq 'O') {
    $hashref->{send_datafile} = 0;
   }
   elsif ($hashref->{send_datatype} eq 'DI') {
    $hashref->{send_datafile} = 0;
   }
   elsif ($hashref->{send_datatype} eq 'ZB') {
    $hashref->{send_datafile} = 0;
   }
   elsif ($hashref->{send_datatype} eq 'ZC') {
    $hashref->{send_datafile} = 1;
   }
   elsif ($hashref->{send_datatype} eq 'ZD') {
    $hashref->{send_datafile} = 2;
   }
   elsif ($hashref->{send_datatype} eq 'ZE') {
    $hashref->{send_datafile} = 3;
   }
  }

   if ( ( !defined $hashref->{send_datatype})
     || ( !defined $hashref->{send_datafile}) ) {
    $PLC::errorcode = 36; # Invalid data address
    return undef;
   }

  if (defined $Xword) {
   ($hashref->{send_dataword},$Xsubelement,$junk) = split '\.', $Xword;
   if (defined $junk) {	 # too many dots
    $PLC::errorcode = 36; # Invalid data address
    return undef;
   }

   if (defined $Xsubelement) {
    $hashref->{send_element} = $hashref->{send_dataword};
    ($hashref->{send_subelement},$hashref->{send_bit},$junk)
      = split '/',$Xsubelement;
   }
   else { ($hashref->{send_element},$hashref->{send_bit},$junk)
    = split '/',$hashref->{send_dataword} }
   if ( ( !defined $hashref->{send_element}) || (defined $junk ) ) {
    $PLC::errorcode = 36; # Invalid data address
    return undef;
   }

# SfcStatus (SC)
   if ($hashref->{send_datatype} eq 'SC') { # SfcStatus
    if ( (defined $hashref->{send_bit})
    && (!defined $hashref->{send_subelement}) ) {
     $hashref->{send_subelement} = 0;
    }
    if ( defined $hashref->{send_subelement} ) {
     $hashref->{send_subelement} =~ s/CTL/0/; # Control Bits
     $hashref->{send_subelement} =~ s/PRE/1/; # Preset
     $hashref->{send_subelement} =~ s/TIM/2/; # Accumulated
    }
    if ( defined $hashref->{send_bit} ) {
     $hashref->{send_bit} =~ s/SA/15/; # 
     $hashref->{send_bit} =~ s/FS/14/; # 
     $hashref->{send_bit} =~ s/LS/13/; # 
     $hashref->{send_bit} =~ s/OV/12/; # Overflow
     $hashref->{send_bit} =~ s/ER/11/; # Error
     $hashref->{send_bit} =~ s/DN/10/; # Done
    }
   }
# Timers & Counters
   if ( ($hashref->{send_datatype} eq 'T')
   || ($hashref->{send_datatype} eq 'C') ) {
    if ( (defined $hashref->{send_bit})
    && (!defined $hashref->{send_subelement}) ) {
     $hashref->{send_subelement} = 0;
    }
    if ( defined $hashref->{send_subelement} ) {
     $hashref->{send_subelement} =~ s/CTL/0/; # Control Bits
     $hashref->{send_subelement} =~ s/PRE/1/; # Preset
     $hashref->{send_subelement} =~ s/ACC/2/; # Accumulated
    }
    if ( defined $hashref->{send_bit} ) {
     $hashref->{send_bit} =~ s/EN/15/; # Enable
     $hashref->{send_bit} =~ s/CU/15/; # Count Up
     $hashref->{send_bit} =~ s/TT/14/; # Timing
     $hashref->{send_bit} =~ s/CD/14/; # Count Down
     $hashref->{send_bit} =~ s/DN/13/; # Done
     $hashref->{send_bit} =~ s/OV/12/; # Overflow
     $hashref->{send_bit} =~ s/UN/11/; # Underflow
    }
   }
# Control Blocks
   elsif ( $hashref->{send_datatype} eq 'R' ) {
    if ( (defined $hashref->{send_bit})
    && (!defined $hashref->{send_subelement}) ) {
     $hashref->{send_subelement} = 0;
    }
    if ( defined $hashref->{send_subelement} ) {
     $hashref->{send_subelement} =~ s/CTL/0/; # Control Bits
     $hashref->{send_subelement} =~ s/LEN/1/; # Length
     $hashref->{send_subelement} =~ s/POS/2/; # Position
    }
    if ( defined $hashref->{send_bit} ) {
     $hashref->{send_bit} =~ s/EN/15/; # Enable Load
     $hashref->{send_bit} =~ s/EU/14/; # Enable Unload
     $hashref->{send_bit} =~ s/DN/13/; # Done
     $hashref->{send_bit} =~ s/EM/12/; # Empty
     $hashref->{send_bit} =~ s/ER/11/; # Error
     $hashref->{send_bit} =~ s/UL/10/; # 
     $hashref->{send_bit} =~ s/IN/9/;  # Inhibit
     $hashref->{send_bit} =~ s/FD/8/;  # Found
    }
   }
# PID Blocks ### (PID subelements are floating point) ###
   elsif ( $hashref->{send_datatype} eq 'PD' ) {
    if ( (defined $hashref->{send_bit})
    && (!defined $hashref->{send_subelement}) ) {
     $hashref->{send_subelement} = 0;
    }
    if ( defined $hashref->{send_subelement} ) {
     $hashref->{send_subelement} =~ s/CTL/0/;     # Control Bits
     $hashref->{send_subelement} =~ s/CTL0/0/;    # Control Bits
     $hashref->{send_subelement} =~ s/CTL1/1/;    # Control Bits
     $hashref->{send_subelement} =~ s/SP/2/;      # Setpoint
     $hashref->{send_subelement} =~ s/KP/4/;      # Proportional Constant
     $hashref->{send_subelement} =~ s/KI/6/;      # Integral Constant
     $hashref->{send_subelement} =~ s/KD/8/;      # Differential Constant
     $hashref->{send_subelement} =~ s/BIAS/10/;   # 
     $hashref->{send_subelement} =~ s/MAXS/12/;   # 
     $hashref->{send_subelement} =~ s/MINS/14/;   # 
     $hashref->{send_subelement} =~ s/DB/16/;     # 
     $hashref->{send_subelement} =~ s/SO/18/;     # 
     $hashref->{send_subelement} =~ s/MAXO/20/;   # 
     $hashref->{send_subelement} =~ s/MINO/22/;   # 
     $hashref->{send_subelement} =~ s/UPD/24/;    # 
     $hashref->{send_subelement} =~ s/PV/26/;     # Process Value
     $hashref->{send_subelement} =~ s/ERR/28/;    # 
     $hashref->{send_subelement} =~ s/OUT/30/;    # 
     $hashref->{send_subelement} =~ s/PVH/32/;    # 
     $hashref->{send_subelement} =~ s/PVL/34/;    # 
     $hashref->{send_subelement} =~ s/DVP/36/;    # 
     $hashref->{send_subelement} =~ s/DVN/38/;    # 
     $hashref->{send_subelement} =~ s/PVDB/40/;   # 
     $hashref->{send_subelement} =~ s/DVDB/42/;   # 
     $hashref->{send_subelement} =~ s/MAXI/44/;   # 
     $hashref->{send_subelement} =~ s/MINI/46/;   # 
     $hashref->{send_subelement} =~ s/TIE/48/;    # 
    }
    if ( defined $hashref->{send_bit} ) {
     $hashref->{send_bit} =~ s/PE/0/;     # 
     $hashref->{send_bit} =~ s/MO/1/;     # 
     $hashref->{send_bit} =~ s/CA/2/;     # 
     $hashref->{send_bit} =~ s/SWM/4/;    # 
     $hashref->{send_bit} =~ s/DO/6/;     # 
     $hashref->{send_bit} =~ s/PVT/7/;    # 
     $hashref->{send_bit} =~ s/CL/8/;     # 
     $hashref->{send_bit} =~ s/CT/9/;     # 
     $hashref->{send_bit} =~ s/DN/13/;    # Done
     $hashref->{send_bit} =~ s/EN/15/;    # Enable
     $hashref->{send_bit} =~ s/PVHA/16/;  # 
     $hashref->{send_bit} =~ s/PVLA/17/;  # 
     $hashref->{send_bit} =~ s/DVPA/18/;  # 
     $hashref->{send_bit} =~ s/DVNA/19/;  # 
     $hashref->{send_bit} =~ s/EWD/24/;   # 
     $hashref->{send_bit} =~ s/OLH/25/;   # 
     $hashref->{send_bit} =~ s/OLL/26/;   # 
     $hashref->{send_bit} =~ s/SPOR/27/;  # Setpoint Out of Range
     $hashref->{send_bit} =~ s/INI/28/;   # 
    }
   }
# Block Transfer Blocks
   elsif ( $hashref->{send_datatype} eq 'BT' ) {
    if ( (defined $hashref->{send_bit})
    && (!defined $hashref->{send_subelement}) ) {
     $hashref->{send_subelement} = 0;
    }
    if ( defined $hashref->{send_subelement} ) {
     $hashref->{send_subelement} =~ s/CTL/0/;  # Control Bits
     $hashref->{send_subelement} =~ s/RLEN/1/; # Requested Word Count
     $hashref->{send_subelement} =~ s/DLEN/2/; # Transmitted Word Count
     $hashref->{send_subelement} =~ s/FILE/3/; # Data File #
     $hashref->{send_subelement} =~ s/ELEM/4/; # Element #
     $hashref->{send_subelement} =~ s/RGS/5/;  # RackGroupSlot #
    }
    if ( defined $hashref->{send_bit} ) {
     $hashref->{send_bit} =~ s/EN/15/; # Enable
     $hashref->{send_bit} =~ s/ST/14/; # 
     $hashref->{send_bit} =~ s/DN/13/; # Done
     $hashref->{send_bit} =~ s/ER/12/; # Error
     $hashref->{send_bit} =~ s/CO/11/; # 
     $hashref->{send_bit} =~ s/EW/10/; # 
     $hashref->{send_bit} =~ s/NR/9/;  # 
     $hashref->{send_bit} =~ s/TO/8/;  # Timeout
     $hashref->{send_bit} =~ s/RW/7/;  # 
    }
   }
# Message Blocks
   elsif ( $hashref->{send_datatype} eq 'MG') {
    if ( (defined $hashref->{send_bit})
    && (!defined $hashref->{send_subelement}) ) {
     $hashref->{send_subelement} = 0;
    }
    if ( defined $hashref->{send_subelement} ) {
     $hashref->{send_subelement} =~ s/CTL/0/;  # Control Bits
     $hashref->{send_subelement} =~ s/ERR/1/;  # Error Code
     $hashref->{send_subelement} =~ s/RLEN/2/; # Requested Word Count
     $hashref->{send_subelement} =~ s/DLEN/3/; # Transmitted Word Count
    }
    if ( defined $hashref->{send_bit} ) {
     $hashref->{send_bit} =~ s/NR/9/;  # No Response
     $hashref->{send_bit} =~ s/TO/8/;  # Timeout
     $hashref->{send_bit} =~ s/EN/7/;  # Enable
     $hashref->{send_bit} =~ s/ST/6/;  # Start
     $hashref->{send_bit} =~ s/DN/5/;  # Done
     $hashref->{send_bit} =~ s/ER/4/;  # Error
     $hashref->{send_bit} =~ s/CO/3/;  # Continue
     $hashref->{send_bit} =~ s/EW/2/;  # Enable-Waiting
     $hashref->{send_bit} =~ s/SD/1/;  # 
     $hashref->{send_bit} =~ s/SE/0/;  # 
    }
   }
# String Data
   elsif ($hashref->{send_datatype} eq 'ST') {
    if ( defined $hashref->{send_subelement} ) {
     $hashref->{send_subelement} =~ s/LEN/0/; # String Length
    }
   }
# convert bit numbers > 15 to word+bit
   if ( ( ($hashref->{send_datatype} eq 'B')
   || ($hashref->{send_datatype} eq 'PD') )
   && (defined $hashref->{send_bit}) ) {
    while ($hashref->{send_bit} >= 16) {
     $hashref->{send_bit} -= 16;
     if ( defined $hashref->{send_subelement} ) {
      $hashref->{send_subelement}++;
     }
     else {$hashref->{send_element}++}
    }
   }
  }

  if    ( ($hashref->{send_datatype} eq 'PN') # Program Name
       || ($hashref->{send_datatype} eq 'RG') # Rung Data
        ) {
           $hashref->{send_section} = 1;
           $hashref->{send_IDvalue} = 0x0D;
           $hashref->{send_datasize} = 1;
          }
  elsif ( ($hashref->{send_datatype} eq 'FO') # Output Force Table
       || ($hashref->{send_datatype} eq 'FI') # Input Force Table
       || ($hashref->{send_datatype} eq 'FF') # Force File Section
        ) {
           $hashref->{send_section} = 2;
           $hashref->{send_IDvalue} = 0x04;
           $hashref->{send_datasize} = 2;
          }
  elsif (  $hashref->{send_datatype} eq 'XA') # Section #3
          {
           $hashref->{send_section} = 3;
           $hashref->{send_IDvalue} = 0x04;
           $hashref->{send_datasize} = 2;
          }
  elsif (  $hashref->{send_datatype} eq 'XB') # Section #4
          {
           $hashref->{send_section} = 4;
           $hashref->{send_IDvalue} = 0x04;
           $hashref->{send_datasize} = 2;
          }
  elsif (  $hashref->{send_datatype} eq 'XC') # Section #5
          {
           $hashref->{send_section} = 5;
           $hashref->{send_IDvalue} = 0x04;
           $hashref->{send_datasize} = 2;
          }
  elsif (  $hashref->{send_datatype} eq 'XD') # Section #6
          {
           $hashref->{send_section} = 6;
           $hashref->{send_IDvalue} = 0x04;
           $hashref->{send_datasize} = 2;
          }
  elsif ( ( ( $hashref->{send_datatype} eq 'ST')  # String Data
         || ( $hashref->{send_datatype} eq 'A') ) # ASCII Data
         && ( !defined $hashref->{send_subelement}) )
          {
           $hashref->{send_section} = 0;
           $hashref->{send_IDvalue} = 0x03;
           $hashref->{send_datasize} = 1;
          }
  elsif ( ( $hashref->{send_datatype} eq 'T') # Timer Data
         && ( !defined $hashref->{send_subelement}) )
          {
           $hashref->{send_section} = 0;
           $hashref->{send_IDvalue} = 0x05;
           $hashref->{send_datasize} = 6; ##### 10 ?????
          }
  elsif ( ( $hashref->{send_datatype} eq 'SC') # SC = SfcStatus
         && ( !defined $hashref->{send_subelement}) )
          {
           $hashref->{send_section} = 0;
           $hashref->{send_IDvalue} = 0x1D;
           $hashref->{send_datasize} = 6;
          }
  elsif ( ( $hashref->{send_datatype} eq 'C') # Counter Data
         && ( !defined $hashref->{send_subelement}) )
          {
           $hashref->{send_section} = 0;
           $hashref->{send_IDvalue} = 0x06;
           $hashref->{send_datasize} = 6;
          }
  elsif ( ( $hashref->{send_datatype} eq 'R') # Control Block Data
         && ( !defined $hashref->{send_subelement}) )
          {
           $hashref->{send_section} = 0;
           $hashref->{send_IDvalue} = 0x07;
           $hashref->{send_datasize} = 6;
          }
  elsif (  $hashref->{send_datatype} eq 'F') # Floating Point
          {
           $hashref->{send_section} = 0;
           $hashref->{send_IDvalue} = 0x08;
           $hashref->{send_datasize} = 4;
          }
  elsif (  $hashref->{send_datatype} eq 'D') # BCD or Hex Data
          {
           $hashref->{send_section} = 0;
           $hashref->{send_IDvalue} = 0x10;
           $hashref->{send_datasize} = 2;
          }
  elsif ( ( $hashref->{send_datatype} eq 'PD') # PID Block Data
         && ( !defined $hashref->{send_subelement}) )
          {
           $hashref->{send_section} = 0;
           $hashref->{send_IDvalue} = 0x15;
           $hashref->{send_datasize} = 164;
          }
  elsif ( ( $hashref->{send_datatype} eq 'PD') # PID Block Data Subelement
         && ( defined $hashref->{send_subelement}) )
          {
           $hashref->{send_section} = 0;
           $hashref->{send_IDvalue} = 0x15;
           if ($hashref->{send_subelement} <= 1) {
           $hashref->{send_IDvalue} = 0x04; # integer data
            $hashref->{send_datasize} = 2;  # 2 bytes/subelement
           }
           else {
           $hashref->{send_IDvalue} = 0x08; # floating point data
            $hashref->{send_datasize} = 4;  # 4 bytes/subelement
           }
          }
  elsif ( ( $hashref->{send_datatype} eq 'MG') # Message Block Data
         && ( !defined $hashref->{send_subelement}) )
          {
           $hashref->{send_section} = 0;
           $hashref->{send_IDvalue} = 0x16;
           $hashref->{send_datasize} = 112;
          }
  elsif ( ( $hashref->{send_datatype} eq 'BT') # Block Transfer Data
         && ( !defined $hashref->{send_subelement}) )
          {
           $hashref->{send_section} = 0;
           $hashref->{send_IDvalue} = 0x20;
           $hashref->{send_datasize} = 12;
          }
  else    {				       # Misc. Data Table Data
           $hashref->{send_section} = 0;
           $hashref->{send_IDvalue} = 0x04;
           $hashref->{send_datasize} = 2;
          }

# For PLC5's, I/O (files 0 and 1) are specified in octal, so convert it
   if ( ($hashref->{PLCtype} =~ /^PLC/i)
   &&   ($hashref->{send_section} == 0) # data file section
   &&   ($hashref->{send_datafile} =~ /^\d+$/)
   &&   ($hashref->{send_datafile} <= 1) ) { # 0=Inputs, 1=Outputs
    if (defined $hashref->{send_element}) {
     if ($hashref->{send_element} !~ /^[0-7]+$/) {
      qprinterr "\n++ PLC5 I/O must be specified in octal ++\n\n";
      $PLC::errorcode = 36; # Invalid data address
      return undef;
     } # end if not octal
     $hashref->{send_element} = oct($hashref->{send_element});
    } # end if defined element
    if (defined $hashref->{send_subelement}) {
     if ($hashref->{send_subelement} !~ /^[0-7]+$/) {
      qprinterr "\n++ PLC5 I/O must be specified in octal ++\n\n";
      $PLC::errorcode = 36; # Invalid data address
      return undef;
     } # end if not octal
     $hashref->{send_subelement} = oct($hashref->{send_subelement});
    } # end if defined subelement
    if (defined $hashref->{send_bit}) {
     if ($hashref->{send_bit} !~ /^[0-7]+$/) {
      qprinterr "\n++ PLC5 I/O must be specified in octal ++\n\n";
      $PLC::errorcode = 36; # Invalid data address
      return undef;
     } # end if not octal
     $hashref->{send_bit} = oct($hashref->{send_bit});
    } # end if defined sendbit
   } # end if PLC

  if ($hashref->{send_datatype}) {
   if ($debug) {
    $s = ($hashref->{send_elements} == 1) ? '' : 's';
    print "\n----- $hashref->{PLCname} ---- $hashref->{send_data_addr} ";
    print "($hashref->{send_elements} element$s)\n";
    print "Datatype = $hashref->{send_datatype}, ";
    print "Section = $hashref->{send_section}, ";
    print "File = $hashref->{send_datafile}";
    if ( defined $hashref->{send_element}) {
     print ", Element = $hashref->{send_element}";
    }
    else { print ", no Element";}
    if ( defined $hashref->{send_subelement}) {
     print ", Sub-element = $hashref->{send_subelement}";
    }
    else { print ", no Sub-element";}
   } # end if $debug
   if ( (defined $hashref->{send_bit})
    && ( ($hashref->{send_bit} < 0)
    || ($hashref->{send_bit} > 15) ) ) { undef $hashref->{send_bit} }

   if (defined $hashref->{send_bit}) {
    $hashref->{send_bitmask} = (1 << $hashref->{send_bit});
    if ($debug) {
     printf(",\n Bit = %d, Bit mask = 0x%04X",
      $hashref->{send_bit}, $hashref->{send_bitmask});
    }
    if ($hashref->{send_elements} > 1 ) {
     if ($debug) {
      print "\n=== bit-level data specified, setting elements to 1 ===\n";
     }
     $hashref->{send_elements} = 1;
    }
   }
   print "\n" if $debug;

# assemble PLC system address
   DF1addrpack1($hashref) or return undef;
  }

 }

######################## generate command packets ########################
################ make sure all required fields are defined ###############

 if (CmdFnc($hashref) eq "6-00") {
 print "\n\nGenerating 6-00 echo command packet .........\n\n" if $PLC::debug;
# 6-00 echo command
# assemble PCCC packet

  undef $hashref->{send_bit};
  packdf1a($hashref) or return undef; # generate "base" PCCC packet 
  $hashref->{send_df1packet}
   .= (pack ("C*", @{$hashref->{send_data_array}}) );
 } # end of 6-00 echo command

 elsif (CmdFnc($hashref) eq "6-03") {
  print "\n\nGenerating 6-03 diagnostic status packet ...\n\n" if $PLC::debug;
# 6-03 diagnostic status command
# assemble PCCC packet
  undef $hashref->{send_bit};
  packdf1a($hashref) or return undef; # generate "base" PCCC packet 
 } # end of 6-03 diagnostic status command


 elsif (CmdFnc($hashref) eq "F-00") { # (PLC only)
 print "\n\nGenerating F-00 word range write command packet ...\n\n"
  if $PLC::debug;
# word range write (totaltrans is in words)

 if (defined $hashref->{send_bit}) {
  qprinterr "\n\n++ Can't specify write data to bit level ++\n\n";
  $PLC::errorcode = 12; # Can't specify write to bit level
  return undef;
 }

  $hashref->{send_totaltrans}
   = $hashref->{send_elements} * $hashref->{send_datasize} / 2;
  $hashref->{send_bytes} = $hashref->{send_totaltrans} * 2;

  packwritedata($hashref) or return undef; # packs data into send_writedatapkt

# assemble PCCC packet
  packdf1a($hashref) or return undef; # generate "base" PCCC packet 
  $hashref->{send_df1packet} .= (pack "SS",
    ($hashref->{send_offset}, $hashref->{send_totaltrans}) )
 . $hashref->{send_sys_addr}
   . $hashref->{send_writedatapkt};
############################ TEST WITH BYTE DATA ######################
 } # end of F-00 word range write command


 elsif (CmdFnc($hashref) eq "F-01") { # (PLC only)
 print "\n\nGenerating F-01 word range read command packet ...\n\n"
  if $PLC::debug;
# word range read (totaltrans is in words)
  $hashref->{send_totaltrans}
   = $hashref->{send_elements} * $hashref->{send_datasize} / 2;
  $hashref->{send_bytes} = $hashref->{send_totaltrans} * 2;
# assemble PCCC packet
  packdf1a($hashref) or return undef; # generate "base" PCCC packet 
  $hashref->{send_df1packet} .= (pack "SS",
    ($hashref->{send_offset}, $hashref->{send_totaltrans}) )
   . $hashref->{send_sys_addr} . chr($hashref->{send_bytes}) ;
 } # end of F-01 word range read command


 elsif (CmdFnc($hashref) eq "F-29") {
 print "\n\nGenerating F-29 read section size command packet .........\n\n"
  if $PLC::debug;
# read section size
# assemble PCCC packet
  packdf1a($hashref) or return undef; # generate "base" PCCC packet 
  $hashref->{send_df1packet} .= $hashref->{send_sys_addr};
 } # end of F-29 read section size command


 elsif (CmdFnc($hashref) eq "F-3A") {
 print "\n\nGenerating F-3A PLC setmode command packet ...\n\n" if $PLC::debug;
# F-3A PLC set mode command
# assemble PCCC packet

  undef $hashref->{send_bit};
  packdf1a($hashref) or return undef; # generate "base" PCCC packet 
  $hashref->{send_df1packet}
   .= (pack ("C*", @{$hashref->{send_data_array}}) );
 } # end of F-3A PLC set mode command


 elsif (CmdFnc($hashref) eq "F-67") {
  print "\n\nGenerating F-67 data write command packet ...\n\n"
   if $PLC::debug;
# typed write (totaltrans is in elements)
  if (defined $hashref->{send_bit}) {
   qprinterr "\n\n++ Can't specify write data to bit level ++\n\n";
   $PLC::errorcode = 12; # Can't specify write to bit level
   return undef;
  }
  $hashref->{send_totaltrans} = $hashref->{send_elements} || 1;
  packwritedata($hashref) or return undef; # packs data into send_writedatapkt

# assemble PCCC packet
  packdf1a($hashref) or return undef; # generate "base" PCCC packet 
  $hashref->{send_df1packet} .= (pack "SS",
    ($hashref->{send_offset}, $hashref->{send_totaltrans}) )
   . $hashref->{send_sys_addr}
   . $hashref->{send_typedatapkt}
   . $hashref->{send_writedatapkt};
 } # end of F-67 typed write command


 elsif (CmdFnc($hashref) eq "F-68") {
  print "\n\nGenerating F-68 data read command packet ...\n\n" if $PLC::debug;
# typed read (totaltrans is in elements)
  $hashref->{send_totaltrans} = $hashref->{send_elements};
# assemble PCCC packet
  packdf1a($hashref) or return undef; # generate "base" PCCC packet 

 $hashref->{send_df1packet} .= (pack "SS",
  ($hashref->{send_offset}, $hashref->{send_totaltrans}) )
  . $hashref->{send_sys_addr}
  . (pack "S", ($hashref->{send_elements}) );
 } # end of F-68 typed read command


 elsif (CmdFnc($hashref) eq "F-80") {
 print "\n\nGenerating F-80 SLC setmode command packet ...\n\n" if $PLC::debug;
# F-80 SLC set mode command
  undef $hashref->{send_bit};

# assemble PCCC packet
  packdf1a($hashref) or return undef; # generate "base" PCCC packet 
  $hashref->{send_df1packet}
   .= (pack ("C*", @{$hashref->{send_data_array}}) );
 } # end of F-80 SLC set mode command


 elsif (CmdFnc($hashref) eq "F-A2") { # SLC typed logical read
  print "\n\nGenerating F-A2 SLC typed log. read packet ...\n\n"
   if $PLC::debug;

  my $SLCdatatype;
  if    ($hashref->{send_datatype} eq 'O')  {$SLCdatatype = 0x82}
  elsif ($hashref->{send_datatype} eq 'I')  {$SLCdatatype = 0x83}
  elsif ($hashref->{send_datatype} eq 'S')  {$SLCdatatype = 0x84}
  elsif ($hashref->{send_datatype} eq 'B')  {$SLCdatatype = 0x85}
  elsif ($hashref->{send_datatype} eq 'T')  {$SLCdatatype = 0x86}
  elsif ($hashref->{send_datatype} eq 'C')  {$SLCdatatype = 0x87}
  elsif ($hashref->{send_datatype} eq 'R')  {$SLCdatatype = 0x88}
  elsif ($hashref->{send_datatype} eq 'N')  {$SLCdatatype = 0x89}
  elsif ($hashref->{send_datatype} eq 'F')  {$SLCdatatype = 0x8A}
  elsif ($hashref->{send_datatype} eq 'ST') {$SLCdatatype = 0x8D}
  elsif ($hashref->{send_datatype} eq 'A')  {$SLCdatatype = 0x8E}
  elsif ($hashref->{send_datatype} eq 'D')  {$SLCdatatype = 0x8F}
  elsif ($hashref->{send_datatype} eq 'PN') { # Program Name
   $SLCdatatype = 0x00;
  }
  elsif ($hashref->{send_datatype} eq 'DI')  { # Directory ???
   $SLCdatatype = 0x01;
   $PLC::hex = 1;
  }
  elsif ($hashref->{send_datatype} eq 'FO') { # Output Force File
   $SLCdatatype = 0xA1;
   $hashref->{send_datasize} = 1;
   $PLC::hex = 1;
  }
  elsif ($hashref->{send_datatype} eq 'FI') { # Input Force File
   $SLCdatatype = 0xA2;
   $hashref->{send_datasize} = 1;
   $PLC::hex = 1;
  }
  elsif ($hashref->{send_datatype} eq 'RG') { ### Rung Data ??????
   $SLCdatatype = 0x22;
  }
  elsif ($hashref->{send_datatype} eq 'ZA') { ### ??? I/O Config ???
   $SLCdatatype = 0x60;
   $PLC::hex = 1;
  }
  elsif ($hashref->{send_datatype} eq 'ZB') { ### Channel Config - General
   $SLCdatatype = 0x42;
   $PLC::hex = 1;
  }
  elsif ($hashref->{send_datatype} eq 'ZC') { ### Channel 0 User Config
   $SLCdatatype = 0x43;
   $PLC::hex = 1;
  }
  elsif ($hashref->{send_datatype} eq 'ZD') { ### Channel 0 System Config
   $SLCdatatype = 0x44;
   $PLC::hex = 1;
  }
  elsif ($hashref->{send_datatype} eq 'ZE') { ### Channel 1 System Config
   $SLCdatatype = 0x45;
   $PLC::hex = 1;
  }
  else {$SLCdatatype = 0x89} # default to integer data

  my $size = $hashref->{send_datasize} * $hashref->{send_elements};

  if ($hashref->{send_datatype} eq 'ST') {
   if (defined $hashref->{send_subelement}) {
    $PLC::hex = 1;
   }
   else  {
    $size = 80; # String data, read 80 bytes, trim to length later
   }
  }

  my ($file,$element,$subelem)
   = ( $hashref->{send_datafile},
       $hashref->{send_element},
       $hashref->{send_subelement} );

  $subelem ||= 0;

  my $data_addr =
   (($file < 255) ? (pack "C",($file)) : (pack "CS",(0xFF,$file)))
   . (pack "C",($SLCdatatype))
   . (($element < 255) ? (pack "C",($element)) : (pack "CS",(0xFF,$element)))
   . (($subelem < 255) ? (pack "C",($subelem)) : (pack "CS",(0xFF,$subelem)));

# assemble PCCC packet
  packdf1a($hashref) or return undef; # generate "base" PCCC packet 

  $hashref->{send_df1packet} .= (pack "C", ($size)) . $data_addr;

 } # end F-A2 SLC typed logical read


 elsif (CmdFnc($hashref) eq "F-AA") { # SLC typed logical write
  if ($PLC::debug) {
   print "\n\nGenerating F-AA SLC typed log. wrt. packet ...\n\n";
  }
  my $SLCdatatype;
  if    ($hashref->{send_datatype} eq 'O')  {$SLCdatatype = 0x82}
  elsif ($hashref->{send_datatype} eq 'I')  {$SLCdatatype = 0x83}
  elsif ($hashref->{send_datatype} eq 'S')  {$SLCdatatype = 0x84}
  elsif ($hashref->{send_datatype} eq 'B')  {$SLCdatatype = 0x85}
  elsif ($hashref->{send_datatype} eq 'T')  {$SLCdatatype = 0x86}
  elsif ($hashref->{send_datatype} eq 'C')  {$SLCdatatype = 0x87}
  elsif ($hashref->{send_datatype} eq 'R')  {$SLCdatatype = 0x88}
  elsif ($hashref->{send_datatype} eq 'N')  {$SLCdatatype = 0x89}
  elsif ($hashref->{send_datatype} eq 'F')  {$SLCdatatype = 0x8A}
  elsif ($hashref->{send_datatype} eq 'ST') {$SLCdatatype = 0x8D}
  elsif ($hashref->{send_datatype} eq 'A')  {$SLCdatatype = 0x8E}
  elsif ($hashref->{send_datatype} eq 'D')  {$SLCdatatype = 0x8F}
  elsif ($hashref->{send_datatype} eq 'PN') { # Program Name
   $SLCdatatype = 0x00;
  }
  elsif ($hashref->{send_datatype} eq 'DI')  { # Directory ???
   $SLCdatatype = 0x01;
   $PLC::hex = 1;
  }
  elsif ($hashref->{send_datatype} eq 'FO') { # Output Force File
   $SLCdatatype = 0xA1;
   $hashref->{send_datasize} = 1;
   $PLC::hex = 1;
  }
  elsif ($hashref->{send_datatype} eq 'FI') { # Input Force File
   $SLCdatatype = 0xA2;
   $hashref->{send_datasize} = 1;
   $PLC::hex = 1;
  }
  elsif ($hashref->{send_datatype} eq 'RG') { ### Rung Data ??????
   $SLCdatatype = 0x22;
  }
  elsif ($hashref->{send_datatype} eq 'ZA') { ### ??? I/O Config ???
   $SLCdatatype = 0x60;
   $PLC::hex = 1;
  }
  elsif ($hashref->{send_datatype} eq 'ZB') { ### Channel Config - General
   $SLCdatatype = 0x42;
   $PLC::hex = 1;
  }
  elsif ($hashref->{send_datatype} eq 'ZC') { ### Channel 0 User Config
   $SLCdatatype = 0x43;
   $PLC::hex = 1;
  }
  elsif ($hashref->{send_datatype} eq 'ZD') { ### Channel 0 System Config
   $SLCdatatype = 0x44;
   $PLC::hex = 1;
  }
  elsif ($hashref->{send_datatype} eq 'ZE') { ### Channel 1 System Config
   $SLCdatatype = 0x45;
   $PLC::hex = 1;
  }
  else {$SLCdatatype = 0x89} # default to integer data

  my ($file,$element,$subelem)
   = ( $hashref->{send_datafile},
       $hashref->{send_element},
       $hashref->{send_subelement} );

  $subelem ||= 0;

  my $data_addr =
   (($file < 255) ? (pack "C",($file)) : (pack "CS",(0xFF,$file)))
   . (pack "C", ($SLCdatatype))
   . (($element < 255) ? (pack "C",($element)) : (pack "CS",(0xFF,$element)))
   . (($subelem < 255) ? (pack "C",($subelem)) : (pack "CS",(0xFF,$subelem)));

  packwritedata($hashref) or return undef; # packs data into send_writedatapkt

# assemble PCCC packet
  packdf1a($hashref) or return undef; # generate "base" PCCC packet 

  my $size = $hashref->{send_datasize} * $hashref->{send_elements};
    if ( ($hashref->{send_datatype} eq 'A')      # ASCII data
  ||   ($hashref->{send_datatype} eq 'ST') ) { # String data
   $size = length $hashref->{send_writedatapkt};
  }

  $hashref->{send_df1packet} .= (pack "C", ($size)) . $data_addr
   . $hashref->{send_writedatapkt};
 } # end F-AA SLC typed logical write


 else {
  printf(STDERR "\n\n+++ Unsupported function %02X-%02X +++\n\n",
   $hashref->{send_cmd},$hashref->{send_fnc});
  $PLC::errorcode = 13; # Unsupported CMD-FNC
  return undef;
 }
 if ($PLC::debug) {
  print "DF1 packet:\n";
  PLC::PrintBytes($hashref->{send_df1packet});
  print "\nPLCname = $hashref->{PLCname}\n";
  if ( defined $hashref->{send_data_addr}) {
   print "Data Address = $hashref->{send_data_addr}\n";
  }
  if ( defined $hashref->{send_elements}) {
   print "elements = $hashref->{send_elements}\n";
  }
  if ( defined $hashref->{send_writedata}) {
   print "Write Data = ", join(' ',@{$hashref->{send_writedata}}), "\n";
  }
  printf "command = 0x%02X\n",$hashref->{send_cmd};
  printf "function = 0x%02X\n",$hashref->{send_fnc};
  PLC::PrintBytes($hashref->{send_df1packet});
  print "\n\n";
 }
 unless ($hashref->{send_df1packet}) {
  $PLC::errorcode = 14; # Unable to generate DF1 packet
  return undef;
 }
 return $hashref->{send_df1packet};
} # end sub gen_df1

sub sendDF1pkt {
##################
# generate RS232 or Ethernet or ControlLogix Gateway data packet,
# send packet via RS232 or Ethernet and get reply
##################
 print "\n--- sub PLC::sendDF1pkt\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $plc = shift;
 unless ($plc) {
  $PLC::errorcode = 15; # No hash ref in subroutine call
  return undef;
 }
 my $PLCname = $plc->{PLCname};
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 my @return_array;
### my $tries = $plc->{retries} || $PLC::retries;
### my $retry_delay = $plc->{retry_delay} || $PLC::retry_delay;
 my $tries = 1; # Let PLCcommand do the re-trying for now
 do {{ # double-curly-do allows 'next' + 'last' statements
  $PLC::errorcode = 0; # No Error
  if (Protocol($PLCname) =~ /^CIP_encap/i) { # ControlLogix Gateway or NET_ENI
   $plc->{recv_raw_pkt}
    = ABgate3::sendDF1pkt($PLCname,$plc->{send_df1packet});
  }
  elsif (Protocol($PLCname) =~ /^Eth/i) { # Ethernet PLC (or SLC)
   $plc->{recv_raw_pkt} = ABeth1::ethsendDF1($plc); # get response from PLC
  }
  else { # default to RS232 if Protocol unknown
   $plc->{send_rs232_pkt} = $plc->{send_df1packet};
# now send it and get reply, put in df1_pkt
   $plc->{recv_raw_pkt}
    = DF1a::DF1sendpkt($PLCname, $plc->{send_rs232_pkt}); ## CHECK AUTO-INIT
   $plc->{recv_df1_pkt} =  $plc->{recv_raw_pkt};
  }

  if ($PLC::errorcode) {
##   if ($PLC::debug) {
    qprint "\n++ ERROR - $PLC::errormsg[$PLC::errorcode]\n";
    qprint "++ PLC = $PLCname\n";
    qprint "Re-trying ...\n" if $tries > 1;
##   }
   PLC::snooze(1);
   unless (defined gen_df1($PLCproc{$PLCname})) { # regenerate all values
    return undef;
   }
  } # end if errorcode
  else { # no $PLC::errorcode, but check return status (STS) from PLC
   @return_array = extract_recv_data($PLCproc{$PLCname});
   if ($PLCproc{$PLCname}->{recv_sts}) {
##    if ($PLC::debug) {
     qprint "\n++ Received: ", ABerrormessage($PLCproc{$PLCname})," ++\n";
     qprint "Re-trying ...\n" if $tries > 1;
##    }
    PLC::snooze(1);
    unless (defined gen_df1($PLCproc{$PLCname})) { # regenerate all values
     return undef;
    }
    $PLC::errorcode = 6; # PLC returned error code
   } # end if recv_sts (if plc or interface returned a PCCC error code)
  }

 }} while ( ($PLC::errorcode) && (--$tries > 0) );

 if ($PLC::errorcode) {
  return undef;
 }
 return extract_recv_data($PLCproc{$PLCname});
## return @return_array;
} # end sub sendDF1pkt

sub CmdFnc {
# convert cmd & fnc codes into a string (i.e. "F-68" = cmd 0x0F, fnc 0x68)
 print "\n--- sub PLC::CmdFnc\n" if ($PLC::debug and $PLC::debug >= 9);
 my $CmdFncStr;
 $PLC::errorcode = 0; # No Error
 my $hashref = shift or return 'N/A';
 return 'N/A' if (!defined $hashref->{send_cmd}
              ||  !defined $hashref->{send_fnc});
 $CmdFncStr = sprintf("%X-%02X",$hashref->{send_cmd},$hashref->{send_fnc});
 return $CmdFncStr;
} # end sub CmdFnc

sub PrintBytes {
## print "\n--- sub PLC::PrintBytes\n" if ($PLC::debug and $PLC::debug >= 5);
 if ( defined $_[0]) {
  print "Data:\n"; my $i=0;
  for (unpack "C*", ($_[0]))
   {printf("%02X ",$_); if (++$i > 15) {print "\n"; $i=0} }
  print "\n";
 }
}

sub Protocol {
 print "\n--- sub PLC::Protocol\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 my $Protocol;
 $Protocol = $PLCproc{$PLCname}->{'Protocol'} if $PLCname;
 $Protocol ||= 'UNKNOWN';
 return $Protocol;
} # end sub Protocol

sub CommType {
 print "\n--- sub PLC::CommType\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 my $CommType;
 $CommType = $PLCproc{$PLCname}->{'CommType'} if $PLCname;
 $CommType ||= 'UNKNOWN';
 if    ($CommType =~ /^Control/i) { $CommType = "ControlLogixGateway" }
 elsif ($CommType =~ /^Eth/i)     { $CommType = "EthernetPLC" }
 elsif ($CommType =~ /^RS232/i)   { $CommType = "RS232" }
 return $CommType;
} # end sub CommType

sub printreturndata {
 print "\n--- sub PLC::printreturndata\n"
  if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $plc = shift;
 my $is_string = 0;
 unless (ref $plc) {
  $PLC::errorcode = 15; # No hash ref in subroutine call
  return undef;
 }
 $plc->{send_datafile} = '' unless defined $plc->{send_datafile};
 $plc->{send_datatype} = '' unless defined $plc->{send_datatype};
 unless (defined $plc->{send_writedata}) {
  if (CmdFnc($plc) eq "6-03") { # Diagnostic Status data
   ABeth1::getstatusdata($plc) or return undef;
  }
  elsif ( (defined $plc->{recv_data_array})
  && (scalar @{$plc->{recv_data_array}}) ) {
   if ($plc->{send_datafile} =~ /^\d+$/) {
    print "\n$plc->{send_data_addr} :\n" unless $PLC::quiet;
    if ( ($plc->{send_section} == 0) && ($plc->{send_datafile} <= 1) ) { # I/O
     unless ( ($PLC::hex) || (defined $plc->{send_bit}) ) {
      $PLC::bin = 1; # default to binary format for printing
     }
    }
   }
   $i = 0;
   for (@{$plc->{recv_data_array}}) {
    if ($PLC::hex) {
     if ($_ < 0) { $_ += 65536}
     printf("0x%04X ", $_); # print it in hexadecimal
    }
    elsif ($PLC::bin) { # print it in binary ( 1 word/line)
     print((unpack "B16",(pack "n",$_)),"\n");
    }
    elsif ( ($plc->{send_datatype} eq 'RG') # Rung Data #QQQQQ undefined???
    || ( CmdFnc($plc) eq "6-00" ) ) { # 6-00 echo command
     if ($_ < 0) { $_ += 65536}
     printf("0x%02X ", $_); # display in hex bytes
    }
    elsif ( ($plc->{send_datatype} eq 'PN') # Program Name
    || ( ( !defined $plc->{send_subelement})
    && ( ( $plc->{send_datatype} eq 'ST' )      # String Data
    || ( $plc->{send_datatype} eq 'A' ) ) ) ) { # ASCII Data
     $is_string = 1; # alphanumeric data
     print( chr($_) ); # display in ASCII
    }
    elsif ( ( $plc->{send_datatype} eq 'F' )  # Floating point value
    || ( ( $plc->{send_datatype} eq "PD")     # PID Block Data Subelement
       && ( defined $plc->{send_subelement})
       && ( $plc->{send_subelement} >= 2) ) ) {
     print "$_ ";
    }
    else {printf("%6d ",$_)}
    if ( !$is_string && (++$i >= 10) ) { $i=0; print "\n"; }
   }
  }
  else {print "[no data returned] "}
 } # end unless defined writedata
 if ((defined $plc->{recv_sts}) && $plc->{recv_sts} == 0) {
  if ($PLC::quiet) { print "\n" }
  else { print "\n\nDone.\n" }
 }
 return LinuxRules;
} # end sub printreturndata

sub extract_recv_data {
# convert received packet to df1 hashref values
# with all df1 fields including data_array
# automatically allowing for various df1 functions
# returns array data_array
# returns an array of data values (or undef if there is an error)
 print "\n--- sub PLC::extract_recv_data\n"
  if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $hashref = shift;
 unless ($hashref) {
  $PLC::errorcode = 15; # No hash ref in subroutine call
  return undef;
 }

 my ($floatdata,$signbit,$exponent,$mantissa,$roundfactor,$value,$datapkt);
 my @sign = (1, -1); # make number positive/negative with sign bit (0/1)
 $hashref->{send_datatype} ||= '';
 my $is_string = ( ($hashref->{send_datatype} eq 'ST')
   && (!defined $hashref->{send_subelement}) ) ? 1 : 0; # ST data type

 unless ( $hashref->{recv_raw_pkt} ) {
  qprinterr "\n\n++ No data received ++\n\n";
  $PLC::errorcode = 7; # Read Timeout
  return undef;
 }

 if ($hashref->{Protocol} =~ /^Eth/i) { # Ethernet PLC
  print "\nProtocol=EthernetPLC - Extracting data\n" if $PLC::debug;
  if (length($hashref->{recv_raw_pkt}) < 36) {
   qprinterr "\n\n++ Received packet is too short ++\n\n";
   PLC::PrintBytes($hashref->{recv_raw_pkt});
   $PLC::errorcode = 7; # Read Timeout
   return undef;
  }
  ($hashref->{recv_mode},$hashref->{recv_submode},
   $hashref->{recv_pccc_length},
   $hashref->{recv_connhandle},$hashref->{recv_status} )
   = unpack "CCnNN", $hashref->{recv_raw_pkt};
  $hashref->{recv_df1_pkt} = substr($hashref->{recv_raw_pkt}, 28);
 ($hashref->{recv_dst},$hashref->{recv_control},$hashref->{recv_src},
  $hashref->{recv_lsap},$hashref->{recv_cmd},$hashref->{recv_sts},
  $hashref->{recv_tns} )
  = unpack("CCCCCCS", $hashref->{recv_df1_pkt}); # eth_df1 only
  @{$hashref->{recv_byte_array}}
   = unpack "C*", substr($hashref->{recv_df1_pkt},8); # bytes after tns
  @{$hashref->{recv_word_array}}
   = unpack "S*", substr($hashref->{recv_df1_pkt},8); # words after tns
 }

 elsif ($hashref->{Protocol} =~ /^CIP_encap/i) { # CL Gateway or NET_ENI
  my $MinPktLen;
  if (PLC::istrue($PLCproc{$PLCname}->{NET_ENI})) { $MinPktLen = 42}
  else {$MinPktLen = 58}
  print "\nProtocol=CIP_encapsulated - Extracting data\n" if $PLC::debug;
  $hashref->{recv_raw_pkt} ||= '';
  if (length($hashref->{recv_raw_pkt}) < $MinPktLen) {
   qprinterr "\n\n++ Received packet is too short ++\n\n";
   PLC::PrintBytes($hashref->{recv_raw_pkt});
   $PLC::errorcode = 7; # Read Timeout
   return undef;
  }

  if (PLC::istrue($PLCproc{$PLCname}->{NET_ENI})) {
   $hashref->{recv_df1_pkt} = (pack "CC", (0,0))
    . substr($hashref->{recv_raw_pkt}, 41); ##ZZZZZ
  }
  else {
   $hashref->{recv_df1_pkt} = substr($hashref->{recv_raw_pkt}, 52); ##ZZZZZ
  }

 ($hashref->{recv_dst},$hashref->{recv_src},
  $hashref->{recv_cmd},$hashref->{recv_sts},
  $hashref->{recv_tns} )
  = unpack("CCCCS", $hashref->{recv_df1_pkt});
  @{$hashref->{recv_byte_array}}
   = unpack "C*", substr($hashref->{recv_df1_pkt},6); # bytes after tns
  @{$hashref->{recv_word_array}}
   = unpack "S*", substr($hashref->{recv_df1_pkt},6); # words after tns
 }

 else { # RS232
  print "\nProtocol=RS232 - Extracting data\n" if $PLC::debug;
  if ( !$hashref->{recv_df1_pkt}
  || (length($hashref->{recv_df1_pkt}) < 6) ) {
   qprinterr "\n\n++ Received packet is too short ++\n\n";
   PLC::PrintBytes($hashref->{recv_df1_pkt});
   $PLC::errorcode = 7; # Read Timeout
   return undef;
  }
  if (length $hashref->{recv_df1_pkt} >= 6) {
  ($hashref->{recv_dst},$hashref->{recv_src},
   $hashref->{recv_cmd},$hashref->{recv_sts},
   $hashref->{recv_tns} )
   = unpack("CCCCS", $hashref->{recv_df1_pkt});
  }
  if (length $hashref->{recv_df1_pkt} > 6) {
   @{$hashref->{recv_byte_array}}
    = unpack "C*", substr($hashref->{recv_df1_pkt},6); # bytes after tns
   @{$hashref->{recv_data_array}} = @{$hashref->{recv_byte_array}}; # default
   @{$hashref->{recv_word_array}}
    = unpack "S*", substr($hashref->{recv_df1_pkt},6); # words after tns
  }
 }

 if ( $hashref->{recv_cmd} & 0x40 ) { # reply packet
  if ($hashref->{recv_sts}) {
   if ($hashref->{recv_sts} == 0xF0) {
    $hashref->{recv_extsts} = $hashref->{recv_byte_array}[0];
   }
   qprinterr "++ ERROR - ", ABerrormessage($hashref)," ++\n";
   $PLC::errorcode = 6; # PLC returned error code
   return undef;
  }
 }
 else { # command packet ######

 }

# if fnc 01,
#  use datasize, etc to populate {data_array} arrayref
# analyse received (command 0F function 01) reply packet
 if (CmdFnc($hashref) eq "F-01") { # (PLC only)
# F-01 word range read
  print "\nAnalysing received data from word range read ...\n" if $PLC::debug;
  @{$hashref->{recv_data_array}} = ();
  while (@{$hashref->{recv_word_array}}) {
   if ( $hashref->{send_datatype} eq 'F' ) { # Floating point value
    $floatdata = (65536 * shift (@{$hashref->{recv_word_array}}))
     + (shift (@{$hashref->{recv_word_array}}));
    $signbit = ($floatdata >> 31) & 1;
    $exponent = (($floatdata >> 23) & 0xFF) - 127;
    $mantissa = 1 + (($floatdata & 0x7FFFFF) / 0x800000);
    $value = $mantissa * ( 2 ** $exponent);
    if ( ($value < (2 ** -126)) || ($value > 99999) ) { $roundfactor = 1}
    else { $roundfactor = 10 ** ( 5 - int( (log $value)/(log 10) )) }
    $value = (int(($value * $roundfactor) + 0.5) / $roundfactor)
     * $sign[$signbit]; # round to 6 significant digits
   }
   else {
    $value = shift (@{$hashref->{recv_word_array}});
    if ( $value > 32767 ) {$value -= 65536}
   }
   push @{$hashref->{recv_data_array}}, $value;
  } # end while @{$hashref->{recv_word_array}}
 } # end if word range read (F-01)


# if fnc A2,
#  use datasize, etc to populate {data_array} arrayref
# analyse received (command 0F function A2) SLC typed log. read reply packet
 elsif (CmdFnc($hashref) eq "F-A2") { # (SLC only)
# F-A2 SLC typed logical read
  print "\nAnalysing data from SLC typed logical read ...\n" if $PLC::debug;

  if ( ($hashref->{send_datatype} eq 'ST')	   # Need to shuffle bytes ...
  &&   (!defined $hashref->{send_subelement}) )  { # for SLC strings
   my ($len,@str) = @{$hashref->{recv_word_array}};
   my $str = substr((pack "n*",@str), 0, $len);
   @{$hashref->{recv_data_array}} = unpack("C*", $str);
  }
  elsif ($hashref->{send_datatype} eq 'A') {       # Need to shuffle bytes ...
   my $str = (pack "C*",@{$hashref->{recv_byte_array}}); # for SLC ASCII data
   if (length($str) % 2 == 1) { $str .= chr(0) } # need even # of bytes
   @{$hashref->{recv_word_array}} = unpack("S*",$str);
   $str = (pack "n*",@{$hashref->{recv_word_array}}); # for SLC ASCII data
   @{$hashref->{recv_data_array}} = unpack("C*", $str);
  }
  elsif ($hashref->{send_datasize} == 1) { # byte data
   @{$hashref->{recv_data_array}} =  @{$hashref->{recv_byte_array}};
  }
  else { # word or floating point data
   @{$hashref->{recv_data_array}} = ();
   while (@{$hashref->{recv_word_array}}) {
    if ( $hashref->{send_datatype} eq 'F' ) { # Floating point value
#    $floatdata = (65536 * shift (@{$hashref->{recv_word_array}}))
#     + (shift (@{$hashref->{recv_word_array}}));
     $floatdata = (shift (@{$hashref->{recv_word_array}}))
      + (65536 * shift (@{$hashref->{recv_word_array}}));
     $signbit = ($floatdata >> 31) & 1;
     $exponent = (($floatdata >> 23) & 0xFF) - 127;
     $mantissa = 1 + (($floatdata & 0x7FFFFF) / 0x800000);
     $value = $mantissa * ( 2 ** $exponent);
     if ( ($value < (2 ** -126)) || ($value > 99999) ) { $roundfactor = 1}
     else { $roundfactor = 10 ** ( 5 - int( (log $value)/(log 10) )) }
     $value = (int(($value * $roundfactor) + 0.5) / $roundfactor)
      * $sign[$signbit]; # round to 6 significant digits
    }
    else {
     $value = shift (@{$hashref->{recv_word_array}});
     if ( $value > 32767 ) {$value -= 65536}
    }
    push @{$hashref->{recv_data_array}}, $value;
   } # end while @{$hashref->{recv_word_array}}
  }

 } # end if word range read (F-A2)


# analyse received (command 0F function 68) reply packet
 elsif ( CmdFnc($hashref) eq "F-68" ) {
# F-68 typed data read

  print "\nAnalysing received data from typed data read ...\n" if $PLC::debug;

  @{$hashref->{recv_data_array}} = ();
  $hashref->{recv_typedata1} = shift @{$hashref->{recv_byte_array}};
  printf "typedata1 = 0x%02X\n",$hashref->{recv_typedata1} if $PLC::debug;
  if ($hashref->{recv_typedata1} & 0x80) {
   $hashref->{recv_IDbytes} = ($hashref->{recv_typedata1} >> 4) & 0x07;
   if ($hashref->{recv_IDbytes} != 1) {
    qprinterr("++ ERROR - IDbytes = $hashref->{recv_IDbytes}",
     " (unsupported) ++\n");
    $PLC::errorcode = 37; # Received data in unknown format
    return undef;
   }
   $hashref->{recv_IDvalue1} = shift @{$hashref->{recv_byte_array}};
  }
  else {
   $hashref->{recv_IDvalue1} = ($hashref->{recv_typedata1} >> 4) & 0x07;
  }
  printf "hashref->{recv_IDvalue1} = 0x%02X\n",$hashref->{recv_IDvalue1}
   if $PLC::debug;

  if ($hashref->{recv_typedata1} & 0x08) {
   $hashref->{recv_sizebytes} = $hashref->{recv_typedata1} & 0x07;
   if ($hashref->{recv_sizebytes} > 2) {
    qprinterr "++ ERROR - hashref->{recv_sizebytes} = ";
    qprinterr "$hashref->{recv_sizebytes} (unsupported) ++\n";
    $PLC::errorcode = 37; # Received data in unknown format
    return undef;
   }
   $hashref->{recv_sizevalue1} = 0;
   for (0 .. ($hashref->{recv_sizebytes} - 1)) {
    $hashref->{recv_sizevalue1}
     += (shift @{$hashref->{recv_byte_array}}) * (256 ** $_);
   }
  }
  else {
   $hashref->{recv_sizevalue1} = $hashref->{recv_typedata1} & 0x07;
  }
  printf "sizevalue1 = 0x%02X\n",$hashref->{recv_sizevalue1}
   if $PLC::debug;

  if ($hashref->{recv_IDvalue1} == 9) {
# type 9 = array, so there is a descriptor byte
   $hashref->{recv_typedata2} = shift @{$hashref->{recv_byte_array}};
   printf "typedata2 = 0x%02X\n",$hashref->{recv_typedata2} if $PLC::debug;
   if ($hashref->{recv_typedata2} & 0x80) {
    $hashref->{recv_IDbytes} = ($hashref->{recv_typedata2} >> 4) & 0x07;
    if ($hashref->{recv_IDbytes} != 1) {
     qprinterr "++ ERROR - IDbytes = $hashref->{recv_IDbytes}";
     qprinterr " (unsupported) ++\n";
     $PLC::errorcode = 37; # Received data in unknown format
     return undef;
    }
    $hashref->{recv_IDvalue2} = shift @{$hashref->{recv_byte_array}};
   }
   else {
    $hashref->{recv_IDvalue2}
     = ($hashref->{recv_typedata2} >> 4) & 0x07;
   }
   printf "IDvalue2 = 0x%02X\n",$hashref->{recv_IDvalue2} if $PLC::debug;
 
   if ($hashref->{recv_typedata2} & 0x08) {
    $hashref->{recv_sizebytes} = $hashref->{recv_typedata2} & 0x07;
    if ($hashref->{recv_sizebytes} > 2) {
     qprinterr "++ ERROR - hashref->{recv_sizebytes} = ";
     qprinterr "$hashref->{recv_sizebytes} (unsupported) ++\n";
     $PLC::errorcode = 37; # Received data in unknown format
     return undef;
    }
    $hashref->{recv_sizevalue2} = 0;
    for (0 .. ($hashref->{recv_sizebytes} - 1)) {
     $hashref->{recv_sizevalue2}
      += (shift @{$hashref->{recv_byte_array}}) * (256 ** $_);
    }
   }
   else {
    $hashref->{recv_sizevalue2} = $hashref->{recv_typedata2} & 0x07;
   }
   printf "sizevalue2 = 0x%02X\n",$hashref->{recv_sizevalue2}
    if $PLC::debug;
   $hashref->{recv_IDvalue} = $hashref->{recv_IDvalue2};
  }
  else {$hashref->{recv_IDvalue} = $hashref->{recv_IDvalue1}}

  if ($PLC::debug) {
   print "Data type is $DataTypes[$hashref->{recv_IDvalue1}]";
   if ($hashref->{recv_IDvalue1} == 9) { # array
    print " of $DataTypes[$hashref->{recv_IDvalue2}]";
   }
   print "\nTotal bytes to read = $hashref->{recv_sizevalue1}";
   if ($hashref->{recv_IDvalue1} == 9) {
    print " (including descriptor), ";
    print "$hashref->{recv_sizevalue2} bytes/element";
   }
   print "\n";
  }

# now we use IDvalue to determine data size & extract values

  $datapkt = pack "C*", (@{$hashref->{recv_byte_array}});

  if ($PLC::debug) {
   PrintBytes($datapkt);
   print "\n";
   print "IDvalue = $hashref->{recv_IDvalue}\n";
  }

  if ( ($hashref->{send_datasize} == 1) ) { # byte data
   @{$hashref->{recv_data_array}} = @{$hashref->{recv_byte_array}};
  }
  elsif ( ( $hashref->{send_datatype} eq "PD") # PID Block Data Subelement
        && ( defined $hashref->{send_subelement})
        && ( $hashref->{send_subelement} <= 1) ) {
   @{$hashref->{recv_data_array}} = unpack("S*", ($datapkt ) );
  }
  else {
   if ( ($hashref->{recv_IDvalue} > 3)
   && ($hashref->{recv_IDvalue} != 9) ) {
    if ( ($hashref->{recv_IDvalue2} == 5)	 # for timers with ...
    &&   ($hashref->{recv_sizevalue2} == 10) ) { # 10 bytes/element, ...

     my $pattern = "SLL" x ((length $datapkt) / $hashref->{recv_sizevalue2} );
     @array = unpack ($pattern, $datapkt);	 # PRE and ACC are ...
						 # 4 bytes each, not 2
    }
    else {
     @array =  unpack "S*", $datapkt;
    }
    while (@array) {
     if ( ($hashref->{recv_IDvalue} == 8)      # Floating point value
     || ( ( $hashref->{send_datatype} eq "PD") # PID Block Data Subelement
        && ( defined $hashref->{send_subelement})
        && ( $hashref->{send_subelement} >= 2) ) ) {
###   $floatdata = (65536 * shift (@array)) + (shift (@array));
      $floatdata =  (shift (@array)) + (65536 * shift (@array));
      $signbit = ($floatdata >> 31) & 1;
      $exponent = (($floatdata >> 23) & 0xFF) - 127;
      $mantissa = 1 + (($floatdata & 0x7FFFFF) / 0x800000);
      $value = $mantissa * ( 2 ** $exponent);
      if ( ($value < (2 ** -126)) || ($value > 99999) ) { $roundfactor = 1}
      else { $roundfactor = 10 ** ( 5 - int( (log $value)/(log 10) )) }
      $value = (int(($value * $roundfactor) + 0.5) / $roundfactor)
       * $sign[$signbit]; # round to 6 significant digits
     }
     else {
      $value = shift (@array);
      if ( $value > 32767 ) {$value -= 65536}
     }
# @{$hashref->{recv_data_array}} is an array containing received data
     push @{$hashref->{recv_data_array}}, $value;
    } # end while @array
   }
  }
 } # end of F-68 typed data read


# analyse received (command 06 function 00) reply packet
 elsif (CmdFnc($hashref) eq "6-00") {
# 6-00 echo
  print "\nAnalysing received data from echo command ...\n" if $PLC::debug;
  @{$hashref->{recv_data_array}} = @{$hashref->{recv_byte_array}};
 } # end of 6-00 echo


# analyse received (command 0F function 29) reply packet
 elsif (CmdFnc($hashref) eq "F-29") {
# F-29 read section size
  print "\nAnalysing received data from read section size command ...\n"
   if $PLC::debug;
  $PLC::hex = 1;

  if (scalar @{$hashref->{recv_byte_array}} >= 9) {
   @{$hashref->{recv_data_array}} =
    (unpack "LSxxC*", (pack "C*", @{$hashref->{recv_byte_array}}) );
  }
  else {
   @{$hashref->{recv_data_array}} =
    (unpack "SSC*", (pack "C*", @{$hashref->{recv_byte_array}}) );
  }

  $hashref->{recv_datasize} = $hashref->{recv_data_array}[0];
  $hashref->{recv_elements} = $hashref->{recv_data_array}[1];

  if ($hashref->{recv_data_array}[2] == 0x90) { # type code in next byte
   $hashref->{recv_datatype} = $hashref->{recv_data_array}[3];
  }
  else {
   $hashref->{recv_datatype} = $hashref->{recv_data_array}[2] >> 4;
  }
  @{$hashref->{recv_data_array}} =
  ($hashref->{recv_datasize},$hashref->{recv_elements},
   $hashref->{recv_datatype});

 } # end of F-29 read section size

 if ( defined $hashref->{send_bit} ) { # bit data, so mask it
  for ( 0 .. ( (scalar @{$hashref->{recv_data_array}}) - 1) ) {
   $hashref->{recv_data_array}[$_]
   = ( $hashref->{recv_data_array}[$_] & $hashref->{send_bitmask} ) ? 1 : 0;
  }
 }

 if ( $PLC::debug && ($PLC::debug >= 2) ) {
  superdebugprint($hashref) if ($PLC::debug >= 2);
 }

 if (defined @{$hashref->{recv_data_array}}) {
  if ($is_string) {
   $hashref->{recv_string} = (pack 'C*', @{$hashref->{recv_data_array}});
   return $hashref->{recv_string};
###   return (pack 'C*', @{$hashref->{recv_data_array}});
  }
  else { return @{$hashref->{recv_data_array}} }
 }
### elsif ($hashref->{recv_sts} == 0) { return 'OK' } # ????
 elsif ($hashref->{recv_sts} == 0) { return '' } # ????
 else {
  $PLC::errorcode = 6; # PLC returned error code
  return undef;
 }
} # end sub extract_recv_data

sub ABerrormessage {
 print "\n--- sub PLC::ABerrormessage\n"
  if ($PLC::debug and $PLC::debug >= 5);
 my ($ABsts, $ABextsts);
 if (ref $_[0]) { # Called with hash ref
  my $hashref = shift;
  ($ABsts, $ABextsts) = ($hashref->{recv_sts},$hashref->{recv_extsts});
 }
 elsif ( scalar(@_) == 1) { # Called with PLC name
  my $PLCname = shift or return 'UNKNOWN';
  ($ABsts, $ABextsts)
  = ($PLCproc{$PLCname}->{recv_sts},$PLCproc{$PLCname}->{recv_extsts});
 }
 else { # Called with status code (STS) and extended status code (EXTSTS)
  ($ABsts, $ABextsts) = @_;
 }
 my $ABstsmsg = "";
 if ( !$ABsts )
  {$ABstsmsg = "-- No Error --"}
 elsif ( $ABsts == 0xF0 ) {
# The following EXT STS messages are valid for DH/DH+ CMD 0F and DH485(some)
  if ($ABextsts == 0x01)
   {$ABstsmsg = "A field has an illegal value"}
  elsif ($ABextsts == 0x02)
   {$ABstsmsg = "Not enough levels specified in address"}
  elsif ($ABextsts == 0x03)
   {$ABstsmsg = "Too many levels specified in address"}
  elsif ($ABextsts == 0x04)
   {$ABstsmsg = "Symbol not found"}
  elsif ($ABextsts == 0x05)
   {$ABstsmsg = "Symbol is of improper format"}
  elsif ($ABextsts == 0x06)
   {$ABstsmsg = "Address does not point to something usable"}
  elsif ($ABextsts == 0x07)
   {$ABstsmsg = "File is wrong size"}
  elsif ($ABextsts == 0x08)
   {$ABstsmsg = "Cannot comply, situation has changed"}
  elsif ($ABextsts == 0x09)
   {$ABstsmsg = "Data or file is too large"}
  elsif ($ABextsts == 0x0A)
   {$ABstsmsg = "Transaction size plus word address is too large"}
  elsif ($ABextsts == 0x0B)
   {$ABstsmsg = "Access denied, improper privilege"}
  elsif ($ABextsts == 0x0C)
   {$ABstsmsg = "Condition cannot be generated, resource is not available"}
  elsif ($ABextsts == 0x0D)
   {$ABstsmsg = "Condition already exists, resource is already available"}
  elsif ($ABextsts == 0x0E)
   {$ABstsmsg = "Command cannot be executed"}
  elsif ($ABextsts == 0x0F)
   {$ABstsmsg = "Histogram Overflow"}
  elsif ($ABextsts == 0x10)
   {$ABstsmsg = "No access"}
  elsif ($ABextsts == 0x11)
   {$ABstsmsg = "Illegal data type"}
  elsif ($ABextsts == 0x12)
   {$ABstsmsg = "Invalid parameter or invalid data"}
  elsif ($ABextsts == 0x13)
   {$ABstsmsg = "Address reference exists to deleted area"}
  elsif ($ABextsts == 0x14)
   {$ABstsmsg = "Command execution failure, reason unknown"}
  elsif ($ABextsts == 0x15)
   {$ABstsmsg = "Data conversion error"}
  elsif ($ABextsts == 0x16)
   {$ABstsmsg = "Scanner unable to communicate with 1771 rack adapter"}
  elsif ($ABextsts == 0x17)
   {$ABstsmsg = "Type mismatch"}
  elsif ($ABextsts == 0x18)
   {$ABstsmsg = "1771 module response was not valid"}
  elsif ($ABextsts == 0x19)
   {$ABstsmsg = "Duplicated label"}
  elsif ($ABextsts == 0x1A)
   {$ABstsmsg = "File is open, another node owns it"}
  elsif ($ABextsts == 0x1B)
   {$ABstsmsg = "Another node is the program owner"}
  elsif ($ABextsts == 0x1E)
   {$ABstsmsg = "Data table element protection violation"}
  elsif ($ABextsts == 0x1F)
   {$ABstsmsg = "Temporary internal problem"}
  elsif ($ABextsts == 0x22)
   {$ABstsmsg = "Remote rack fault"}
  elsif ($ABextsts == 0x23)
   {$ABstsmsg = "Timeout"}
  elsif ($ABextsts == 0x24)
   {$ABstsmsg = "Unknown error"}
  else {
   $ABstsmsg = sprintf("Error code 0x%02X, Extended error code 0x%02X",
    $ABsts,$ABextsts);
  }
  printf ("-- ERROR EXTSTS 0x%02X -- $ABstsmsg\n",$ABextsts) if $PLC::debug;
 }
 elsif ($ABsts == 0x01)
  {$ABstsmsg = "Destination node is out of buffer space"}
 elsif ($ABsts == 0x02)
  {$ABstsmsg = "Cannot guarantee delivery - remote node did not acknowledge"}
 elsif ($ABsts == 0x03)
  {$ABstsmsg = "Duplicate token holder detected"}
 elsif ($ABsts == 0x04)
  {$ABstsmsg = "Local port is disconnected"}
 elsif ($ABsts == 0x05)
  {$ABstsmsg = "Application layer timed out waiting for a response"}
 elsif ($ABsts == 0x06)
  {$ABstsmsg = "Duplicate node detected"}
 elsif ($ABsts == 0x07)
  {$ABstsmsg = "Station is offline"}
 elsif ($ABsts == 0x08)
  {$ABstsmsg = "Hardware fault"}
 elsif ($ABsts == 0x10)
  {$ABstsmsg = "Illegal Command or Format"}
 elsif ($ABsts == 0x20)
  {$ABstsmsg = "Host will not communicate"}
 elsif ($ABsts == 0x30)
  {$ABstsmsg = "Host missing, disconnected, or shut down"}
 elsif ($ABsts == 0x40)
  {$ABstsmsg = "Host has a hardware fault"}
 elsif ($ABsts == 0x50)
  {$ABstsmsg = "Addressing problem or Memory Protect rungs"}
 elsif ($ABsts == 0x60)
  {$ABstsmsg = "Function not allowed due to protection"}
 elsif ($ABsts == 0x70)
  {$ABstsmsg = "Processor is in Program mode"}
 elsif ($ABsts == 0x80)
  {$ABstsmsg = "Compatibility Mode file missing"}
 elsif ($ABsts == 0x90)
  {$ABstsmsg = "Remote Node cannot buffer command"}
 elsif ($ABsts == 0xA0)
  {$ABstsmsg = "Wait ACK (1775-KA buffer full)"}
 elsif ($ABsts == 0xB0)
  {$ABstsmsg = "Remote node problem due to download"}
 elsif ($ABsts == 0xC0)
  {$ABstsmsg = "Wait ACK (1775-KA buffer full)"}
 elsif ($ABsts != 0) {$ABstsmsg = "Unknown Error"}
 return $ABstsmsg;
} # end sub ABerrormessage

# End of module AB

#-------------------------------------------------------------------------#
package ABeth1;

use constant LinuxRules => 1; # always true

# datasize is in bytes (not words)

# f-00 & f-01 totaltrans=words, size=bytes (totaltrans * 2)
# f-67 & f-68 totaltrans=elements, size=elements (same as totaltrans)

use strict;
use IO::Socket;
use POSIX;
STDOUT->autoflush(1);

use vars qw($junk);

sub qprint {
 print @_ unless $PLC::quiet;
}

sub qprinterr {
 print STDERR @_ unless $PLC::quiet && $PLC::quiet >= 2;
}


####################################
sub YesNo {
 print "\n--- sub ABeth1::YesNo\n" if ($PLC::debug and $PLC::debug >= 5);
 my ($bool) = @_;
 if    (lc $bool eq "no")  {return "No"}
 elsif (lc $bool eq "yes") {return "Yes"}
 elsif ($bool == 0) {return "No"} else {return "Yes"}
}

####################################
sub getstatusdata {
 print "\n--- sub ABeth1::getstatusdata\n"
  if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $hashref = shift;
 unless ($hashref) {
  $PLC::errorcode = 15; # No hash ref in subroutine call
  return undef;
 }

 $hashref->{recv_RECVDATApkt} = pack "C*", @{$hashref->{recv_byte_array}};

 unless ( $hashref->{recv_RECVDATApkt} ) {
  qprinterr "\n\n++ No data received ++\n\n";
  $PLC::errorcode = 7; # Read Timeout
  return undef;
 }

 if ($PLC::debug) {
  print "\nDiag. Status ";
  PLC::PrintBytes($hashref->{recv_RECVDATApkt});
  print "\n";
 }

# detect PLC type automatically and print details
 ($hashref->{recv_XModeStatus},$hashref->{recv_XTypeExtender},
  $hashref->{recv_XExtendedInterfaceType},
  $hashref->{recv_XExtendedProcessorType},
  $hashref->{recv_XSeriesRevision},
  $hashref->{recv_X1},$hashref->{recv_X2},
  $hashref->{recv_X3},$hashref->{recv_X4},
  $hashref->{recv_X5},$hashref->{recv_X6},
  $hashref->{recv_X7},$hashref->{recv_X8},
  $hashref->{recv_X9},$hashref->{recv_X0},
  $hashref->{recv_X},$hashref->{recv_XMajorErrorWord},
  $hashref->{recv_XProcessorModeWord},$hashref->{recv_XProgramID},
  $hashref->{recv_XRAMSize},$hashref->{recv_XProgramOwnerNodeAddress})
   = unpack "CCCCCCCCCCCCCCCCSSSCC", $hashref->{recv_RECVDATApkt};

 if ( ( ($hashref->{recv_XTypeExtender} == 0xEE)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x1F) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEE)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x20) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEE)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x31) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEE)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x34) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEE)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x4A) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEE)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x36) ) )
   {$hashref->{recv_SLC_MicroLogix} = "Yes"}
 else {$hashref->{recv_SLC_MicroLogix} = "No"}

 if ( ($hashref->{recv_XTypeExtender} == 0xF3)
   && ($hashref->{recv_XExtendedInterfaceType} == 0x00) )
   {$hashref->{recv_KF2} = "Yes"}
 else {$hashref->{recv_KF2} = "No"}

 if ( ($hashref->{recv_XTypeExtender} == 0xFE)
   && ($hashref->{recv_XExtendedInterfaceType} == 0x2C) )
   {$hashref->{recv_KF3} = "Yes"}
 else {$hashref->{recv_KF3} = "No"}

 if ( ($hashref->{recv_XTypeExtender} == 0xEE)
   && ($hashref->{recv_XExtendedInterfaceType} == 0x3D) )
   {$hashref->{recv_DHRIO} = "Yes"}
 else {$hashref->{recv_DHRIO} = "No"}

 if ( ( ($hashref->{recv_XTypeExtender} == 0xEB)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x13) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEB)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x14) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEB)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x15) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEB)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x22) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEB)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x23) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEB)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x28) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEB)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x29) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEB)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x31) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEB)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x32) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEB)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x33) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEB)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x4A) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEB)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x4B) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEB)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x55) )
  ||  ( ($hashref->{recv_XTypeExtender} == 0xEB)
     && ($hashref->{recv_XExtendedInterfaceType} == 0x59) ) )
   {$hashref->{recv_PLC5} = "Yes"; $hashref->{recv_PLC5_250} = "No"}
 elsif ( ($hashref->{recv_XTypeExtender} == 0xDE)
    ||   ($hashref->{recv_XTypeExtender} == 0x0E)
    ||   ($hashref->{recv_XTypeExtender} == 0xD0) )
   {$hashref->{recv_PLC5} = "Yes"; $hashref->{recv_PLC5_250} = "Yes"}
 else {$hashref->{recv_PLC5} = "No"; $hashref->{recv_PLC5_250} = "No"}

 if ( ($hashref->{recv_SLC_MicroLogix} eq "No") &&
      ($hashref->{recv_PLC5} eq "No") ) {
  printf(
   "Type Extender: 0x%02X, InterfaceType: 0x%02X, Processor Type 0x%02X\n\n",
   $hashref->{recv_XTypeExtender},$hashref->{recv_XExtendedInterfaceType},
   $hashref->{recv_XExtendedProcessorType});
 }

 if ($hashref->{recv_SLC_MicroLogix} eq "Yes") {
  if (length($hashref->{recv_RECVDATApkt}) < 16) {
   qprinterr "\n\n++ Received packet is too short ++\n\n";
   PLC::PrintBytes($hashref->{recv_RECVDATApkt});
   $PLC::errorcode = 7; # Read Timeout
   return undef;
  }
  $hashref->{recv_BulletinName} = substr($hashref->{recv_RECVDATApkt},5,11);
  $hashref->{recv_XDirectoryFileCorrupted}
   = $hashref->{recv_XProgramOwnerNodeAddress} & 0x01;
  $hashref->{recv_XProgramOwnerNodeAddress}
   = $hashref->{recv_XProgramOwnerNodeAddress} >> 2;

  $hashref->{recv_TestingEdits}
   = ($hashref->{recv_XModeStatus} & 0x40) >> 6;
  $hashref->{recv_EditsInProcessor}
   = ($hashref->{recv_XModeStatus} & 0x80) >> 7;
  if ($hashref->{recv_XExtendedInterfaceType} == 0x20)
   {$hashref->{recv_InterfaceType} = "DH485 on Port 0 (RS-232)"}
  elsif ($hashref->{recv_XExtendedInterfaceType} == 0x31)
   {$hashref->{recv_InterfaceType} = "DH+ on Port 1 via Domino Plug";}
  elsif ($hashref->{recv_XExtendedInterfaceType} == 0x34)
   {$hashref->{recv_InterfaceType} = "DF1 full-duplex protocol on Port 0"}
  elsif ($hashref->{recv_XExtendedInterfaceType} == 0x36)
   {$hashref->{recv_InterfaceType} = "DF1 half-duplex protocol on Port 0"}
  elsif ($hashref->{recv_XExtendedInterfaceType} == 0x4A)
   {$hashref->{recv_InterfaceType} = "Ethernet (type 0x4A)"}
  else {$hashref->{recv_InterfaceType}
        = sprintf("0x%02X",$hashref->{recv_XExtendedInterfaceType})}
  if ($hashref->{recv_XExtendedProcessorType} == 0x1A)
   {$hashref->{recv_ProcessorType}
    = "20, 30, and 40 I/O SLC Fixed Controllers";}
  elsif ($hashref->{recv_XExtendedProcessorType} == 0x18)
   {$hashref->{recv_ProcessorType}
    = "1747-L511, -L514 SLC 5/01 Controller";}
  elsif ($hashref->{recv_XExtendedProcessorType} == 0x25)
   {$hashref->{recv_ProcessorType}
    = "1747-L524 and SLC 5/02 Controller";}
  elsif (($hashref->{recv_XExtendedProcessorType} == 0x49) 
      || ($hashref->{recv_XExtendedProcessorType} == 0x5B))
   {$hashref->{recv_ProcessorType} = "SLC-5/03, 5/04 rack type 1747-L534";}
  elsif ($hashref->{recv_XExtendedProcessorType} == 0x77)
   {$hashref->{recv_ProcessorType}
    = "1747-L551 SLC 5/05 Controller";}
  elsif ($hashref->{recv_XExtendedProcessorType} == 0x78)
   {$hashref->{recv_ProcessorType}
    = "1747-L552 SLC 5/05 Controller";}
  elsif ($hashref->{recv_XExtendedProcessorType} == 0x79)
   {$hashref->{recv_ProcessorType}
    = "1747-L553 SLC 5/05 Controller";}
  else {$hashref->{recv_ProcessorType}
        = sprintf("0x%02X",$hashref->{recv_XExtendedProcessorType})}
  $hashref->{recv_FirmwareRelease}
   = $hashref->{recv_XSeriesRevision} & 0x1F;
  $hashref->{recv_XSeries} = $hashref->{recv_XSeriesRevision} >> 5;
  if ( $hashref->{recv_BulletinName} =~ /1761 Micro/i) { # MicroLogix
   $hashref->{recv_FirmwareRelease}++;
   $hashref->{recv_XSeries}++;
  }
  ($hashref->{recv_Series}) = pack "C", ($hashref->{recv_XSeries}+0x41);
  $hashref->{recv_XProcessorMode}
   = $hashref->{recv_XProcessorModeWord} & 0x1F;
  if ($hashref->{recv_XProcessorMode} == 0)
   {$hashref->{recv_ProcessorMode} = "Download"}
  elsif ($hashref->{recv_XProcessorMode} == 1)
   {$hashref->{recv_ProcessorMode} = "Remote PROG"}
  elsif ($hashref->{recv_XProcessorMode} == 3)
   {$hashref->{recv_ProcessorMode} = "Idle due to SUS Instruction"}
  elsif ($hashref->{recv_XProcessorMode} == 6)
   {$hashref->{recv_ProcessorMode} = "Remote RUN"}
  elsif ($hashref->{recv_XProcessorMode} == 7)
   {$hashref->{recv_ProcessorMode} = "Remote TEST; continuous scan"}
  elsif ($hashref->{recv_XProcessorMode} == 8)
   {$hashref->{recv_ProcessorMode} = "Remote TEST; single scan"}
  elsif ($hashref->{recv_XProcessorMode} == 9)
   {$hashref->{recv_ProcessorMode} = "Remote TEST; continuous step"}
  elsif ($hashref->{recv_XProcessorMode} == 16)
   {$hashref->{recv_ProcessorMode} = "Download"}
  elsif ($hashref->{recv_XProcessorMode} == 17)
   {$hashref->{recv_ProcessorMode} = "PROG"}
  elsif ($hashref->{recv_XProcessorMode} == 27)
   {$hashref->{recv_ProcessorMode} = "Idle due to SUS Instruction"}
  elsif ($hashref->{recv_XProcessorMode} == 30)
   {$hashref->{recv_ProcessorMode} = "RUN"}
  else {$hashref->{recv_ProcessorMode} = "--UNKNOWN--"}
  $hashref->{recv_ForcesActive}
   = YesNo($hashref->{recv_XProcessorModeWord} & 0x0020);
  $hashref->{recv_ForcesInstalled}
   = YesNo($hashref->{recv_XProcessorModeWord} & 0x0040);
  $hashref->{recv_CommActive}
   = YesNo($hashref->{recv_XProcessorModeWord} & 0x0080);
  $hashref->{recv_ProtPowerLoss}
   = YesNo($hashref->{recv_XProcessorModeWord} & 0x0100);
  $hashref->{recv_StartupProtectFault}
   = YesNo($hashref->{recv_XProcessorModeWord} & 0x0200);
  $hashref->{recv_LoadMemModuleOnError}
   = YesNo($hashref->{recv_XProcessorModeWord} & 0x0400);
  $hashref->{recv_LoadMemModuleAlways}
   = YesNo($hashref->{recv_XProcessorModeWord} & 0x0800);
  $hashref->{recv_LoadMemModuleAndRun}
   = YesNo($hashref->{recv_XProcessorModeWord} & 0x1000);
  $hashref->{recv_MajorErrorHalted}
   = YesNo($hashref->{recv_XProcessorModeWord} & 0x2000);
  $hashref->{recv_Locked}
   = YesNo($hashref->{recv_XProcessorModeWord} & 0x4000);
  $hashref->{recv_FirstPassBit}
   = YesNo($hashref->{recv_XProcessorModeWord} & 0x8000);

   print "BulletinName = $hashref->{recv_BulletinName}\n";
   printf("Testing Edits: %s\n",YesNo($hashref->{recv_TestingEdits}));
   printf("Edits in Processor: %s\n",YesNo($hashref->{recv_TestingEdits}));
   printf("Type Extender: 0x%02X\n",$hashref->{recv_XTypeExtender});
   print "Extended Interface Type: $hashref->{recv_InterfaceType}\n";
   print "Extended Processor Type: $hashref->{recv_ProcessorType}\n";
   print "Series/Revision ";
   print "$hashref->{recv_Series}-$hashref->{recv_FirmwareRelease}\n";
   printf("Major Error Word: 0x%04X\n",$hashref->{recv_XMajorErrorWord});
   print "Processor Mode: $hashref->{recv_ProcessorMode}\n";
   if ($hashref->{recv_XRAMSize}) {
    printf("RAM Size: %d KBytes\n", $hashref->{recv_XRAMSize});
   }
   printf("Program ID: 0x%04X\n",$hashref->{recv_XProgramID});
   if ($hashref->{recv_XDirectoryFileCorrupted} != 0)
    {print "Directory File Corrupted\n"}

   print "Forces Active\n" if ($hashref->{recv_ForcesActive} eq "Yes");
   print "Forces Installed\n"
    if ($hashref->{recv_ForcesInstalled} eq "Yes");
   print "DH+ (or Ethernet) Communication Active\n"
    if ($hashref->{recv_CommActive} eq "Yes");
   print "Protection Power Loss Active\n"
    if ($hashref->{recv_ProtPowerLoss} eq "Yes");
   print "Startup Protect Fault Active\n"
    if ($hashref->{recv_StartupProtectFault} eq "Yes");
   print "Load Memory Module On Error\n"
    if ($hashref->{recv_LoadMemModuleOnError} eq "Yes");
   print "Load Memory Module Always\n"
    if ($hashref->{recv_LoadMemModuleAlways} eq "Yes");
   print "Load Memory Module And Run\n"
    if ($hashref->{recv_LoadMemModuleAndRun} eq "Yes");
   print "Major Error - Processor Halted\n"
    if ($hashref->{recv_MajorErrorHalted} eq "Yes");
   print "Locked\n" if ($hashref->{recv_Locked} eq "Yes");
   print "First Pass Bit\n" if ($hashref->{recv_FirstPassBit} eq "Yes");

   if ($hashref->{recv_XProgramOwnerNodeAddress} != 0x3F) {
    printf("Program Owner Node Address: 0x%02X\n",
     $hashref->{recv_XProgramOwnerNodeAddress});
   }

   $hashref->{'PLCtype'} ||= 'SLC';
   $hashref->{recv_ProgName} = PLC::read_program_name($hashref->{PLCname});
   unless (defined $hashref->{recv_ProgName}) {
    $hashref->{recv_ProgName} = 'UNKNOWN';
   }
   print "Program Name: $hashref->{recv_ProgName}\n";
  } # End SLC/MicroLogix}-only stuff

  elsif ( ($hashref->{recv_PLC5} eq "Yes")
       && ($hashref->{recv_PLC5_250} ne "Yes") ) {
  ($hashref->{recv_XModeStatus},
   $hashref->{recv_XTypeExtender},
   $hashref->{recv_XPLC5Type},
   $hashref->{recv_XRAMSize},
   $hashref->{recv_XSeriesRevision},
   $hashref->{recv_XPLC5DHPlusNode},
   $hashref->{recv_XPLC5All_IOAddr},
   $hashref->{recv_XPLC5IOComm},
   $hashref->{recv_XPLC5LastProgramFile},
   $hashref->{recv_XPLC5LastDataFile},
   $hashref->{recv_XPLC5ForcesX},
   $hashref->{recv_XPLC5MemProt}, $hashref->{recv_XPLC5BadRAM},
   $hashref->{recv_XPLC5DebugMode},
   $hashref->{recv_XPLC5HoldPointFile},
   $hashref->{recv_XPLC5HoldPointElement},
   $hashref->{recv_XPLC5EditTimeStampYear},
   $hashref->{recv_XPLC5EditTimeStampMonth},
   $hashref->{recv_XPLC5EditTimeStampDay},
   $hashref->{recv_XPLC5EditTimeStampHour},
   $hashref->{recv_XPLC5EditTimeStampMin},
   $hashref->{recv_XPLC5EditTimeStampSecs},
   $hashref->{recv_XPLC5PortNumber})
   = unpack "CCCLCCCCSSCCCCSSSSSSSSC", $hashref->{recv_RECVDATApkt};

   $hashref->{recv_XPLC5LastProgramFile}--;
   $hashref->{recv_XPLC5LastDataFile}--;

# If $hashref->{recv_PLC1785LT} eq "Yes", there is no data after
# $hashref->{recv_XPLC5BadRAM}
   $hashref->{recv_PLC1785LT}
    = YesNo($hashref->{recv_XPLC5Type} < 0x15); # Old PLC5's

   $hashref->{recv_XProcessorMode} = $hashref->{recv_XModeStatus} & 0x07;

   if ($hashref->{recv_XProcessorMode} == 0)
    {$hashref->{recv_ProcessorMode} = "PROGRAM Mode"}
   elsif ($hashref->{recv_XProcessorMode} == 2)
    {$hashref->{recv_ProcessorMode} = "RUN Mode"}
   elsif ($hashref->{recv_XProcessorMode} == 4)
    {$hashref->{recv_ProcessorMode} = "Remote PROGRAM Mode"}
   elsif ($hashref->{recv_XProcessorMode} == 5)
    {$hashref->{recv_ProcessorMode} = "Remote TEST Mode"}
   elsif ($hashref->{recv_XProcessorMode} == 6)
    {$hashref->{recv_ProcessorMode} = "Remote RUN Mode"}
   else {$hashref->{recv_ProcessorMode} = "--UNKNOWN--"}

   $hashref->{recv_TestingEdits}
    = YesNo($hashref->{recv_XModeStatus} & 0xC0);
   $hashref->{recv_MajorErrorHalted}
    = YesNo($hashref->{recv_XModeStatus} & 0x08);
   $hashref->{recv_PLC5DownloadMode}
    = YesNo($hashref->{recv_XModeStatus} & 0x10);

   if ($hashref->{recv_XPLC5Type} == 0x13)
    {$hashref->{recv_ProcessorType} = "1785-LT3 (PLC-5/12)";}
   elsif ($hashref->{recv_XPLC5Type} == 0x14)
    {$hashref->{recv_ProcessorType} = "1785-LT2 (PLC-5/25)";}
   elsif ($hashref->{recv_XPLC5Type} == 0x15)
    {$hashref->{recv_ProcessorType} = "1785-L40B (PLC-5/40)";}
   elsif ($hashref->{recv_XPLC5Type} == 0x22)
    {$hashref->{recv_ProcessorType} = "1785-LT4 (PLC-5/10)";}
   elsif ($hashref->{recv_XPLC5Type} == 0x23)
    {$hashref->{recv_ProcessorType} = "1785-L60B (PLC-5/60)";}
   elsif ($hashref->{recv_XPLC5Type} == 0x28)
    {$hashref->{recv_ProcessorType} = "1785-L40L (PLC-5/40)";}
   elsif ($hashref->{recv_XPLC5Type} == 0x29)
    {$hashref->{recv_ProcessorType} = "1785-L60L (PLC-5/60)";}
   elsif ($hashref->{recv_XPLC5Type} == 0x31)
    {$hashref->{recv_ProcessorType} = "1785-L11B (PLC-5/11)";}
   elsif ($hashref->{recv_XPLC5Type} == 0x32)
    {$hashref->{recv_ProcessorType} = "1785-L20B (PLC-5/20)";}
   elsif ($hashref->{recv_XPLC5Type} == 0x33)
    {$hashref->{recv_ProcessorType} = "1785-L30B (PLC-5/30)";}
   elsif ($hashref->{recv_XPLC5Type} == 0x4A)
    {$hashref->{recv_ProcessorType} = "1785-L20E (PLC-5/20E)";}
   elsif ($hashref->{recv_XPLC5Type} == 0x4B)
    {$hashref->{recv_ProcessorType} = "1785-L40E (PLC-5/40E)";}
   elsif ($hashref->{recv_XPLC5Type} == 0x55)
    {$hashref->{recv_ProcessorType} = "1785-L80B (PLC-5/80)";}
   elsif ($hashref->{recv_XPLC5Type} == 0x59)
    {$hashref->{recv_ProcessorType} = "1785-L80E (PLC-5/80E)";}
   else {$hashref->{recv_ProcessorType} = "--UNKNOWN--";}

   ($hashref->{recv_FirmwareRelease})
    = pack "C", (($hashref->{recv_XSeriesRevision} & 0x1F)+0x41);
   $hashref->{recv_XSeries} = $hashref->{recv_XSeriesRevision} >> 5;
   ($hashref->{recv_Series})
    = pack "C", ($hashref->{recv_XSeries}+0x41);

   print "Processor Type: $hashref->{recv_ProcessorType}\n";
   print "Series $hashref->{recv_Series},";
   print " Firmware Revision $hashref->{recv_FirmwareRelease}\n";
   print "Processor Mode: $hashref->{recv_ProcessorMode}\n";

   printf("RAM Size: %d Bytes (%d Words)\n",
    ($hashref->{recv_XRAMSize}), ($hashref->{recv_XRAMSize}/2) );

   printf("Testing Edits: %s\n",YesNo($hashref->{recv_TestingEdits}));
   print "Major Processor Fault\n"
    if ($hashref->{recv_MajorErrorHalted} eq "Yes");
   print "Download Mode\n" if ($hashref->{recv_PLC5DownloadMode} eq "Yes");

 # Processor Number} on DH+ link
   printf("DH+ Node Address: 0x%02X\n",$hashref->{recv_XPLC5DHPlusNode});

 # all IO address (0xFD if scanner)
   printf("All I/O Address: 0x%02X\n",$hashref->{recv_XPLC5All_IOAddr});

   $hashref->{recv_PLC5DblDensity}
    = YesNo( !($hashref->{recv_XPLC5IOComm} & 0x01));
   $hashref->{recv_PLC5AdapterMode}
    = YesNo($hashref->{recv_XPLC5IOComm} & 0x02);
   $hashref->{recv_PLC5ModG4TopHalf}
    = YesNo($hashref->{recv_XPLC5IOComm} & 0x04);
   $hashref->{recv_PLC5AdapterHalfRack}
    = YesNo($hashref->{recv_XPLC5IOComm} & 0x20);
   $hashref->{recv_PLC5DHPlus115K}
    = YesNo($hashref->{recv_XPLC5IOComm} & 0x40);
   print "DH+ Communication Double Density\n"
    if ($hashref->{recv_PLC5DblDensity} eq "Yes");
   print "Adapter Mode\n"
    if ($hashref->{recv_PLC5AdapterMode} eq "Yes");
   print "Module Group 4 Top Half\n"
    if ($hashref->{recv_PLC5ModG4TopHalf} eq "Yes");
   print "Adapter is Half Rack\n"
    if ($hashref->{recv_PLC5AdapterHalfRack} eq "Yes");
   print "Data Highway Plus at 115K baud\n"
    if ($hashref->{recv_PLC5DHPlus115K} eq "Yes");

   print "Last Data Table File: $hashref->{recv_XPLC5LastDataFile}\n";
   print "Last Program File:    $hashref->{recv_XPLC5LastProgramFile}\n";
   $hashref->{recv_ForcesActive}
    = YesNo($hashref->{recv_XPLC5ForcesX} & 0x01);
   $hashref->{recv_ForcesInstalled}
    = YesNo($hashref->{recv_XPLC5ForcesX} & 0x10);
   print "Forces Active\n"
    if ($hashref->{recv_ForcesActive} eq "Yes");
   print "Forces Present\n"
    if ($hashref->{recv_ForcesInstalled} eq "Yes");
   $hashref->{recv_XPLC5SubRevision}
    = ($hashref->{recv_XPLC5ForcesX} & 0x0E) >> 1;
   print "PLC5 processor sub-revision number:";
   print " $hashref->{recv_XPLC5SubRevision}\n";

   print "Memory Protect Active}\n"
    if ($hashref->{recv_XPLC5MemProt} > 0);
   print "Bad RAM indication\n" if ($hashref->{recv_XPLC5BadRAM} > 0);

   if ($hashref->{recv_PLC1785LT} eq "No") { # Newer PLC5's
    if ($hashref->{recv_XPLC5DebugMode} > 0) {
     print "Debug Mode\n";
     print "Hold Point File:    $hashref->{recv_XPLC5HoldPointFile}\n";
     print "Hold Point Element: $hashref->{recv_XPLC5HoldPointElement}\n";
    }
    print "Edit Time Stamp Year: $hashref->{recv_XPLC5EditTimeStampYear}\n";
    print "Edit Time Stamp Month:";
    print "  $hashref->{recv_XPLC5EditTimeStampMonth}\n";
    print "Edit Time Stamp Day:";
    print "    $hashref->{recv_XPLC5EditTimeStampDay}\n";
    print "Edit Time Stamp Hour:";
    print "   $hashref->{recv_XPLC5EditTimeStampHour}\n";
    print "Edit Time Stamp Minute:";
    print " $hashref->{recv_XPLC5EditTimeStampMin}\n";
    print "Edit Time Stamp Second:";
    print " $hashref->{recv_XPLC5EditTimeStampSecs}\n";
    print "Port number this command was received on:";
    print " $hashref->{recv_XPLC5PortNumber}\n";
   } # End if newer PLC5's

   $hashref->{'PLCtype'} ||= 'PLC';
   $hashref->{recv_ProgName} = PLC::read_program_name($hashref->{PLCname});
   unless (defined $hashref->{recv_ProgName}) {
    $hashref->{recv_ProgName} = 'UNKNOWN';
   }
   print "Program Name: $hashref->{recv_ProgName}\n";

  } # End PLC5-only stuff (Not 5/250)

  elsif ($hashref->{recv_PLC5_250} eq "Yes") {

  ($hashref->{recv_XModeStatus}, $hashref->{recv_XTypeExtender},
  $hashref->{recv_XPLC5_250ID}, $hashref->{recv_XPLC5_250X4},
  $hashref->{recv_XPLC5DHPlusNode}, $hashref->{recv_XPLC5_250ProgSeq},
  $hashref->{recv_XPLC5_250DataSeq}, $hashref->{recv_XPLC5_250UserSeq})
   = unpack "CCCCCSSS", $hashref->{recv_RECVDATApkt};

   $hashref->{recv_XProcessorMode} = $hashref->{recv_XModeStatus} & 0x07;

   if ($hashref->{recv_XProcessorMode} == 0)
    {$hashref->{recv_ProcessorMode} = "PROGRAM Mode"}
   elsif ($hashref->{recv_XProcessorMode} == 2)
    {$hashref->{recv_ProcessorMode} = "RUN Mode"}
   elsif ($hashref->{recv_XProcessorMode} == 4)
    {$hashref->{recv_ProcessorMode} = "Remote PROGRAM Mode"}
   elsif ($hashref->{recv_XProcessorMode} == 5)
    {$hashref->{recv_ProcessorMode} = "Remote TEST Mode"}
   elsif ($hashref->{recv_XProcessorMode} == 6)
    {$hashref->{recv_ProcessorMode} = "Remote RUN Mode"}
   else {$hashref->{recv_ProcessorMode} = "--UNKNOWN--"}

   $hashref->{recv_TestingEdits}
    = YesNo($hashref->{recv_XModeStatus} & 0xC0);
   $hashref->{recv_MajorErrorHalted}
    = YesNo($hashref->{recv_XModeStatus} & 0x08);
   $hashref->{recv_PLC5DownloadMode}
    = YesNo($hashref->{recv_XModeStatus} & 0x10);
   $hashref->{recv_PLC5_250_Offline}
    = YesNo($hashref->{recv_XModeStatus} & 0x20);
   $hashref->{recv_ProcessorType} = "PLC-5/250";
   $hashref->{recv_ForcesActive}
    = YesNo($hashref->{recv_XPLC5_250X4} & 0x01);
   $hashref->{recv_BackupActive}
    = YesNo($hashref->{recv_XPLC5_250X4} & 0x02);
   $hashref->{recv_PartnerActive}
    = YesNo($hashref->{recv_XPLC5_250X4} & 0x04);
   $hashref->{recv_EditAlloc}
    = YesNo($hashref->{recv_XPLC5_250X4} & 0x08);
   $hashref->{recv_OutputsReset}
    = YesNo($hashref->{recv_XPLC5_250X4} & 0x10);
   $hashref->{recv_MemoryInvalid}
    = YesNo($hashref->{recv_XPLC5_250X4} & 0x20);
   $hashref->{recv_SFC_ForcesActive}
    = YesNo($hashref->{recv_XPLC5_250X4} & 0x40);
   $hashref->{recv_EditAllocByUser}
    = YesNo($hashref->{recv_XPLC5_250X4} & 0x80);

   print "Processor Type: $hashref->{recv_ProcessorType}\n";
   print "Processor Mode: $hashref->{recv_ProcessorMode}\n";

   printf("Testing Edits: %s\n",YesNo($hashref->{recv_TestingEdits}));
   print "Major Processor Fault\n"
    if ($hashref->{recv_MajorErrorHalted} eq "Yes");
   print "Download Mode\n" if ($hashref->{recv_PLC5DownloadMode} eq "Yes");
   print "Offline\n" if ($hashref->{recv_PLC5_250_Offline} eq "Yes");
   printf("DH+ Node Address: 0x%02X\n",$hashref->{recv_XPLC5DHPlusNode});
   printf("ID: 0x%02X\n",$hashref->{recv_XPLC5_250ID});
   printf("Program Change Sequence Count: 0x%04X\n",
    $hashref->{recv_XPLC5_250ProgSeq});
   printf("Data Change Sequence Count: 0x%04X\n",
    $hashref->{recv_XPLC5_250DataSeq});
   printf("User-defined Data Change Sequence Count: 0x%04X\n",
    $hashref->{recv_XPLC5_250UserSeq});
   print "Forces Active}\n" if ($hashref->{recv_ForcesActive} eq "Yes");
   print "Backup System Active}\n" if ($hashref->{recv_BackupActive} eq "Yes");
   print "Partner System Active}\n"
    if ($hashref->{recv_PartnerActive} eq "Yes");
   print "Edit Resource Allocated\n"
    if ($hashref->{recv_EditAlloc} eq "Yes");
   print "Outputs Reset\n" if ($hashref->{recv_OutputsReset} eq "Yes");
   print "Memory Invalid\n" if ($hashref->{recv_MemoryInvalid} eq "Yes");
   print "SFC Forces Active}\n"
    if ($hashref->{recv_SFC_ForcesActive} eq "Yes");
   print "Edit Resource Allocated by User\n"
    if ($hashref->{recv_EditAllocByUser} eq "Yes");
  }
  elsif ($hashref->{recv_KF2} eq "Yes") {
   ($junk,$junk,$junk,$junk,$junk,$junk,$hashref->{recv_DiagAddr},
    $hashref->{recv_XSeriesRevision}, $hashref->{recv_X1})
    = unpack "CCCCCCSCC", $hashref->{recv_RECVDATApkt};
   print "Module Type: 1770-KF2\n";
   print "Data Highway Plus speed: 57600 bits/second\n";
   printf("Diagnostic Address: 0x%04X\n",$hashref->{recv_DiagAddr});
   $hashref->{recv_FirmwareRelease}
    = $hashref->{recv_XSeriesRevision} & 0x1F;
   $hashref->{recv_XSeries} = $hashref->{recv_XSeriesRevision} >> 5;
   ($hashref->{recv_Series})
    = pack "C", ($hashref->{recv_XSeries}+0x41-3);
   $hashref->{recv_SW1_1}
    = ($hashref->{recv_X1} & 0x08) ? 0 : 1; # 1 = switch up
   $hashref->{recv_SW1_2}
    = ($hashref->{recv_X1} & 0x10) ? 0 : 1; # 1 = switch up
   $hashref->{recv_SW1_3}
    = ($hashref->{recv_X1} & 0x20) ? 0 : 1; # 1 = switch up
   $hashref->{recv_SW1_4}
    = ($hashref->{recv_X1} & 0x40) ? 0 : 1; # 1 = switch up
   $hashref->{recv_SW1_5}
    = ($hashref->{recv_X1} & 0x80) ? 0 : 1; # 1 = switch up
   print "Series/Revision";
   print " $hashref->{recv_Series}-$hashref->{recv_FirmwareRelease}\n";
   print ($hashref->{recv_SW1_3} ? "Ignore" : "Accept");
   print " Duplicate Messages\n";
   print "Handshaking: ", ( $hashref->{recv_SW1_4} ? "On" : "Off" );
   print "\n";

   print( ($hashref->{recv_SW1_5} && ($hashref->{recv_SW1_1}
       || !$hashref->{recv_SW1_2}) ) ? "Half" : "Full" );
   print " Duplex\n";

   print( ($hashref->{recv_SW1_2}
    && $hashref->{recv_SW1_5} ) ? "CRC" : "BCC" );
   print " Error Detection\n";
   print
    ( ( ($hashref->{recv_SW1_2}
      && $hashref->{recv_SW1_5})
     || !$hashref->{recv_SW1_1} ) ? "No" : "Even" ); print " Parity\n";
   print "Embedded Responses: ";
   print( ( ($hashref->{recv_SW1_1}
          && $hashref->{recv_SW1_5})
         || !$hashref->{recv_SW1_2} ) ? "No\n" : "Yes\n");
  }
  elsif ($hashref->{recv_KF3} eq "Yes") {
   ($junk,$junk,$junk,$junk,$hashref->{recv_XSeriesRevision})
    = unpack "CCCCC", $hashref->{recv_RECVDATApkt};
   print "Module Type: 1770-KF3\n";
   $hashref->{recv_FirmwareRelease} = $hashref->{recv_XSeriesRevision} & 0x1F;
   $hashref->{recv_FirmwareRelease}++;
   $hashref->{recv_XSeries} = $hashref->{recv_XSeriesRevision} >> 5;
   ($hashref->{recv_Series}) = pack "C", ($hashref->{recv_XSeries}+0x41);
   print "Series/Revision";
   print " $hashref->{recv_Series}-$hashref->{recv_FirmwareRelease}\n";
  }
  elsif ($hashref->{recv_DHRIO} eq "Yes") {
   ($junk,$junk,$junk,$junk,$hashref->{recv_XSeriesRevision})
    = unpack "CCCCC", $hashref->{recv_RECVDATApkt};
   if (length $hashref->{recv_RECVDATApkt} > 14) {
   ($hashref->{recv_DHnodeA},$hashref->{recv_DHnodeB})
    = unpack "CC", substr($hashref->{recv_RECVDATApkt},11);
    $hashref->{recv_BulletinName} = substr($hashref->{recv_RECVDATApkt},14);
   }
   else {
    $hashref->{recv_BulletinName} = "1756-DHRIO (?)";
   }
   if ($hashref->{recv_BulletinName} =~ /^[\x20-\x7E]*$/) {
    print "Module Type: $hashref->{recv_BulletinName}\n";
    if (defined $hashref->{recv_DHnodeB}) {
     printf("Channel A(1) DH node = %00o (octal)\n", $hashref->{recv_DHnodeA});
     printf("Channel B(2) DH node = %00o (octal)\n", $hashref->{recv_DHnodeB});
    }
   }
  }
  else {print "Unsupported Processor Type\n"}
 return LinuxRules; # always true
} # end getstatusdata

sub ABsyswrite {
 print "\n--- sub ABeth1::ABsyswrite\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $hashref = shift;
 unless ($hashref) {
  $PLC::errorcode = 15; # No hash ref in subroutine call
  return undef;
 }
 syswrite($hashref->{sock}, $_[0], length($_[0]));
 return LinuxRules;
} # end sub ABsyswrite

sub ABsysread {
 print "\n--- sub ABeth1::ABsysread\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $hashref = shift;
 unless ($hashref) {
  $PLC::errorcode = 15; # No hash ref in subroutine call
  return undef;
 }
 my $len = shift || 10000;
 my $read_timeout = shift || 0.5;
 my $totalbytes;
 my ($nfound,$timeleft,$bytes,$str0);
 my ($rin,$rin0,$ein,$ein0);
 $rin0 = "";
 vec($rin0, fileno($hashref->{sock}), 1) = 1;
 $ein0 = $rin0;
  ($nfound, $timeleft) = select($rin=$rin0, undef, $ein=$ein0,$read_timeout);
  $bytes = sysread($hashref->{sock}, $str0, $len);
  $totalbytes = length $str0;
 
 if ( defined $bytes ) {
  if ($bytes == 0) { # closed socket
   close ($hashref->{sock});
   undef $hashref->{sock};
   $PLC::errorcode = 21; # Lost Socket
   return undef;
  }
  else {
   print "received $totalbytes bytes\n" if $PLC::debug;
   PLC::PrintBytes($str0) if $PLC::debug;
   return $str0;
  }
 }
 if ( !$totalbytes ) {
  $PLC::errorcode = 7; # Read Timeout
  return undef;
 } # Read Timeout
} # end sub ABsysread


#--------------------------------------------------------

sub eth_connect {
 print "\n--- sub ABeth1::ethconnect\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $hashref = shift;
 unless ($hashref) {
  $PLC::errorcode = 15; # No hash ref in subroutine call
  return undef;
 }
 my $port = 2222; # TCP port for connecting to Ethernet PLC's

 unless ($hashref->{remoteIP}) {
  $PLC::errorcode = 22; # No IP specified for this PLC
  return undef;
 }
 if ($hashref->{remoteIP} =~ /^([@-_\w.]+)$/) {
  $hashref->{remoteIP} = $&; # untaint remoteIP
 }
 if ($hashref->{remoteIP} =~  /^(\d+\.\d+\.\d+\.\d+)$/) {
  $hashref->{remoteIP} = $&; # untaint remoteIP if IPV4 dotted quad
 }

 if ($PLC::PingFirst) { # ping first to avoid error message from IO::Socket
  my $pingdata = `ping -c 1 $hashref->{remoteIP}`;
  if ($pingdata =~ /100\% packet loss/i) {
    $PLC::errorcode = 3; # No Socket
    return undef;
  }
 }
 my $save_warning_state = $^W; $^W = 0;
 unless (
  $hashref->{sock} = new IO::Socket::INET( PeerAddr => $hashref->{remoteIP},
					   PeerPort => $port,
					   Proto    => 'tcp',
					   Timeout  => 10 ) ) {
  $PLC::errorcode = 3; # No Socket
  return undef;
 }
 $^W = $save_warning_state;
 $hashref->{sock}->autoflush(1);
 fcntl( $hashref->{sock}, F_SETFL(), O_NONBLOCK() );

######################## try different values, add comments #######
 $hashref->{eth_send_pkt0} = pack "CCnNNnnNNN", (1,1,0,0,0,4,5,0,0,0);
# $hashref->{eth_send_pkt0} = pack "CCnNNnnNNN", (1,1,0,0,0,4,0x0028,0,0,0);

 if ($PLC::debug) {
  print "\nSending attach packet to $hashref->{remoteIP}:\n";
  PLC::PrintBytes($hashref->{eth_send_pkt0});
 }
 ABsyswrite($hashref,$hashref->{eth_send_pkt0}) or return undef;

 $hashref->{eth_reply_pkt0} = ABsysread($hashref); # read reply packet from PLC
 unless ( ($hashref->{eth_reply_pkt0})
 &&      ( length($hashref->{eth_reply_pkt0}) >= 8 ) ) {
  $PLC::errorcode = 23; # Failed to get connection handle from ethernet PLC
  return undef;
 }
 ($hashref->{connhandle}) = unpack("N",substr($hashref->{eth_reply_pkt0},4));
 printf("\nconnhandle=%08X\n",$hashref->{connhandle}) if $PLC::debug;
 return $hashref->{connhandle};
} # end sub eth_connect

sub check_tns {
 print "\n--- sub ABeth1::check_tns\n" if ($PLC::debug and $PLC::debug >= 5);
# return 1 if tns matches, 0 if no match, undef if error
 my $hashref = shift;
 unless ($hashref) { return undef }

 my $recv_tns;
 my $recv_df1_pkt;
 my $tns_offset;
 my $str0;

 unless ( $hashref->{recv_raw_pkt} ) { return undef }

 $str0 = $hashref->{recv_raw_pkt};
 $str0 ||= '';
 $tns_offset = 34;

 if (length($str0) < ($tns_offset + 2)) { return undef }
 ($recv_tns) = unpack("S", substr($str0, $tns_offset) );
 $hashref->{xrecv_tns} = $recv_tns;
 return ($recv_tns == $hashref->{send_tns}) ? 1 : 0;

} # end sub check_tns

####------------------------------------------------------------------

sub ethsendDF1 {
 print "\n--- sub ABeth1::ethsendDF1\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $hashref = shift;
 my ($tns_tries,$tns_ok);
 unless ($hashref) {
  $PLC::errorcode = 15; # No hash ref in subroutine call
  return undef;
 }
# make connection to PLC unless it is already established
 unless ( defined $hashref->{connhandle} ) {
  ABeth1::eth_connect($hashref) or return undef;
 }

# now we are attached to the PLC, so get ready to send the request packet
 $hashref->{send_eth_pkt} = (pack "CCnNNNNNN",
  (1,7,(length $hashref->{send_df1packet}),
   $hashref->{connhandle},0,0x00003AED,0,0,0))
  .  $hashref->{send_df1packet};

while ( ABeth1::ABsysread($hashref, 1000, 0.1) ) {} # flush socket

 if ($PLC::debug) {
  print "\nSending data request packet:\n";
  PLC::PrintBytes($hashref->{send_eth_pkt});
 }
# send data request to PLC
 ABeth1::ABsyswrite($hashref,$hashref->{send_eth_pkt}) or return undef;

 print "\n\nWaiting for reply .........................\n\n" if $PLC::debug;
# get response from PLC

 $tns_tries = 3;
 do {
  $hashref->{recv_raw_pkt} = ABeth1::ABsysread($hashref);
  $tns_ok = check_tns($hashref);
  return undef if !(defined $tns_ok);
  if ($PLC::debug && !$tns_ok) {
   print "\n++ ERROR - Sent TNS = $hashref->{send_tns}, ";
   print "Received TNS = $hashref->{xrecv_tns} ++\n\n";
  }
 } until ($tns_ok || --$tns_tries <= 0);

 return $hashref->{recv_raw_pkt};
} # end sub ethsendDF1

# End of module ABeth1

#-------------------------------------------------------------------------#
package ABgate3;

use strict;
use IO::Socket;
use POSIX;
STDOUT->autoflush(1);

use constant LinuxRules => 1; # always true
my ($str6,$str6b,$bytes_to_follow);
my ($pktctr1a,$ctr5a,$ctr5b,$ctr2);
my (%SessionHandle,%handlechan,%sock,%NoTimeout);
use vars qw($junk);
my $timeouts;

my @letter = ('0','A','B','C','D','E','F','G','H','I','J','K','L');
my $pktctr1 = 4;
my $ctr5 = 2;
my $z1 = chr(0);
my $z2 = $z1 x 2;
my $z3 = $z1 x 3;
my $z4 = $z1 x 4;
my $z8 = $z1 x 8;

# command codes:
my $SendRRData = 0x006F; # unconnected message
my $SendUnitData = 0x0070; # connected message

# service codes: (for requests, replies have bit 7 set)
my $GetAttributeAll = 0x01;
my $SetAttributeAll = 0x02;
my $ResetCode = 0x05;
my $StartCode = 0x06; # ControlLogix RemoteRun ???
my $StopCode = 0x07;  # ControlLogix RemoteProgram ???
my $GetAttributeSingle = 0x0E;
my $SetAttributeSingle = 0x10;
my $ExecutePCCC = 0x4B;
my $ConnectedDataTableRead = 0x4C;
my $FwdClose = 0x4C;
my $UnconnectedSend = 0x52;
my $FwdOpen = 0x54;

# Object Classes: (preceded by 0x20 in connection path)
my $CLASS = 0x20;
my $INSTANCE = 0x24;
my $ATTRIBUTE = 0x30;
my $ObIdentity = 0x01;
my $ObMessageRouter = 0x02;
my $ObAssembly = 0x04;
my $ObConnectionManager = 0x06;
my $ObRegister = 0x07;
my $ObParameter = 0x0F;
my $ObParameterGroup = 0x10;
my $ObPCCC = 0x67;
my $ObDHplusxxx = 0xA6;  ##### 1756-DHRIO DH+ object ?????????????
my $ObControlNet = 0xF0;

sub qprint {
 print @_ unless $PLC::quiet;
}

sub qprinterr {
 print STDERR @_ unless $PLC::quiet && $PLC::quiet >= 2;
}

sub flushsock {
 my $IP = shift;
 my $timeout = shift || 5;
 unless ($sock{$IP}) {
  $PLC::errorcode = 3; # No Socket
  return undef;
 }
 do { $junk = ABsysread($IP, 1000, $timeout) } while $junk;
}

sub ABsysread {
 print "\n--- sub ABgate3::ABsysread\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $IP = shift;
 unless ($sock{$IP}) {
  $PLC::errorcode = 3; # No Socket
  return undef;
 }
 my $len = shift || 10000;
 my $read_timeout = shift || 1.5;
 my $totalbytes;
 my ($nfound,$timeleft,$bytes,$str0);
 my ($rin,$rin0,$ein,$ein0);
 $rin0 = "";
 vec($rin0, fileno($sock{$IP}), 1) = 1;
 $ein0 = $rin0;
  ($nfound, $timeleft) = select($rin=$rin0, undef, $ein=$ein0,$read_timeout);
  $bytes = sysread($sock{$IP}, $str0, $len);
  $totalbytes = length $str0;
 
 if ( defined $bytes ) {
  if ($bytes == 0) { # closed socket
   close ($sock{$IP});
   undef $sock{$IP};
   $PLC::errorcode = 21; # Lost Socket
   return undef;
  }
  else {
   print "received $totalbytes bytes\n" if $PLC::debug;
   PLC::PrintBytes($str0) if $PLC::debug;
   return $str0;
  }
 }
 if ( !$totalbytes ) {
  $PLC::errorcode = 7; # Read Timeout
  return undef;
 }
} # end sub ABsysread

sub sndpktstr {
 print "\n--- sub ABgate3::sndpktstr\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $sock = shift;
 PLC::PrintBytes($_[0]) if $PLC::debug && length($_[0]);
 unless ($sock) {
  $PLC::errorcode = 3; # No Socket
  return undef;
 }
 syswrite($sock, $_[0], length($_[0]));
}

sub ABcleanup {
 print "\n--- sub ABgate3::ABcleanup\n" if ($PLC::debug and $PLC::debug >= 5);
 %handlechan = ();
 %NoTimeout = ();
 %SessionHandle = ();
 close (values %sock) if (values %sock);
 %sock = ();
}

sub ABinit {
 print "\n--- sub ABgate3::ABinit\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $IP = shift;
 unless ($IP) {
  $PLC::errorcode = 17; # No Gateway IP
  return undef;
 }
# initialize socket to ControlLogix Gateway
 my $port = 44818; # TCP port for connecting to ControlLogix Ethernet module
 my ($str1,$str2,$str3,$str4,$xlen);
 my ($str1hdr,$str2hdr,$str3hdr,$str4hdr);
 my ($str1data,$str2data,$str3data,$str4data);
 my $snd1 = pack "S*", (0x0001,0,0,0,0,0,0,0,0,0,0,0); # CommandCode 1 ###
 my $snd2 = pack "S*", (0x0004,0,0,0,0,0,0,0,0,0,0,0); # CommandCode 4 ###
 my $snd3 = pack "S*", (0x0064,0,0,0,0,0,0,0,0,0,0,0); # CommandCode 0x64 ###
 my $snd4 = pack "S*", (0x0065,0x0004,0,0,0,0,0,0,0,0,0,0,1,0); # Code 0x65
 $timeouts = 0;
 if ($PLC::debug) {
  print 
   "Initializing CIP_encapsulated communications with IP address $IP ...\n";
 }

 if ($IP =~ /^([@-_\w.]+)$/) {
  $IP = $&; # untaint IP
 }
 if ($IP =~  /^(\d+\.\d+\.\d+\.\d+)$/) {
  $IP = $&; # untaint IP if IPV4 dotted quad
 }

 if ($PLC::PingFirst) { # ping first to avoid error message from IO::Socket
  my $pingdata = `ping -c 1 $IP`;
  if ($pingdata =~ /100\% packet loss/i) {
    $PLC::errorcode = 3; # No Socket
    return undef;
  }
 }
 my $save_warning_state = $^W; $^W = 0;
 unless (
 $sock{$IP} = new IO::Socket::INET( PeerAddr => $IP,
				    PeerPort => $port,
				    Proto    => 'tcp',
				    Timeout  => 10 ) ) {
  $PLC::errorcode = 3; # No Socket
  return undef;
 }
 $^W = $save_warning_state;
 $sock{$IP}->autoflush(1);
 fcntl( $sock{$IP}, F_SETFL(), O_NONBLOCK() );

 print "\nsending Packet #1 (Command Code 0x01)...\n" if $PLC::debug;
 sndpktstr($sock{$IP},$snd1);
 print "\nsending Packet #2 (Command Code 0x04)...\n" if $PLC::debug;
 sndpktstr($sock{$IP},$snd2);
 print "\nsending Packet #3 (Command Code 0x64)...\n" if $PLC::debug;
 sndpktstr($sock{$IP},$snd3);

 print "\nwaiting for Packet #1 Header\n" if $PLC::debug;
 unless ( length ($str1hdr = ABsysread($IP,24)) == 24 ) {
  $PLC::errorcode = 31; # Failed to get first init packet from gateway
  return undef;
 }
 $xlen = unpack "S", substr($str1hdr, 2, 2); # length of Encapsulation Data
 print "\nwaiting for Packet #1 Data ($xlen bytes)\n" if $PLC::debug;
 unless ( length ($str1data = ABsysread($IP,$xlen)) == $xlen ) {
  $PLC::errorcode = 31; # Failed to get first init packet from gateway
  return undef;
 }
 $str1 = $str1hdr . $str1data;

 print "\nwaiting for Packet #2 Header\n" if $PLC::debug;
 unless ( length ($str2hdr = ABsysread($IP,24)) == 24 ) {
  $PLC::errorcode = 32; # Failed to get second init packet from gateway
  return undef;
 }
 $xlen = unpack "S", substr($str2hdr, 2, 2); # length of Encapsulation Data
 print "\nwaiting for Packet #2 Data ($xlen bytes)\n" if $PLC::debug;
 unless ( length ($str2data = ABsysread($IP,$xlen)) == $xlen ) {
  $PLC::errorcode = 32; # Failed to get second init packet from gateway
  return undef;
 }
 $str2 = $str2hdr . $str2data;

 print "\nwaiting for Packet #3 Header\n" if $PLC::debug;
 unless ( length ($str3hdr = ABsysread($IP,24)) == 24 ) {
  $PLC::errorcode = 33; # Failed to get third init packet from gateway
  return undef;
 }
 $xlen = unpack "S", substr($str3hdr, 2, 2); # length of Encapsulation Data
 print "\nwaiting for Packet #3 Data ($xlen bytes)\n" if $PLC::debug;
 unless ( length ($str3data = ABsysread($IP,$xlen)) == $xlen ) {
  $PLC::errorcode = 33; # Failed to get third init packet from gateway
  return undef;
 }
 $str3 = $str3hdr . $str3data;

 print "\nsending Packet #4 (Command Code 0x65)...\n" if $PLC::debug;
 sndpktstr($sock{$IP},$snd4);

 print "\nwaiting for Packet #4 Header\n" if $PLC::debug;
 unless ( length ($str4hdr = ABsysread($IP,24)) == 24 ) {
  $PLC::errorcode = 34; # Failed to get connection handle from gateway
  return undef;
 }
 $xlen = unpack "S", substr($str4hdr, 2, 2); # length of Encapsulation Data
 print "\nwaiting for Packet #4 Data ($xlen bytes)\n" if $PLC::debug;
 unless ( length ($str4data = ABsysread($IP,$xlen)) == $xlen ) {
  $PLC::errorcode = 34; # Failed to get connecion handle from gateway
  return undef;
 }
 $str4 = $str4hdr . $str4data;

# bytes 4-7 are connection ID string for this Gateway (IP address):
 $SessionHandle{$IP} = substr($str4, 4, 4);
 if ($PLC::debug) {
  printf("Session Handle = 0x%08X\n",(unpack "L", $SessionHandle{$IP}));
 }
 return LinuxRules; # always true
} # end sub ABinit

sub ABinitchan {
# parameters: IP, slotnum,channel  returns: channel handle
 print "\n--- sub ABgate3::ABinitchan\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my ($IP, $slotnum, $channel) = @_;
 unless ($IP) {
  $PLC::errorcode = 17; # No Gateway IP
  return undef;
 }

 if (PLC::istrue($PLC::PLCproc{$PLC::PLCname}->{NET_ENI})) { ###ZZZ
  $slotnum ||= 0; $channel ||= 0;
 } # end if NET_ENI   ###ZZZ

 unless (defined $slotnum) {
  $PLC::errorcode = 18; # No Gateway Slot #
  return undef;
 }
 unless (defined $channel) {
  $PLC::errorcode = 19; # No Gateway Channel
  return undef;
 }

 if ($PLC::UseConnectedMessages == 0) { ###ZZZ
  return undef;
 } # end if $PLC::UseConnectedMessages == 0  ###ZZZ

 my ($str5,$str5x,$x3,$snd5,$handlechan);
 my ($ok);

###############ZZZ
 unless ($SessionHandle{$IP}) {
  print "\n### Auto-initializing ControlLogix Gateway at IP=$IP\n\n"
   if $PLC::debug;
  ABinit($IP) or return undef;
 }

  if ($PLC::debug) {
   print
    "Initializing $IP DH+ channel $slotnum","$letter[$channel] ...\n";
  }

 $ok = 0;
 my $xtries = $PLC::retries;
#### my $xtries = $PLC::PLCproc{$procname}->{retries} || $PLC::retries;
 do {{ # double-curly-do allows 'next' + 'last' statements
  $pktctr1a = pack "S*", ( $pktctr1,0,0,0 ); # Encap. header transaction ID
  $ctr5a = pack "S", ( $ctr5 );
  $ctr5b = pack "S", ( $ctr5 - 1 );

#ZZZZZ
my $CommandCode = $SendRRData; # (0x006F) for encapsulation header
my $EncapStatus = 0; # encap header status (0 in request)
my $EncapOptions = 0; # encap header options (unused)
my $EncapHandle = 0; # encap data handle (0 for encapsulated CIP)
my $EncapTimeout = 0x10;
my $CPF_ItemCount = 2;
my $CPF_AdrItemType = 0; # item type (address item in CPF)
my $CPF_AdrItemDataLength = 0; # item data length (address item in CPF)
my $CPF_DataItemType = 0xB2; # item type (data item in CPF)
my $ServiceCode = $FwdOpen;
my $PathSize = 0x02; # (size of request path in words)
my $TickTime = 7;
my $TimeoutTicks = 0x9B;
my $O_T_ConnID = 0x80000000 + $ctr5;
my $T_O_ConnID = 0x80000000 + $ctr5 - 1;
my $ConnSN = $ctr5;
my $OrigVendorID = 0; # originator vendor ID (rslinx=4D)
my $OrigSN = $$ & 0xFFFF; # originator serial number
my $TimeoutMult = 2; # inactivity timeout
my $O_T_RPI = 1250000; # requested O-T RPI in uS
my $O_T_Parm = 0x4302; # O-T connection parameters
my $T_O_RPI = 1250000; # requested T-O RPI in uS
my $T_O_Parm = 0x4302; # T-O connection parameters
my $TransClassTrigger = 0xA3; # server transport, class 3, application trigger
my $ConnPathSize = 4; # # of 16bit words in connection path

# start of T-PDU (Data Item Data)
my $T_PDU =  (pack "C*", ($ServiceCode,$PathSize, 0x20,6, 0x24,1));
  $T_PDU .=  (pack "C*", ($TickTime,$TimeoutTicks));
  $T_PDU .= (pack "LLS", ($O_T_ConnID,$T_O_ConnID,$ConnSN));
  $T_PDU .= (pack "SLCCCC", ($OrigVendorID,$OrigSN,$TimeoutMult,0,0,0));
  $T_PDU .= (pack "LSLS", ($O_T_RPI,$O_T_Parm,$T_O_RPI,$T_O_Parm));
  $T_PDU .=  (pack "C*", ($TransClassTrigger,$ConnPathSize));
# start of connection path:
  $T_PDU .=  (pack "C*", (1)); # backplane port of 1756-ENET
  $T_PDU .= (pack "C", ($slotnum));
  $T_PDU .= pack "C*", ($CLASS,$ObDHplusxxx);
  $T_PDU .= pack "C*", ($INSTANCE,$channel);
  $T_PDU .= pack "C*", (0x2C,0x01); # ?????????????????????
my $CPF_DataItemDataLength = length($T_PDU); # data item data length

# start of encapsulation data:
my $EncapData = (pack "L", ($EncapHandle));
  $EncapData .= (pack "S", ($EncapTimeout));
# start of CPF (in encapsulation data)
my $CPF  =  (pack "S*", ($CPF_ItemCount));
  $CPF .=  (pack "S*", ($CPF_AdrItemType,$CPF_AdrItemDataLength));
  $CPF .=  (pack "S*", ($CPF_DataItemType,$CPF_DataItemDataLength));
  $CPF .= $T_PDU;
  $EncapData .= $CPF;
my $DataLength = length($EncapData); # for encapsulation header

############## check for uninitialized variables here: ####
# start of encapsulation header:
my  $EncapHeader = (pack "S*", ($CommandCode,$DataLength));
  $EncapHeader    .= $SessionHandle{$IP};
  $EncapHeader    .= (pack "L", ($EncapStatus));
  $EncapHeader    .= $pktctr1a;
  $EncapHeader    .= (pack "L", ($EncapOptions));

  $snd5 = $EncapHeader . $EncapData;

  print "\nsending Packet #5 (FwdOpen request for T3 conn.)...\n"
   if $PLC::debug;
  sndpktstr($sock{$IP},$snd5);
  $ctr2 = 1;
########################  $ctr2++; if ($ctr2 > 65535) {$ctr2 = 1}
  $pktctr1++; if ($pktctr1 > 65535) {$pktctr1 = 1}
  $ctr5++; if ($ctr5 > 65535) {$ctr5 = 1}

  print "\nwaiting for #5 Encap. Header (first 24 bytes)\n" if $PLC::debug;
  $str5 = ABsysread($IP,24); # Read Encapsulation Header of reply packet
  unless ( defined $str5 ) {
   print "\n### Auto-initializing ControlLogix Gateway at IP=$IP\n\n"
    if $PLC::debug;
   ABinit($IP);
   next;
  }
  unless ( length($str5) && ($str5 =~ /^\x6F/) ) {
   flushsock($IP); # flush socket
   next;
  }
  if (length $str5 < 24) { # didn't get entire encapsulation header?
   if ($PLC::debug) {print "\nReceived Short Packet ";PLC::PrintBytes($str5);}
   flushsock($IP); # flush socket
   next;
  }

  $x3 = unpack "S", substr($str5, 2, 2); # length of Encapsulation Data

  print "\nwaiting for #5 Encapsulation Data (last $x3 bytes)\n"
   if $PLC::debug;
  unless ( length ($str5x = ABsysread($IP,$x3) ) ) {
   next;
  }
  if (length $str5x < 46) { # expect 46 bytes for 0x6F reply
   if ($PLC::debug) { PLC::PrintBytes($str5x)}
   next;
  }
  $ok = 1;
 }} until ( ($ok == 1) or ($xtries-- == 0) );

 if (!$ok) { $PLC::errorcode = 5; return undef } # Can't get channel handle
 $str5 .= $str5x;
 $handlechan = substr($str5, 44, 4); # O-T Conn. ID from reply packet
 return $handlechan;
} # end sub ABinitchan

sub sendDF1pkt { ############# edit for unconnected PCCC requests #####
# parameters: procname,dhpkt  return: df1 return packet
 print "\n--- sub ABgate3::sendDF1pkt\n"
  if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my ($procname, $dhpkt) = @_;
 unless ($procname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 unless ($dhpkt) {
  $PLC::errorcode = 30; # No packet to send
  return undef;
 }
 $PLC::PLCname = $procname; ###ZZZZZZZZZZZZ

 my ($snd6,$data_addr,$size,$datatype);

 my $slotnum  = $PLC::PLCproc{$procname}{GatewaySlot};
 my $channel  = $PLC::PLCproc{$procname}{GatewayChan};
 my $dhnode   = $PLC::PLCproc{$procname}{DHnode};
 my $proctype = $PLC::PLCproc{$procname}{PLCtype};
 my $IP       = $PLC::PLCproc{$procname}{GatewayIP};
 my $NET_ENI  = $PLC::PLCproc{$procname}{NET_ENI};
 $proctype ||= 'UNKNOWN';
 unless ($IP) {
  $PLC::errorcode = 17; # No Gateway IP
  return undef;
 }
 if (PLC::istrue($NET_ENI)) { ###ZZZ
  $slotnum ||= 0; $channel ||= 0; $dhnode  ||= 0;
 } # end if NET_ENI  ###ZZZ
 else { # (not NET_ENI, must be a real ControlLogix Gateway)
  unless (defined $slotnum) {
   $PLC::errorcode = 18; # No Gateway Slot #
   return undef;
  }
  unless (defined $channel) {
   $PLC::errorcode = 19; # No Gateway Channel
   return undef;
  }
  unless (defined $dhnode) {
   $PLC::errorcode = 20; # No DH+ node
   return undef;
  }
 }

 flushsock($IP, 0.1); # flush socket

 if ($PLC::UseConnectedMessages) { ###ZZZ
  if ($PLC::AlwaysInitGatewayChannel) {
   $NoTimeout{$IP}{$slotnum}{$channel} = 0;
   delete $handlechan{$IP}{$slotnum}{$channel};
  }
  unless ($NoTimeout{$IP}{$slotnum}{$channel} && ($timeouts < 10) ) {
   if ($PLC::debug) {
    print "\n ---- Initializing $IP slot $slotnum, channel $channel\n";
   }
   print("\n### Auto-initializing DHRIO at IP=$IP, slot=$slotnum, ",
    "channel=$channel\n\n") if $PLC::debug;
   $handlechan{$IP}{$slotnum}{$channel} = ABinitchan($IP, $slotnum, $channel);
   if ($PLC::errorcode) {return undef}
  }
 } # end if $PLC::UseConnectedMessages ###ZZZ


 else { # Using Unconnected messages  ZZZZ
   ABinit($IP); #######ZZZZZZZZZZZ
   if ($PLC::errorcode) {return undef}
 }



 if ($PLC::debug) {
  print "\n\n$procname type= $proctype, IP= $IP,";
  print " channel=$slotnum",$letter[$channel];
  printf(", node= %02o (octal)\n", $dhnode);
  if ($PLC::UseConnectedMessages) { print "Using T3-Connected Messages\n"}
  else {print "Using Unconnected Messages -- EXPERIMENTAL\n"}
 }

 $pktctr1a = pack "S*", ( $pktctr1,0,0,0 ); # Encap. header transaction ID
### $pktctr1a = pack "S", ( $pktctr1 );

 if ($PLC::debug) { print "\nReady to send dhpkt:\n";PLC::PrintBytes($dhpkt) }

my ($CommandCode,$CPF_AdrItemType,$CPF_AdrItemDataLength,$CPF_DataItemType);
my (@CPF_AdrItemData);
if ($PLC::UseConnectedMessages) { ###ZZZ
 $CommandCode = $SendUnitData; # (0x0070) for encapsulation header
 $CPF_AdrItemType = 0xA1; # item type (address item in CPF)
 $CPF_AdrItemDataLength = 4; # item data length (address item in CPF)
 $CPF_DataItemType = 0xB1; # item type (data item in CPF)
} else {
 $CommandCode = $SendRRData; # (0x006F) for encapsulation header
 $CPF_AdrItemType = 0x81; # item type (address item in CPF)
 @CPF_AdrItemData = (0); # address item data (address item in CPF)
 $CPF_AdrItemDataLength = scalar(@CPF_AdrItemData);
 $CPF_DataItemType = 0x91; # item type (data item in CPF)
}
my $EncapStatus = 0; # encap header status (0 in request)
my $EncapOptions = 0; # encap header options (unused)
my $EncapHandle = 0; # encap data handle (0 for encapsulated CIP)
my $EncapTimeout = 0x10;
my $CPF_ItemCount = 2;
###my $ServiceCode = $FwdOpen;
my $PathSize = 0x02; # (size of request path in words)
my $TickTime = 7;
my $TimeoutTicks = 0x9B;
my $O_T_ConnID = 0x80000000 + $ctr5;
my $T_O_ConnID = 0x80000000 + $ctr5 - 1;
my $ConnSN = $ctr5;
my $OrigVendorID = 0; # originator vendor ID (rslinx=4D)
my $OrigSN = $$ & 0xFFFF; # originator serial number
my $TimeoutMult = 2; # inactivity timeout
my $O_T_RPI = 1250000; # requested O-T RPI in uS
my $O_T_Parm = 0x4302; # O-T connection parameters
my $T_O_RPI = 1250000; # requested T-O RPI in uS
my $T_O_Parm = 0x4302; # T-O connection parameters
my $TransClassTrigger = 0xA3; # server transport, class 3, application trigger
my $ConnPathSize = 4; # # of 16bit words in connection path

# start of T-PDU (Data Item Data)
#my $T_PDU =  (pack "C*", ($ServiceCode,$PathSize, 0x20,6, 0x24,1));
#  $T_PDU .=  (pack "C*", ($TickTime,$TimeoutTicks));
#  $T_PDU .= (pack "LLS", ($O_T_ConnID,$T_O_ConnID,$ConnSN));
#  $T_PDU .= (pack "SLCCCC", ($OrigVendorID,$OrigSN,$TimeoutMult,0,0,0));
#  $T_PDU .= (pack "LSLS", ($O_T_RPI,$O_T_Parm,$T_O_RPI,$T_O_Parm));
#  $T_PDU .=  (pack "C*", ($TransClassTrigger,$ConnPathSize));
## start of connection path:
#  $T_PDU .=  (pack "C*", (1)); # backplane port of 1756-ENET
#  $T_PDU .= (pack "C", ($slotnum));
#  $T_PDU .= pack "C*", ($CLASS,$ObDHplusxxx);
#  $T_PDU .= pack "C*", ($INSTANCE,$channel);
#  $T_PDU .= pack "C*", (0x2C,0x01); # ?????????????????????

my $T_PDU;

if ($PLC::UseConnectedMessages) { ###ZZZ
 my $ctr2a = pack "S", ( $ctr2 );
 $T_PDU = $ctr2a . $z2 . chr($dhnode) . $z3 . $dhpkt; ###ZZZ
} else {
 $T_PDU = substr($dhpkt,2); ###ZZZ
}
my $CPF_DataItemDataLength = length($T_PDU); # data item data length

# start of encapsulation data:
my $EncapData = (pack "L", ($EncapHandle));
  $EncapData .= (pack "S", ($EncapTimeout));
# start of CPF (in encapsulation data)
my $CPF  =  (pack "S*", ($CPF_ItemCount));
  $CPF .=  (pack "S*", ($CPF_AdrItemType,$CPF_AdrItemDataLength));
  if ($PLC::UseConnectedMessages) { ###ZZZ
   $CPF .= $handlechan{$IP}{$slotnum}{$channel}; # ConnectionID 4bytes
  }
  else { $CPF .=  (pack "C*", (@CPF_AdrItemData));}
  $CPF .=  (pack "S*", ($CPF_DataItemType,$CPF_DataItemDataLength));
  $CPF .= $T_PDU;
  $EncapData .= $CPF;
my $DataLength = length($EncapData); # for encapsulation header

############## check for uninitialized variables here: ####
# start of encapsulation header:
my  $EncapHeader = (pack "S*", ($CommandCode,$DataLength));
  $EncapHeader    .= $SessionHandle{$IP};
  $EncapHeader    .= (pack "L", ($EncapStatus));
  $EncapHeader    .= $pktctr1a;
  $EncapHeader    .= (pack "L", ($EncapOptions));

 $snd6 = $EncapHeader . $EncapData;

 if ( $CommandCode == $SendUnitData) {
  print "Sending T3-Connected Encapsulated PCCC request:\n" if ($PLC::debug);
 }
 else {
  print "Sending Unconnected Encapsulated PCCC request:\n" if ($PLC::debug);
 }

 sndpktstr($sock{$IP},$snd6);

 $pktctr1 += 2;
 $ctr2++; if ($ctr2 > 65535) {$ctr2 = 1}

 if ($PLC::debug) {print "\nWaiting for header ..."}
# first read 24 bytes (encapsulation header) of T3 connected reply
 $str6 = ABsysread($IP,24);
 unless ( defined $str6 && (length($str6) == 24) ) {
  $NoTimeout{$IP}{$slotnum}{$channel} = 0;
  $timeouts++;
  delete $handlechan{$IP}{$slotnum}{$channel};
  print "\n++ No response from ControlLogix Gateway ++\n" if $PLC::debug;
  $PLC::errorcode = 7; # Read Timeout
  return undef;
 }
# error unless it starts with 0x70 (T3 connected ONLY) or 6F (unconnected)
 my ($junk,$morebytes) = unpack "SS",$str6;
 unless (($junk == 0x0070) || ($junk == 0x6F)) {
  print "\n++ Bad Packet from ControlLogix Gateway ++\n" if $PLC::debug;
   flushsock($IP); # flush socket
  $PLC::errorcode = 7;
  return undef;
 }
# next 2 bytes = length of Encapsulation Data section
 unless ($morebytes) {  $PLC::errorcode = 7; return undef}
 if ($PLC::debug) {print "\nWaiting for data ($morebytes bytes) ..."}
  unless ( length ($str6b = ABsysread($IP,$morebytes)) == $morebytes ) {
   $NoTimeout{$IP}{$slotnum}{$channel} = 0;
   $timeouts++;
   delete $handlechan{$IP}{$slotnum}{$channel};
   print "\n++ No response from ControlLogix Gateway ++\n" if $PLC::debug;
   $PLC::errorcode = 7; # Read Timeout
   return undef;
  }
 $str6 .= $str6b; # append to first 24 bytes

 $NoTimeout{$IP}{$slotnum}{$channel} = 1;

 if ($PLC::debug) { print "Received Packet "; PLC::PrintBytes($str6); }
 return $str6;
} # end sub sendDF1pkt

LinuxRules; # End the module with something that is always true

# End of module ABgate3


#-------------------------------------------------------------------------#
package DF1a;

# Serial Communication - Allen-Bradley DF1 Protocol
# with autodetection of BCC/CRC, bits/second, and parity
# For SLC, MicroLogix, and PLC5 processors only (Not PLC-5/15 or VME)

use strict;
STDOUT->autoflush(1);

use constant LinuxRules => 1; # always true

unless ($PLC::DisableSerialPortStuff) {
use vars qw($KF2 $KF3 $DiagAddr $port_error);
my @yesno = ("No", "Yes");
my $redials;
my (@xpkt,$value,@values);
my $dhnode = 1;
my ($dhpkt,@xxx);
my ($totaltrans,$totaltrans_a);
my ($data_addr, $slc_data_addr, $fileword, $offset, $bit, $size);
my $datatype;
my $filenum = -1;
my ($file,$word);
my ($i,$junk,$found,$str);
my $proctype;
my $returndata;
my $writedata;
my $writepkt;
my @readcheck;
my $wrt_ok;
my $response;
my $dialprefix;
use vars qw($OS_win);
$OS_win = $PLC::OS_win;

my $prevparmfile = "/plc/df1parms";
my $bps; my $parity;
my ($prevbps,$prevcrc_bcc,$prevparity);

my $SOH = chr(0x01);
my $STX = chr(0x02);
my $ETX = chr(0x03);
my $EOT = chr(0x04);
my $ENQ = chr(0x05);
my $ACK = chr(0x06);
my $LF  = chr(0x0A);
my $FF  = chr(0x0C);
my $CR  = chr(0x0D);
my $DLE = chr(0x10);
my $NAK = chr(0x15);
my $SYN = chr(0x16);
my $CAN = chr(0x18);
my $ESC = chr(0x1B);
my $RS  = chr(0x1E);
my $DLE_STX = $DLE . $STX;
my $DLE_ETX = $DLE . $ETX;
my $DLE_ACK = $DLE . $ACK;
my $DLE_NAK = $DLE . $NAK;
my $DLE_ENQ = $DLE . $ENQ;
my $DLE_DLE = $DLE . $DLE;

my $SLC_MicroLogix;
my ($bcc,$CRCBCC,$dst,$src,$cmd,$sts,$fnc);
my ($ext_sts,$EXT_STS);
my $RECVPKT; my $RECVCRCBCC;
my $extstsmsg;
my ($recvsrc,$recvdst,$recvcmd,$recvsts,$recvtns,$recvextsts);
my $RECVDATA;
my ($bytes,$timeleft,$tterm,$sterm);
my ($cflags,$oflag,$iflag,$lflag,$fdterm,%commport,%lockfile);
my (%default_bps,%default_parity,%default_crc_bcc);
my ($nfound,$x);
my @buf; my $BUF;
my @pkt; my $PKT;
my @data; my $DATA;
my (@msg,$MSG);
my $CH;
my ($XModeStatus,$XTypeExtender,$XExtendedInterfaceType,
 $XExtendedProcessorType,$XSeriesRevision,
 $X1,$X2,$X3,$X4,$X5,$X6,$X7,$X8,$X9,$X0,$X,$XMajorErrorWord,
 $XProcessorModeWord,$XProgramID,$XRAMSize,$XProgramOwnerNodeAddress);
my ($XPLC5Type, $XPLC5DHPlusNode,$XPLC5All_IOAddr,$XPLC5IOComm,
 $XPLC5DataTableFiles, $XPLC5DataTypeFiles, $XPLC5ForcesX,
 $XPLC5MemProt, $XPLC5BadRAM, $XPLC5DebugMode, $XPLC5HoldPointFile,
 $XPLC5HoldPointElement, $XPLC5EditTimeStampSecs, $XPLC5EditTimeStampMin,
 $XPLC5EditTimeStampHour, $XPLC5EditTimeStampDay, $XPLC5EditTimeStampMonth,
 $XPLC5EditTimeStampYear, $XPLC5PortNumber, $PLC1785LT, $PLC5_250);
my ($XPLC5_250ID,$XPLC5_250X4,$XPLC5_250ProgSeq,
 $XPLC5_250DataSeq,$XPLC5_250UserSeq);
my $PLC5; my $ProcessorMode; my $XProcessorMode; my $TestingEdits;
my $MajorErrorHalted; my $ProcessorType; my $ProgName;
my $ForcesActive; my $ForcesInstalled;
my $error = 0;
my $crc_bcc;
my @speeds; my @errordetect; my @parity;
my @sign = ( 1, -1);
my ($floatsign,$floatx,$floatexponent,$floatword);
my ($floatmantissa,$floatdata,$mantissa,$signbit,$exponent,$roundfactor);
my $using_modem = 0;

my @autobps = (19200, 2400, 9600, 4800, 1200, 300); # autobaud speeds to try

$dst = 1; # default destination (remote address) = 1
$src = 0; # default source (local address) = 0

sub qprint {
 print @_ unless $PLC::quiet;
}

sub qprinterr {
 print STDERR @_ unless $PLC::quiet && $PLC::quiet >= 2;
}

####################################
#sub initRS232defaults {
# print "\n--- sub DF1a::initRS232defaults\n"
#  if ($PLC::debug and $PLC::debug >= 5);
# $PLC::errorcode = 0; # No Error
# my $PLCname = shift;
# unless ($PLCname) {
#  $PLC::errorcode = 1; # No PLC name
#  return undef;
# }
# my $device = $PLC::PLCproc{$PLCname}->{'RS232device'} or return undef;
# my $commport = $commport{$device} or return undef;
# my $bps = $PLC::PLCproc{$PLCname}->{'RS232bps'};
# my $parity = $PLC::PLCproc{$PLCname}->{'RS232parity'};
# my $crc_bcc = $PLC::PLCproc{$PLCname}->{'RS232crc_bcc'};
# if ( ($default_bps{$device}) && (!defined $bps) ) {
#  $PLC::PLCproc{$PLCname}->{'RS232bps'} = $default_bps{$device};
# }
# if ( ($default_parity{$device}) && (!defined $parity) ) {
#  $PLC::PLCproc{$PLCname}->{'RS232parity'} = $default_parity{$device};
# }
# if ( ($default_crc_bcc{$device}) && (!defined $crc_bcc) ) {
#  $PLC::PLCproc{$PLCname}->{'RS232crc_bcc'} = $default_crc_bcc{$device};
# }
#}

####################################
sub YesNo {
 print "\n--- sub DF1a::YesNo\n" if ($PLC::debug and $PLC::debug >= 5);
 my ($bool) = @_;
 if    (lc $bool eq "no")  {return "No"}
 elsif (lc $bool eq "yes") {return "Yes"}
 elsif ($bool == 0) {return "No"} else {return "Yes"}
}

####################################
sub writeserial {
 print "\n--- sub DF1a::writeserial\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 my $device = $PLC::PLCproc{$PLCname}->{'RS232device'};
 unless ($device) {
  $PLC::errorcode = 24; # No serial port device
  return undef;
 }
 my $commport = $commport{$device};
 unless ($commport) {
  $PLC::errorcode = 25; # No commport object
  return undef;
 }
 my $wrtdata = shift;
 unless (defined $wrtdata) {
  $PLC::errorcode = 16; # No write data specified
  return undef;
 }
 my $delay = shift || $PLC::writeserial_delay;
 while(readserial($PLCname, 1, $delay)) { # flush read buffer
  $PLC::errorcode = 0; # No Error
 }
 select(undef,undef,undef,$delay);
 unless ($commport->write($wrtdata)) {
  $PLC::errorcode = 27; # RS232 write error
  return undef;
 }
 select(undef,undef,undef,$PLC::scanfor_delay);
} # end sub writeserial

####################################
sub readserial {
 print "\n--- sub DF1a::readserial\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 my $device = $PLC::PLCproc{$PLCname}->{'RS232device'};
 unless ($device) {
  $PLC::errorcode = 24; # No serial port device
  return undef;
 }
 my $commport = $commport{$device};
 unless ($commport) {
  $PLC::errorcode = 25; # No commport object
  return undef;
 }
 my $Read_data;
 my $Databytes;
 my ($read_bytes,$wait_timeout) = @_;
### $wait_timeout ||= 0.5;
 $wait_timeout ||= 0.1;
 $commport->read_interval(100) if $^O eq "MSWin32";
 $commport->read_const_time(1000 * $wait_timeout);
 ($Databytes,$Read_data) = $commport->read($read_bytes);
 unless (defined $Read_data) {
  $PLC::errorcode = 26; # RS232 read error
  return undef;
 }
 PLC::PrintBytes $Read_data if ($PLC::debug and $PLC::debug >= 6);
 return ($Databytes,$Read_data);
} # end sub readserial

#####################################
sub waitfor {
 print "\n--- sub DF1a::waitfor\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 my $device = $PLC::PLCproc{$PLCname}->{'RS232device'};
 unless ($device) {
  $PLC::errorcode = 24; # No serial port device
  return undef;
 }
 my $commport = $commport{$device};
 unless ($commport) {
  $PLC::errorcode = 25; # No commport object
  return undef;
 }
 my $timeout=$commport->get_tick_count + (1000 * shift);
 my @matchesx = @_;
 $commport->are_match(@matchesx);
 $commport->lookclear;  # clear buffers
 my $gotit = "";

 for (;;) {
  unless (defined ($gotit = $commport->lookfor)) {
   $PLC::errorcode = 26; # RS232 read error
   return undef;
  }
  if ($gotit ne "") {
   my ($found, $end) = $commport->lastlook;
   $PLC::errorcode = 0; # No Error
   return ($gotit, $found);
  }
  if ( ($commport->reset_error)
  ||   ($commport->get_tick_count > $timeout) ) {
   $PLC::errorcode = 26; # RS232 read error
   return undef;
  }
 }
} # end sub waitfor

#####################################
sub scanfor {
 print "\n--- sub DF1a::scanfor\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 my $foundit = undef;
 my $str = '';
#### select(undef,undef,undef,$PLC::scanfor_delay);
 do {
  ($bytes,$CH) = readserial($PLCname,1,$PLC::scanfor_delay);
  $str .= $CH if defined $CH;
  for (@_) {
   if ( index($str,$_) > -1) {$foundit = $_}
  }
 } until (($bytes == 0) || $foundit );
 return $foundit;
} # end sub scanfor

#####################################
sub attr_restore {
 print "\n--- sub DF1a::attr_restore\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 my $device = $PLC::PLCproc{$PLCname}->{'RS232device'};
 unless ($device) {
  $PLC::errorcode = 24; # No serial port device
  return undef;
 }
 my $commport = $commport{$device};
 unless ($commport) {
  $PLC::errorcode = 25; # No commport object
  return undef;
 }
# ======================================================================
# hang up modem (if it is used)
 if ($using_modem) {
  $commport->dtr_active(0);
  my $i = 0;
  while ( ($commport->modemlines & $commport->MS_RLSD_ON)  && ($i++ < 20)) {
   select(undef,undef,undef,0.25);
  }
  select(undef,undef,undef,1);
# send ATH0 + \r
 }
 undef $commport;
 return LinuxRules; # always true
} # End of attr_restore

####################################
sub init {
 print "\n--- sub DF1a::init\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $commport;
 my $PLCname = shift;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 my ($device, $bps, $crc_bcc, $parity, $plctype) = (0,0,0,0,0);
 $RECVDATA = '';
 if ($_[0]) {
  $device = $_[0];
 }
 elsif ($PLC::PLCproc{$PLCname}->{'RS232device'}) {
  $device = $PLC::PLCproc{$PLCname}->{'RS232device'}
 }
 if ($_[1]) {
  $bps = $_[1];
 }
 elsif ($PLC::PLCproc{$PLCname}->{'RS232bps'}) {
  $bps = $PLC::PLCproc{$PLCname}->{'RS232bps'}
 }
 if ($_[2]) {
  $crc_bcc = $_[2];
 }
 elsif ($PLC::PLCproc{$PLCname}->{'RS232crc_bcc'}) {
  $crc_bcc = $PLC::PLCproc{$PLCname}->{'RS232crc_bcc'}
 }
 if ($_[3]) {
  $parity = $_[3];
 }
 elsif ($PLC::PLCproc{$PLCname}->{'RS232parity'}) {
  $parity = $PLC::PLCproc{$PLCname}->{'RS232parity'}
 }
 if ($_[4]) {
  $plctype = $_[4];
 }
 elsif ($PLC::PLCproc{$PLCname}->{'PLCtype'}) {
  $plctype = $PLC::PLCproc{$PLCname}->{'PLCtype'}
 }

 if ($crc_bcc) {@errordetect = ($crc_bcc)}
 else {@errordetect = qw(CRC BCC)}
 if ($bps) {@speeds = ($bps)}
 else {@speeds = @autobps}
 if ($parity) {@parity = ($parity)}
 #else {@parity = qw(none even odd)}
 else {@parity = qw(none even)}

 unless ($device) {
  $PLC::errorcode = 24; # No serial port device
  return undef;
 }

##################### edit for windows ####### (if $OS_win)
 if ( ($device =~ /^\/dev\/([-\w.]+)$/) && (-c $device) ) {
  $device = $&; # untaint $device
 }
 else { PLC::abort1(5, "++ Invalid device name $device ++")}

##################### edit for windows ####### (if $OS_win)
 $PLC::lockdir .= '/' unless $PLC::lockdir =~ /\/$/;
 my $lockfile = $PLC::lockdir . 'LCK..'
  . substr( $device, ( rindex( $device, '/' ) ) + 1 );

 print "Initializing serial port $device\n" if $PLC::debug;

 my $ignorelock = YesNo($PLC::PLCproc{$PLCname}->{'ignorelock'} || 0);
 my $waitforlock = YesNo($PLC::PLCproc{$PLCname}->{'waitforlock'} || 0);

 if ($ignorelock eq 'Yes') {
  print("Ignoring existing lock file (if any)\n") if $PLC::debug;
  if ($OS_win) { $commport = Win32::SerialPort->new ($device)}
  else { $commport = Device::SerialPort->new ($device)}
 }
 else {
  if ($waitforlock eq 'Yes') {
   print("Wait for device lockfile (if found) to go away.\n") if $PLC::debug;
  }
  if ($OS_win) { $commport = Win32::SerialPort->new ($device,1,$lockfile)}
  else { $commport = Device::SerialPort->new ($device,1,$lockfile)}
 }

 $port_error = $^E;
 if ($port_error =~ /File exists/i) {
  undef $commport;
##################### edit for windows ####### (if $OS_win)
  print("Found existing lock file\n") if $PLC::debug;
# if stale_lock_file, unlink it
  if (open LOCKFILE, $lockfile) {
   my $lockpid = <LOCKFILE>;
   my $lockowner = "";
   close LOCKFILE;
   $lockpid = int($lockpid);
   if ($lockpid =~ /^(\d+)$/) {$lockpid = $&} # untaint lock pid
   else {$lockpid = 'invalid'}
   print "lockpid=$lockpid\n" if $PLC::debug;
   if ($lockpid == $$) {
    print "Hey! that's my pid!!! I guess I already own $device\n"
     if $PLC::debug;
    if (unlink $lockfile) {
     print("Removed stale lock file\n") if $PLC::debug;
     if ($OS_win) { $commport = Win32::SerialPort->new ($device,1,$lockfile)}
     else { $commport = Device::SerialPort->new ($device,1,$lockfile)}
    }
    else {
     print "Failed to delete my own lockfile!!!\n" if $PLC::debug;
    }
   }
   else {
    if ( open (PS, "/bin/ps p $lockpid 2>/dev/null |") ) { 
     my $psheader = <PS>; chomp $psheader; # get header line
     my $psline = <PS> || ""; chomp $psline; close PS;
     print "psline=$psline\n" if $PLC::debug;
     my $psindex = index $psheader,"COMMAND";
     $lockowner = substr $psline, $psindex - 1 if $psline;
     if ($lockowner ) {
      qprint("$device locked by process ($lockowner) (pid $lockpid)\n");
     }
     elsif (!$psline && (unlink $lockfile) ) {
      print("Removed stale lock file\n") if $PLC::debug;
      if ($OS_win) { $commport = Win32::SerialPort->new ($device,1,$lockfile)}
      else { $commport = Device::SerialPort->new ($device,1,$lockfile)}
     }
    } else {print("Unable to run /bin/ps\n") if $PLC::debug}
   }
  } else {qprinterr "Unable to open existing lock file\n" }

# if -w option, wait for it to go away
  if ( ($waitforlock eq 'Yes') && !$commport) {
   qprint "Device $device in use, waiting for it to become available ...\n";
   alarm 600; # limit wait to 10 minutes
   do {
    PLC::snooze(5);
    if ($OS_win) { $commport = Win32::SerialPort->new ($device,1,$lockfile)}
    else { $commport = Device::SerialPort->new ($device,1,$lockfile)}
    $port_error = $^E;
   } while ($port_error =~ /File exists/i);
  }
 }

 unless ($commport) {
  if ($port_error =~ /File exists/i) {
   qprinterr "$port_error\n";
   PLC::abort1(6, "++ Failed creating lock file $lockfile ++");
  }
  qprinterr "$port_error\n"  if $port_error;
  PLC::abort1(7,"++ Can't open serial port $device ++");
 }

 alarm 0;

 unless ($commport) {
  $PLC::errorcode = 2; # Can't open serial device
  return undef;
 }

 $commport{$device} = $commport;
 $lockfile{$device} = $lockfile;

 if ($commport) {
  $commport->user_msg(1);	# misc. warnings
  $commport->error_msg(1);	# hardware and data errors
  $commport->baudrate(19200);
  $commport->parity("none");
### $commport->parity_enable(1);   # for any parity except "none"
  $commport->databits(8);
  $commport->stopbits(1);
  $commport->handshake('none');
  $commport->write_settings;
  $commport->dtr_active(1);
  $commport->read_interval(100) if $^O eq "MSWin32";
  $commport->read_const_time(2000);
 }

 if ($bps || $crc_bcc || $parity) {
   setparity($PLCname, $parity) if $parity;
   setspeed($PLCname, $bps) if $bps;
 }
 else {
# try using previous parameters (saved in file $prevparmfile)
#  if (open (PREVPARMS, "$prevparmfile")) {
#   ($prevbps,$prevcrc_bcc,$prevparity) = <PREVPARMS>;
#   chomp ($prevbps,$prevcrc_bcc,$prevparity);
#   close (PREVPARMS);
#  }
  if ($prevbps && $prevcrc_bcc && $prevparity) {
   $crc_bcc = $prevcrc_bcc;
   setparity($PLCname, $prevparity);
   setspeed($PLCname, $prevbps);
   print "Trying previous settings: " if $PLC::debug;
   print "$bps bits/second (parity=$parity) using $crc_bcc...\n"
    if $PLC::debug;
   $RECVDATA = DF1send($PLCname, 6,3); # Get diagnostic status
  }
 }

 if ($crc_bcc) {@errordetect = ($crc_bcc)}
 else {@errordetect = qw(CRC BCC)}
 if ($bps) {@speeds = ($bps)}
 else {@speeds = @autobps}
 if ($parity) {@parity = ($parity)}
 #else {@parity = qw(none even odd)}
 else {@parity = qw(none even)}

 unless ( ($bps && $crc_bcc && $parity) 
 || ($prevbps && $prevcrc_bcc && $prevparity && ($error == 0)) ) {
  qprint "\nAutodetecting ";
  qprint "bits/second" unless $bps;
  qprint ", " unless ($crc_bcc && $parity) || $bps;
  qprint "parity" unless $parity;
  qprint ", " unless $crc_bcc || ($bps && $parity);
  qprint "crc_bcc" unless $crc_bcc;
  qprint "\n";
  for (@parity) {
   $parity = $_;
   for (@speeds) { # this is a list of speeds to try (in order)
    $bps = $_;
    for (@errordetect) {
     $crc_bcc = $_;
     if ($PLC::debug) {
      qprint "Trying $bps bits/second (parity=$parity) using $crc_bcc...\n";
     }
     $PLC::PLCproc{$PLCname}->{'RS232crc_bcc'} = $crc_bcc;
     setparity($PLCname, $parity);
     setspeed($PLCname, $bps);
     $RECVDATA = DF1send($PLCname, 6,3); # Get diagnostic status
     last if ($error == 0); # found error detection type
    } # End for @errordetect
    last if ($error == 0); # found serial speed
   } # End for @speeds
   last if ($error == 0); # found parity type
  } # End for @parity
 }

 if ($error) {
  qprinterr "------- No Reply -------\n\n";
  attr_restore($PLCname);
  $PLC::errorcode = 26; # RS232 read error
  return undef;
 }
## Save speed, error detection, and parity for next time
# if (open (PREVPARMS, "> $prevparmfile")) {
#  print PREVPARMS "$bps\n$crc_bcc\n$parity\n";
#  close (PREVPARMS);
# }

# These will be defaults for other devices using this serial port:
 $default_bps{$commport}     = $bps;
 $default_parity{$commport}  = $parity;
 $default_crc_bcc{$commport} = $crc_bcc;

 if ($PLC::debug) {
  print "\nUsing serial port $device\n";
  print "Serial port speed: $bps bits/second (parity=$parity)\n";
  print "Using $crc_bcc error detection\n\n";
 }

 $PLC::PLCproc{$PLCname}->{'RS232bps'} = $bps;
 $PLC::PLCproc{$PLCname}->{'RS232parity'} = $parity;
 $PLC::PLCproc{$PLCname}->{'RS232crc_bcc'} = $crc_bcc;

 return LinuxRules;
} # End of init

####################################
sub setspeed {
 print "\n--- sub DF1a::setspeed\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 my $device = $PLC::PLCproc{$PLCname}->{'RS232device'};
 unless ($device) {
  $PLC::errorcode = 24; # No serial port device
  return undef;
 }
 my $commport = $commport{$device};
 unless ($commport) {
  $PLC::errorcode = 25; # No commport object
  return undef;
 }
######## Don't change speed if modem is connected ####################
 ($bps) = @_;
 if ($bps) {
#  print "BPS=$bps --------------------";
  $commport->baudrate("$bps");
  $commport->write_settings;
#  my $speed = $commport->baudrate;
#  print " BAUDRATE=$speed\n";
 }
 return LinuxRules;
}

####################################
sub setparity {
 print "\n--- sub DF1a::setparity\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 my $device = $PLC::PLCproc{$PLCname}->{'RS232device'};
 unless ($device) {
  $PLC::errorcode = 24; # No serial port device
  return undef;
 }
 my $commport = $commport{$device};
 unless ($commport) {
  $PLC::errorcode = 25; # No commport object
  return undef;
 }
 ($parity) = @_;
 if ($parity) {
#  print "PARITY=$parity -----------------";
  if ($parity =~ /^[eE]/) {
   $commport->parity('even');
   $commport->parity_enable(1);
  }
  elsif ($parity =~ /^[oO]/) {
   $commport->parity('odd');
   $commport->parity_enable(1);
  }
  else {
   $commport->parity('none');
   $commport->parity_enable(0);
  }
  $commport->write_settings;
#  my $par = $commport->parity;
#  print " PARITY=$par\n";
 }
 return LinuxRules;
}

####################################
sub calc_bcc {
 print "\n--- sub DF1a::calc_bcc\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $MSG = shift or return undef;
 my @msg; my $calcbcc; my $CALCBCC;
 my $isum = 0;
 @msg = unpack "C*", $MSG;
 for (@msg) {$isum += $_;}
 $calcbcc = (0x100 - ($isum & 0xFF)) & 0xFF;
 printf("\nBCC is 0x%02X\n", $calcbcc) if ($PLC::debug and $PLC::debug >= 2);
 $CALCBCC = chr($calcbcc);
 return $CALCBCC;
} # End of calc_bcc

####################################
sub calc_crc {
 print "\n--- sub DF1a::calc_crc\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $MSG = shift or return undef;
 my $Kcrc = 0xA001; # CRC calculation constant
 my @msg; my $CALCCRC; my $xorit;
 my $crcreg = 0;
 @msg = (unpack "C*", $MSG . $ETX);
 for (@msg) {
  $crcreg ^= $_;
  for (1..8) {
   $xorit = (($crcreg & 1) == 1);
   $crcreg = $crcreg >> 1;
   if ($xorit) {$crcreg ^= $Kcrc}
  }
 }
 printf("\nCRC is 0x%04X\n", $crcreg) if ($PLC::debug and $PLC::debug >= 2);
 $CALCCRC = pack "S", $crcreg;
 return $CALCCRC;
} # End of calc_crc

####################################
sub calc_crc_bcc {
 print "\n--- sub DF1a::calc_crc_bcc\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 my $crc_bcc = $PLC::PLCproc{$PLCname}->{'RS232crc_bcc'};
 print "\nUsing $crc_bcc\n" if ($PLC::debug and $PLC::debug >= 5);
 my $MSG = shift;
 unless ($crc_bcc && $MSG) {
  $PLC::errorcode = 29; # Error generating CRC or BCC
  return undef;
 }
 my $CRCBCC;
 if ($crc_bcc eq "CRC") {$CRCBCC = calc_crc($MSG)}
 else {$CRCBCC = calc_bcc($MSG)}
 unless (defined $CRCBCC) {
  $PLC::errorcode = 29; # Error generating CRC or BCC
  return undef;
 }
 return $CRCBCC;
} # End of calc_crc_bcc

####################################
sub flushDF1 {
 print "\n--- sub DF1a::flushDF1\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 my $found;
 do {
# scan past DLE ACK
  $found = scanfor($PLCname, $DLE_ACK,$DLE_NAK,$DLE_ENQ,$DLE_ETX);
  if (defined($found) && $found eq $DLE_ACK) {
   print "DF1a::flushDF1 Received DLE ACK\n" if $PLC::debug;
  }
  elsif (defined($found) && $found eq $DLE_NAK) {
   print "DF1a::flushDF1 Received DLE NAK\n" if $PLC::debug;
  }
  elsif (defined($found) && $found eq $DLE_ENQ) {
   print "DF1a::flushDF1 Received DLE ENQ\n" if $PLC::debug;
   print "DF1a::flushDF1 Sending DLE ACK\n" if ($PLC::debug);
   writeserial($PLCname,$DLE_ACK);
  }
  elsif (defined($found) && $found eq $DLE_ETX) {
   print "DF1a::flushDF1 Received DLE ETX\n" if $PLC::debug;
   print "DF1a::flushDF1 Sending DLE ACK\n" if ($PLC::debug);
   writeserial($PLCname,$DLE_ACK);
  }
 } while (defined $found);
} # end sub flushDF1

####################################
sub DF1send {
 print "\n--- sub DF1a::DF1send\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 my $device = $PLC::PLCproc{$PLCname}->{'RS232device'};
 unless ($device) {
  $PLC::errorcode = 24; # No serial port device
  return undef;
 }
 $src = $PLC::PLCproc{$PLCname}->{'RS232src'};
 $dst = $PLC::PLCproc{$PLCname}->{'RS232dst'};
 $dst = $PLC::PLCproc{$PLCname}->{'DHnode'} unless defined $dst;
 my $commport = $commport{$device};
 unless ($commport) {
  init($PLCname) or return undef;
  print "Initialized serial port $device\n" if $PLC::debug;
 }
 my $done = 0;
 my ($PREVCH,$CH) = ('','');
 my $RECVDATA = '';
 ($cmd,$fnc,$DATA) = @_;
 my ($rin,$win,$ein)    = (0,0,0);
 my ($rout,$wout,$eout) = (0,0,0);
 $BUF = '';
 use vars qw($tns);

 if ($PLC::debug) {
  printf("CMD-FNC = %X-%02X\n",$cmd,$fnc);
  PLC::PrintBytes($DATA) if defined $DATA;
 }

 $error = 0;
 $dst ||= 1;
 $src ||= 0;
 $sts = 0;

 unless (defined $cmd) {
  $PLC::errorcode = 10; # No CMD
  return undef;
 }
 unless (defined $fnc) {
  $PLC::errorcode = 11; # No FNC
  return undef;
 }

 my $tries = 3;
 do {{ # double-curly-do allows 'next' + 'last' statements
  $tns = PLC::tns();
  $MSG = pack "CCCCSC",($dst,$src,$cmd,$sts,$tns,$fnc);
  $MSG .= $DATA if $DATA;
  $CRCBCC = calc_crc_bcc($PLCname, $MSG);
  return undef unless defined $CRCBCC;
  $MSG =~ s/$DLE/$DLE_DLE/g;
  my $SENDPKT = $DLE_STX . $MSG . $DLE_ETX . $CRCBCC;

###  flushDF1($PLCname);
  print "\nSending " if ($PLC::debug);
  PLC::PrintBytes($SENDPKT) if ($PLC::debug);
  writeserial($PLCname,$SENDPKT);

# scan past DLE ACK
  $found = scanfor($PLCname, $DLE_ACK,$DLE_NAK,$DLE_ENQ);
  if (defined($found) && $found eq $DLE_ACK) {
   print "Received DLE ACK\n" if $PLC::debug;
  }
  elsif (defined($found) && $found eq $DLE_NAK) {
   print "Received DLE NAK\n" if $PLC::debug;
   next;
  }
  elsif (defined($found) && $found eq $DLE_ENQ) {
   print "Received DLE ENQ\n" if $PLC::debug;
   print "Sending DLE ACK\n" if ($PLC::debug);
   writeserial($PLCname,$DLE_ACK);
   next;
  }
  else {
   print "DLE ACK header not found\n" if $PLC::debug;
   next;
  }

# scan past DLE STX
  $found = scanfor($PLCname, $DLE_STX);
  if (defined($found) && ($found eq $DLE_STX)) {
   print "Received DLE STX\n" if $PLC::debug;
  }
  else {
   print "DLE STX header not found\n" if $PLC::debug;
   flushDF1($PLCname);
   next;
  }

# read characters into data buffer until DLE ETX
  $RECVPKT = '';
  my $databytes;
  do {
   ($databytes,$CH) = readserial($PLCname,1);
   $RECVPKT .= $CH if $databytes;
  } until ( ($databytes == 0) || ($RECVPKT =~ /$DLE_ETX/) );

  $RECVPKT =~ s/$DLE_ETX// or next;

# eliminate double 0x10's
  $RECVPKT =~ s/$DLE_DLE/$DLE/g;

# read BCC or CRC char(s)
  ($bytes,$RECVCRCBCC) = readserial($PLCname,length $CRCBCC);
  if ($bytes < length($CRCBCC)) {qprint "Bad CRC/BCC\n";}
  next if ($bytes < 1);

  $crc_bcc = $PLC::PLCproc{$PLCname}->{'RS232crc_bcc'};
  $CRCBCC = calc_crc_bcc($PLCname, $RECVPKT);
  return undef unless defined $CRCBCC;
  if ($RECVCRCBCC eq $CRCBCC) {$done = 1;}
  else {
   printf("Received   $crc_bcc is 0x%04X\n",ord($RECVCRCBCC)) if ($PLC::debug);
   printf("Calculated $crc_bcc is 0x%04X\n",ord($CRCBCC)) if ($PLC::debug);
  }
 }} until ( ($done) || (--$tries == 0) );

 if (!$done) {
  $error = 1;
  print "Sending DLE ACK\n" if ($PLC::debug);
  writeserial($PLCname,$DLE_ACK);
  $PLC::errorcode = 26; # RS232 read error
  return undef;
 }
 else { # done
  my $i = 0;
  print "Received Packet:\n" if ($PLC::debug);
  PLC::PrintBytes($RECVPKT) if ($PLC::debug);
  print "Sending DLE ACK\n" if ($PLC::debug);
  writeserial($PLCname,$DLE_ACK);
 }

# extract src,dst,cmd,sts,tns,(extsts),(data)
 ($recvsrc,$recvdst,$recvcmd,$recvsts,$recvtns,$recvextsts,$RECVDATA) =
  unpack "CCCCSCA*", $RECVPKT;
 if ($recvsts == 0) {$recvextsts = 0;}

 if ($PLC::debug) {
  printf("\nTransmitted SRC = 0x%02X, DST = 0x%02X, CMD = 0x%02X\n",
   $src, $dst, $cmd);
  printf("            STS = 0x%02X, TNS = 0x%04X\n", $sts, $tns);
  printf("Received    SRC = 0x%02X, DST = 0x%02X, CMD = 0x%02X\n",
   $recvsrc, $recvdst, $recvcmd);
  printf("            STS = 0x%02X, TNS = 0x%04X, CRCBCC = 0x%04X,
   EXT.STS = 0x%02X\n", $recvsts, $recvtns, ord($RECVCRCBCC), $recvextsts);
 }

 if ($recvcmd != ($cmd | 0x40)) {qprint "CMD field does not match\n";}
 if ($recvtns != $tns)          {
  qprint "TNS field does not match (sent $tns, got $recvtns)\n";
 }

 if ($recvsts) {
  qprinterr("++ ERROR - ", PLC::ABerrormessage($recvsts,$recvextsts),
   " ++\n");
##  print "\n";
 }

 if ($PLC::debug) {
  print "\nReceived Data:\n";
  if (length($RECVDATA) == 0) {print "NONE\n";}
  else {PLC::PrintBytes($RECVDATA);}
 }
 return $RECVDATA;
} # End of DF1send

####################################
sub DF1sendpkt {
 print "\n--- sub DF1a::DF1sendpkt\n" if ($PLC::debug and $PLC::debug >= 5);
 $PLC::errorcode = 0; # No Error
 my $PLCname = shift;
 unless ($PLCname) {
  $PLC::errorcode = 1; # No PLC name
  return undef;
 }
 my $device = $PLC::PLCproc{$PLCname}->{'RS232device'};
 unless ($device) {
  $PLC::errorcode = 24; # No serial port device
  return undef;
 }
 my $commport = $commport{$device};
 $src = $PLC::PLCproc{$PLCname}->{'RS232src'};
 $dst = $PLC::PLCproc{$PLCname}->{'RS232dst'};
 $dst = $PLC::PLCproc{$PLCname}->{'DHnode'} unless defined $dst;

 if ( !$commport			# no commport object open yet
 &&    $PLC::ForceRS232autodetect ) {	# ignore RS232 settings in rc file
  undef $PLC::PLCproc{$PLCname}->{'RS232bps'};
  undef $PLC::PLCproc{$PLCname}->{'RS232parity'};
  undef $PLC::PLCproc{$PLCname}->{'RS232crc_bcc'};
 }

 if ($commport) { # if port already initialized for another PLC
  $PLC::PLCproc{$PLCname}->{'RS232bps'}     ||= $default_bps{$commport};
  $PLC::PLCproc{$PLCname}->{'RS232parity'}  ||= $default_parity{$commport};
  $PLC::PLCproc{$PLCname}->{'RS232crc_bcc'} ||= $default_crc_bcc{$commport};
 }

 unless ( ($commport)
 && ($PLC::PLCproc{$PLCname}->{'RS232bps'})
 && ($PLC::PLCproc{$PLCname}->{'RS232parity'})
 && ($PLC::PLCproc{$PLCname}->{'RS232crc_bcc'}) ) {
  init($PLCname) or return undef;
 }

 my $done = 0;
 my ($PREVCH,$CH) = ('','');
 my $RECVDATA = '';
 my ($DATApkt) = @_;
 my ($rin,$win,$ein)    = (0,0,0);
 my ($rout,$wout,$eout) = (0,0,0);
 $BUF = '';

 $error = 0;

 my $tries = 1;
 do {{ # double-curly-do allows 'next' + 'last' statements
  $MSG = $DATApkt; ####

  $crc_bcc = $PLC::PLCproc{$PLCname}->{'RS232crc_bcc'};
  $CRCBCC = calc_crc_bcc($PLCname, $MSG);
  return undef unless defined $CRCBCC;
  $MSG =~ s/$DLE/$DLE_DLE/g;
  my $SENDPKT = $DLE_STX . $MSG . $DLE_ETX . $CRCBCC;

###  flushDF1($PLCname);
  print "\nSending " if ($PLC::debug);
  PLC::PrintBytes($SENDPKT) if ($PLC::debug);
  writeserial($PLCname,$SENDPKT);

# scan past DLE ACK
  $found = scanfor($PLCname, $DLE_ACK,$DLE_NAK,$DLE_ENQ);
  if (defined($found) && $found eq $DLE_ACK) {
   print "Received DLE ACK\n" if $PLC::debug;
  }
  elsif (defined($found) && $found eq $DLE_NAK) {
   print "Received DLE NAK\n" if $PLC::debug;
   next;
  }
  elsif (defined($found) && $found eq $DLE_ENQ) {
   print "Received DLE ENQ\n" if $PLC::debug;
   print "Sending DLE ACK\n" if ($PLC::debug);
   writeserial($PLCname,$DLE_ACK);
   next;
  }
  else {
   print "DLE ACK header not found\n" if $PLC::debug;
   next;
  }

# scan past DLE STX
  $found = scanfor($PLCname, $DLE_STX);
  if (defined($found) && $found eq $DLE_STX) {
   print "Received DLE STX\n" if $PLC::debug;
  }
  else {
   print "DLE STX header not found\n" if $PLC::debug;
   flushDF1($PLCname);
   next;
  }

# read characters into data buffer until DLE ETX
  $RECVPKT = '';
  my $databytes;
  do {
   ($databytes,$CH) = readserial($PLCname,1);
   $RECVPKT .= $CH if $databytes;
  } until ( ($databytes == 0) || ($RECVPKT =~ /$DLE_ETX/) );

  $RECVPKT =~ s/$DLE_ETX// or next;

# eliminate double 0x10's
  $RECVPKT =~ s/$DLE_DLE/$DLE/g;

# read BCC or CRC char(s)
  ($bytes,$RECVCRCBCC) = readserial($PLCname,length $CRCBCC);
  if ($bytes < length($CRCBCC)) {qprint "Bad CRC/BCC\n";}
  next if ($bytes < 1);

  $CRCBCC = calc_crc_bcc($PLCname, $RECVPKT);
  return undef unless defined $CRCBCC;
  if ($RECVCRCBCC eq $CRCBCC) {$done = 1;}
  else {
   printf("Received   $crc_bcc is 0x%04X\n",ord($RECVCRCBCC)) if ($PLC::debug);
   printf("Calculated $crc_bcc is 0x%04X\n",ord($CRCBCC)) if ($PLC::debug);
  }
 }} until ( ($done) || (--$tries == 0) );

 if (!$done) {
  $error = 1;
  $PLC::errorcode = 26; # RS232 read error
  return undef;
 }
 else {
  my $i = 0;
  print "Received Packet:\n" if ($PLC::debug);
  PLC::PrintBytes($RECVPKT) if ($PLC::debug);
  print "Sending DLE ACK\n" if ($PLC::debug);
  writeserial($PLCname,$DLE_ACK);
 }
 return $RECVPKT;
} # End of DF1sendpkt

} # end unless ($PLC::DisableSerialPortStuff)

LinuxRules; # End the module with something that is always true

# End of module DF1a

# End of PLC.pm
