#ifndef _FTVT_H_
#define _FTVT_H_
/*
 *      Copyright (C) 1997-2000 Claus-Justus Heine

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 * $RCSfile: ftvt.h,v $
 * $Revision: 1.8 $
 * $Date: 2000/07/22 18:53:01 $
 *
 *     This file contains the definitions for the volume table
 *     manipulation tool for the the QIC-40/80/3010/3020 floppy-tape
 *     driver "ftape" for Linux.
 */

#ifdef __cplusplus
extern "C" {
#endif

#include <linux/ftape.h>
#include <linux/zftape.h>

#define FTVT_SIZE 128 /* bytes */

/* The following are offsets in the vtbl.  */
#define FTVT_SIG   0
#define FTVT_START 4
#define FTVT_END   6
#define FTVT_DESC  8
#define FTVT_DATE  52
#define FTVT_FLAGS 56
#define FTVT_FL_VENDOR_SPECIFIC (1<<0)
#define FTVT_FL_MUTLI_CARTRIDGE (1<<1)
#define FTVT_FL_NOT_VERIFIED    (1<<2)
#define FTVT_FL_REDIR_INHIBIT   (1<<3)
#define FTVT_FL_SEG_SPANNING    (1<<4)
#define FTVT_FL_DIRECTORY_LAST  (1<<5)
#define FTVT_FL_RESERVED_6      (1<<6)
#define FTVT_FL_RESERVED_7      (1<<7)
#define FTVT_M_NO  57
#define FTVT_EXT   58
#define FTVT_EXT_ZFT_SIG     0
#define FTVT_EXT_ZFT_BLKSZ  10
#define FTVT_EXT_ZFT_CMAP   12
#define FTVT_EXT_ZFT_QIC113 13
#define FTVT_PWD   84
#define FTVT_DIR_SIZE 92
#define FTVT_DATA_SIZE 96
#define FTVT_OS_VERSION 104
#define FTVT_SRC_DRIVE  106
#define FTVT_DEV        122
#define FTVT_RESERVED_1 123
#define FTVT_CMPR       124
#define FTVT_CMPR_UNREG 0x3f
#define FTVT_CMPR_USED  0x80
#define FTVT_FMT        125
#define FTVT_RESERVED_2 126
#define FTVT_RESERVED_3 127
/* compatability with pre revision K */
#define FTVT_K_CMPR     120 

/*  the next is used by QIC-3020 tapes with format code 6 (>2^16
 *  segments) It is specified in QIC-113, Rev. G, Section 5 (SCSI
 *  volume table). The difference is simply, that we only store the
 *  number of segments used, not the starting segment. The starting
 *  and ending segment is stored at offset 76 and 80. The vendor
 *  extension data starts at 62.
 *
 */
#define VTBL_6_SEGS      4  /* is a 4 byte value */
#define VTBL_6_MAJOR     58 /* set to 113 */
#define VTBL_6_MINOR     60 /* Revision G, i.e. 7 */
#define VTBL_6_EXT       62 /* 14 bytes */
#define EXT_6_ZFT_SIG       0 /* signature, 3 bytes */
#define EXT_6_ZFT_SIGV      3 /* version, 1 byte */
#define EXT_6_ZFT_BLKSZ     4 /* block size, 2 bytes */
#define EXT_6_ZFT_RES       6 /* reserved, 8 bytes */
#define VTBL_6_START     76
#define VTBL_6_END       80

/*  one vtbl is 128 bytes, that results in a maximum number of
 *  29*1024/128 = 232 volumes.
 */
#define FTVT_MAX_VOLUMES (FT_SEGMENT_SIZE/FTVT_SIZE)
#define FTVT_IDS { "VTBL", "XTBL", "UTID", "EXVT" } /* valid ids */

/* zftape extension signature */
#define ZFT_SIG     "LINUX ZFT"  /* QIC-80/3010/3020 vendor extension */
#define ZFT_SIGLEN  10
#define ZFT_SIG2    "ZFT"        /* QIC-113 vendor extension */
#define ZFT_SIGLEN2 3

typedef struct zft_ext_qic80 {
	u_int8_t  sig[ZFT_SIGLEN];
	u_int16_t blksz;
	u_int8_t  cmap;
	u_int8_t  qic113;
	u_int8_t  reserved[26-ZFT_SIGLEN-2-1-1];
} zft_ext_qic80_t __attribute__ ((packed));

typedef struct zft_ext_qic113 {
	u_int8_t  sig[ZFT_SIGLEN2];
	u_int8_t  version;
	u_int16_t blksz;
	u_int8_t  reserved[8];
} zft_ext_qic113_t __attribute__ ((packed));

/*  for bsm map stuff
 */
#define FTVT_EMPTY_SEGMENT           (0xffffffff)

typedef struct ftvt_entry {
	u_int8_t sig[4];
	union {
		struct { u_int16_t start, end; } se __attribute__ ((packed));
		u_int32_t fmt6_segs;
	} size __attribute__ ((packed)); 
	u_int8_t  desc[44];
	u_int32_t date;
	u_int8_t vendor_specific:1;
	u_int8_t multi_cartridge:1;
	u_int8_t not_verified:1;
	u_int8_t inhibit_redirection:1;
	u_int8_t segment_spanning:1;
	u_int8_t directory_last:1;
	u_int8_t fl_reserved_6:1;
	u_int8_t fl_reserved_7:1;
	u_int8_t m_no;
	union {
		u_int8_t ext[26];
		struct {
			u_int16_t major;
			u_int16_t minor;
			u_int8_t  ext[14];
			u_int8_t start[4];
			u_int8_t end[4];
		} qic113 __attribute__ ((packed));
	} ext __attribute__ ((packed));
	u_int8_t pwd[8];
	u_int32_t dir_size;
	u_int64_t data_size;
	u_int16_t os_version;
	u_int8_t  source_drive[16];
	u_int8_t  device;
	u_int8_t  reserved_1;
	u_int8_t  cmpr;
	u_int8_t  format;
	u_int8_t  reserved_2;
	u_int8_t  reserved_3;
} ftvt_entry __attribute__ ((packed));

typedef struct ftvt {
	int num;
	int fmt_code;
	int start, end, num_segments;
	unsigned int modified:1;
	unsigned int zft_bug:1;
	ftvt_entry entry;
#if 0
	u_int8_t   bytes[128];
#else
	u_int8_t   *bytes;
#endif
} ftvt;

#define ftvt_start     entry.size.se.start
#define ftvt_6_start   entry.ext.qic113.start
#define ftvt_end       entry.size.se.end
#define ftvt_6_end     entry.ext.qic113.end
#define ftvt_space     entry.size.fmt6_segs
#define ftvt_sig       entry.sig
#define ftvt_desc      entry.desc
#define ftvt_date      entry.date

/* volume table entry extension "XTBL" */

/* offsets within the volume table entry */
#define XTBL_SIG    0 /* signature, "XTBL", 4 bytes */
#define XTBL_NAME   4 /* volume name, unicode, 88 bytes */
#define XTBL_PWD   92 /* password, unicode, 16 bytes */
#define XTBL_RES  108 /* reserved, 20 bytes, set to 0 */

/* volume table extension for unicode tape names */

/* offsets within the volume table entry */
#define UTID_SIG    0 /* signature, "UTID", 4 bytes */
#define UTID_NAME   4 /* tape name, unicode, 88 bytes, space filled */
#define UTID_RES   92 /* reserved, 36 bytes, set to 0 */

/* volume table overflow extension */

/* offsets within the volume table entry */
#define EXVT_SIG    0 /* signature, "EXVT", 4 bytes */
#define EXVT_PARENT 4 /* parent segment, 2 bytes, little endian */
#define EXVT_CHILD  6 /* child segment, 2 bytes, little endian */
#define EXVT_RES    8 /* the standard says this is 119 bytes. So what
					   * happened to the last byte? :-)
					   */

#define FTAPE_DEF_TAPE "/dev/rawft0"

/*  Control flags
 */
#define FTVT_VERBOSE (1 << 0)
#define FTVT_TAGGED  (1 << 1)

#define NR_ITEMS(x)     (int)(sizeof(x)/ sizeof(*x))

/* This is return by ftvt_get_ctrl
 */
typedef struct ftvt_ctrl {
	const char *version;
	unsigned long flags;
	int fmt_code;
	int first_seg;
	int last_seg;
	int max_volumes;
	const char *errstr;
    const char *drivetype;
} ftvt_ctrl;

/*  format code 4 bad sector entry, ftape uses this
 *  internally for all format codes
 */
typedef u_int32_t SectorMap;

/* exported functions
 */

extern int ftvt_read_header_segment(int tape_fd, u_int8_t *hseg);
extern void ftvt_set_ctrl(unsigned long flags);
extern const struct ftvt_ctrl *ftvt_get_ctrl(void);
extern int ftvt_open(const char *name, mode_t mode);
extern int ftvt_close(int tape_fd);
extern int ftvt_read(int tape_fd, ftvt *volumes, u_int8_t *buffer);
extern int ftvt_read_vtbl(int tape_fd, u_int8_t *hseg, int fmt_code,
						  ftvt *volumes, u_int8_t *buffer);
extern int ftvt_write(int tape_fd, const ftvt *volumes, u_int8_t *buffer,
					  int vtbl_cnt, int do_write);
extern void ftvt_print(const ftvt *volumes, int maxnum);
extern void ftvt_print_one(const ftvt *volume);
extern void ftvt_print_one_tagged(const ftvt *volume);
extern int ftvt_parse_tagged(ftvt *volumes);
extern int ftvt_add_volume(ftvt *volumes, int num_volumes);
extern char *ftvt_decode_date(u_int32_t timestamp);
extern int ftvt_set_date(ftvt *volumes, int maxnum,
						 const char *date, int vtbl_no);
extern int ftvt_set_label(ftvt *volumes, int maxnum,
						  const char *desc,int vtbl_no);
extern int ftvt_set_bounds(ftvt *volumes, int maxnum, 
						   int start, int end, int vtbl_no);
extern int ftvt_set_id(ftvt *volumes, int maxnum, const char *id, int vtbl_no);

/* bsm functions */ 
void      ftvt_init_bsm(u_int8_t * buffer, ft_format_type code, int erase);
void      ftvt_put_bsm_entry(int segment_id, SectorMap new_map);
SectorMap ftvt_get_bsm_entry(int segment_id);
void      ftvt_print_bad_sector_map(int verbose);

#ifdef __cplusplus
}
#endif

#endif _FTVT_H_

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
