/*
 *      Copyright (C) 1997 Claus-Justus Heine

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 *      This program contains the user level floppy tape formatting
 *      stuff for the QIC-40/80/3010/3020 floppy-tape driver "ftape"
 *      for Linux.
 */

char src[] = "$RCSfile: ftformat.c,v $";
char rev[] = "$Revision: 1.9 $";
char dat[] = "$Date: 1999/03/17 11:25:42 $";

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <locale.h>
#include <libintl.h>
#define _(String) gettext (String)

#include <linux/ftape.h>
#include <linux/ftape-header-segment.h>

#include "ftformat.h"
#include "ftfmt-options.h"
#include "ftfmt-bsm.h"
#include "ftfmt-tapelib.h"

int main(int argc, char *argv[])
{
    struct opt_parms opts;
    ftfmt_tpparms_t tpparms;
    int tape_fd = -1;
    void *dma_buffer = NULL;
    u_int8_t hseg[FT_SEGMENT_SIZE];

    setlocale (LC_ALL, "");
    bindtextdomain (PACKAGE, LOCALEDIR);
    textdomain (PACKAGE);

    /*  These two shouldn't fail, I think.
     */
    fflush(stdout);
    (void)setvbuf(stdout, NULL, _IONBF, 0);
    fflush(stderr);
    (void)setvbuf(stderr, NULL, _IONBF, 0); /* shouldn't be needed,
					     * stderr should
					     * be unbuffered, AFAIK
					     */

    /* parse options will abort if there are problems
     */
    parse_options(argc, argv, &opts, &tpparms.fmt_parms);

    tape_fd = tape_open(opts.tape_dev, &opts, &tpparms);
    if (tape_fd == -1) {
	exit(1);
    }
    if (check_api_version(tape_fd, &opts) == -1) {
	tape_close(tape_fd, dma_buffer, opts.dma_size);
	exit(1);
    }
    if (!(tpparms.drive_status & QIC_STATUS_REFERENCED)) {
	if (!opts.write_bursts) {
	    fprintf(stderr,
		    _("Cannot omit writing of the reference bursts "
		      "with an unformatted cartridge!\n")
		);
	    tape_close(tape_fd, dma_buffer, opts.dma_size);
	    exit(1);
	}
	opts.read_hseg = opts.erase = 0; /* wouldn't work anyway */
    }
    /* read header segment now. If this tape drive can't format
     * cartridges, then get_format_parms() will use the contents of
     * hseg rather then the system ioctl call. Also, it will verify
     * the probed parameters with the parameters found in hseg.
     */
    if (opts.read_hseg) {
	if (read_header_segment(tape_fd, &tpparms, hseg)) {
	    tape_close(tape_fd, dma_buffer, opts.dma_size);
	    exit (1);
	}
    } else {
	memset(hseg, 0, FT_SEGMENT_SIZE);
	hseg[FT_FMT_ERROR] = 0xff; /* flag as invalid */
    }
    if (opts.mode != FORCE) {
	if (!opts.do_format) { /* verify only */
	    if (get_format_parms(tape_fd,
				 opts.read_hseg ? hseg : NULL, &tpparms)) {
		tape_close(tape_fd, dma_buffer, opts.dma_size);
		exit(1);
	    }
	    /* get_parms() doesn't set QIC_WIDE as the kernel driver
	     * doesn't remember this flag as it doesn't need it
	     */
	    tpparms.fmt_parms.ft_qicstd |= tpparms.tape_status & QIC_TAPE_WIDE;
	} else if (autodetect(tape_fd, &opts, &tpparms)) {
	    tape_close(tape_fd, dma_buffer, opts.dma_size);
	    exit(1);
	}
    }
    report_format_parms(stdout, &tpparms.fmt_parms);

    /* initialize the bsm library. Don't erase the bad sector map if
     * we aren't actually formatting the cartridge.
     */
    init_bsm(hseg, tpparms.fmt_parms.ft_fmtcode, opts.keepbsm == 0);

    if (opts.mode == PROBING) {
	if (opts.print_hseg) {
	    printf(_("Dump of header segment:\n\n"));
	    print_header_segment(hseg, &tpparms);
	    printf("\nDump of bad sector map:\n\n");
	    print_bad_sector_map(&tpparms, 1 /* verbose */ );
	}
	tape_close(tape_fd, dma_buffer, opts.dma_size);
	exit (0);
    }

    if (opts.erase && erase_cartridge(tape_fd,
				      &opts, tpparms.fmt_parms.ft_tpc)) {
	tape_close(tape_fd, dma_buffer, opts.dma_size);
	exit (1);
    }

    if (opts.do_format) {

	/* We actually want to format the tape drive, let's map 
	 * in the dma buffers!
	 */
	if ((tape_fd = tape_reopen_rw(opts.tape_dev, tape_fd)) == -1) {
	    (void)close(tape_fd);
	    return -1;
	}
	if (set_format_parms(tape_fd, &tpparms)) {
	    tape_close(tape_fd, dma_buffer, opts.dma_size);
	    exit (1);
	}
	if ((dma_buffer = tape_mmap(tape_fd, opts.dma_size)) == NULL) {
	    (void)close(tape_fd);
	    return -1;
	}
	if (opts.write_bursts) {
	    /* the point of no return:
	     */
	    if (write_reference_burst(tape_fd)) {
		tape_close(tape_fd, dma_buffer, opts.dma_size);
		exit (1);
	    }
	}
	if (format_cartridge(tape_fd, dma_buffer, opts.dma_size,
			     &tpparms.fmt_parms, &opts)) {
	    tape_close(tape_fd, dma_buffer, opts.dma_size);
	    exit (1);
	}
    }
    /* Need to open it a second time to clear the mmaped status of the
     * dma buffers. The driver can't handle this yet properly via
     * munmap()
     */
    if ((opts.write_hseg || opts.zero_tape) &&
	(tape_fd = tape_reopen_rw(opts.tape_dev, tape_fd)) == -1) {
	(void)close(tape_fd);
	return -1;
    }

    if (verify_cartridge(tape_fd, &tpparms, &opts, hseg)) {
	tape_close(tape_fd, dma_buffer, opts.dma_size);
	exit (1);
    }

    if (compose_header_segment(hseg, opts.label, &tpparms)) {
	tape_close(tape_fd, dma_buffer, opts.dma_size);
	exit (1);
    }

    if (opts.write_hseg && write_header_segments(tape_fd, hseg, &tpparms)) {
	tape_close(tape_fd, dma_buffer, opts.dma_size);
	exit (1);
    }

    tape_close(tape_fd, dma_buffer, opts.dma_size);

    if (opts.print_hseg) {
	printf(_("Dump of header segment:\n\n"));
	print_header_segment(hseg, &tpparms);
	printf("\nDump of bad sector map:\n\n");
	print_bad_sector_map(&tpparms, 1 /* verbose */ );
    }    

    printf(_("Done formatting.\n"
	     "Tracks  : %8d\n"
	     "Segments: %8d\n"
	     "Capacity: %8ldM\n"),
	   tpparms.fmt_parms.ft_tpc,
	   tpparms.fmt_parms.ft_spt,
	   tape_capacity(&tpparms.fmt_parms)>>10);

    exit (0);
}

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-basic-offset: 4
 * End:
 */
