/*
 *      Copyright (C) 1994-1996 Bas Laarhoven,
 *                (C) 1996-1997 Claus Heine.

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 * $RCSfile: ftfmt-bsm.c,v $
 * $Revision: 1.5 $
 * $Date: 1999/03/17 11:25:42 $
 *
 *      This file contains the bad-sector map handling code for
 *      the QIC-117 floppy tape driver for Linux.
 *      QIC-40, QIC-80, QIC-3010 and QIC-3020 maps are implemented.
 * 
 *      This is a clone of drivers/ftape/ftape-bsm.c      
 */

#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <libintl.h>
#define _(String) gettext (String)

#include <assert.h>

#include <asm/types.h>
#include <linux/ftape.h>
#include <linux/ftape-header-segment.h>

#include "ftformat.h"
#include "ftfmt-options.h"
#include "ftfmt-bsm.h"

/*  maximum (format code 4) bad sector map size (bytes).
 */
#define BAD_SECTOR_MAP_SIZE     (FT_SEGMENT_SIZE - FT_HEADER_END)

/*  variable and 1100 ft bad sector map entry. These three bytes represent
 *  a single sector address measured from BOT. 
 */
typedef struct NewSectorMap {          
    u_int8_t bytes[3];
} SectorCount __attribute__((packed));

static u_int8_t       *bad_sector_map = NULL;
static u_int8_t       *bad_sector_end = NULL;
static SectorCount    *bsm_hash_ptr   = NULL;
static ft_format_type  format_code    = fmt_normal;

static inline void put_sector(SectorCount *ptr, unsigned int sector)
{
    ptr->bytes[0] = sector & 0xff;
    sector >>= 8;
    ptr->bytes[1] = sector & 0xff;
    sector >>= 8;
    ptr->bytes[2] = sector & 0xff;
}

static inline unsigned int get_sector(SectorCount *ptr)
{
    unsigned int sector;

    sector  = ptr->bytes[0];
    sector += ptr->bytes[1] <<  8;
    sector += ptr->bytes[2] << 16;

    return sector;
}

void init_bsm(u_int8_t * buffer, ft_format_type code, int erase)
{

    format_code = code;
    if (erase) {
	memset(buffer + FT_HEADER_END, 0, FT_SEGMENT_SIZE - FT_HEADER_END);
    }

    bad_sector_end = buffer + FT_SEGMENT_SIZE;

    if (format_code == fmt_var || format_code == fmt_big) {
	/* QIC-3010/3020 and wide QIC-80 tapes no longer have a failed
	 * sector log but use this area to extend the bad sector map.
	 */
	bad_sector_map = &buffer[FT_HEADER_END];
    } else {
	/* non-wide QIC-80 tapes have a failed sector log area that
	 * mustn't be included in the bad sector map.
	 */
	bad_sector_map = &buffer[FT_FSL + FT_FSL_SIZE];
    }
    if (format_code == fmt_1100ft || 
	format_code == fmt_var    ||
	format_code == fmt_big) {
	bsm_hash_ptr = (SectorCount *)bad_sector_map;
    } else {
	bsm_hash_ptr = NULL;
    }
}

static inline SectorMap cvt2map(unsigned int sector)
{
    return 1 << (((sector & 0x7fffff) - 1) % FT_SECTORS_PER_SEGMENT);
}

static inline int cvt2segment(unsigned int sector)
{
    return ((sector & 0x7fffff) - 1) / FT_SECTORS_PER_SEGMENT;
}

static int forward_seek_entry(int segment_id, 
			      SectorCount **ptr, 
			      SectorMap *map)
{
    unsigned int sector;
    int segment;

    do {
	sector = get_sector((*ptr)++);
	segment = cvt2segment(sector);
    } while (sector != 0 && segment < segment_id);
    (*ptr) --; /* point to first sector >= segment_id */

    assert((u_int8_t *)(*ptr) < bad_sector_end);

    /*  Get all sectors in segment_id
     */
    if (sector == 0 || segment != segment_id) {
	*map = 0;
	return 0;
    } else if ((sector & 0x800000) &&
	       (format_code == fmt_var || format_code == fmt_big)) {
	*map = EMPTY_SEGMENT;
	return FT_SECTORS_PER_SEGMENT;
    } else {
	int count = 1;
	SectorCount *tmp_ptr = (*ptr) + 1;

	assert((u_int8_t *)tmp_ptr < bad_sector_end);
		
	*map = cvt2map(sector);
	while ((sector = get_sector(tmp_ptr++)) != 0 &&
	       (segment = cvt2segment(sector)) == segment_id) {
	    *map |= cvt2map(sector);
	    ++count;
	}
	return count;
    }
}

static int backwards_seek_entry(int segment_id,
				SectorCount **ptr,
				SectorMap *map)
{
    unsigned int sector;
    int segment; /* max unsigned int */

    if (*ptr <= (SectorCount *)bad_sector_map) {
	*map = 0;
	return 0;
    }
    do {
	sector  = get_sector(--(*ptr));
	segment = cvt2segment(sector);
    } while (*ptr > (SectorCount *)bad_sector_map && segment > segment_id);
    if (segment > segment_id) { /*  at start of list, no entry found */
	*map = 0;
	return 0;
    } else if (segment < segment_id) {
	/*  before smaller entry, adjust for overshoot */
	(*ptr) ++;
	*map = 0;
	return 0;
    } else if ((sector & 0x800000) &&
	       (format_code == fmt_big || format_code == fmt_var)) {
	*map = EMPTY_SEGMENT;
	return FT_SECTORS_PER_SEGMENT;
    } else { /*  get all sectors in segment_id */
	int count = 1;

	*map = cvt2map(sector);
	while(*ptr > (SectorCount *)bad_sector_map) {
	    sector = get_sector(--(*ptr));
	    segment = cvt2segment(sector);
	    if (segment != segment_id) {
		break;
	    }
	    *map |= cvt2map(sector);
	    ++count;
	}
	if (segment < segment_id) {
	    (*ptr) ++;
	}
	return count;
    }
}

/* Check for buffer overflows in case there are too many bad sectors!!!
 */
void put_bsm_entry(int segment_id, SectorMap new_map)
{
    SectorCount *ptr = (SectorCount *)bad_sector_map;
    int count;
    int new_count;
    SectorMap map;

    if (bsm_hash_ptr == NULL) {
	((SectorMap *) bad_sector_map)[segment_id] = new_map;
	return;
    }

    count = forward_seek_entry(segment_id, &ptr, &map);
    new_count = count_ones(new_map);
    /* If format code == 4 put empty segment instead of 32
     * bad sectors.
     */
    if (format_code == fmt_var || format_code == fmt_big) {
	if (new_count == FT_SECTORS_PER_SEGMENT) {
	    new_count = 1;
	}
	if (count == FT_SECTORS_PER_SEGMENT) {
	    count = 1;
	}
    }
    if (count != new_count) {
	/* insert (or delete if < 0) new_count - count
	 * entries.  Move trailing part of list
	 * including terminating 0.
	 */
	SectorCount *hi_ptr = ptr;
	size_t gap;

	do {
	} while (get_sector(hi_ptr++) != 0);
	/* check for buffer overflow. If there are too many
	 * bad sectors, then exit with an error message.
	 */
	gap = (size_t)(hi_ptr - (ptr + count))*sizeof(SectorCount);
	if (gap < 0) {
	    fprintf(stderr,
		    _("Too many bad sectors, "
		      "no space left in the bad sector map"));
	    exit(1);
	} else {
	    /*  Note: ptr is of type byte *, and each bad
	     *  sector consumes 3 bytes.
	     */

	    assert((u_int8_t *)ptr < bad_sector_end);

	    memmove(ptr + new_count, ptr + count, gap);
	}
    }

    assert((u_int8_t *)ptr < bad_sector_end);

    if (new_count == 1 && new_map == EMPTY_SEGMENT) {
	put_sector(ptr++, (0x800001 + 
			   segment_id * FT_SECTORS_PER_SEGMENT));
    } else {
	int i = 0;

	while (new_map) {
	    if (new_map & 1) {
		put_sector(ptr++, 
			   1 + segment_id * 
			   FT_SECTORS_PER_SEGMENT + i);
	    }
	    ++i;
	    new_map >>= 1;
	}
    }
}

SectorMap get_bsm_entry(int segment_id)
{
    static int last_reference = -1;
    static SectorMap map = 0;
	
    if (bsm_hash_ptr == NULL) {
	return ((SectorMap *) bad_sector_map)[segment_id];
    }
    /*  Invariants:
     *    map - mask value returned on last call.
     *    bsm_hash_ptr - points to first sector greater or equal to
     *          first sector in last_referenced segment.
     *    last_referenced - segment id used in the last call,
     *                      sector and map belong to this id.
     *  This code is designed for sequential access and retries.
     *  For true random access it may have to be redesigned.
     */
    if (segment_id > last_reference) {
	/*  Skip all sectors before segment_id
	 */
	forward_seek_entry(segment_id, &bsm_hash_ptr, &map);
    } else if (segment_id < last_reference) {
	/* Skip backwards until begin of buffer or
	 * first sector in segment_id 
	 */
	backwards_seek_entry(segment_id, &bsm_hash_ptr, &map);
    }		/* segment_id == last_reference : keep map */
    last_reference = segment_id;
    return map;
}

void print_bad_sector_map(const ftfmt_tpparms_t *tpparms, int verbose)
{
    unsigned int good_sectors;
    unsigned int total_bad = 0;
    SectorMap map;
    int i;

    /* we use our helper function to print the bad sector map on a
     * segment-by-segment basis
     */
    for (i=tpparms->frst_data_segment; i <= tpparms->last_data_segment; ++i) {

	map = get_bsm_entry(i);
	
	if (map) {
	    if (verbose) {
		printf("bsm[%4d] = 0x%08x\n", i, (unsigned int)map);
	    }
	    total_bad += ((map == EMPTY_SEGMENT)
			  ? FT_SECTORS_PER_SEGMENT - 3
			  : count_ones(map));
	}
    }
    good_sectors = ((1 + 
		     tpparms->last_data_segment - tpparms->frst_data_segment)
		    * (FT_SECTORS_PER_SEGMENT - 3)) - total_bad;
    printf("\ntotal number of sectors: %d\n", good_sectors + total_bad);
    printf("number of bad sectors  : %d (%2.2f%%)\n", total_bad,
	   (double)total_bad/(double)(good_sectors + total_bad)*100.0);
    printf("number of good sectors : %d (%2.2f%%)\n", good_sectors,
	   (double)good_sectors/(double)(good_sectors + total_bad)*100.0);    
}

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-basic-offset: 4
 * End:
 */
