/* File: ip_big_brother.c
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/stat.h>
#include <time.h>

#define CONFIGURATIONFILE         "ip_big_brother.conf"
#define ALTCONFIGURATIONFILE      "/etc/ip_big_brother.conf"

/*****************************************************************************/
/* Typedefs
*/

typedef struct free_day {
  int day;
  int month;
  int weekday;
  struct free_day *next;
} free_day;

typedef struct weight {
  char *variable;
  char *r_value;
  struct weight *next;
} weight;

typedef struct assignment {
  weight *variable;
  weight *r_value;
  struct assignment *next;
} assignment;

typedef struct ip_class {
  char *class_name;
  struct assignment *assignments;
  struct ip_class *next;
} ip_class;

typedef struct tax_interval {
  int hour_start;
  int hour_end;
  ip_class *tax_class;
  struct tax_interval *next;
} tax_interval;

typedef struct network {
  char *address;
  char *mask;
  weight *net_weight;
  struct network *next;
} network;

/*****************************************************************************/
/* Globals
*/

free_day 	*free_day_head = (free_day *) 0;
ip_class 	*ip_class_head = (ip_class *) 0;
tax_interval 	*tax_interval_head = (tax_interval *) 0;
weight 		*weight_head = (weight *) 0;
network		*network_head = (network *) 0;
ip_class	*free_ip_class = (ip_class *) 0;

char 		 logFile[1024] = "/usr/adm/ipacctlog";
char 		 acctFile[1024] = "/etc/ip_acct_user";
char 		 updateCommand[1024] = "/usr/local/sbin/ipacct -a";

/*****************************************************************************/
/* Purpose: Log events...
*/

void RegisterLog(char *message) {
  struct tm *timeStruct;
  time_t timeNow;
  FILE *log;

  if ((log = fopen(logFile,"a")) == NULL)
    return;
  timeNow = time(NULL);
  timeStruct = localtime(&timeNow);
  fprintf(log,"%04d/%02d/%02d, %02d:%02d:%02d: %s\n",timeStruct->tm_year + 1900,timeStruct->tm_mon +
1,timeStruct->tm_mday,timeStruct->tm_hour,timeStruct->tm_min,timeStruct->tm_sec,message);
  fclose(log);
}

/*****************************************************************************/
/* Purpose: Parse a command line
*/

int ParseCommandLine(char *inBuff,char *argv[],int maxArg) {
  char *tmp;
  int index,argc;

  for(index = strlen(inBuff),argc = 0,tmp = inBuff;index && (argc <
maxArg);tmp = inBuff) {
    for(argv[argc++] = tmp;!isspace(*inBuff) && index;inBuff++,index--);
    *(inBuff++) = (char) 0;
    if (!index || !(--index) || (*inBuff == '\n') || (*inBuff == '\r'))
      break;
    for(;isspace(*inBuff) && index && (*inBuff != '\n') && (*inBuff !=
'\r');inBuff++,index--);
    if (!index || (*inBuff == '\n') || (*inBuff == '\r'))
      break;
  }
  argv[argc] = (char *) 0;
  return argc;
}

/*****************************************************************************/
/* Purpose: Verify if it's a number
*/

int IsANumber(char *aNumber) {
  char *tmp;

  for(tmp = aNumber;*tmp;tmp++)
    if (!isdigit(*tmp))
      return 0;
  return 1;
}

/*****************************************************************************/
/* Purpose: Read configuration file
*/

int ReadConfigurationFile(char *filename) {
  char buffer[4096],*tmp,*args[33];
  free_day     *free_day_ptr;
  ip_class     *ip_class_ptr;
  assignment   *assignment_ptr;
  tax_interval *tax_interval_ptr;
  weight       *weight_ptr,*weight_ptr1;
  network      *network_ptr,*network_ptr1;
  FILE *myFile;
  int nArgs,line,index;

  if ((myFile = fopen(filename,"rb")) == NULL)
    return -1;
  for(line = 1;fgets(buffer,4096,myFile);line++) {
    for(tmp = buffer + strlen(buffer) - 1;(tmp >= buffer) &&
isspace(*tmp);tmp--)
      *tmp = (char) 0;
    nArgs = ParseCommandLine(buffer,args,32);
    if ((nArgs > 0) && (args[0][0] != '#')) {
      if (!strcasecmp("FreeIpDay",args[0])) {
        if ((nArgs != 3) || !IsANumber(args[1]) || !IsANumber(args[2]))
          fprintf(stderr,"ip_big_brother, line %d: (warning) invalid syntax
in option FreeIpDay\n",line);
        else {
          free_day_ptr = (free_day *) malloc(sizeof(free_day));
          free_day_ptr->day = atoi(args[2]);
          free_day_ptr->month = atoi(args[1]) - 1;
          free_day_ptr->weekday = -1;
          free_day_ptr->next = free_day_head;
          free_day_head = free_day_ptr;
        }
      } else if (!strcasecmp("FreeIpAtSunday",args[0])) {
        if (nArgs != 1)
          fprintf(stderr,"ip_big_brother, line %d: (warning) invalid syntax
in option FreeIpAtSunday\n",line);
        else {
          free_day_ptr = (free_day *) malloc(sizeof(free_day));
          free_day_ptr->day = -1;
          free_day_ptr->month = -1;
          free_day_ptr->weekday = 0;
          free_day_ptr->next = free_day_head;
          free_day_head = free_day_ptr;
        }
      } else if (!strcasecmp("FreeIpAtMonday",args[0])) {
        if (nArgs != 1)
          fprintf(stderr,"ip_big_brother, line %d: (warning) invalid syntax
in option FreeIpAtMonday\n",line);
        else {
          free_day_ptr = (free_day *) malloc(sizeof(free_day));
          free_day_ptr->day = -1;
          free_day_ptr->month = -1;
          free_day_ptr->weekday = 1;
          free_day_ptr->next = free_day_head;
          free_day_head = free_day_ptr;
        }
      } else if (!strcasecmp("FreeIpAtTuesday",args[0])) {
        if (nArgs != 1)
          fprintf(stderr,"ip_big_brother, line %d: (warning) invalid syntax
in option FreeIpAtTuesday\n",line);
        else {
          free_day_ptr = (free_day *) malloc(sizeof(free_day));
          free_day_ptr->day = -1;
          free_day_ptr->month = -1;
          free_day_ptr->weekday = 2;
          free_day_ptr->next = free_day_head;
          free_day_head = free_day_ptr;
        }
      } else if (!strcasecmp("FreeIpAtWednesday",args[0])) {
        if (nArgs != 1)
          fprintf(stderr,"ip_big_brother, line %d: (warning) invalid syntax
in option FreeIpAtWednesday\n",line);
        else {
          free_day_ptr = (free_day *) malloc(sizeof(free_day));
          free_day_ptr->day = -1;
          free_day_ptr->month = -1;
          free_day_ptr->weekday = 3;
          free_day_ptr->next = free_day_head;
          free_day_head = free_day_ptr;
        }
      } else if (!strcasecmp("FreeIpAtThursday",args[0])) {
        if (nArgs != 1)
          fprintf(stderr,"ip_big_brother, line %d: (warning) invalid syntax
in option FreeIpAtThursday\n",line);
        else {
          free_day_ptr = (free_day *) malloc(sizeof(free_day));
          free_day_ptr->day = -1;
          free_day_ptr->month = -1;
          free_day_ptr->weekday = 4;
          free_day_ptr->next = free_day_head;
          free_day_head = free_day_ptr;
        }
      } else if (!strcasecmp("FreeIpAtFriday",args[0])) {
        if (nArgs != 1)
          fprintf(stderr,"ip_big_brother, line %d: (warning) invalid syntax
in option FreeIpAtFriday\n",line);
        else {
          free_day_ptr = (free_day *) malloc(sizeof(free_day));
          free_day_ptr->day = -1;
          free_day_ptr->month = -1;
          free_day_ptr->weekday = 5;
          free_day_ptr->next = free_day_head;
          free_day_head = free_day_ptr;
        }
      } else if (!strcasecmp("FreeIpAtSaturday",args[0])) {
        if (nArgs != 1)
          fprintf(stderr,"ip_big_brother, line %d: (warning) invalid syntax
in option FreeIpAtSaturday\n",line);
        else {
          free_day_ptr = (free_day *) malloc(sizeof(free_day));
          free_day_ptr->day = -1;
          free_day_ptr->month = -1;
          free_day_ptr->weekday = 6;
          free_day_ptr->next = free_day_head;
          free_day_head = free_day_ptr;
        }
      } else if (!strcasecmp("Class",args[0])) {
        if (nArgs < 3)
          fprintf(stderr,"ip_big_brother, line %d: (warning) invalid syntax
in option Class\n",line);
        else {
          for(ip_class_ptr = ip_class_head;ip_class_ptr;ip_class_ptr =
ip_class_ptr->next)
            if (!strcasecmp(args[1],ip_class_ptr->class_name))
              break;
          if (ip_class_ptr)
            fprintf(stderr,"ip_big_brother, line %d: (warning) class %s
already defined in option Class\n",line,args[1]);
          else {
            ip_class_ptr = (ip_class *) malloc(sizeof(ip_class));
            ip_class_ptr->class_name = strdup(args[1]);
            ip_class_ptr->assignments = (assignment *) 0;
            for(index = 2;index < nArgs;index++)
              if (!(tmp = strchr(args[index],'=')))
                fprintf(stderr,"ip_big_brother, line %d: (warning) invalid
assignment %s in option Class\n",line,args[index]);
              else {
                *tmp = (char) 0;
                for(assignment_ptr =
ip_class_ptr->assignments;assignment_ptr;assignment_ptr = assignment_ptr->next)
                  if
(!strcasecmp(args[index],assignment_ptr->variable->variable))
                    break;
                if (assignment_ptr)
                  fprintf(stderr,"ip_big_brother, line %d: (warning)
assignment %s already exists in option Class\n",line,args[index]);
                else {
                  for(weight_ptr = weight_head;weight_ptr;weight_ptr =
weight_ptr->next)
                    if (!strcasecmp(args[index],weight_ptr->variable))
                      break;
                  if (!weight_ptr)
                    fprintf(stderr,"ip_big_brother, line %d: (warning)
assignment weight %s not defined in option Class\n",line,args[index]);
                  else if (weight_ptr->r_value)
                    fprintf(stderr,"ip_big_brother, line %d: (warning)
assignment weight %s is not variable in option Class\n",line,args[index]);
                  else if (*(tmp + 1) != '$') 
                    fprintf(stderr,"ip_big_brother, line %d: (warning)
invalid assigned weight %s in option Class\n",line,tmp + 1);
                  else {
                    for(weight_ptr1 = weight_head;weight_ptr1;weight_ptr1 =
weight_ptr1->next)
                      if (!strcasecmp(tmp + 2,weight_ptr1->variable))
                        break;
                    if (!weight_ptr1)
                      fprintf(stderr,"ip_big_brother, line %d: (warning)
assigned weight %s not defined in option Class\n",line,tmp + 2);
                    else if (!weight_ptr1->r_value)
                      fprintf(stderr,"ip_big_brother, line %d: (warning)
assigned weight %s is not constant in option Class\n",line,tmp + 2);
                    else  {
                      assignment_ptr = (assignment *)
malloc(sizeof(assignment));
                      assignment_ptr->variable = weight_ptr;
                      assignment_ptr->r_value = weight_ptr1;
                      assignment_ptr->next = ip_class_ptr->assignments;
                      ip_class_ptr->assignments = assignment_ptr;
                    }
                  }
                }
              }
            ip_class_ptr->next = ip_class_head;
            ip_class_head = ip_class_ptr;
          }
        }
      } else if (!strcasecmp("IpTaxationInterval",args[0])) {
        if ((nArgs != 4) || !IsANumber(args[1]) || !IsANumber(args[2]))
          fprintf(stderr,"ip_big_brother, line %d: (warning) invalid syntax
in option IpTaxationInterval\n",line);
        else {
          for(ip_class_ptr = ip_class_head;ip_class_ptr;ip_class_ptr =
ip_class_ptr->next)
            if (!strcasecmp(args[3],ip_class_ptr->class_name))
              break;
          if (!ip_class_ptr)
            fprintf(stderr,"ip_big_brother, line %d: (warning) class %s not
found in TaxationInterval\n",line,args[3]);
          else {
            tax_interval_ptr = (tax_interval *) malloc(sizeof(tax_interval));
            tax_interval_ptr->hour_start = atoi(args[1]);
            tax_interval_ptr->hour_end = atoi(args[2]);
            tax_interval_ptr->tax_class = ip_class_ptr;
            tax_interval_ptr->next = tax_interval_head;
            tax_interval_head = tax_interval_ptr;
          }
        }
      } else if (!strcasecmp("Weight",args[0])) {
        if (nArgs != 3)
          fprintf(stderr,"ip_big_brother, line %d: (warning) invalid syntax
in option Weight\n",line);
        else {
          for(weight_ptr = weight_head;weight_ptr;weight_ptr = weight_ptr->next)
            if (!strcasecmp(args[1],weight_ptr->variable))
              break;
          if (weight_ptr)
            fprintf(stderr,"ip_big_brother, line %d: (warning) weight %s
already defined in option Weight\n",line,args[1]);
          else {
            weight_ptr = (weight *) malloc(sizeof(weight));
            weight_ptr->variable = strdup(args[1]);
            weight_ptr->r_value = !strcmp(args[2],"?") ? (char *) 0 :
strdup(args[2]);
            weight_ptr->next = weight_head;
            weight_head = weight_ptr;
          }
        }
      } else if (!strcasecmp("Network",args[0])) {
        if (nArgs != 4)
          fprintf(stderr,"ip_big_brother, line %d: (warning) invalid syntax
in option Network\n",line);
        else {
          if (*args[3] != '$')
            fprintf(stderr,"ip_big_brother, line %d: (warning) invalid
weight %s in option Network\n",line,args[3]);
          else {
            for(weight_ptr = weight_head;weight_ptr;weight_ptr =
weight_ptr->next)
              if (!strcasecmp(args[3] + 1,weight_ptr->variable))
                break;
            if (!weight_ptr)
              fprintf(stderr,"ip_big_brother, line %d: (warning) weight %s
not defined in option Network\n",line,args[3] + 1);
            else {
              network_ptr = (network *) malloc(sizeof(network));
              network_ptr->address = strdup(args[1]);
              network_ptr->mask = strdup(args[2]);
              network_ptr->net_weight = weight_ptr;
              if (!network_head) {
                network_ptr->next = network_head;
                network_head = network_ptr;
              } else {
                for(network_ptr1 =
network_head;network_ptr1->next;network_ptr1 = network_ptr1->next);
                network_ptr->next = (network *) 0;
                network_ptr1->next = network_ptr;
              }
            }
          }
        }
      } else if (!strcasecmp("FreeIpClass",args[0])) {
        if (nArgs != 2)
          fprintf(stderr,"ip_big_brother, line %d: (warning) invalid syntax
in option FreeIpClass\n",line);
        else {
          for(ip_class_ptr = ip_class_head;ip_class_ptr;ip_class_ptr =
ip_class_ptr->next)
            if (!strcasecmp(args[1],ip_class_ptr->class_name))
              break;
          if (!ip_class_ptr)
            fprintf(stderr,"ip_big_brother, line %d: (warning) class %s not
defined in option FreeIpClass\n",line,args[1]);
          else
            free_ip_class = ip_class_ptr;
        }
      } else if (!strcasecmp("LogFile",args[0])) {
        if (nArgs != 2)
          fprintf(stderr,"ip_big_brother, line %d: (warning) invalid syntax
in option LogFile\n",line);
        else {
          strcpy(logFile,args[1]);
        }
      } else if (!strcasecmp("AcctFile",args[0])) {
        if (nArgs != 2)
          fprintf(stderr,"ip_big_brother, line %d: (warning) invalid syntax
in option AcctFile\n",line);
        else {
          strcpy(acctFile,args[1]);
        }
      } else if (!strcasecmp("UpdateCommand",args[0])) {
        if (nArgs < 2)
          fprintf(stderr,"ip_big_brother, line %d: (warning) invalid syntax
in option FlushCommand\n",line);
        else {
          strcpy(updateCommand,args[1]);
          for(index = 2;index < nArgs;index++) {
            strcat(updateCommand," ");
            strcat(updateCommand,args[index]);
          }
        }
      } else
        fprintf(stderr,"ip_big_brother, line %d: (warning) unknown option
%s\n",line,args[0]);
    }
  }
  fclose(myFile);
  return 0;
}

/*****************************************************************************/
/* Purpose: Cleans all the dynamic data
*/

int DestroyStructures() {
  free_day *free_day_ptr,*free_day_ptr1;
  ip_class *ip_class_ptr,*ip_class_ptr1;
  tax_interval *tax_interval_ptr,*tax_interval_ptr1;
  assignment *assignment_ptr,*assignment_ptr1;
  weight *weight_ptr,*weight_ptr1;
  network *network_ptr,*network_ptr1;

  for(free_day_ptr = free_day_head;free_day_ptr;free_day_ptr = free_day_ptr1) {
    free_day_ptr1 = free_day_ptr->next;
    free(free_day_ptr);
  }
  for(ip_class_ptr = ip_class_head;ip_class_ptr;ip_class_ptr = ip_class_ptr1) {
    ip_class_ptr1 = ip_class_ptr->next;
    free(ip_class_ptr->class_name);
    for(assignment_ptr =
ip_class_ptr->assignments;assignment_ptr;assignment_ptr = assignment_ptr1) {
      assignment_ptr1 = assignment_ptr->next;
      free(assignment_ptr->r_value);
    }
    free(ip_class_ptr);
  }
  for(tax_interval_ptr = tax_interval_head;tax_interval_ptr;tax_interval_ptr
= tax_interval_ptr1) {
    tax_interval_ptr1 = tax_interval_ptr->next;
    free(tax_interval_ptr);
  }
  for(weight_ptr = weight_head;weight_ptr;weight_ptr = weight_ptr1) {
    weight_ptr1 = weight_ptr->next;
    free(weight_ptr->variable);
    if (weight_ptr->r_value)
      free(weight_ptr->r_value);
  }
  for(network_ptr = network_head;network_ptr;network_ptr = network_ptr1) {
    network_ptr1 = network_ptr->next;
    free(network_ptr->address);
    free(network_ptr->mask);
  }
  return 0;
}

/*****************************************************************************/
/* Purpose: See if an hour is in an interval
*/

int HourBelongsToInterval(int anHour,int lInterval,int rInterval) {

  return (lInterval < rInterval) ?
         ((anHour >= lInterval) && (anHour < rInterval)) :
         (((anHour >= lInterval) && (anHour < 24)) || ((anHour >= 0) &&
(anHour < rInterval)));
}

/*****************************************************************************/
/* Purpose: Calculates the actual class
*/

ip_class *GetActualClass() {
  free_day *free_day_ptr;
  tax_interval *tax_interval_ptr;
  struct tm *timeStruct;
  time_t timeNow;

  timeNow = time(NULL);
  timeStruct = localtime(&timeNow);

  /* Search for free days... */

  for(free_day_ptr = free_day_head;free_day_ptr;free_day_ptr =
free_day_ptr->next)
    if (free_day_ptr->weekday == -1) {
      if ((free_day_ptr->day == timeStruct->tm_mday) &&
          (free_day_ptr->month == timeStruct->tm_mon))
        return free_ip_class;
    } else {
      if (free_day_ptr->weekday == timeStruct->tm_wday)
        return free_ip_class;
    }

  /* Search for other days... */

  for(tax_interval_ptr = tax_interval_head;tax_interval_ptr;tax_interval_ptr
= tax_interval_ptr->next)
    if
(HourBelongsToInterval(timeStruct->tm_hour,tax_interval_ptr->hour_start,tax_interval_ptr->hour_end))
      break;
  if (tax_interval_ptr)
    return tax_interval_ptr->tax_class;
  return (ip_class *) 0;
}

/*****************************************************************************/
/* Purpose: Updates account files related
*/

int UpdateAccountInformation(ip_class *actual_class) {
  assignment *assignment_ptr;
  network *network_ptr;
  FILE *out;

  /* Update assignments */

  for(assignment_ptr =
actual_class->assignments;assignment_ptr;assignment_ptr = assignment_ptr->next)
    assignment_ptr->variable->r_value = assignment_ptr->r_value->r_value;

  if ((out = fopen(acctFile,"w")) == NULL)
    return -1;
  fprintf(out,"#\n");
  fprintf(out,"# Generated automagically by ip_big_brother (cs)\n");
  fprintf(out,"#\n");
  fprintf(out,"# Net-Addr              Net-Mask                Weight\n");
  fprintf(out,"#\n");
  for(network_ptr = network_head;network_ptr;network_ptr = network_ptr->next)
    if (!network_ptr->net_weight->r_value) {
      fclose(out);
      return -2;
    } else
      fprintf(out,"%-23s %-23s
%s\n",network_ptr->address,network_ptr->mask,network_ptr->net_weight->r_value);
  fclose(out);
  if (system(updateCommand))
    return -4;
  return 0;
}

/*****************************************************************************/
/* Purpose: Main program segment
*/

int main(int argc,char **argv) {
  ip_class *actual_class;
  char buffer[1024];
  int status;

  umask(0137);
  if ((argc == 3) && !strcasecmp(argv[1],"-c")) {
    if (ReadConfigurationFile(argv[2]) == -1) {
      fprintf(stderr,"ip_big_brother: (error) could not find the
configuration file %s.\n",argv[2]);
      exit(-1);
    }
  } else if (ReadConfigurationFile(CONFIGURATIONFILE) == -1) {
    if (ReadConfigurationFile(ALTCONFIGURATIONFILE) == -1) {
      fprintf(stderr,"ip_big_brother: (error) could not find the default
configuration file.\n");
      exit(-1);
    }
  }
  if (!(actual_class = GetActualClass()))
    RegisterLog("Could not get actual class!");
  else if ((status = UpdateAccountInformation(actual_class)) != 0) {
    sprintf(buffer,"Could not update account information properly, status =
%d",status);
    RegisterLog(buffer);
  } else {
    sprintf(buffer,"Using class %s for ip accounting",actual_class->class_name);
    RegisterLog(buffer);
  }
  DestroyStructures();
  return 0;
}
