/*
 * vt100.c
 *
 * Code to emulate vt100 escape sequences on mono monitor.
 * This is part of mda.c.
 *
 * This file is based heavily on /usr/src/linux/drivers/char/console.c
 * by Linus Torvalds. I've just removed whole chunks I thought that didn't
 * seem appropriate or that I was too lazy to modify.
 *
 */

static enum
  {
    ESnormal, ESesc, ESsquare, ESgetpars, ESgotpars, ESfunckey,
    EShash, ESsetG0, ESsetG1, ESpercent, ESignore
  } vc_state;


static int attr;

static int x,y;
static int intensity;
static int reverse;
static int blink;
static int underline;

static int saved_x, saved_y;
static int s_intensity;
static int s_underline;
static int s_blink;
static int s_reverse;
static int s_offset;

#ifndef NPAR
#define NPAR 10
#endif
static int npar;
static int par[NPAR];

#define ERASE_CHARACTER (ATTR_NORMAL<<8 | ' ')

static void gotoxy(new_x, new_y)
{
    x = new_x;
    y = new_y;

    offset = (y*VIDEO_COLS + x) * 2;
    set_cursor(offset);
}


static void lf ()
{

/* If we are at the end of a line then take care to LF to the next line
   rather than the one after (offset points to the start of the next line) */
    
    if (x == VIDEO_COLS)
    {
	offset = offset - (offset % VIDEO_LINELENGTH);
    }
    else
    {
	offset = offset - (offset % VIDEO_LINELENGTH) + VIDEO_LINELENGTH;
    }
    x = 0;
    y++;
}

static void tab()
{
    char* videomem = (char*)VIDEOMEM_START;
    int new_voffset;
    int j;
	
    new_voffset = offset - (offset % (TAB_SIZE*2)) + (TAB_SIZE*2);
    
/* Clear out the gap before the tab stop */	    
    for (j=offset; j<new_voffset; j++,j++)
    {
	writew(' '|ATTR_NORMAL<<8, videomem + base + j);
    }
    offset = new_voffset;
    x = x - (x % (TAB_SIZE*2)) + (TAB_SIZE*2);
}

/*
 * Reverse index (Cursor up to you)
 */
static void ri ()
{
    if (offset > VIDEO_LINELENGTH) offset = offset - (offset % VIDEO_LINELENGTH) - VIDEO_LINELENGTH;
    y--;
}

static inline void cr ()
{
    offset = offset - (offset % VIDEO_LINELENGTH);
    x = 0;
}

static inline void bs ()
{
    if (offset > base)
    {
	offset--;
	offset--;
	x--;
    }
}

static inline void del ()
{
    char* videomem = (char*)VIDEOMEM_START;
    
    if (offset > base)
    {
	writew(' '|ATTR_NORMAL<<8, videomem + base + offset);
	x--;
	offset--,offset--;
    }
}

static void csi_J (int vpar)
{
  unsigned long i;
  char* videomem = (char*)VIDEOMEM_START;
  
  switch (vpar)
  {
    case 0:			/* erase from cursor to end of display */
      for (i=offset; i<VIDEOSCREEN_SIZE; i++,i++)
      {
	  writew(' '|ATTR_NORMAL<<8, videomem + base+i);
      }
      break;
    case 1:			/* erase from start to cursor */
      for (i=0; i<offset; i++,i++)
      {
	  writew(' '|ATTR_NORMAL<<8, videomem + base+i);
      }
      break;
    case 2:			/* erase whole display and reset origin */
      for (i=0; i<VIDEOSCREEN_SIZE; i++,i++)
      {
	  writew(' '|ATTR_NORMAL<<8, videomem + i);
      }
      base = 0;
      set_origin(0);
      set_cursor(offset);
      break;
      
    default:
      return;
  }

}

/* Line erase */
static void csi_K (int vpar)
{
  char* videomem = (char*)VIDEOMEM_START;
  unsigned long count;
  unsigned short start = offset - (offset % VIDEO_LINELENGTH);
  unsigned short end = start + VIDEO_LINELENGTH;

  switch (vpar)
    {
    case 0:			/* erase from cursor to end of line */
	for (count = offset; count < end; count++,count++)
	{
	    writew(' '|ATTR_NORMAL<<8, videomem+base+count);
	}
      break;
    case 1:			/* erase from start of line to cursor */
	for (count = start; count <= offset; count++, count++)
	{
	    writew(' '|ATTR_NORMAL<<8, videomem+base+count);
	}
      break;
    case 2:			/* erase whole line */
	for (count = start; count <= end; count++, count++)
	{
	    writew(' '|ATTR_NORMAL<<8, videomem+base+count);
	}
	break;
    default:
      return;
    }
}

static void csi_X (int vpar)   	/* erase the following vpar positions */
{

}

static void update_attr ()
{
    if (intensity == 1)
	attr |= ATTR_NORMAL;
    else
	attr &= ~ATTR_NORMAL;

    if (underline == 1)
	attr |= ATTR_UNDERLINE;
    else
	attr &= ~ATTR_UNDERLINE;

    if (intensity == 2)
	attr |= ATTR_BOLD;
    else
	attr &= ~ATTR_BOLD;
    
#ifdef DO_BLINK
    if (blink == 1)
	attr |= ATTR_BLINK;
    else
	attr &= ~ATTR_BLINK;
#else
    if (reverse == 1)
	attr |= ATTR_BLINK;
    else
	attr &= ~ATTR_BLINK;
#endif
    
}

static void default_attr ()
{
  intensity = 1;
  underline = 0;
  reverse = 0;
  blink = 0;
  update_attr();
}

static void csi_m ()
{
  int i;

  for (i = 0; i <= npar; i++)
    switch (par[i])
      {
      case 0:			/* all attributes off */
	default_attr ();
	break;
      case 1:
	intensity = 2;
	break;
      case 2:
	intensity = 0;
	break;
      case 4:
	underline = 1;
	break;
      case 5:
	blink = 1;
	break;
      case 7:
	reverse = 1;
	break;
      case 10:

	break;
      case 11:

	break;
      case 12:

	break;
      case 21:
      case 22:
	intensity = 1;
	break;
      case 24:
	underline = 0;
	break;
      case 25:
	blink = 0;
	break;
      case 27:
	reverse = 0;
	break;
      case 38:

	break;
      case 39:

	break;
      case 49:

	break;
      default:
	break;
      }
  update_attr ();
}


static inline void status_report ()
{
   
}

static inline void respond_ID ()
{
    /*
  respond_string (VT102ID, tty);
  */
}


static void set_mode (int on_off)
{

}

static void insert_char ()
{
    unsigned int i = x;
    unsigned short tmp, old = ERASE_CHARACTER;
    unsigned short * p = (unsigned short *)VIDEOMEM_START+offset/2;

    while (i++ < VIDEO_COLS)
    {
	tmp = readb(p);
	writeb(old, p);
	old = tmp;
	p++;
    }
}

static void insert_line ()
{
    printk("insert_line"             );
}

static void delete_char ()
{
    printk("delete_char"             );
}

static void delete_line ()
{
    printk("delete_line"             );
}

static void csi_at (unsigned int nr)
{
 
    if (nr > VIDEO_COLS)
	nr = VIDEO_COLS;
    else if (!nr)
	nr = 1;
    while (nr--)
	insert_char();
      
}

static void csi_L (unsigned int nr)
{
 
	if (nr > VIDEO_ROWS)
		nr = VIDEO_ROWS;
	else if (!nr)
		nr = 1;
	while (nr--)
		insert_line();
}

static void csi_P (unsigned int nr)
{
  
	if (nr > VIDEO_COLS)
		nr = VIDEO_COLS;
	else if (!nr)
		nr = 1;
	while (nr--)
		delete_char();

}

static void csi_M (unsigned int nr)
{
    if (nr > VIDEO_ROWS)
	nr = VIDEO_ROWS;
    else if (!nr)
	nr=1;
    while (nr--)
	delete_line();

}

static void save_cur ()
{
  saved_x = x;
  saved_y = y;
  s_intensity = intensity;
  s_underline = underline;
  s_blink = blink;
  s_reverse = reverse;
  s_offset = offset;
}

static void restore_cur ()
{
  gotoxy (saved_x, saved_y);
  intensity = s_intensity;
  underline = s_underline;
  blink = s_blink;
  reverse = s_reverse;
  offset = s_offset;
  update_attr ();
  
}

static void reset_terminal ()
{
  attr = 0;
  npar = 0;
  vc_state = ESnormal;

  default_attr ();
  update_attr ();

  x = y = 0;
  gotoxy (0, 0);
  save_cur ();
  csi_J (2);
}

static void vt100_write (unsigned char c)
{
    char* videomem = (char*)VIDEOMEM_START;

    if ((vc_state == ESnormal) && (c >= ' ') && (c < DEL_CHAR))
    {
	writew(c|attr<<8, videomem+base+offset);
	offset += 2;
	x++;
    }
    else
    {
	if (vc_state == ESnormal)
	{
	    switch (c)
	    {
	    case 7:
		/* No bell */
		break;
	    case 8:
		bs ();
		break;
	    case 9:
		tab ();
		break;
	    case 10:
	    case 11:
	    case 12:
		lf ();
		break;
	    case 13:
		cr ();
		break;
	    case 14:
		/* No charset flipping */
		break;
	    case 15:
		/* No charset flipping */
		break;
	    case 24:
	    case 26:
		vc_state = ESnormal;
		break;
	    case 27:
		vc_state = ESesc;
		break;
	    case 127:
		del ();
		break;
	    case 128 + 27:
		vc_state = ESsquare;
		break;
	    }
	}
	else
	{
	    switch (vc_state)
	    {
	    case ESesc:
		vc_state = ESnormal;
		switch (c)
		{
		case '[':
		    vc_state = ESsquare;
		    break;
		case '%':
		    vc_state = ESpercent;
		    break;
		case 'E':
		    cr ();
		    lf ();
		    break;
		case 'M':
		    ri ();
		    break;
		case 'D':
		    lf ();
		    break;
		case 'H':

		    break;
		case 'Z':
		    respond_ID ();
		    break;
		case '7':
		    save_cur ();
		    break;
		case '8':
		    restore_cur ();
		    break;
		case '(':
		    vc_state = ESsetG0;
		    break;
		case ')':
		    vc_state = ESsetG1;
		    break;
		case '#':
		    vc_state = EShash;
		    break;
		case 'c':
		    reset_terminal ();
		    break;
		case '>':	/* Numeric keypad */
		    break;
		case '=':	/* Appl. keypad */
		    break;
		}
		break;
	    case ESsquare:
		for (npar = 0; npar < NPAR; npar++)
		    par[npar] = 0;
		npar = 0;
		vc_state = ESgetpars;
		if (c == '[')
		{		/* Function key */
		    vc_state = ESfunckey;
		    break;
		}
	    case ESgetpars:
		if (c == ';' && npar < NPAR - 1)
		{
		    npar++;
		    break;
		}
		else if (c >= '0' && c <= '9')
		{
		    par[npar] *= 10;
		    par[npar] += c - '0';
		    break;
		}
		else
		    vc_state = ESgotpars;
	    case ESgotpars:
		vc_state = ESnormal;
		switch (c)
		{
		case 'h':
		    set_mode ( 1);
		    break;
		case 'l':
		    set_mode ( 0);
		    break;
		case 'n':
		    break;
		}

		switch (c)
		{
		case 'G':
		case '`':
		    if (par[0])
			par[0]--;
		    gotoxy ( par[0], y);
		    break;
		case 'A':
		    if (!par[0])
			par[0]++;
		    gotoxy ( x, y - par[0]);
		    break;
		case 'B':
		case 'e':
		    if (!par[0])
			par[0]++;
		    gotoxy ( x, y + par[0]);
		    break;
		case 'C':
		case 'a':
		    if (!par[0])
			par[0]++;
		    gotoxy ( x + par[0], y);
		    break;
		case 'D':
		    if (!par[0])
			par[0]++;
		    gotoxy ( x - par[0], y);
		    break;
		case 'E':
		    if (!par[0])
			par[0]++;
		    gotoxy ( 0, y + par[0]);
		    break;
		case 'F':
		    if (!par[0])
			par[0]++;
		    gotoxy ( 0, y - par[0]);
		    break;
		case 'd':
		    if (par[0])
			par[0]--;
		    gotoxy ( x, par[0]);
		    break;
		case 'H':
		case 'f':
		    if (par[0])
			par[0]--;
		    if (par[1])
			par[1]--;
		    gotoxy ( par[1], par[0]);
		    break;
		case 'J':
		    csi_J ( par[0]);
		    break;
		case 'K':
		    csi_K ( par[0]);
		    break;
		case 'L':
		    csi_L ( par[0]);
		    break;
		case 'M':
		    csi_M ( par[0]);
		    break;
		case 'P':
		    csi_P ( par[0]);
		    break;
		case 'c':
		    break;
		case 'g':
		    break;
		case 'm':
		    csi_m ();
		    break;
		case 'q':
		    break;
		case 'r':
		    break;
		case 's':
		    save_cur ();
		    break;
		case 'u':
		    restore_cur ();
		    break;
		case 'X':
		    csi_X ( par[0]);
		    break;
		case '@':
		    csi_at ( par[0]);
		    break;
		case ']':
		    /*	  setterm_command (); */
		    break;
		}
		break;
	    case ESpercent:
		vc_state = ESnormal;
		break;
	    case ESfunckey:
		vc_state = ESnormal;
		break;
	    case EShash:
		vc_state = ESnormal;
		break;
	    case ESsetG0:
		vc_state = ESnormal;
		break;
	    case ESsetG1:
		vc_state = ESnormal;
		break;
	    default:
		vc_state = ESnormal;
	    }
	}
    }    
    /* Do we need to scroll? */
    
    if (offset >= VIDEOSCREEN_SIZE)
    {
	offset = scroll_screen();
    }
    set_cursor(offset);
    
    return;
}
