/*
 * mda.c
 *
 * Kernel module to drive MDA nonitor
 *
 * v0.7 1995 Patrick J Caulfield
 *
 * Revision History:
 *
 * v0.1   3-Jun-1995  PJC  First implementation as a kernel module
 * v0.2   4-Jun-1995  PJC  Allowed multiple opens to be active
 * v0.3   4-Jun-1995  PJC  Added hardware scrolling support
 * v0.4   6-Jun-1995  PJC  Fixed odd scrolling bug.
 * v0.5   9-Jul-1995  PJC  Added DEL/BS support
 * v0.6  18-Jul-1995  PJC  Now include autoconf.h to get CONFIG_MODVERSIONS
 *						 Started vt100 support. 
 * v0.7  21-Jul-1995  PJC  Added card detection and reverse video in place of
 *						 flashing.
 * v0.8  15-Jan-1996  JBE  Integrated support for Hercules graphics.
 *
 ******************************************************************************

	(c) 1995 P.J. Caulfield	 patrick@pandh.demon.co.uk
								pcaulfield@cix.compulink.co.uk
	
	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
 ******************************************************************************
 */
/* Linux module includes */
#include <linux/module.h>
#include <linux/version.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/major.h>
#include <linux/sched.h>
#include <linux/ioport.h>
#include <linux/fcntl.h>
#include <linux/delay.h>
#ifdef HAVE_AUTOCONF
#include <linux/autoconf.h>  /* for CONFIG_MODVERSIONS */
#endif
#include <asm/io.h>
#include <linux/kdev_t.h>
#include <linux/fs.h>
#include <linux/mman.h>
#include "hercules.h"

static mode=0; /* >0 == text <0 == graphics */

static int mono_lseek(struct inode *inode, struct file *file, off_t offset, int whence)
{
	int m=MINOR(inode->i_rdev);
	if(m==1) return graphic_lseek(inode, file, offset, whence);
	else return -ESPIPE;
}

static int mono_read(struct inode *inode, struct file *file, const char *buf, int count)
{
	int m=MINOR(inode->i_rdev);
	if(m==1) return graphic_read(inode, file, buf, count);
	else return -EINVAL;
}

static int mono_write(struct inode *inode, struct file *file, const char *buf, int count)
{
	int m=MINOR(inode->i_rdev);
	if(m==1) return graphic_write(inode, file, buf, count);
	else if(m==2) return text_write(inode, file, buf, count);
}

static int mono_mmap(struct inode *inode, struct file *file, struct vm_area_struct *vma)
{
	int m=MINOR(inode->i_rdev);
	if(m==1) return graphic_mmap(inode, file, vma);
	else return -EINVAL;
}
static int mono_open(struct inode *inode, struct file *file)
{
	int m=MINOR(inode->i_rdev);
	if(m==1) {
		if(mode>=0) {
			MOD_INC_USE_COUNT;
			mode++;
			graphic_mode();
			return graphic_open(inode,file);
		} else return -EBUSY;
	} else if(m==2) {
		if(mode<=0) {
			MOD_INC_USE_COUNT;
			mode--;
			text_mode();
			return text_open(inode,file);
		} else return -EBUSY;
	} else return -ENODEV;
}
static void mono_release(struct inode *inode, struct file *file)
{
	int m=MINOR(inode->i_rdev);
	if(m==2) ++mode;
	if(m==1) --mode;
	MOD_DEC_USE_COUNT;
}
	
/* ------------------------- Module support stuff -------------------------- */

/* File operations to register with the kernel */

static struct file_operations mono_fops = {
	mono_lseek,
	mono_read,
	mono_write,
	NULL,		/* _readdir */
	NULL,		/* _select  */
	NULL,		/* _ioctl   */
	mono_mmap,	
	mono_open,
	mono_release
};
/* char kernel_version[]= UTS_RELEASE; */

/*
 * Detect card.
 * Returns 0, for no MDA, 1 for MDA present...more???
 * Fixme: We punt here and assume that an MDA is hercules compatable
 */
static int detect_card()
{
    unsigned char saved_pos, inbyte;
    unsigned char arbit_val = 0x66;
    
    outb(15, video_port_reg); /* Select cursor low register */
    saved_pos = inb(video_port_val);

/* Write an arbitrary value to the cursor register & read it back */
    outb(arbit_val, video_port_val);

    udelay(10); /* Wait a bit */
    
    inbyte = inb(video_port_val);
    if (arbit_val == inbyte)
    {
	return 1;
    }
    else
    {
	return 0;
    }
}
/*
 * Called at insmod time
 */
int init_module(void)
{

    /* Is there are card in the system ? */
    if (!detect_card())
    {
	printk("mda: unable to find mono adapter card\n");
	return -EIO;
    }
    
    if (register_chrdev(MONO_MAJOR,"mono",&mono_fops))
    {
	printk("mda: unable to get major %d for mono monitor\n", MONO_MAJOR);
	return -EIO;
    }
    
#ifndef DO_BLINK
    clear_blink_flag();
#endif
    text_mode();
    reset_terminal();
    return 0;
}

/*
 * Called at rmmod time
 */
void cleanup_module(void)
{
    if (MOD_IN_USE)
	printk("mono: busy - remove delayed\n");
    else
	unregister_chrdev(MONO_MAJOR,"mono");
}
