/*
 * mda.c
 *
 * Kernel module to drive MDA nonitor
 *
 * v0.7 1995 Patrick J Caulfield
 *
 * Revision History:
 *
 * v0.1   3-Jun-1995  PJC  First implementation as a kernel module
 * v0.2   4-Jun-1995  PJC  Allowed multiple opens to be active
 * v0.3   4-Jun-1995  PJC  Added hardware scrolling support
 * v0.4   6-Jun-1995  PJC  Fixed odd scrolling bug.
 * v0.5   9-Jul-1995  PJC  Added DEL/BS support
 * v0.6  18-Jul-1995  PJC  Now include autoconf.h to get CONFIG_MODVERSIONS
 *						 Started vt100 support. 
 * v0.7  21-Jul-1995  PJC  Added card detection and reverse video in place of
 *						 flashing.
 * v0.8  15-Jan-1996  JBE  Integrated support for Hercules graphics.
 *
 ******************************************************************************

	(c) 1995 P.J. Caulfield	 patrick@pandh.demon.co.uk
								pcaulfield@cix.compulink.co.uk
	
	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
 ******************************************************************************
 */
/* Parts of this code are adapted from code bearing this copyright:          */
/*****************************************************************************/
/*	Written by:    Ben Bederson   (c) 1988				     */
/*		       bbb7889@acf5.NYU.EDU				     */
/*									     */
/*		       222 East 19th St.  #3H				     */
/*		       New York, NY  10003				     */
/*		       (212) 260-2667					     */
/*									     */
/*	These routines were written by myself for my own purposes and I make */
/*	no guarantees, warentees, expressed, or implied, etc.  However,      */
/*	I hope they are useful.  Everybody is free and encouraged to copy    */
/*	and use these routines under the condition that this statement is    */
/*	kept at the top without modification.  I would appreciate any	     */
/*	comments or criticisms or improvement/ideas about these routines.    */
/*									     */
/*	Good luck.							     */
/*****************************************************************************/

/* Linux module includes */
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/major.h>
#include <linux/sched.h>
#include <linux/ioport.h>
#include <linux/fcntl.h>
#include <linux/delay.h>
#include <linux/mm.h>
#include <asm/io.h>
#include "hercules.h"


/* Make sure we have a major device number to use */
#ifndef MONO_MAJOR
#error No Major number specified
#endif

/* Function prototypes */

static void   set_mode(int);
static void   set_cursor(unsigned short); /* Only in text? */
static inline void set_origin(unsigned short); /* Only in text? */

static int isgraphics = 0;

/* Globals */
static unsigned offset = 0; /* Current offset into video memory */
static unsigned base   = 0; /* Current location of top of display */

int gr6845[] = {0x38, 0x2d, 0x30, 0x08, 0x5a, 0x00, 0x57, 0x57, 0x02,
		0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
int te6845[] = {0x61, 0x50, 0x52, 0x0f, 0x19, 0x06, 0x19, 0x19, 0x02,
		0x0d, 0x0b, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

/*****************************************************************************/
/*									     */
/*	I enter graphics mode by directly accessing the 6845 graphics	     */
/*	controller chip.  I found the register addresses and data by	     */
/*	debugging a sample graphics program that came with my Everex	     */
/*	Hercules compatible graphics card.  I have tried them on both	     */
/*	Leading Edge and Five Star computers with Hercules-compatible	     */
/*	graphics cards with 100% success, so I expect that it should	     */
/*	work.								     */
/*									     */
/*****************************************************************************/

graphic_mode()
{
  int i;

  outb(0x00,video_mode_reg);
  outb(0x03,video_cfg_reg);
  for (i=0; i<18; i++) {
    outb(       i ,video_port_reg);
    outb(gr6845[i],video_port_val);
  }
  outb(0x06,video_mode_reg);
  /* grcls(); */
  outb(0x0e,video_mode_reg);
}

/*****************************************************************************/
/*									     */
/*	Read the comment before about the graphic() routine to find	     */
/*	about this routine.						     */
/*									     */
/*****************************************************************************/

text_mode()
{
  int i;

  outb(0x00,video_cfg_reg);
  outb(0x28,video_mode_reg);
  for (i=0; i<18; i++) {
    outb(       i ,video_port_reg);
    outb(te6845[i],video_port_val);
  }
  /* cls(); */
}

static void set_mode(int g)
{
	if(g) graphic_mode();
	else text_mode();
}

/*
 * Write a single character to the screen.
 */
static void graphic_write_char(char outchar)
{
	char *videomem=(char*)VIDEOMEM_START;
	videomem[offset++]=outchar;
}

/*------------------------- Basic file operations ---------------------------*/

int graphic_mmap(struct inode * inode, struct file * file, struct vm_area_struct * vma)
{
	if (vma->vm_offset & ~PAGE_MASK) return -ENXIO;
	if (vma->vm_end-vma->vm_start+vma->vm_offset>0x8000) return -EFAULT;
	vma->vm_offset+=VIDEOMEM_START;
	if (remap_page_range(vma->vm_start, vma->vm_offset, vma->vm_end - vma->vm_start, vma->vm_page_prot))
		return -EAGAIN;
	vma->vm_inode = inode;
	inode->i_count++;
	return 0;
}

int graphic_read(struct inode *inode, struct file*file, const char*buf, int count)
{
	int		i;
	const char *temp = buf;
	char	   c;
	char *videomem=(char*)VIDEOMEM_START;
	offset=file->f_pos;
	if(offset>0x7fff) { return -1; }
	for (i=0; i<count; i++) {
	 	if((offset+i)>0x7fff) break;
		c = videomem[offset+i];
		put_fs_byte(c,temp+i);
	}
	file->f_pos=offset;
	return i;
}

/*
 * Write a string to the screen
 */
int graphic_write(struct inode * inode, struct file * file, const char * buf, int count)
{
	int	i;
	const char *temp = buf;
	char	c;
	
	offset=file->f_pos;
	if(offset>0x7fff) { return -1; }
	for (i=0; i<count; i++) {
	 	if(offset>0x7fff) break;
		c = get_fs_byte(temp+i);
		graphic_write_char(c);
	}
	file->f_pos=offset;
	return i;
}

/*
 * lseek?? okay.
 */
int graphic_lseek(struct inode * inode, struct file * file,
		 		  off_t newoffset, int origin)
{
	offset=file->f_pos;
	if(origin==0/*SEEK_SET*/) {
		offset=newoffset;
	} else if(origin==1/*SEEK_CUR*/) {
		offset+=newoffset;
	} else if(origin==2/*SEEK_END*/) {
		offset=0x2000+newoffset;
	} else return -EINVAL;
	file->f_pos=offset;
	return offset;
}

/*
 * Open -- text/graphic sharing check done in main.c
 */
int graphic_open(struct inode * inode, struct file * file)
{
	graphic_mode();
	return 0;
}

/*
 * Close
 */
void graphic_release(struct inode * inode, struct file * file)
{
}
