/*
 * mda.c
 *
 * Kernel module to drive MDA nonitor
 *
 * v0.14 1998 Patrick J Caulfield
 *            Console mods by David Woodhouse
 *
 * Revision History:
 *
 * v0.1   3-Jun-1995  PJC  First implementation as a kernel module
 * v0.2   4-Jun-1995  PJC  Allowed multiple opens to be active
 * v0.3   4-Jun-1995  PJC  Added hardware scrolling support
 * v0.4   6-Jun-1995  PJC  Fixed odd scrolling bug.
 * v0.5   9-Jul-1995  PJC  Added DEL/BS support
 * v0.6  18-Jul-1995  PJC  Now include autoconf.h to get CONFIG_MODVERSIONS
 *                         Started vt100 support. 
 * v0.7  21-Jul-1995  PJC  Added card detection and reverse video in place of
 *                         flashing.
 * v0.8  25-Apr-1996  PJC  Added support for dynamic major number allocation
 * v0.9  22-Jun-1996  PJC  Fixed boldface bug.
 * v0.10 24-Dec-1996  PJC  Fixed scroll flashing (possibly).
 *                         Made to work with 2.1.x kernels
 * v0.11 17-May-1997  PJC  More 2.1.x mods
 *                    DMW  ifdefs for using as console
 * v0.12 23-Nov-1997  PJC  Mods for 2.1.60+ kernels
 *                         added read functions
 * v0.13 26-Dec-1997  DMW  Diffs for mda console on 2.1.71
 *                    PJC  Some typos fixed.
 * v0.14 15-Feb-1998  PJC  Made console mode selectable at insmod time
 *                         provided we are compiled for 2.1.82 or higher
 *                         Added other options.
 *       16-May-1998  MM   Bug fixes in vt100.c
 *
 ******************************************************************************

    (c) 1995-1998 P.J. Caulfield          patrick@pandh.demon.co.uk
                                          pcaulfield@cix.compulink.co.uk
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
 ******************************************************************************
 */

/* Linux module includes */
#ifdef HAVE_AUTOCONF
#include <linux/autoconf.h>  /* for CONFIG_MODVERSIONS */
#endif


#include <linux/module.h>
#ifdef CONFIG_MODVERSIONS
#include <linux/modversions.h>
#endif
#include <linux/version.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/major.h>
#include <linux/sched.h>
#include <linux/ioport.h>
#include <linux/fcntl.h>
#include <linux/delay.h>
#if LINUX_VERSION_CODE > 0x020100
#include <asm/uaccess.h>
#endif
#if LINUX_VERSION_CODE > 0x020152
#include <linux/console.h>
#endif
#include <asm/io.h>
#include "mda.h"

/* Our name in the kernel */
#define MODULE_NAME "mono"

/* Make sure we have a major device number to use */
#ifndef MONO_MAJOR
#error No Major number specified
#endif

/* Function prototypes */
static void   set_cursor(unsigned short);
static int    scroll_screen(void);
static void   setup_card(void);
static inline void set_origin(unsigned short);

#if LINUX_VERSION_CODE > 0x20152
static void mda_console_write (struct console *cons, const char *text, unsigned int len);
static kdev_t mda_console_device (struct console *cons);
#endif 

static unsigned char tbl[] = {0x61, 0x50, 0x52, 0x0F, 0x19, 0x06, 0x19, 0x19,
			      0x02, 0x0D, 0x0B, 0x0C, 0x00, 0x00, 0x00, 0x28};

/* Globals */
static unsigned short offset = 0; /* Current offset into video memory */
static unsigned short base   = 0; /* Current location of top of display */
static unsigned short major  = 0; /* Major device number assigned */
static unsigned short cardType;   /* MDA or hercules */
static unsigned long  MemEnd = VIDEOMEM_END;

/* Globals that can be assigned at insmod time */
static unsigned int console[1] = {0};
static unsigned int mem[1] = {8};

#if LINUX_VERSION_CODE > 0x20152
MODULE_PARM(console, "1i");
MODULE_PARM_DESC(console, "register as a console");
MODULE_PARM(mem, "1i");
MODULE_PARM_DESC(mem, "memory on the mono card (in Kbytes)");
#endif

/* Include the ANSI/vt100 emulation module */
#include "vt100.c"

/*
 * Clearing the blink flag in the control register makes the BLINK
 * attribute show reverse instead.
 * I *much* prefer this. If you don't then define DO_BLINK in mda.h
 * Also set up my Hercules card so that I can see the text(?)
 */
static void setup_card()
{
    unsigned int set_val;

#ifdef DO_BLINK
    set_val = 0x29;
#else
    set_val = 0x09;
#endif

/* If we are driving a hercules card then enable graphics mode.
   This is essential to get my card working but is probably not necessary
   for anybody elses because my HGC+ card works fine on all other MBs I've
   tried it on apart from my TC430HX. Thanks Intel :-(
 */       
    if (cardType == HERCPLUS_CARD)
    {
	unsigned char i;
	
	for (i=0; i<16; i++)
	{
	    outb(i, 0x3b4);
	    outb(tbl[i], 0x3b5);
	}
	outb(3, video_config);
    }
    outb(set_val, video_mode_reg);
}

/*
 * Write a single character to the screen.
 */
static void mono_write_char(char outchar)
{
    vt100_write(outchar);
}

/*
 * Scroll up by one line
 */
static int scroll_screen()
{
    int i,j;
    char* videomem = (char*)VIDEOMEM_START;

    offset = VIDEOSCREEN_SIZE - VIDEO_LINELENGTH; /* Stay on the bottom line */
    base  += VIDEO_LINELENGTH;

/* Reached the end of video memory? -- go back to the beginning and start again */	
    if (base+VIDEOSCREEN_SIZE+VIDEOMEM_START >= MemEnd)
    {
	for (j=0; j< VIDEOSCREEN_SIZE-VIDEO_LINELENGTH; j++,j++)
	{
	    writew(readw(videomem+j+base), videomem+j);
	}
	base = 0;
    }
    
    /* Clear the bottom line */
    for (i=VIDEOSCREEN_SIZE-VIDEO_LINELENGTH; i<VIDEOSCREEN_SIZE; i++,i++)
    {
	writew(' '|ATTR_NORMAL<<8, videomem+base+i);
    }
    
#ifdef DEBUG
    printk("mda: video base address is: %d\n", base);
#endif
    set_origin(base);

    return offset;
}

/*
 * Move the offset into video memory -- Hardware scrolling
 *
 * Thanks to Paul Gortmaker (see /usr/src/linux/driver/char/vga.c) for
 * this piece of code that eliminates the odd "flashing" at the RHS of
 * the screen during heavy scrolling.
 *
 * If it causes a problem you can get the old behaviour
 * by #define OLD_SCROLL.
 *
 */
static inline void set_origin(unsigned short offset)
{

    base = offset;

#ifdef OLD_SCROLL
    outb(13, video_port_reg);
    outb(offset >> 1, video_port_val);
    outb(12, video_port_reg);
    outb(offset >> 9, video_port_val);
#else
    {
	unsigned int v1, v2;
	unsigned int reg = 12;
	offset >>= 1;

	v1 = reg + (offset & 0xff00);
	v2 = reg + 1 + ((offset << 8) & 0xff00);
	outw(v1, video_port_reg);
	outw(v2, video_port_reg);
    }
#endif
}


/*
 * Position the cursor
 */
void set_cursor(unsigned short pos)
{
    int video_mem_base = VIDEOMEM_START;    
    
    outb(14, video_port_reg);
    outb(0xff&((pos-video_mem_base+base)>>9), video_port_val); 
    outb(15, video_port_reg);
    outb(0xff&((pos-video_mem_base+base)>>1), video_port_val);
}

/*
 * Detect card.
 * Returns 0, for no MDA, 1 for MDA present.
 * As a side effect sets the cardType.
 */
static int detect_card()
{
    unsigned char saved_pos, inbyte;
    unsigned char arbit_val = 0x66;
    unsigned int  count;
    
    outb(15, video_port_reg); /* Select cursor low register */
    saved_pos = inb(video_port_val);

/* Write an arbitrary value to the cursor register & read it back */
    outb(arbit_val, video_port_val);

    udelay(10); /* Wait a bit */
    
    inbyte = inb(video_port_val);
    if (arbit_val != inbyte)
    {
	return 0;
    }
/*
   OK, now we know there's a card in the system. we need to know if it is
   a hercules or just an MDA
 */
    inbyte = saved_pos = inb(video_status)&0x80;
    for (count=0; count < 32767 && inbyte == saved_pos; count++)
	inbyte = inb(video_status)&0x80;

/* If that bit has changed then we have a Hercules card of some description */
    if (inbyte != saved_pos)
    {
	cardType = HERCULES_CARD;
	
	inbyte = inb(video_status) & 0x70;
	if (inbyte != 0)
	{
	    if (inbyte == 0x10)
		cardType = HERCPLUS_CARD;
	    else
		cardType = INCOLOUR_CARD;
	}
    }
    else
    {
	cardType = MDA_CARD;
    }

/* 
  How much memory does it have ? Check for memory wrap at 4K, assume 8 if
  not. This call all be overridden on the command-line 
 */
    if (mem[0] == 0)
    {
	writew(0x0000, VIDEOMEM_START + 0x1000);
	writew(0xAAAA, VIDEOMEM_START);
	
	if (readw(VIDEOMEM_START + 0x1000) == 0xAAAA)
	{
	    mem[0] = 4;
	}
    }
    return 1;
}

/*------------------------- Basic file operations ---------------------------*/

/*
 * Write a string to the screen
 */
#if LINUX_VERSION_CODE > 0x2013C
static ssize_t mono_write(struct file * file,
			  const char * buf, 
			  size_t count, 
			  loff_t *offset)
#else
static long mono_write(struct inode *inode, 
		       struct file *file,
		       const char *buf, 
		       unsigned long count)
#endif
{
    int        i;
    const char *temp = buf;
    char       c;
    
    for (i=0; i<count; i++)
    {
#if LINUX_VERSION_CODE > 0x20100
        get_user(c,temp++);
#else
	c = get_fs_byte(temp++);
#endif
	mono_write_char(c);
    }
    return count;
}

#if LINUX_VERSION_CODE > 0x2013C
static ssize_t mono_read(struct file *file,
			 char *buf, 
			 size_t count,
			 loff_t *offset)
#else
static long mono_read(struct inode *inode, 
		      struct file *file,
		      char *buf, 
		      unsigned long count)
#endif
{
    short *startmem = (short *)(char *)(VIDEOMEM_START + base);
    short *endmem = (short *)(char *)(VIDEOMEM_START + base + VIDEOSCREEN_SIZE);
    char *userbuf = buf;
    int   i;

    for (i=0; (i<count) && (startmem+file->f_pos < endmem); i++)
    {
	put_user(readb(file->f_pos+startmem++), userbuf++);
    }
    file->f_pos += i;    
    return i;
}

/*
 * Open -- no sharing check otherwise we can't redirect
 * both stderr and stdout to the mono monitor.
 */
static int mono_open(struct inode * inode, struct file * file)
{
    MOD_INC_USE_COUNT;
    file->f_pos = 0;
    return 0;
}

/*
 * Close
 */
static int mono_release(struct inode * inode, struct file * file)
{
    MOD_DEC_USE_COUNT;
    return 0L;
}
    
/* ------------------------- Module support stuff -------------------------- */

/* File operations to register with the kernel */

#if LINUX_VERSION_CODE > 0x20152
static struct console mda_console = {
        "mono",
       mda_console_write,
       NULL,
       mda_console_device,
       NULL,
       NULL,
       NULL,
       CON_PRINTBUFFER | CON_ENABLED,
       -1,
       0,
       NULL
};
#endif

static struct file_operations mono_fops = {
        NULL,           /* seek - default */
	mono_read,
	mono_write,
	NULL,		/* _readdir */
	NULL,		/* _select  */
	NULL,           /* _ioctl   */
	NULL,		/* _mmap    */
	mono_open,
	mono_release
};

char kernel_version[] = UTS_RELEASE;

/*
 * Called at insmod time
 */
int init_module(void)
{
    
    /* Is there a card in the system ? */
    if (!detect_card())
    {
	printk("mda: unable to find mono adapter card\n");
	return -EIO;
    }
    

#if MONO_MAJOR == DYNAMIC
    major = register_chrdev(0, MODULE_NAME, &mono_fops);
    if (major <= 0)
    {
	printk("mda: unable to get a major number for mono monitor, retcode = %d\n", major);
	return -EIO;
    }
#else
    major = register_chrdev(MONO_MAJOR, MODULE_NAME, &mono_fops);
    if (major != 0)
    {
	printk("mda: unable to register driver for mono monitor, retcode = %d\n", major);
	return -EIO;
    }
#endif

    if (mem[0] != 0)
    {
	MemEnd = VIDEOMEM_START + mem[0]*1024 - 1;
    }
    
    setup_card();
    reset_terminal();

#if LINUX_VERSION_CODE > 0x20152
    if (console[0] == 1)
    {
	register_console(&mda_console);
    }
#endif
    return 0;
}

/*
 * Called at rmmod time
 */
void cleanup_module(void)
{
    if (MOD_IN_USE)
    {
	printk("mda: busy - remove delayed\n");
	return;
    }
#if LINUX_VERSION_CODE > 0x20152
    if (console[0] == 1)
    {
	if (unregister_console(&mda_console))
	    printk("mono: can't unregister console device");
    }
#endif

#if MONO_MAJOR == DYNAMIC
    unregister_chrdev(major, MODULE_NAME);
#else
    unregister_chrdev(MONO_MAJOR, MODULE_NAME);
#endif    
}

#if LINUX_VERSION_CODE > 0x20152
static void mda_console_write (struct console *cons, const char *text,
			       unsigned int len)
{
    unsigned int c=0;
        
    for (c=0; c < len; c++)
        mono_write_char(*(text++));
  }

static kdev_t mda_console_device (struct console *cons)
{ 
    return MKDEV(MONO_MAJOR,0);
}
#endif
