/* cm206.c. A linux-driver for the cm206 cdrom player with cm260 adapter card.
   Copyright (c) 1995, 1996 David van Leeuwen.
   
     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.
     
     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.
     
     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

History:
 Started 25 jan 1994. Waiting for documentation...
 22 feb 1995: 0.1a first reasonably safe polling driver.
	      Two major bugs, one in read_sector and one in 
	      do_cm206_request, happened to cancel!
 25 feb 1995: 0.2a first reasonable interrupt driven version of above.
              uart writes are still done in polling mode. 
 25 feb 1995: 0.21a writes also in interrupt mode, still some
	      small bugs to be found... Larger buffer. 
  2 mrt 1995: 0.22 Bug found (cd-> nowhere, interrupt was called in
              initialization), read_ahead of 16. Timeouts implemented.
	      unclear if they do something...
  7 mrt 1995: 0.23 Start of background read-ahead.
 18 mrt 1995: 0.24 Working background read-ahead. (still problems)
 26 mrt 1995: 0.25 Multi-session ioctl added (kernel v1.2).
              Statistics implemented, though separate stats206.h.
	      Accessible trough ioctl 0x1000 (just a number).
	      Hard to choose between v1.2 development and 1.1.75.
	      Bottom-half doesn't work with 1.2...
	      0.25a: fixed... typo. Still problems...
  1 apr 1995: 0.26 Module support added. Most bugs found. Use kernel 1.2.n.
  5 apr 1995: 0.27 Auto-probe for the adapter card base address.
              Auto-probe for the adaptor card irq line.
  7 apr 1995: 0.28 Added lilo setup support for base address and irq.
              Use major number 32 (not in this source), officially
	      assigned to this driver.
  9 apr 1995: 0.29 Added very limited audio support. Toc_header, stop, pause,
              resume, eject. Play_track ignores track info, because we can't 
	      read a table-of-contents entry. Toc_entry is implemented
	      as a `placebo' function: always returns start of disc. 
  3 may 1995: 0.30 Audio support completed. The get_toc_entry function
              is implemented as a binary search. 
 15 may 1995: 0.31 More work on audio stuff. Workman is not easy to 
              satisfy; changed binary search into linear search.
	      Auto-probe for base address somewhat relaxed.
  1 jun 1995: 0.32 Removed probe_irq_on/off for module version.
 10 jun 1995: 0.33 Workman still behaves funny, but you should be
              able to eject and substitute another disc.

 An adaptation of 0.33 is included in linux-1.3.7 by Eberhard Moenkeberg

 18 jul 1995: 0.34 Patch by Heiko Eissfeldt included, mainly considering 
              verify_area's in the ioctls. Some bugs introduced by 
	      EM considering the base port and irq fixed. 

 18 dec 1995: 0.35 Add some code for error checking... no luck...

 We jump to reach our goal: version 1.0 in the next stable linux kernel.

 19 mar 1996: 0.95 Different implementation of CDROM_GET_UPC, on
	      request of Thomas Quinot. 
 25 mar 1996: 0.96 Interpretation of opening with O_WRONLY or O_RDWR:
	      open only for ioctl operation, e.g., for operation of
	      tray etc.
 4 apr 1996:  0.97 First implementation of layer between VFS and cdrom
              driver, a generic interface. Much of the functionality
	      of cm206_open() and cm206_ioctl() is transferred to a
	      new file cdrom.c and its header ucdrom.h. 

	      Upgrade to Linux kernel 1.3.78. 

 11 apr 1996  0.98 Upgrade to Linux kernel 1.3.85
              Made it more uniform.

 14 mar 1998  2.0.30 Linux kernel, added support for cm250
              the cm205ms and cm206ms are reportedly interchangeable
              on the cm250 and cm260 adapters
              by Sudarshan P. Bhat 03-14-1998
 * 
 * Parts of the code are based upon lmscd.c written by Kai Petzke,
 * sbpcd.c written by Eberhard Moenkeberg, and mcd.c by Martin
 * Harriss, but any off-the-shelf dynamic programming algorithm won't
 * be able to find them.
 *
 * The cm206 drive interface and the cm260 adapter card seem to be 
 * sufficiently different from their cm205/cm250 counterparts
 * in order to write a complete new driver.
 * 
 * I call all routines connected to the Linux kernel something
 * with `cm206' in it, as this stuff is too series-dependent. 
 * 
 * Currently, my limited knowledge is based on:
 * - The Linux Kernel Hacker's guide, v. 0.5, by Michael K. Johnson
 * - Linux Kernel Programmierung, by Michael Beck and others
 * - Philips/LMS cm206 and cm226 product specification
 * - Philips/LMS cm260 product specification
 *
 *                       David van Leeuwen, david@tm.tno.nl.  */
#define VERSION "$Id: cm206.c,v 0.99.1.1 1996/08/11 10:35:01 david Exp $"

#include <linux/module.h>	

#include <linux/errno.h>	/* These include what we really need */
#include <linux/delay.h>
#include <linux/string.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/timer.h>
#include <linux/cdrom.h>
#include <linux/ioport.h>
#include <linux/mm.h>
#include <linux/malloc.h>

#include <linux/ucdrom.h>

#include <asm/io.h>

#define MAJOR_NR CM206_CDROM_MAJOR
#include <linux/blk.h>

#undef DEBUG
#define DEBUG 1
#define STATISTICS		/* record times and frequencies of events */
#undef AUTO_PROBE_MODULE
#define USE_INS

#define CM250
#include <linux/cm206.h>

/* This variable defines whether or not to probe for adapter base port 
   address and interrupt request. It can be overridden by the boot 
   parameter `auto'.
*/
static int auto_probe=1;	/* Yes, why not? */

static int cm206_base    = CM206_BASE;
static int cm206_irq     = CM206_IRQ; 
static int cm206_adapter = CM206_ADAPTER;

#define POLLOOP 10000
#define READ_AHEAD 1		/* defines private buffer, waste! */
#define BACK_AHEAD 1		/* defines adapter-read ahead */
#define DATA_TIMEOUT (3*HZ)	/* measured in jiffies (10 ms) */
#define UART_TIMEOUT (5*HZ/100)
#define DSB_TIMEOUT (7*HZ)	/* time for the slowest command to finish */

#define LINUX_BLOCK_SIZE 512	/* WHERE is this defined? */
#define RAW_SECTOR_SIZE 2352	/* ok, is also defined in cdrom.h */
#define ISO_SECTOR_SIZE 2048
#define BLOCKS_ISO (ISO_SECTOR_SIZE/LINUX_BLOCK_SIZE) /* 4 */
#define CD_SYNC_HEAD 16		/* CD_SYNC + CD_HEAD */

#ifdef STATISTICS		/* keep track of errors in counters */
#define stats(i) { ++cd->stats[st_ ## i]; \
		     cd->last_stat[st_ ## i] = cd->stat_counter++; \
		 }
#else
#define stats(i) (void) 0
#endif

#if DEBUG > 0			/* from lmscd.c */
#define debug(a) printk a
#else
#define debug(a) (void) 0
#endif
#if DEBUG > 1			/* from lmscd.c */
#define debug2(a) printk a
#else
#define debug2(a) (void) 0
#endif
#if DEBUG > 2			/* from lmscd.c */
#define debug3(a) printk a
#else
#define debug3(a) (void) 0
#endif

typedef unsigned char  uch;	/* 8-bits */
typedef unsigned short ush;	/* 16-bits */

struct toc_struct{		/* private copy of Table of Contents */
  uch track, fsm[3], q0;
};

struct cm206_struct {
/* cm250/cm260 i/o */
  ush  (*in) (ush port);
  void (*ins)(ush port, void *dest, ulong count);
  void (*out)(ush value, ush port);
/* cm250/cm260 port addresses */
  ush r_data_status;
  ush r_uart_receive;
  ush r_fifo_output_buffer;
  ush r_line_status;
  ush r_data_control;
  ush r_uart_transmit;
  ush r_test_clock;
  ush r_test_control;
/* cm250/cm260 data_status flags */
  ush ds_channel_mask;
  ush ds_data_ready;
  ush ds_fifo_overflow;
  ush ds_data_error;
  ush ds_crc_error;
  ush ds_sync_error;
  ush ds_fifo_empty;
  ush ds_toc_ready;
  ush ds_ram_size;
/* Define all used cm250/cm260 data control register flag combinations here */
  ush dc_normal_unmask_xmit;
  ush dc_normal;
  ush dc_reset_adapter;
  ush dc_reset_drive;

  ush intr_ds;	 /* data status read on last interrupt */
  ush intr_ls;	 /* uart line status read on last interrupt*/
  uch intr_ur;			/* uart receive buffer */
  uch dsb, cc;	 /* drive status byte and condition (error) code */
  uch fool;
  int command;			/* command to be written to the uart */
  int openfiles;
  ush sector[READ_AHEAD*RAW_SECTOR_SIZE/2]; /* buffered cd-sector */
  int sector_first, sector_last;	/* range of these sector */
  struct wait_queue * uart;	/* wait for interrupt */
  struct wait_queue * data;
  struct timer_list timer;	/* time-out */
  char timed_out;
  signed char max_sectors;
  char wait_back;		/* we're waiting for a background-read */
  char background;		/* is a read going on in the background? */
  int adapter_first;		/* if so, that's the starting sector */
  int adapter_last;
  char fifo_overflowed;
  uch disc_status[7];		/* result of get_disc_status command */
#ifdef STATISTICS
  int stats[NR_STATS];
  int last_stat[NR_STATS];	/* `time' at which stat was stat */
  int stat_counter;
#endif  
  struct toc_struct toc[101];	/* The whole table of contents + lead-out */
  uch q[10];			/* Last read q-channel info */
  uch audio_status[5];		/* last read position on pause */
  uch media_changed;		/* record if media changed */
};

#define DISC_STATUS cd->disc_status[0]
#define FIRST_TRACK cd->disc_status[1]
#define LAST_TRACK  cd->disc_status[2]
#define PAUSED      cd->audio_status[0] /* misuse this memory byte! */
#define PLAY_TO     cd->toc[0]       	/* toc[0] records end-time in play */

static struct cm206_struct * cd; /* the main memory structure */

/* define 8-bit i/o for cm250 */
ush cm250_in(ush port)
{
  return (ush) inb(port);
}
void cm250_ins(ush port, void *dest, ulong count)
{
#ifdef USE_INS
  insb(port, ((uch *)dest)+cd->fifo_overflowed, count-cd->fifo_overflowed);
#else
/* duplicates what was in earlier cm206.c */
  int i;
  uch * d;
  for (i=0, d=((uch *)dest)+cd->fifo_overflowed; i<count; i++, d++)
    *d = inb(port);
#endif
}
void cm250_out(ush value, ush port)
{
  outb((uch)value, port);
}

/* define 16-bit i/o for cm260 */
ush cm260_in(ush port)
{
  return inw(port);
}
void cm260_ins(ush port, void *dest, ulong count)
{
#ifdef USE_INS
  insw(port,((ush *)dest)+cd->fifo_overflowed,count/2-cd->fifo_overflowed);
#else
/* this routine implements insw(,,). There was a time i had the
   impression that there would be any difference in error-behaviour. */
  int i;
  ush * d;
  for (i=0, d=((ush *)dest)+cd->fifo_overflowed; i<count/2-cd->fifo_overflowed; i++, d++)
    *d = inw(port);
#endif
}
void cm260_out(ush value, ush port)
{
  outw(value, port);
}

void cd_struct_init(void)
{
  if(cm206_adapter == CM250_HOST)
  {
    /* cm250 i/o */
    cd->in  = cm250_in;
    cd->ins = cm250_ins;
    cd->out = cm250_out;
    /* cm250 port addresses */
    cd->r_data_status        = r_data_status_cm250;
    cd->r_uart_receive       = r_uart_receive_cm250;
    cd->r_fifo_output_buffer = r_fifo_output_buffer_cm250;
    cd->r_line_status        = r_line_status_cm250;
    cd->r_data_control       = r_data_control_cm250;
    cd->r_uart_transmit      = r_uart_transmit_cm250;
    cd->r_test_clock         = r_test_clock_cm250;
    cd->r_test_control       = r_test_control_cm250;
    /* cm250 data_status flags */
    cd->ds_channel_mask  = ds_channel_mask_cm250;
    cd->ds_data_ready    = ds_data_ready_cm250;
    cd->ds_fifo_overflow = ds_fifo_overflow_cm250;
    cd->ds_data_error    = ds_data_error_cm250;
    cd->ds_crc_error     = ds_crc_error_cm250;
    cd->ds_sync_error    = 0;
    cd->ds_fifo_empty    = 0;
    cd->ds_toc_ready     = 0;
    cd->ds_ram_size      = 0;
    /* Define all used cm250 data control register flag combinations here */
    cd->dc_normal_unmask_xmit =  dc_channel_cm250;
    cd->dc_normal             = (dc_mask_transmit_ready_cm250 |
                                 dc_channel_cm250);
    cd->dc_reset_adapter      = (dc_initialize_cm250 |
                                 dc_mask_transmit_ready_cm250 |
                                 dc_channel_cm250);
    cd->dc_reset_drive        = (dc_break_cm250 |
                                 dc_mask_transmit_ready_cm250 |
                                 dc_channel_cm250);
  }
  else
  {
    /* cm260 i/o */
    cd->in  = cm260_in;
    cd->ins = cm260_ins;
    cd->out = cm260_out;
    /* cm260 port addresses */
    cd->r_data_status        = r_data_status_cm260;
    cd->r_uart_receive       = r_uart_receive_cm260;
    cd->r_fifo_output_buffer = r_fifo_output_buffer_cm260;
    cd->r_line_status        = r_line_status_cm260;
    cd->r_data_control       = r_data_control_cm260;
    cd->r_uart_transmit      = r_uart_transmit_cm260;
    cd->r_test_clock         = r_test_clock_cm260;
    cd->r_test_control       = r_test_control_cm260;
    /* cm260 data_status flags */
    cd->ds_channel_mask  = ds_channel_mask_cm260;
    cd->ds_data_ready    = ds_data_ready_cm260;
    cd->ds_fifo_overflow = ds_fifo_overflow_cm260;
    cd->ds_data_error    = ds_data_error_cm260;
    cd->ds_crc_error     = ds_crc_error_cm260;
    cd->ds_sync_error    = ds_sync_error_cm260;
    cd->ds_fifo_empty    = ds_fifo_empty_cm260;
    cd->ds_toc_ready     = ds_toc_ready_cm260;
    cd->ds_ram_size      = ds_ram_size_cm260;
    /* Define all used cm260 data control register flag combinations here */
    cd->dc_normal_unmask_xmit = (dc_mask_sync_error_cm260 |
                                 dc_no_stop_on_error_cm260 |
                                 dc_channel_cm260);
    cd->dc_normal             = (dc_mask_sync_error_cm260 |
                                 dc_no_stop_on_error_cm260 |
                                 dc_mask_transmit_ready_cm260 |
                                 dc_channel_cm260);
    cd->dc_reset_adapter      = (dc_mask_sync_error_cm260 |
                                 dc_no_stop_on_error_cm260 |
                                 dc_initialize_cm260 |
                                 dc_mask_transmit_ready_cm260 |
                                 dc_channel_cm260);
    cd->dc_reset_drive        = (dc_mask_sync_error_cm260 |
                                 dc_no_stop_on_error_cm260 |
                                 dc_break_cm260 |
                                 dc_mask_transmit_ready_cm260 |
                                 dc_channel_cm260);
  }
}

/* First, we define some polling functions. These are actually
   only being used in the initialization. */

void send_command_polled(int command)
{
  int loop=POLLOOP;
  while (!((*cd->in)(cd->r_line_status) & ls_transmitter_buffer_empty) && loop>0) 
    --loop;
  (*cd->out)(command, cd->r_uart_transmit);
}

uch receive_echo_polled(void)
{
  int loop=POLLOOP;
  while (!((*cd->in)(cd->r_line_status) & ls_receive_buffer_full) && loop>0)
    --loop;
  return inb(cd->r_uart_receive);
}

uch send_receive_polled(int command)
{
  send_command_polled(command);
  return receive_echo_polled();
}

/* The interrupt handler. When the cm260 generates an interrupt, very
   much care has to be taken in reading out the registers in the right
   order; in case of a receive_buffer_full interrupt, first the
   uart_receive must be read, and then the line status again to
   de-assert the interrupt line. It took me a couple of hours to find
   this out:-( 

   The function reset_cm206 appears to cause an interrupt, because
   pulling up the INIT line clears both the uart-write-buffer /and/
   the uart-write-buffer-empty mask. We call this a `lost interrupt,'
   as there seems so reason for this to happen.
*/

static void cm206_interrupt(int sig, void *dev_id, struct pt_regs * regs) 
/* you rang? */
{
  volatile ush fool;
  cd->intr_ds = (*cd->in)(cd->r_data_status); /* resets data_ready, data_error,
					      crc_error, sync_error, toc_ready 
					      interrupts */
  cd->intr_ls = (*cd->in)(cd->r_line_status); /* resets overrun bit */
  if (cd->intr_ls & ls_attention) stats(attention);
  /* receive buffer full? */
  if (cd->intr_ls & ls_receive_buffer_full) {	
    cd->intr_ur = inb(cd->r_uart_receive); /* get order right! */
    cd->intr_ls = (*cd->in)(cd->r_line_status); /* resets rbf interrupt */
    if (!cd->background && cd->uart) wake_up_interruptible(&cd->uart);
  }
  /* data ready in fifo? */
  else if (cd->intr_ds & cd->ds_data_ready) { 
    if (cd->background) ++cd->adapter_last;
    if ((cd->wait_back || !cd->background) && cd->data) 
      wake_up_interruptible(&cd->data);
    stats(data_ready);
  }
  /* ready to issue a write command? */
  else if (cd->command) {
    if(cd->intr_ls & ls_transmitter_buffer_empty) {
       (*cd->out)(cd->dc_normal, cd->r_data_control);
       (*cd->out)(cd->command, cd->r_uart_transmit);
       cd->command=0;
       if (!cd->background) wake_up_interruptible(&cd->uart);
    }
    else
    {
       (*cd->out)(cd->dc_normal_unmask_xmit, cd->r_data_control);
    }
  }
  /* now treat errors (at least, identify them for debugging) */
  else if (cd->intr_ds & cd->ds_fifo_overflow) {
    debug(("Fifo overflow at sectors 0x%x\n", cd->sector_first));
    fool = (*cd->in)(cd->r_fifo_output_buffer);	/* de-assert the interrupt */
    cd->fifo_overflowed=1;	/* signal one word less should be read */
    stats(fifo_overflow);
  }
  else if (cd->intr_ds & cd->ds_data_error) {
    debug(("Data error at sector 0x%x\n", cd->sector_first));
    stats(data_error);
  }
  else if (cd->intr_ds & cd->ds_crc_error) {
    debug(("CRC error at sector 0x%x\n", cd->sector_first));
    stats(crc_error);
  }
  else if (cd->intr_ds & cd->ds_sync_error) {
    debug(("Sync at sector 0x%x\n", cd->sector_first));
    stats(sync_error);
  }
  else if (cd->intr_ds & cd->ds_toc_ready) {
    /* do something appropriate */
  }
  /* couldn't see why this interrupt, maybe due to init */
  else {			
    (*cd->out)(cd->dc_normal, cd->r_data_control);
    stats(lost_intr);
  }
  if (cd->background && (cd->adapter_last-cd->adapter_first == cd->max_sectors
			 || cd->fifo_overflowed))
    mark_bh(CM206_BH);	/* issue a stop read command */
  stats(interrupt);
}

/* we have put the address of the wait queue in who */
void cm206_timeout(unsigned long who)
{
  cd->timed_out = 1;
  wake_up_interruptible((struct wait_queue **) who);
}

/* This function returns 1 if a timeout occurred, 0 if an interrupt
   happened */
int sleep_or_timeout(struct wait_queue ** wait, int timeout)
{
  cd->timer.data=(unsigned long) wait;
  cd->timer.expires = jiffies + timeout;
  add_timer(&cd->timer);
  interruptible_sleep_on(wait);
  del_timer(&cd->timer);
  if (cd->timed_out) {
    cd->timed_out = 0;
    return 1;
  }
  else return 0;
}

void cm206_delay(int jiffies) 
{
  struct wait_queue * wait = NULL;
  sleep_or_timeout(&wait, jiffies);
}

void send_command(int command)
{
  if (!((*cd->in)(cd->r_line_status) & ls_transmitter_buffer_empty)) {
    cd->command = command;
    cli();			/* don't interrupt before sleep */
    (*cd->out)(cd->dc_normal_unmask_xmit, cd->r_data_control);
    /* interrupt routine sends command */
    if (sleep_or_timeout(&cd->uart, UART_TIMEOUT)) {
      debug(("Time out on write-buffer\n"));
      stats(write_timeout);
      (*cd->out)(command, cd->r_uart_transmit);
    }
  }
  else (*cd->out)(command, cd->r_uart_transmit);
}

uch receive_echo(void)
{
  if (!((*cd->in)(cd->r_line_status) & ls_receive_buffer_full) &&
      sleep_or_timeout(&cd->uart, UART_TIMEOUT)) {
    debug(("Time out on receive-buffer\n"));
    stats(receive_timeout);
    return inb(cd->r_uart_receive);
  }
  return cd->intr_ur;
}

inline uch send_receive(int command)
{
  send_command(command);
  return receive_echo();
}

uch wait_dsb(void)
{
  if (!((*cd->in)(cd->r_line_status) & ls_receive_buffer_full) &&
      sleep_or_timeout(&cd->uart, DSB_TIMEOUT)) {
    debug2(("state %02X%02X:",(*cd->in)(cd->r_data_status),(*cd->in)(cd->r_line_status)));
    debug(("Time out on Drive Status Byte\n"));
    stats(dsb_timeout);
    return inb(cd->r_uart_receive);
  }
  return cd->intr_ur;
}

int type_0_command(int command, int expect_dsb)
{
  int e;
  if (command != (e=send_receive(command))) {
    debug(("command 0x%x echoed as 0x%x\n", command, e));
    stats(echo);
    return -1;
  }
  debug2(("sent %02X:", command));
  if (expect_dsb) {
    cd->dsb = wait_dsb();	/* wait for command to finish */
    debug2(("got %02X %02X:", command, cd->dsb));
  }
  debug2(("state %02X%02X:",(*cd->in)(cd->r_data_status),(*cd->in)(cd->r_line_status)));
  return 0;
}

int type_1_command(int command, int bytes, uch * status) /* returns info */
{
  int i;
#if DEBUG > 1
  debug2(("sent %02X", command));
  for(i=0; i<bytes; i++) 
     debug2((" %02X", c_gimme));
#endif

  if (type_0_command(command,0)) return -1;
  for(i=0; i<bytes; i++) 
    status[i] = send_receive(c_gimme);

#if DEBUG > 1
  debug2((":got %02X", command));
  for(i=0; i<bytes; i++) 
     debug2((" %02X", status[i]));
  debug2((":"));
#endif
  debug2(("state %02X%02X:",(*cd->in)(cd->r_data_status),(*cd->in)(cd->r_line_status)));
  return 0;
}  

/* This function resets the adapter card. We'd better not do this too */
/* often, because it tends to generate `lost interrupts.' */
void reset_cm260(void)
{
  (*cd->out)(cd->dc_reset_adapter, cd->r_data_control);
  udelay(10);			/* 3.3 mu sec minimum */
  (*cd->out)(cd->dc_normal, cd->r_data_control);
  debug2(("card reset state %02X%02X:",(*cd->in)(cd->r_data_status),(*cd->in)(cd->r_line_status)));
}

/* fsm: frame-sec-min from linear address */
void fsm(int lba, uch * fsm) 
{
  fsm[0] = lba % 75;
  lba /= 75; lba += 2;
  fsm[1] = lba % 60; fsm[2] = lba / 60;
}

inline int fsm2lba(uch * fsm) 
{
  return fsm[0] + 75*(fsm[1]-2 + 60*fsm[2]);
}

inline int f_s_m2lba(uch f, uch s, uch m)
{
  return f + 75*(s-2 + 60*m);
}

int start_read(int start) 
{
  uch read_sector[4] = {c_read_data, };
  int i, e;

  fsm(start, &read_sector[1]);
  for (i=0; i<4; i++) 
    if (read_sector[i] != (e=send_receive(read_sector[i]))) {
      debug(("read_sector: %x echoes %x\n", read_sector[i], e));
      stats(echo);
      return -1;
    }
  debug2(("sent %02X %02X %02X %02X:", read_sector[0], read_sector[1], read_sector[2], read_sector[3]));
  debug2(("state %02X%02X:",(*cd->in)(cd->r_data_status),(*cd->in)(cd->r_line_status)));
  return 0;
}

int stop_read(void)
{
  type_0_command(c_stop,0);
  if(receive_echo() != 0xff) {
    debug(("c_stop didn't send 0xff\n"));
    stats(stop_0xff);
    return -1;
  }
  return 0;
}  

/* This function starts to read sectors in adapter memory, the
   interrupt routine should stop the read. In fact, the bottom_half
   routine takes care of this. Set a flag `background' in the cd
   struct to indicate the process. */

int read_background(int start, int reading)
{
  if (cd->background) return -1; /* can't do twice */
  (*cd->out)(cd->dc_normal, cd->r_data_control);
  if (!reading && start_read(start)) return -2;
  cd->adapter_first = cd->adapter_last = start; 
  cd->background = 1;		/* flag a read is going on */
  return 0;
}

int read_sector(int start)
{
  if (cd->background) {
    cd->background=0;
    cd->adapter_last = -1;	/* invalidate adapter memory */
    stop_read();
  }
  cd->fifo_overflowed=0;
  reset_cm260();		/* empty fifo etc. */
  if (start_read(start)) return -1;
  if (sleep_or_timeout(&cd->data, DATA_TIMEOUT)) {
    debug(("Read timed out sector 0x%x\n", start));
    stats(read_timeout);
    stop_read();
    return -3;		
  }
  (*cd->ins)(cd->r_fifo_output_buffer, (void *)cd->sector, 
		 READ_AHEAD*RAW_SECTOR_SIZE);
#if DEBUG > 2
  debug3(("read %2X %2X %2X %2X",start,cd->adapter_first,cd->adapter_last,READ_AHEAD*RAW_SECTOR_SIZE));
  {
    int i;
    for(i=0; i<16; i++) 
       debug3((" %02X", ((uch *)cd->sector)[i]));
    debug3((":"));
  }
#endif
  if (read_background(start+READ_AHEAD,1)) stats(read_background);
  cd->sector_first = start; cd->sector_last = start+READ_AHEAD;
  stats(read_restarted);
  return 0;
}

/* The function of bottom-half is to send a stop command to the drive
   This isn't easy because the routine is not `owned' by any process;
   we can't go to sleep! The variable cd->background gives the status:
   0 no read pending
   1 a read is pending
   2 c_stop waits for write_buffer_empty
   3 c_stop waits for receive_buffer_full: echo
   4 c_stop waits for receive_buffer_full: 0xff
*/

void cm206_bh(void)
{
  debug2(("bh: %d:", cd->background));
  switch (cd->background) {
  case 1:
    debug2(("statei %02X%02X:",cd->intr_ds,cd->intr_ls));
    debug2(("state %02X%02X:",(*cd->in)(cd->r_data_status),(*cd->in)(cd->r_line_status)));
    stats(bh);
    if (!(cd->intr_ls & ls_transmitter_buffer_empty)) {
      cd->command = c_stop;
      (*cd->out)(cd->dc_normal_unmask_xmit, cd->r_data_control);
      cd->background=2;
      break;			/* we'd better not time-out here! */
    }
    else (*cd->out)(c_stop, cd->r_uart_transmit);
    /* fall into case 2: */
  case 2:			
    /* the write has been satisfied by interrupt routine */
    cd->background=3;
    debug2(("state %02X%02X:",(*cd->in)(cd->r_data_status),(*cd->in)(cd->r_line_status)));
    break;
  case 3:
    debug2(("statei %02X%02X:",cd->intr_ds,cd->intr_ls));
    debug2(("state %02X%02X:",(*cd->in)(cd->r_data_status),(*cd->in)(cd->r_line_status)));
    if (cd->intr_ur != c_stop) {
      debug(("cm206_bh: c_stop echoed 0x%x\n", cd->intr_ur));
      stats(echo);
    }
    cd->background++;
    break;
  case 4:
    if (cd->intr_ur != 0xff) {
      debug(("cm206_bh: c_stop reacted with 0x%x\n", cd->intr_ur));
      stats(stop_0xff);
    }
    cd->background=0;
  }
}

/* This command clears the dsb_possible_media_change flag, so we must 
 * retain it.
 */
void get_drive_status(void)
{
  uch status[2];
  type_1_command(c_drive_status, 2, status); /* this might be done faster */
  cd->dsb=status[0];
  cd->cc=status[1];
  cd->media_changed |= 
    !!(cd->dsb & (dsb_possible_media_change | 
		  dsb_drive_not_ready | dsb_tray_not_closed));
}

void get_disc_status(void)
{
  if (type_1_command(c_disc_status, 7, cd->disc_status)) {
    debug(("get_disc_status: error\n"));
  }
}

/* The new open. The real opening strategy is defined in cdrom.c. */

static int cm206_open(kdev_t dev, int purpose) 
{
  if (!cd->openfiles) {		/* reset only first time */
    cd->background=0;
    reset_cm260();
    cd->adapter_last = -1;	/* invalidate adapter memory */
    cd->sector_last = -1;
  }
  ++cd->openfiles; MOD_INC_USE_COUNT;
  stats(open);
  return 0;
}

static void cm206_release(kdev_t dev)
{
  if (cd->openfiles==1) {
    if (cd->background) {
      cd->background=0;
      stop_read();
    }
    cd->sector_last = -1;	/* Make our internal buffer invalid */
    FIRST_TRACK = 0;		/* No valid disc status */
  }
  --cd->openfiles; MOD_DEC_USE_COUNT;
}

/* Empty buffer empties $sectors$ sectors of the adapter card buffer,
 * and then reads a sector in kernel memory.  */
void empty_buffer(int sectors) 
{
  while (sectors>=0) {
  (*cd->ins)(cd->r_fifo_output_buffer, (void *)cd->sector,
                 READ_AHEAD*RAW_SECTOR_SIZE);
#if DEBUG > 2
  {
    int i;
    debug3(("empty %02X %02X %02X %02X",sectors,cd->adapter_first,cd->adapter_last,RAW_SECTOR_SIZE-cd->fifo_overflowed));
    for(i=0; i<16; i++) 
      debug3((" %02X", ((uch *)cd->sector)[i]));
    debug3((":"));
  }
#endif
    --sectors;
    ++cd->adapter_first;	/* update the current adapter sector */
    cd->fifo_overflowed=0;	/* reset overflow bit */
    stats(sector_transferred);
  } 
  cd->sector_first=cd->adapter_first-1;
  cd->sector_last=cd->adapter_first; /* update the buffer sector */
}

/* try_adapter. This function determines if the requested sector is
   in adapter memory, or will appear there soon. Returns 0 upon
   success */
int try_adapter(int sector)
{
  if (cd->adapter_first <= sector && sector < cd->adapter_last) { 
    /* sector is in adapter memory */
    empty_buffer(sector - cd->adapter_first);
    return 0;
  }
  else if (cd->background==1 && cd->adapter_first <= sector
	   && sector < cd->adapter_first+cd->max_sectors) {
    /* a read is going on, we can wait for it */
    cd->wait_back=1;
    while (sector >= cd->adapter_last) {
      if (sleep_or_timeout(&cd->data, DATA_TIMEOUT)) {
	debug(("Timed out during background wait: %d %d %d %d\n", sector, 
	       cd->adapter_last, cd->adapter_first, cd->background));
	stats(back_read_timeout);
	cd->wait_back=0;
	return -1;
      }
    }
    cd->wait_back=0;
    empty_buffer(sector - cd->adapter_first);
    return 0;
  }
  else return -2;
}

/* This is not a very smart implementation. We could optimize for 
   consecutive block numbers. I'm not convinced this would really
   bring down the processor load. */
static void do_cm206_request(void)
{
  long int i, cd_sec_no;
  int quarter, error; 
  uch * source, * dest;
  
  while(1) {	 /* repeat until all requests have been satisfied */
    INIT_REQUEST;
    if (CURRENT == NULL || CURRENT->rq_status == RQ_INACTIVE)
      return;
    if (CURRENT->cmd != READ) {
      debug(("Non-read command %d on cdrom\n", CURRENT->cmd));
      end_request(0);
      continue;
    }
    error=0;
    for (i=0; i<CURRENT->nr_sectors; i++) {
      cd_sec_no = (CURRENT->sector+i)/BLOCKS_ISO; /* 4 times 512 bytes */
      quarter = (CURRENT->sector+i) % BLOCKS_ISO; 
      dest = CURRENT->buffer + i*LINUX_BLOCK_SIZE;
      /* is already in buffer memory? */
      if (cd->sector_first <= cd_sec_no && cd_sec_no < cd->sector_last) {
	source = ((uch *) cd->sector) + 16 + quarter*LINUX_BLOCK_SIZE 
	  + (cd_sec_no-cd->sector_first)*RAW_SECTOR_SIZE;
 	memcpy(dest, source, LINUX_BLOCK_SIZE); 
      }
      else if (!try_adapter(cd_sec_no) || !read_sector(cd_sec_no)) {
	source =  ((uch *) cd->sector)+16+quarter*LINUX_BLOCK_SIZE;
	memcpy(dest, source, LINUX_BLOCK_SIZE); 
      }
      else {
	error=1;
      }
    }
    end_request(!error);
  }
}

/* Audio support. I've tried very hard, but the cm206 drive doesn't 
   seem to have a get_toc (table-of-contents) function, while i'm
   pretty sure it must read the toc upon disc insertion. Therefore
   this function has been implemented through a binary search 
   strategy. All track starts that happen to be found are stored in
   cd->toc[], for future use. 

   I've spent a whole day on a bug that only shows under Workman---
   I don't get it. Tried everything, nothing works. If workman asks
   for track# 0xaa, it'll get the wrong time back. Any other program
   receives the correct value. I'm stymied.
*/

/* seek seeks to address lba. It does wait to arrive there. */
void seek(int lba)
{
  int i;
  uch seek_command[4]={c_seek, };
  
  fsm(lba, &seek_command[1]);
  for (i=0; i<4; i++) type_0_command(seek_command[i], 0);
  cd->dsb = wait_dsb();
}

uch bcdbin(unsigned char bcd)	/* stolen from mcd.c! */
{
  return (bcd >> 4)*10 + (bcd & 0xf);
} 

inline uch normalize_track(uch track) 
{
  if (track<1) return 1;
  if (track>LAST_TRACK) return LAST_TRACK+1;
  return track;
}

/* This function does a binary search for track start. It records all
 * tracks seen in the process. Input $track$ must be between 1 and
 * #-of-tracks+1 */
int get_toc_lba(uch track)
{
  int max=74*60*75-150, min=0;
  int i, lba, l, old_lba=0;
  uch * q = cd->q;
  uch ct;			/* current track */
  int binary=0;
  const skip = 3*60*75;

  for (i=track; i>0; i--) if (cd->toc[i].track) {
    min = fsm2lba(cd->toc[i].fsm);
    break;
  }
  lba = min + skip;		/* 3 minutes */
  do {
    seek(lba); 
    type_1_command(c_read_current_q, 10, q);
    ct = normalize_track(q[1]);
    if (!cd->toc[ct].track) {
      l = q[9]-bcdbin(q[5]) + 75*(q[8]-bcdbin(q[4])-2 + 
				  60*(q[7]-bcdbin(q[3])));
      cd->toc[ct].track=q[1];	/* lead out still 0xaa */
      fsm(l, cd->toc[ct].fsm);
      cd->toc[ct].q0 = q[0];	/* contains adr and ctrl info */
      if (ct==track) return l;
    }
    old_lba=lba;
    if (binary) {
      if (ct < track) min = lba; else max = lba;
      lba = (min+max)/2; 
    } else {
      if(ct < track) lba += skip;
      else {
	binary=1;
	max = lba; min = lba - skip;
	lba = (min+max)/2;
      }
    }
  } while (lba!=old_lba);
  return lba;
}

void update_toc_entry(uch track) 
{
  track = normalize_track(track);
  if (!cd->toc[track].track) get_toc_lba(track);
}

/* return 0 upon success */
int read_toc_header(struct cdrom_tochdr * hp)
{
  if (!FIRST_TRACK) get_disc_status();
  if (hp && DISC_STATUS & cds_all_audio) { /* all audio */
    int i;
    hp->cdth_trk0 = FIRST_TRACK;
    hp->cdth_trk1 = LAST_TRACK;
    cd->toc[1].track=1;		/* fill in first track position */
    for (i=0; i<3; i++) cd->toc[1].fsm[i] = cd->disc_status[3+i];
    update_toc_entry(LAST_TRACK+1);		/* find most entries */
    return 0;
  }
  return -1;
}  

void play_from_to_msf(struct cdrom_msf* msfp)
{
  uch play_command[] = {c_play, 
	   msfp->cdmsf_frame0, msfp->cdmsf_sec0, msfp->cdmsf_min0,
	   msfp->cdmsf_frame1, msfp->cdmsf_sec1, msfp->cdmsf_min1, 2, 2};
  int i;
  for (i=0; i<9; i++) type_0_command(play_command[i], 0);
  for (i=0; i<3; i++) 
    PLAY_TO.fsm[i] = play_command[i+4];
  PLAY_TO.track = 0;		/* say no track end */
  cd->dsb = wait_dsb();
}  

void play_from_to_track(int from, int to)
{
  uch play_command[8] = {c_play, };
  int i;

  if (from==0) {		/* continue paused play */
    for (i=0; i<3; i++) { 
      play_command[i+1] = cd->audio_status[i+2];
      play_command[i+4] = PLAY_TO.fsm[i];
    }
  } else {
    update_toc_entry(from); update_toc_entry(to+1);
    for (i=0; i<3; i++) {
      play_command[i+1] = cd->toc[from].fsm[i];
      PLAY_TO.fsm[i] = play_command[i+4] = cd->toc[to+1].fsm[i];
    }
    PLAY_TO.track = to; 
  }
  for (i=0; i<7; i++) type_0_command(play_command[i],0);
  for (i=0; i<2; i++) type_0_command(0x2, 0); /* volume */
  cd->dsb = wait_dsb();
}

int get_current_q(struct cdrom_subchnl * qp)
{
  int i;
  uch * q = cd->q;
  if (type_1_command(c_read_current_q, 10, q)) return 0;
/*  q[0] = bcdbin(q[0]); Don't think so! */
  for (i=2; i<6; i++) q[i]=bcdbin(q[i]); 
  qp->cdsc_adr = q[0] & 0xf; qp->cdsc_ctrl = q[0] >> 4;	/* from mcd.c */
  qp->cdsc_trk = q[1];  qp->cdsc_ind = q[2];
  if (qp->cdsc_format == CDROM_MSF) {
    qp->cdsc_reladdr.msf.minute = q[3];
    qp->cdsc_reladdr.msf.second = q[4];
    qp->cdsc_reladdr.msf.frame = q[5];
    qp->cdsc_absaddr.msf.minute = q[7];
    qp->cdsc_absaddr.msf.second = q[8];
    qp->cdsc_absaddr.msf.frame = q[9];
  } else {
    qp->cdsc_reladdr.lba = f_s_m2lba(q[5], q[4], q[3]);
    qp->cdsc_absaddr.lba = f_s_m2lba(q[9], q[8], q[7]);
  }
  get_drive_status();
  if (cd->dsb & dsb_play_in_progress) 
    qp->cdsc_audiostatus = CDROM_AUDIO_PLAY ;
  else if (PAUSED) 
    qp->cdsc_audiostatus = CDROM_AUDIO_PAUSED;
  else qp->cdsc_audiostatus = CDROM_AUDIO_NO_STATUS;
  return 0;
}

void invalidate_toc(void)
{
  memset(cd->toc, 0, sizeof(cd->toc));
  memset(cd->disc_status, 0, sizeof(cd->disc_status));
}

/* cdrom.c guarantees that cdte_format == CDROM_MSF */
void get_toc_entry(struct cdrom_tocentry * ep)
{
  uch track = normalize_track(ep->cdte_track);
  update_toc_entry(track);
  ep->cdte_addr.msf.frame = cd->toc[track].fsm[0];
  ep->cdte_addr.msf.second = cd->toc[track].fsm[1];
  ep->cdte_addr.msf.minute = cd->toc[track].fsm[2];
  ep->cdte_adr = cd->toc[track].q0 & 0xf; 
  ep->cdte_ctrl = cd->toc[track].q0 >> 4;
  ep->cdte_datamode=0;
}

/* Audio ioctl.  Ioctl commands connected to audio are in such an
 * idiosyncratic i/o format, that we leave these untouched. Return 0
 * upon success. Memory checking has been done by cdrom_ioctl(), the
 * calling function, as well as LBA/MSF sanitization.
*/
int cm206_audio_ioctl(kdev_t dev, unsigned int cmd, void * arg)
{
  switch (cmd) {
  case CDROMREADTOCHDR: 
    return read_toc_header((struct cdrom_tochdr *) arg);
  case CDROMREADTOCENTRY: 	
    get_toc_entry((struct cdrom_tocentry *) arg);
    return 0;
  case CDROMPLAYMSF: 
    play_from_to_msf((struct cdrom_msf *) arg);
    return 0;
  case CDROMPLAYTRKIND:		/* admittedly, not particularly beautiful */
    play_from_to_track(((struct cdrom_ti *)arg)->cdti_trk0, 
		       ((struct cdrom_ti *)arg)->cdti_trk1);
    return 0;
  case CDROMSTOP: 
    PAUSED=0;
    if (cd->dsb & dsb_play_in_progress) return type_0_command(c_stop, 1);
    else return 0;
  case CDROMPAUSE: 
    get_drive_status();
    if (cd->dsb & dsb_play_in_progress) {
      type_0_command(c_stop, 1);
      type_1_command(c_audio_status, 5, cd->audio_status);
      PAUSED=1;	/* say we're paused */
    }
    return 0;
  case CDROMRESUME:
    if (PAUSED) play_from_to_track(0,0);
    PAUSED=0;
    return 0;
  case CDROMSTART:
  case CDROMVOLCTRL:
    return 0;
  case CDROMSUBCHNL: 
    return get_current_q((struct cdrom_subchnl *)arg);
  default:
    return -EINVAL;
  }
}

/* Ioctl. These ioctls are specific to the cm206 driver. I have made
   some driver statistics accessible through ioctl calls.
 */

static int cm206_ioctl(kdev_t dev, unsigned int cmd, unsigned long arg)
{
  switch (cmd) {
#ifdef STATISTICS
  case CM206CTL_GET_STAT:
    if (arg >= NR_STATS) return -EINVAL;
    else return cd->stats[arg];
  case CM206CTL_GET_LAST_STAT:
    if (arg >= NR_STATS) return -EINVAL;
    else return cd->last_stat[arg];
#endif    
  default:
    debug(("Unknown ioctl call 0x%x\n", cmd));
    return -EINVAL;
  }
}     

int cm206_media_changed(kdev_t dev) 
{
  if (cd != NULL) {
    int r;
    get_drive_status();		/* ensure cd->media_changed OK */
    r = cd->media_changed;
    cd->media_changed = 0;	/* clear bit */
    return r;
  }
  else return -EIO;
}

/* The new generic cdrom support. Routines should be concise, most of
   the logic should be in cdrom.c */

/* returns number of times device is in use */
int cm206_open_files(kdev_t dev)	
{
  if (cd) return cd->openfiles;
  return -1;
}

/* controls tray movement */
int cm206_tray_move(kdev_t dev, int position) 
{
  if (position) {		/* 1: eject */
    type_0_command(c_open_tray,1);
    invalidate_toc();
  } 
  else type_0_command(c_close_tray, 1);	/* 0: close */
  return 0;
}

/* gives current state of the drive */
int cm206_drive_status(kdev_t dev)
{
  get_drive_status();
  if (cd->dsb & dsb_tray_not_closed) return CDS_TRAY_OPEN;
  if (!(cd->dsb & dsb_disc_present)) return CDS_NO_DISC; 
  if (cd->dsb & dsb_drive_not_ready) return CDS_DRIVE_NOT_READY;
  return CDS_DISC_OK;
}
 
/* gives current state of disc in drive */
int cm206_disc_status(kdev_t dev)
{
  uch xa;
  get_drive_status();
  if ((cd->dsb & dsb_not_useful) | !(cd->dsb & dsb_disc_present))
    return CDS_NO_DISC;
  get_disc_status();
  if (DISC_STATUS & cds_all_audio) return CDS_AUDIO;
  xa = DISC_STATUS >> 4;
  switch (xa) {
  case 0: return CDS_DATA_1;	/* can we detect CDS_DATA_2? */
  case 1: return CDS_XA_2_1;	/* untested */
  case 2: return CDS_XA_2_2;
  }
  return 0;
}

/* locks or unlocks door lock==1: lock; return 0 upon success */
int cm206_lock_door(kdev_t dev, int lock)
{
  uch command = (lock) ? c_lock_tray : c_unlock_tray;
  type_0_command(command, 1);	/* wait and get dsb */
  /* the logic calculates the success, 0 means successful */
  return lock ^ ((cd->dsb & dsb_tray_locked) != 0);
}
  
/* Although a session start should be in LBA format, we return it in 
   MSF format because it is slightly easier, and the new generic ioctl
   will take care of the necessary conversion. */
int cm206_get_last_session(kdev_t dev, struct cdrom_multisession * mssp) 
{
  if (!FIRST_TRACK) get_disc_status();
  if (mssp != NULL) {
    if (DISC_STATUS & cds_multi_session) { /* multi-session */
      mssp->addr.msf.frame = cd->disc_status[3];
      mssp->addr.msf.second = cd->disc_status[4];
      mssp->addr.msf.minute = cd->disc_status[5];
      mssp->addr_format = CDROM_MSF;
      mssp->xa_flag = 1;
    } else {
      mssp->xa_flag = 0;
    }
    return 1;
  }
  return 0;
}

int cm206_get_upc(kdev_t dev, struct cdrom_mcn * mcn)
{
  uch upc[10];
  char * ret = mcn->medium_catalog_number;
  int i;
  
  if (type_1_command(c_read_upc, 10, upc)) return -EIO;
  for (i=0; i<13; i++) {
    int w=i/2+1, r=i%2;
    if (r) ret[i] = 0x30 | (upc[w] & 0x0f);
    else ret[i] = 0x30 | ((upc[w] >> 4) & 0x0f);
  }
  ret[13] = '\0';
  return 0;
} 

int cm206_reset(kdev_t dev)
{
  stop_read();
  reset_cm260();
  (*cd->out)(cd->dc_reset_drive, cd->r_data_control);
  udelay(1000);			/* 750 musec minimum */
  (*cd->out)(cd->dc_normal, cd->r_data_control);
  cd->sector_last = -1;		/* flag no data buffered */
  cd->adapter_last = -1;    
  invalidate_toc();
  return 0;
}

static struct cdrom_device_ops cm206_dops = {
  cm206_open,			/* open */
  cm206_release,		/* release */
  cm206_open_files,		/* number of open_files */
  cm206_drive_status,		/* drive status */
  cm206_disc_status,		/* disc status */
  cm206_media_changed,		/* media changed */
  cm206_tray_move,		/* tray move */
  cm206_lock_door,		/* lock door */
  NULL,				/* select speed */
  NULL,				/* select disc */
  cm206_get_last_session,	/* get last session */
  cm206_get_upc,		/* get universal product code */
  cm206_reset,			/* hard reset */
  cm206_audio_ioctl,		/* audio ioctl */
  cm206_ioctl,			/* device-specific ioctl */
  CDC_CLOSE_TRAY | CDC_OPEN_TRAY | CDC_LOCK | CDC_MULTI_SESSION |
    CDC_MEDIA_CHANGED | CDC_MCN | CDC_PLAY_AUDIO, /* capability */
  0,				/* mask flags */
  2,				/* maximum speed */
  1,				/* number of minor devices */
  1,				/* number of discs */
  0,				/* options, ignored */
  0				/* mc_flags, ignored */
};

/* This routine gets called during init if thing go wrong, can be used
 * in cleanup_module as well. */
void cleanup(int level)
{
  switch (level) {
  case 4: 
    if (unregister_cdrom(MAJOR_NR, "cm206")) {
      printk("Can't unregister cdrom cm206\n");
      return;
    }
    if (unregister_blkdev(MAJOR_NR, "cm206")) {
      printk("Can't unregister major cm206\n");
      return;
    }
  case 3: 
    free_irq(cm206_irq, NULL);
  case 2: 
  case 1: 
    kfree(cd);
    release_region(cm206_base, 16);
  default:
  }
}

/* This function probes for the adapter card. It returns the base
   address if it has found the adapter card. One can specify a base 
   port to probe specifically, or 0 which means span all possible
   bases. 

   Linus says it is too dangerous to use writes for probing, so we
   stick with pure reads for a while. Hope that 8 possible ranges,
   check_region, 15 bits of one port and 6 of another make things
   likely enough to accept the region on the first hit...
 */
int probe_base_port(int base)
{
  int b=0x300, e=0x370;		/* this is the range of start addresses */
  volatile int fool, i;

  if (base) b=e=base;
  for (base=b; base<=e; base += 0x10) {
    if (check_region(base, 0x10)) continue;
    for (i=0; i<3; i++) 
      fool = inw(base+2); /* empty possibly uart_receive_buffer */
    if((inw(base+6) & 0xffef) != 0x0001 || /* line_status */
       (inw(base) & 0xad00) != 0) /* data status */
      continue;
    cm206_adapter = CM260_HOST;
    return(base);
  }

  for (base=b; base<=e; base += 0x10) {
    fool=0xff;
    for (i=0; i<3; i++) fool &= inb(base+4+i); /* write regs, read 0xff */
    if(fool != 0xff) continue;
    if(inb(base+3) & 0xac) continue; /* check r_line_status for patterns */
    cm206_adapter = CM250_HOST;
    return(base);
  }

  return 0;
}

#if !defined(MODULE) || defined(AUTO_PROBE_MODULE)
/* Probe for irq# nr. If nr==0, probe for all possible irq's. */
int probe_irq(int nr) {
  int irqs, irq;
  (*cd->out)(cd->dc_normal, cd->r_data_control);	/* disable irq-generation */
  sti(); 
  irqs = probe_irq_on();
  reset_cm260();		/* causes interrupt */
  udelay(100);			/* wait for it */
  irq = probe_irq_off(irqs);
  (*cd->out)(cd->dc_normal, cd->r_data_control);	/* services interrupt */
  if (nr && irq!=nr && irq>0) return 0;	/* wrong interrupt happened */
  else return irq;
}
#endif

int cm206_init(void)
{
  uch e=0;
  long int size=sizeof(struct cm206_struct);

  printk(KERN_INFO VERSION);
  cm206_base = probe_base_port(auto_probe ? 0 : cm206_base);
  if (!cm206_base) {
    printk(" can't find adapter!\n");
    return -EIO;
  }
  if(cm206_adapter == CM250_HOST)
    printk(" CM250 adapter at 0x%x", cm206_base);
  else
    printk(" CM260 adapter at 0x%x", cm206_base);
  request_region(cm206_base, 16, "cm206");
  cd = (struct cm206_struct *) kmalloc(size, GFP_KERNEL);
  if (!cd) return -EIO;
  memset(cd, 0, sizeof(*cd));	/* give'm some reasonable value */
  cd_struct_init();
  debug2(("cd=%X\n",(ulong)cd));
  debug2(("r_data_status=%02X\n",cd->r_data_status));
  debug2(("cd->in=%X\n",(ulong)cd->in));
  /* Now we have found the adaptor card, try to reset it. As we have
   * found out earlier, this process generates an interrupt as well,
   * so we might just exploit that fact for irq probing! */
#if !defined(MODULE) || defined(AUTO_PROBE_MODULE)
  cm206_irq = probe_irq(auto_probe ? 0 : cm206_irq);	
  if (cm206_irq<=0) {
    printk("can't find IRQ!\n");
    cleanup(1);
    return -EIO;
  }
  else printk(" IRQ %d found\n", cm206_irq);
#else
  cli();
  reset_cm260();
  /* Now, the problem here is that reset_cm260 can generate an
     interrupt. It seems that this can cause a kernel oops some time
     later. So we wait a while and `service' this interrupt. */
  udelay(10);
  (*cd->out)(cd->dc_normal, cd->r_data_control);
  sti();
  printk(" using IRQ %d\n", cm206_irq);
#endif
  if (send_receive_polled(c_drive_configuration) != c_drive_configuration) 
    {
      printk(" drive not there\n");
      cleanup(1);
      return -EIO;
    }
  e = send_receive_polled(c_gimme);
  debug2(("sent %02X %02X:", c_drive_configuration, c_gimme));
  debug2(("got %02X %02X:", c_drive_configuration, e));
  printk(KERN_INFO "Firmware revision %d", e & dcf_revision_code);
  if (e & dcf_transfer_rate) printk(" double");
  else printk(" single");
  printk(" speed drive");
  if (e & dcf_motorized_tray) printk(", motorized tray");
  if (request_irq(cm206_irq, cm206_interrupt, 0, "cm206", NULL)) {
    printk("\nUnable to reserve IRQ---aborted\n");
    cleanup(2);
    return -EIO;
  }
  printk(".\n");
  if (register_blkdev(MAJOR_NR, "cm206", &cdrom_fops) != 0) {
    printk("Cannot register for major %d!\n", MAJOR_NR);
    cleanup(3);
    return -EIO;
  }
  if (register_cdrom(MAJOR_NR, "cm206", &cm206_dops) != 0) {
    printk("Cannot register for cdrom %d!\n", MAJOR_NR);
    cleanup(3);
    return -EIO;
  }    
  blk_dev[MAJOR_NR].request_fn = DEVICE_REQUEST;
  read_ahead[MAJOR_NR] = 16;	/* reads ahead what? */
  init_bh(CM206_BH, cm206_bh);

  cd->sector_last = -1;		/* flag no data buffered */
  cd->adapter_last = -1;
  cd->timer.function = cm206_timeout;
  if(cm206_adapter == CM250_HOST)
  {
    cd->max_sectors = 10;
  }
  else
  {
    cd->max_sectors = ((*cd->in)(cd->r_data_status) & cd->ds_ram_size) ? 24 : 97;
  }
  printk(KERN_INFO "%d kB adapter memory available, "  
	 " %ld bytes kernel memory used.\n", cd->max_sectors*2, size);
  return 0;
}

#ifdef MODULE

static int cm206[3] = {0,0,0};	/* for compatible `insmod' parameter passing */

void parse_options(void) 
{
  int i;
  for (i=0; i<3; i++) {
    if (0x300 <= cm206[i] && cm206[i]<= 0x370 && cm206[i] % 0x10 == 0) {
      cm206_base = cm206[i];
      auto_probe=0;
    }
    else if (3 <= cm206[i] && cm206[i] <= 15) {
      cm206_irq = cm206[i];
      auto_probe=0;
    }
    else if (CM250_HOST == cm206[i] || CM260_HOST == cm206[i]) {
      cm206_adapter = cm206[i];
      auto_probe=0;
    }
  }
}

int init_module(void)
{
	parse_options();
#if !defined(AUTO_PROBE_MODULE)
	auto_probe=0;
#endif
	return cm206_init();
}

void cleanup_module(void)
{
  cleanup(4);
  printk(KERN_INFO "cm206 removed\n");
}
      
#else /* !MODULE */

/* This setup function accepts either `auto' or numbers in the range
 * 3--11 (for irq) or 0x300--0x370 (for base port) or
*  CM250_HOST or CM260_HOST to denote cm250 or cm260 host adapter. */
void cm206_setup(char *s, int *p)
{
  int i;
  if (!strcmp(s, "auto")) auto_probe=1;
  for(i=1; i<=p[0]; i++) {
    if (0x300 <= p[i] && p[i]<= 0x370 && p[i] % 0x10 == 0) {
      cm206_base = p[i];
      auto_probe = 0;
    }
    else if (3 <= p[i] && p[i] <= 15) {
      cm206_irq = p[i];
      auto_probe = 0;
    }
    else if (CM250_HOST == p[i] || CM260_HOST == p[i]) {
      cm206_adapter = p[i];
      auto_probe = 0;
    }
  }
}
#endif /* MODULE */
/*
 * Local variables:
 * compile-command: "gcc -DMODULE -D__KERNEL__ -I/usr/src/linux/include/linux -Wall -Wstrict-prototypes -O2 -m486 -c cm206.c -o cm206.o"
 * End:
 */
