
/*
 *    Copyright (C) 1996  Burkhard Kohl
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * 	$Id: hexbin.h,v 0.4 1996/12/07 19:39:56 buk Exp buk $
 *
 *	$Log: hexbin.h,v $
 *	Revision 0.4  1996/12/07 19:39:56  buk
 *	Rename crc_ to chk_ - it was not a crc.
 *	Changed readHexFile to accept an istream.
 *
 *	Revision 0.3  1996/12/04 18:53:03  buk
 *	Added ostream& operator<<(...) for types hexByte, hexWord, hexRec
 *	and changed writeHexFile() to support writing to an ostream.
 *
 *	Revision 0.2  1996/12/02 19:45:47  buk
 *	Checked usage of const.
 *	Some adaptions to g++.
 *
 *	Revision 0.1  1996/11/30 16:06:18  buk
 *	Initial Revision.
 *
 */

#ifndef HEXBIN_H
#define HEXBIN_H

// Classes for generation and operation on Files containing hex records
// according to Intels HEX format description.

#include "types.h"
#include "core.h"
#include <ctype.h>
#include <String.h>
#include <stdio.h>
#include <fstream.h>

// Up to 16 net bytes per record.
// Real hexrecs might be longer.
enum { MAXHEXRECSIZE = 44 } ; //maxHexRecSize_;
enum { MAXBINRECSIZE = 16 } ; //maxBinRecSize_;

// Assuming a minimum of 1 net byte.
enum { MINHEXRECSIZE = 11 } ; //minHexRecSize_;
typedef ushort hexRecType;

// Holds bin value and hex representation of 1 byte.
class hexByte {
	char hexByte1_, hexByte2_;
	bool valid_;
	byte binvalue_;

inline	byte toBin_() const { return (hexToBin(hexByte1_)* 0x10 + hexToBin(hexByte2_)); }
	byte hexToBin(const char) const;

protected:
inline  char digit2hex(const byte akt) const { return akt < 0xA ? akt + '0' : akt + 0x37; }

public:
	hexByte();
	hexByte(const byte);
	hexByte(const char, const char);
	hexByte(const char *);

public:
inline bool valid()    const { return valid_; }
inline byte binvalue() const { return binvalue_; }
friend ostream& operator<<(ostream&, const hexByte&);

};

// Holds bin value and hex representation of one word.
class hexWord {
	hexByte hexWord1_, hexWord2_;
	word    binvalue_;
	bool 	valid_;

inline	word toBin_() const { return ((hexWord1_.binvalue() *0x100) + hexWord2_.binvalue()); }

public:
	hexWord();
	hexWord(const word);
	hexWord(const byte, const byte);
	hexWord(const char, const char, const char, const char);
	hexWord(const char*);
	hexWord(const hexByte);
	hexWord(const hexByte, const hexByte);

inline  const bool valid()    const { return valid_; }
inline  const word binvalue() const { return binvalue_; }
friend ostream& operator<<(ostream&, const hexWord&);

};

// Holds complete Intel hex record representation
// incl. type, offset, length and chk.
class hexRec {
	bool valid_;

	hexByte size_, type_, chk_;
	hexWord offset_;

protected:
	hexByte hexRec_[MAXHEXRECSIZE];

	// Compute chk and compare with chk field value.
	void update();
	void update(const off_t, const off_t, const char *);
	byte calcChk()	   const;
inline 	bool checkChk() const { return (bool)(calcChk() == chk_.binvalue()); }
//	void putByte(const off_t, const byte);
inline  byte getByte(const off_t byteOffset) const { return hexRec_[byteOffset].binvalue(); }

public:
	hexRec();
	hexRec(byte *);
	hexRec(const String& aktString);

inline  const bool valid() const { return valid_; }
inline  hexRecType type()  const { return type_.binvalue(); }
inline  off_t offset()	   const { return offset_.binvalue(); }
inline  off_t size()	   const { return size_.binvalue(); }
	// Fill record completely with new values (reusage).
	void update(const String&);
friend ostream& operator<<(ostream&, const hexRec&);

};


// Allows reading hex records from an open input stream.
// Clients have to handle opening and closing themselves.
// Does not check for valid stream pointers.
// Any error (EOF reached, invalid records i.e) will result
// in returnation of EOF - no provisons for further error handling.
// (This still allows for sufficient external error handling via
// valid-flag, line number, type-, length- and offset values.

class hexFile : protected hexRec {

protected:
	hexFile();
	hexFile(FILE *const);

	FILE  *const fPtr;
	char  aktLine[MAXHEXRECSIZE+2];

	off_t lastRecOffset_;
	bool  coreFilled_;

static	const ushort maxLineLen;

// The following line of members is intended to be used by derived classes.
	bool  EOF_;
	off_t recOffset_, aktOffset_, aktLen_, aktLineNum_;

public:
inline	bool eof()   const { return EOF_; }

};


// Allows to read hex records gathered from a stream.
class readHexFile : protected hexFile {

protected:
	core *const pCore;
	istream& in;

	void acceptFile();
	void acceptLine();
	void getRecord();

public:
//	readHexFile();
	readHexFile(istream&, core *const);

inline  bool valid() const { return hexFile::valid(); }
inline  int  line()  const { return hexFile::aktLineNum_; }
	int getByte();

};


// Allows to write hex records gathered from a stream of binary values.
class writeHexFile : protected hexFile {

protected:
	const core *const pCore;
	ostream& out;
	void writeLine();

public:
//	writeHexFile();
	writeHexFile(ostream&, const core* const);
};

#endif // HEXBIN_H
